"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fingerprint = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const ignore_1 = require("./ignore");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The set of exclude patterns, if defined in `options.exclude`
 * 3. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const ignoreMode = options.ignoreMode || options_1.IgnoreMode.GLOB;
    if (ignoreMode != options_1.IgnoreMode.GLOB) {
        _hashField(hash, 'options.ignoreMode', ignoreMode);
    }
    const ignoreStrategy = ignore_1.IgnoreStrategy.fromCopyOptions(options, fileOrDirectory);
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (!isRootDir && ignoreStrategy.ignores(symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, _contentFingerprint(realPath, stat));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function _contentFingerprint(file, stat) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // eslint-disable-next-line no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    try {
        let read = 0;
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            hash.update(buffer.slice(0, read));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${stat.size}:${hash.digest('hex')}`;
}
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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