"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretValue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK.
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 *
 * @stability stable
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs.
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     *
     * @stability stable
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     *
     * @param secretId The ID or ARN of the secret.
     * @param options Options.
     * @stability stable
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        if (options.versionStage && options.versionId) {
            throw new Error(`verionStage: '${options.versionStage}' and versionId: '${options.versionId}' were both provided but only one is allowed`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager Parameter Store.
     * @param version An integer that specifies the version of the parameter to use.
     * @stability stable
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     * @stability stable
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     * @stability stable
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
_a = JSII_RTTI_SYMBOL_1;
SecretValue[_a] = { fqn: "@aws-cdk/core.SecretValue", version: "1.119.0" };
//# sourceMappingURL=data:application/json;base64,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