"""
(c)  Copyright  [2018-2023]  OpenText  or one of its
affiliates.  Licensed  under  the   Apache  License,
Version 2.0 (the  "License"); You  may  not use this
file except in compliance with the License.

You may obtain a copy of the License at:
http://www.apache.org/licenses/LICENSE-2.0

Unless  required  by applicable  law or  agreed to in
writing, software  distributed  under the  License is
distributed on an  "AS IS" BASIS,  WITHOUT WARRANTIES
OR CONDITIONS OF ANY KIND, either express or implied.
See the  License for the specific  language governing
permissions and limitations under the License.
"""

# Pytest
import pytest

# VerticaPy
from verticapy import drop, set_option
from verticapy.datasets import load_cities, load_world
from verticapy.geo import *

set_option("print_info", False)


@pytest.fixture(scope="module")
def cities_vd():
    cities = load_cities()
    yield cities
    drop(name="public.cities")


@pytest.fixture(scope="module")
def world_vd():
    world = load_world()
    yield world
    drop(name="public.world")


class TestGeo:
    def test_index_create_describe_rename_intersect(self, world_vd, cities_vd):
        world_copy = world_vd.copy()
        world_copy["id"] = "ROW_NUMBER() OVER (ORDER BY pop_est)"
        result = create_index(world_copy, "id", "geometry", "world_polygons", True)
        assert result["polygons"][0] == 177
        assert result["min_x"][0] == pytest.approx(-180.0)
        assert result["min_y"][0] == pytest.approx(-90.0)
        assert result["max_x"][0] == pytest.approx(180.0)
        assert result["max_y"][0] == pytest.approx(83.64513)
        rename_index(
            "world_polygons",
            "world_polygons_rename",
            True,
        )
        result2 = describe_index("world_polygons_rename", True)
        assert result2.shape() == (177, 3)
        cities_copy = cities_vd.copy()
        cities_copy["id"] = "ROW_NUMBER() OVER (ORDER BY city)"
        result3 = intersect(cities_copy, "world_polygons_rename", "id", "geometry")
        assert result3.shape() == (172, 2)
        cities_copy["x"] = "ST_X(geometry)"
        cities_copy["y"] = "ST_Y(geometry)"
        result4 = intersect(cities_copy, "world_polygons_rename", "id", x="x", y="y")
        assert result4.shape() == (172, 2)
        drop("world_polygons_rename", method="geo")

    def test_coordinate_converter(self, cities_vd):
        result = cities_vd.copy()
        result["longitude"] = "ST_X(geometry)"
        result["latitude"] = "ST_Y(geometry)"
        avg1 = result["longitude"].avg()
        avg2 = result["latitude"].avg()
        result = coordinate_converter(result, "longitude", "latitude")
        assert result["longitude"].avg() == pytest.approx(2385.59407399394)
        result = coordinate_converter(result, "longitude", "latitude", reverse=True)
        assert result["longitude"].avg() == pytest.approx(avg1)
        assert result["latitude"].avg() == pytest.approx(avg2)

    def test_split_polygon_n(self):
        p = "POLYGON ((121.334030916 31.5081948415, 121.334030917 31.5079167872, 121.334297575 31.5079213447, 121.334297576 31.5073515614, 121.334587026 31.507351561, 121.334587025 31.5076410113, 121.335136298 31.5076410118, 121.335136298 31.5070803438, 121.335418911 31.5070849027, 121.33541891 31.5068045687, 121.335136297 31.5068114057, 121.335136298 31.5065310726, 121.334862801 31.5065310727, 121.334862801 31.5062507387, 121.334587025 31.5062507383, 121.334587026 31.5059704052, 121.334304413 31.5059726841, 121.334304413 31.5056946296, 121.333748303 31.5057014674, 121.333748304 31.5051407995, 121.333472529 31.5051407995, 121.333472529 31.5045801324, 121.333189916 31.5045824116, 121.333189916 31.5040285814, 121.332640644 31.5040308607, 121.332640644 31.5034701936, 121.33236031 31.5034724719, 121.33236031 31.5031944175, 121.331801922 31.5031944176, 121.331801922 31.5029163632, 121.331528425 31.5029209218, 121.331528425 31.5026405878, 121.330970038 31.502640588, 121.330970037 31.5023602541, 121.330696541 31.5023602545, 121.330696541 31.500694206, 121.330970037 31.5006942063, 121.330970037 31.500416151, 121.331801922 31.50042071, 121.331801922 31.5001403759, 121.333748304 31.500140376, 121.333748303 31.4998600427, 121.335692407 31.4998600419, 121.335692407 31.4995797085, 121.335975019 31.4995842666, 121.335975019 31.4993062127, 121.335692407 31.4993084915, 121.335692406 31.4990304366, 121.335984136 31.4990304369, 121.335984136 31.4987501034, 121.336250794 31.4987501035, 121.336250795 31.4984697691, 121.338753285 31.4984720483, 121.338753286 31.4981939941, 121.339302557 31.4981939944, 121.339302557 31.4979182189, 121.339858667 31.4979182189, 121.339858667 31.4981939945, 121.340141279 31.4981939941, 121.34014128 31.4976378847, 121.340690551 31.497640164, 121.340697389 31.4970794965, 121.342917267 31.4970840549, 121.342917266 31.4968060004, 121.34401809 31.4968105592, 121.344018089 31.4965302252, 121.344856811 31.4965302251, 121.344856811 31.4962498919, 121.345909772 31.4962498911, 121.345969029 31.4959695579, 121.346800913 31.4959718372, 121.346800914 31.4954157277, 121.347359302 31.4954157278, 121.347359302 31.495137673, 121.347641915 31.4951399526, 121.347641914 31.4948596191, 121.348198024 31.4948618981, 121.348198024 31.4945838432, 121.349303405 31.494583843, 121.349303405 31.494027734, 121.351805896 31.494030013, 121.351805896 31.4937496792, 121.352355167 31.4937496797, 121.352355167 31.4934716254, 121.353469665 31.4934739038, 121.353469664 31.4931935708, 121.354025774 31.4931935708, 121.354025774 31.4929155155, 121.354857659 31.4929200746, 121.354857658 31.4923708025, 121.355140271 31.4923708029, 121.355140271 31.4918101356, 121.355422884 31.4918101353, 121.355422884 31.4915298012, 121.355698659 31.4915298017, 121.355698659 31.4912517471, 121.355972156 31.4912517465, 121.355972156 31.4909714133, 121.356247932 31.4909714135, 121.356247931 31.4906933588, 121.356528265 31.4906933583, 121.356528265 31.4904175832, 121.356797203 31.4904198623, 121.356797203 31.4901418076, 121.357086653 31.4901418083, 121.357086653 31.4893122023, 121.357360149 31.4893122023, 121.357360149 31.4890318688, 121.358192033 31.4890318689, 121.358192034 31.4893122024, 121.359580028 31.4893053654, 121.359580028 31.4895834194, 121.360138416 31.4895811402, 121.360138416 31.4898614738, 121.3609703 31.4898614741, 121.360970301 31.4901418077, 121.361252913 31.4901418079, 121.361252913 31.490419862, 121.361809022 31.4904175834, 121.361809022 31.490971413, 121.362351457 31.4909714135, 121.362351457 31.4912517471, 121.362533788 31.4912517467, 121.362640906 31.4912517471, 121.362640907 31.4915298012, 121.363197016 31.4915275222, 121.363197016 31.4920836314, 121.363746288 31.4920813527, 121.363746288 31.4923708029, 121.3640289 31.4923708026, 121.364028901 31.492920074, 121.364304676 31.4929155159, 121.364304675 31.4931935709, 121.364585009 31.493193571, 121.364585009 31.4934739038, 121.365134281 31.4934739044, 121.365134281 31.4931935706, 121.365416894 31.4931935708, 121.365416894 31.4929155157, 121.365692669 31.4929200746, 121.365692669 31.4926397408, 121.366248778 31.4926397408, 121.366248778 31.4923708029, 121.365966165 31.4923708028, 121.365966165 31.4920813524, 121.364860785 31.4920836316, 121.364860785 31.4912494679, 121.36458501 31.4912517469, 121.364585009 31.4909714131, 121.364304676 31.4909714132, 121.364304676 31.490693359, 121.3640289 31.490693359, 121.3640289 31.4904175835, 121.363746287 31.4904198626, 121.363746288 31.4901418075, 121.363472791 31.4901418075, 121.363472791 31.4898614737, 121.362914403 31.4898614743, 121.362914403 31.4895834192, 121.362640907 31.4895834194, 121.362640906 31.4893053651, 121.362351456 31.4893122022, 121.362351456 31.4890318685, 121.362075681 31.4890318689, 121.362075682 31.4884712014, 121.36152641 31.4884712019, 121.361526409 31.4881954263, 121.361252913 31.4881885885, 121.361252913 31.487639317, 121.36152641 31.487639317, 121.36152641 31.4873589833, 121.362082519 31.4873612622, 121.362082518 31.487083208, 121.362358294 31.4870832074, 121.362358293 31.4868051531, 121.362640907 31.4868051536, 121.362640906 31.486529378, 121.362923519 31.4865293779, 121.36292352 31.486249044, 121.363197016 31.4862490439, 121.363197016 31.4856906556, 121.363472791 31.4856952143, 121.363472792 31.4854171599, 121.363746287 31.4854194391, 121.363746288 31.4851391051, 121.365692669 31.4851391055, 121.365692669 31.4848587713, 121.366248779 31.4848610509, 121.366248778 31.4845829962, 121.366807166 31.4845829957, 121.366807167 31.4843072206, 121.367080663 31.4843094998, 121.367080662 31.4840291664, 121.367363275 31.4840291663, 121.367363276 31.4834707782, 121.367636772 31.483473057, 121.367636772 31.4831950023, 121.368468657 31.4831950021, 121.368468657 31.4829169481, 121.369027045 31.4829192272, 121.369027045 31.4826411723, 121.369307379 31.4826411721, 121.369307378 31.4823608385, 121.369583153 31.4823608383, 121.369583154 31.4820805047, 121.370139263 31.4820805054, 121.370139263 31.482360839, 121.370408201 31.4823608383, 121.370408201 31.482641172, 121.370697651 31.4826411721, 121.370697651 31.482919227, 121.370971148 31.4829169477, 121.370971148 31.4834730568, 121.371246923 31.4834707779, 121.371246922 31.4840291657, 121.372917529 31.4840291658, 121.37291753 31.4843094996, 121.373749414 31.4843094995, 121.373749414 31.4840291657, 121.374312361 31.4840291659, 121.374312361 31.4834707778, 121.374032026 31.4834730567, 121.374032027 31.4823608387, 121.374581298 31.4823608388, 121.374581298 31.4826411727, 121.374861632 31.4826388934, 121.374861632 31.483195002, 121.375137408 31.4831950025, 121.375137408 31.4834730568, 121.37542002 31.4834707782, 121.37542002 31.4837488326, 121.375518023 31.4837488326, 121.375969292 31.4837488323, 121.375969292 31.4834890109, 121.376525402 31.4834707781, 121.376525401 31.4837488325, 121.377083789 31.4837488321, 121.37708379 31.4843095001, 121.377357285 31.4843072207, 121.377357286 31.4845829956, 121.377639899 31.4845829962, 121.377639898 31.4848610507, 121.377906557 31.484858771, 121.377906557 31.485139105, 121.378196008 31.4851391054, 121.378196007 31.4848587714, 121.379859777 31.4848610509, 121.379859777 31.4859709895, 121.379303668 31.4859687104, 121.379303667 31.4862490445, 121.379027892 31.4862490438, 121.379027892 31.4865293779, 121.378471783 31.4865293774, 121.378471783 31.4868051531, 121.378196008 31.4868051533, 121.378196008 31.4865293776, 121.377083789 31.4865293778, 121.37708379 31.4862490439, 121.375969292 31.4862490444, 121.375969292 31.4865293782, 121.375137408 31.4865293779, 121.375137407 31.4868051534, 121.374305523 31.486805153, 121.374305523 31.4870832081, 121.373473639 31.4870832077, 121.373473638 31.4873612626, 121.374032027 31.4873589829, 121.374032027 31.4876393166, 121.375137407 31.4876393169, 121.375137408 31.4873589831, 121.37542002 31.4873589834, 121.37542002 31.4876393172, 121.375686679 31.4876393165, 121.37568668 31.4879196503, 121.375969292 31.4879196503, 121.375969292 31.4876393165, 121.377083789 31.4876393173, 121.377083789 31.4879196503, 121.377357286 31.4879173711, 121.377357285 31.4881931467, 121.377639899 31.4881931467, 121.377639898 31.4879173711, 121.378196007 31.4879196504, 121.378196008 31.4876393165, 121.378745279 31.4876393165, 121.378745279 31.4873589834, 121.379303667 31.4873612624, 121.379303667 31.4870832079, 121.379586281 31.487083208, 121.37958628 31.4868051535, 121.380142389 31.4868051528, 121.380142389 31.4865293779, 121.380691661 31.4865293775, 121.380691661 31.4862490443, 121.38124777 31.4862490441, 121.38124777 31.4859687108, 121.381530383 31.4859709897, 121.381530383 31.4856952137, 121.381806158 31.4856952144, 121.381806158 31.4854171597, 121.382079654 31.4854194385, 121.382079655 31.4851391051, 121.382638043 31.485139105, 121.382638042 31.4848587716, 121.384582146 31.4848610509, 121.384582146 31.484582996, 121.384864759 31.484582996, 121.384864759 31.4840268871, 121.385140534 31.4840291663, 121.385140534 31.4837488327, 121.385970139 31.4837488325, 121.385970139 31.4834707775, 121.386528528 31.4834730569, 121.386528527 31.4831950027, 121.386802024 31.4831950025, 121.386802023 31.4829169474, 121.387643025 31.4829192268, 121.387643024 31.4826411726, 121.388192296 31.482641172, 121.388192296 31.4818092877, 121.388465792 31.4818092875, 121.388465793 31.4809705662, 121.388748406 31.4809728451, 121.388748406 31.4806947907, 121.389304514 31.4806947908, 121.389304514 31.4809728453, 121.38958029 31.4809705664, 121.38958029 31.481809288, 121.3901364 31.48180473, 121.3901364 31.4820827844, 121.390419012 31.482080505, 121.390419012 31.4826411722, 121.39068567 31.4826411723, 121.390685671 31.4829192268, 121.390975121 31.4829169474, 121.390975122 31.4834730572, 121.391250896 31.4834707776, 121.391250896 31.483748832, 121.39153351 31.4837488326, 121.39153351 31.4840291657, 121.392082781 31.4840291657, 121.392082781 31.4843094998, 121.392365394 31.4843072207, 121.392365394 31.484861051, 121.392082781 31.4848587717, 121.392082781 31.4851368261, 121.391533509 31.4851391047, 121.391533509 31.4854194387, 121.390975121 31.4854171592, 121.390975121 31.4856952145, 121.390685671 31.4856906558, 121.39068567 31.486249044, 121.390975121 31.4862490444, 121.390975121 31.486805153, 121.392082781 31.4868051534, 121.392082781 31.4865293777, 121.392356277 31.4865293776, 121.392356277 31.4862490445, 121.39263889 31.4862490446, 121.39263889 31.4859687105, 121.393470775 31.4859687106, 121.393470774 31.4862490437, 121.393753387 31.486249044, 121.393753387 31.486805153, 121.394026884 31.4868051529, 121.394026884 31.4870832074, 121.394302659 31.4870832082, 121.394302659 31.4873612627, 121.394576156 31.4873589837, 121.394576156 31.4876393168, 121.394858768 31.4876393166, 121.394858768 31.4879196507, 121.395141381 31.4879173711, 121.395141381 31.4884712015, 121.395417156 31.4884712018, 121.395417156 31.489031869, 121.395690653 31.4890318691, 121.395690652 31.4893122023, 121.395973265 31.4893122024, 121.395973266 31.4898614744, 121.396798313 31.4898614743, 121.396798313 31.4882022634, 121.396529375 31.4881885885, 121.396529375 31.4879173718, 121.396246762 31.4879196508, 121.396246762 31.4876393166, 121.395690653 31.4876393169, 121.395690653 31.4873589837, 121.395417157 31.4873612624, 121.395417157 31.4868051533, 121.395141381 31.486805153, 121.395141381 31.486529378, 121.394858768 31.4865293781, 121.394858768 31.4862490439, 121.394576155 31.4862490443, 121.394576155 31.4854194389, 121.394302659 31.4854194384, 121.394302659 31.4845807164, 121.394026884 31.4845829962, 121.394026883 31.4843072203, 121.393753387 31.4843094993, 121.393753387 31.484029166, 121.393470775 31.4840291659, 121.393470775 31.4837488326, 121.392905549 31.4837488322, 121.39290555 31.4834707777, 121.392638891 31.4834730572, 121.39263889 31.4829169481, 121.392082781 31.4829192273, 121.392082781 31.4826411728, 121.391807005 31.4826411728, 121.391807006 31.4820805046, 121.391524393 31.482085063, 121.391524393 31.4815266753, 121.391250897 31.4815312332, 121.391250896 31.4809705658, 121.390975122 31.4809728447, 121.390975122 31.4804167356, 121.390685671 31.4804304113, 121.390685671 31.4801409606, 121.390419012 31.4801409606, 121.390419012 31.4798606265, 121.389862903 31.4798606267, 121.389862902 31.4795802935, 121.38958029 31.4795848514, 121.38958029 31.4790287424, 121.389862903 31.4790310214, 121.389862903 31.4787506882, 121.390419013 31.4787506878, 121.390419012 31.4782014165, 121.390685671 31.4782014164, 121.390685671 31.4776407489, 121.390975121 31.4776407487, 121.390975121 31.4770800811, 121.391250896 31.4770823603, 121.391250896 31.4768043056, 121.391524393 31.4768043057, 121.391524393 31.4765285306, 121.391250897 31.47653081, 121.391250897 31.4762504757, 121.391533509 31.4762504758, 121.391524393 31.475972421, 121.391807005 31.4759724214, 121.391807006 31.4756943666, 121.392082781 31.475694367, 121.392082781 31.4754163122, 121.39263889 31.4754208708, 121.39263889 31.4751405373, 121.392905549 31.4751405369, 121.392905549 31.4754208705, 121.393470775 31.4754163124, 121.393470774 31.4756943673, 121.393194999 31.4756943672, 121.393195 31.4770823609, 121.393470775 31.4770800816, 121.393470775 31.4773718103, 121.393753387 31.477371811, 121.393753388 31.4776407484, 121.394026884 31.4776407486, 121.394026884 31.4782014166, 121.394302659 31.478194579, 121.39430266 31.4795848513, 121.394576156 31.4795802931, 121.394576156 31.4798606266, 121.394858768 31.4798606267, 121.394858768 31.4801409602, 121.395141381 31.4801409607, 121.395141381 31.4804304107, 121.395417156 31.4804167358, 121.395417156 31.4806947906, 121.395690653 31.4806947901, 121.395690653 31.480972845, 121.395973265 31.4809705659, 121.395973266 31.4812508994, 121.396246762 31.4812508996, 121.396246762 31.481809288, 121.396529374 31.4818047299, 121.396529374 31.4820827838, 121.396798313 31.4820805049, 121.396798313 31.4823608384, 121.397087763 31.4823608391, 121.397087763 31.4831950027, 121.397361259 31.4831950025, 121.397361259 31.4834730573, 121.397637034 31.4834707775, 121.397637034 31.4837488323, 121.397919647 31.4837488323, 121.397919648 31.4843094993, 121.398193144 31.4843072205, 121.398193143 31.4845829957, 121.398475757 31.4845829963, 121.398475757 31.4848610503, 121.398749253 31.4848587712, 121.398749253 31.4851391047, 121.399025029 31.4851368256, 121.399025029 31.485695214, 121.399307641 31.4856906564, 121.399307641 31.4862490438, 121.399581138 31.4862490444, 121.399581138 31.487083208, 121.400139526 31.4870832075, 121.400139525 31.4873612624, 121.40097141 31.4873612625, 121.40097141 31.4870832077, 121.401251744 31.4870832081, 121.401251744 31.4868051531, 121.401527519 31.4868051528, 121.40152752 31.4865293775, 121.402076791 31.4865293781, 121.402076791 31.4862490445, 121.402359404 31.4862490438, 121.402359404 31.4865293775, 121.402642017 31.486529378, 121.402642016 31.4862490443, 121.402915512 31.4862490441, 121.402915513 31.4856906562, 121.402642016 31.4856952138, 121.402642016 31.4854171596, 121.40207679 31.4854194386, 121.402076791 31.4851391048, 121.401251744 31.4851391047, 121.401251744 31.4854194384, 121.40097141 31.4854171594, 121.40097141 31.4856952144, 121.400413022 31.4856952142, 121.400413022 31.4854171593, 121.400139526 31.4854194392, 121.400139526 31.484858771, 121.39986375 31.4848610508, 121.39986375 31.4845829958, 121.399581137 31.4845829956, 121.399581138 31.4840268867, 121.399307641 31.4840291658, 121.399307641 31.4837488322, 121.399581137 31.4837488326, 121.399581137 31.4834707777, 121.400139526 31.4834730572, 121.400139526 31.4831950022, 121.400413022 31.4831950026, 121.400413022 31.4826388934, 121.400139526 31.4826411726, 121.400139526 31.4823608389, 121.399581138 31.4823608391, 121.399581137 31.4826411728, 121.399307641 31.4826411724, 121.399307641 31.4829192274, 121.399025028 31.4829169482, 121.399025028 31.4831950023, 121.398475757 31.4831950025, 121.398475757 31.4829169475, 121.398193144 31.4829192266, 121.398193144 31.4820805054, 121.398475756 31.4820827845, 121.398475757 31.4818047296, 121.399018191 31.4818092877, 121.399018191 31.4815312336, 121.399307641 31.4815312329, 121.399307641 31.4809705664, 121.398749253 31.4809728454, 121.398749252 31.4806947904, 121.397919648 31.4806902319, 121.397919647 31.4812509001, 121.397361259 31.4812508998, 121.39736126 31.4809705662, 121.397087763 31.4809728452, 121.397087763 31.4804167363, 121.396529375 31.4804304109, 121.396529375 31.4798606271, 121.396246762 31.479860627, 121.396246762 31.4795802935, 121.395973265 31.4795848516, 121.395973266 31.4790287427, 121.395690653 31.4790310219, 121.395690653 31.4787506875, 121.395417156 31.478750688, 121.395417157 31.4782014161, 121.395141381 31.4782014164, 121.395141381 31.477080081, 121.395690652 31.4770823605, 121.395690653 31.4768043064, 121.395973265 31.4768043057, 121.395973265 31.4765285303, 121.396246762 31.4765308098, 121.396246762 31.4759701424, 121.395973266 31.4759724212, 121.395973265 31.4751405374, 121.395690653 31.4751405373, 121.395690652 31.4743109312, 121.395973265 31.4743109313, 121.395973265 31.4737502646, 121.396246761 31.4737502643, 121.396246762 31.4734699305, 121.397087763 31.4734722101, 121.397087763 31.4729183801, 121.397361259 31.4729183792, 121.397361259 31.4723599918, 121.398482594 31.4723599916, 121.398482594 31.4726403248, 121.398749253 31.4726403256, 121.398749253 31.47291838, 121.399307641 31.4729183795, 121.399307641 31.4731941553, 121.399581138 31.4731941551, 121.399581138 31.4734722096, 121.399856913 31.4734699307, 121.399856913 31.4740305979, 121.400139526 31.4740283191, 121.400139526 31.4745821484, 121.400406184 31.4745798699, 121.400406184 31.474860203, 121.399581137 31.4748602031, 121.399581138 31.47457987, 121.399018191 31.4745798694, 121.399018191 31.4751405373, 121.399307642 31.4751405371, 121.399307641 31.4765308098, 121.399025028 31.4765285301, 121.399025028 31.4768043058, 121.39986375 31.4768043056, 121.39986375 31.4765285308, 121.401527519 31.4765285308, 121.401527519 31.4768043056, 121.401801016 31.4768043063, 121.401801015 31.4770823608, 121.402359404 31.4770800815, 121.402359404 31.4773718108, 121.402642017 31.4773718108, 121.402642017 31.4776407487, 121.402915513 31.4776407488, 121.402915513 31.477921082, 121.403198125 31.4779165243, 121.403198126 31.4784726333, 121.403747398 31.4784703543, 121.403747398 31.4790310216, 121.40403001 31.4790310213, 121.40403001 31.4798606265, 121.404303507 31.4798606266, 121.404303507 31.4806947909, 121.40458612 31.4806947904, 121.404586119 31.4809728449, 121.405693779 31.480972845, 121.405693779 31.4806947904, 121.406249889 31.4806947909, 121.406249889 31.4801386817, 121.406516548 31.4801409608, 121.406516547 31.4798606265, 121.406249888 31.4798606271, 121.406249889 31.4793113548, 121.405983229 31.4793113548, 121.405983229 31.4790310215, 121.405693779 31.4790310212, 121.405693779 31.4787506879, 121.405418003 31.4787506878, 121.405418004 31.4784703545, 121.405135391 31.4784726329, 121.405135391 31.4781945783, 121.404861895 31.4782014163, 121.404861894 31.4776407487, 121.404586119 31.4776407487, 121.404586119 31.4773718107, 121.403747397 31.477371811, 121.403747398 31.4770800816, 121.403473901 31.4770823601, 121.403473901 31.4765285306, 121.403198126 31.4765308099, 121.403191289 31.4762504762, 121.402915512 31.4762504762, 121.402915513 31.4756898088, 121.402642017 31.475694367, 121.402642017 31.4754163124, 121.402359404 31.4754208709, 121.402359404 31.4751405367, 121.402076791 31.4751405372, 121.402076791 31.4748602031, 121.401801016 31.4748602035, 121.401801016 31.4745798693, 121.401527519 31.4745821484, 121.401527519 31.4743063734, 121.401251744 31.4743109319, 121.401251744 31.4737502645, 121.40097141 31.473750264, 121.400971411 31.47291838, 121.401251744 31.4729183795, 121.401251744 31.4718084407, 121.401520682 31.4718084406, 121.401520682 31.4712409361, 121.403747398 31.4712409358, 121.403747398 31.4715303858, 121.404586119 31.4715303862, 121.404586119 31.471808441, 121.404861895 31.4718084407, 121.404861895 31.4723599912, 121.405135391 31.4723599917, 121.405135391 31.4726403253, 121.40598323 31.4726403247, 121.405983229 31.4729183796, 121.406249889 31.4729183801, 121.406249889 31.4731941554, 121.406525664 31.4731941552, 121.406525664 31.4734722092, 121.406805997 31.4734699305, 121.406805997 31.4737502641, 121.407081773 31.4737502646, 121.407081773 31.4740305982, 121.40735527 31.4740305977, 121.407355269 31.4743109313, 121.407637882 31.4743063734, 121.407637882 31.4748602037, 121.407913657 31.4748602033, 121.407913657 31.4751405369, 121.408203108 31.4751405373, 121.408203108 31.4754208709, 121.408469766 31.4754163122, 121.408469766 31.4756943672, 121.408752379 31.4756943671, 121.408752379 31.4759724211, 121.409028155 31.4759701424, 121.409028155 31.4765308096, 121.409301651 31.4765285309, 121.409308488 31.4768043064, 121.409577427 31.4768020264, 121.409577426 31.4773718112, 121.40985776 31.4773718111, 121.409857761 31.4776407485, 121.410422986 31.477640749, 121.410422986 31.4779210826, 121.410712436 31.477921082, 121.410712436 31.4782014166, 121.410972258 31.4782014165, 121.410972258 31.4804304106, 121.410698761 31.4804167364, 121.410698761 31.4806947904, 121.410422986 31.4806902327, 121.410422986 31.4815312333, 121.411248032 31.4815312331, 121.411248033 31.4812508995, 121.411530645 31.4812508999, 121.411530645 31.4809705663, 121.412636026 31.4809728451, 121.412636027 31.4806947901, 121.412918639 31.4806947901, 121.41291864 31.480416736, 121.413192136 31.4804304109, 121.413192136 31.4801409606, 121.413481586 31.4801409602, 121.413481586 31.4798606265, 121.413750524 31.4798606265, 121.413750524 31.4795802936, 121.414017182 31.4795802936, 121.414017182 31.4798606265, 121.414306633 31.479860627, 121.414306633 31.4806947904, 121.414582408 31.4806947905, 121.414582408 31.4809728447, 121.415412013 31.4809728451, 121.415412013 31.4806947908, 121.416253014 31.4806947906, 121.416253015 31.4804167363, 121.416526511 31.4804304104, 121.416526511 31.4801409608, 121.416809124 31.4801409602, 121.416809124 31.4793113554, 121.416526511 31.4793113551, 121.416526511 31.4790310211, 121.415970402 31.4790310214, 121.415970402 31.4787506874, 121.414855904 31.4787506877, 121.414855905 31.4782014161, 121.413750524 31.478201416, 121.413750523 31.4776407492, 121.413192136 31.477640749, 121.413192136 31.4779210828, 121.412918639 31.477921082, 121.412918639 31.4782014158, 121.412086755 31.4782014166, 121.412086755 31.4779210828, 121.411797305 31.477921082, 121.411797305 31.4776407492, 121.411530646 31.4776407491, 121.411530646 31.4773718107, 121.411248033 31.4773718105, 121.411248033 31.4770800815, 121.410972258 31.4770823604, 121.410972258 31.4768043063, 121.410416148 31.4768043061, 121.410416148 31.4765285307, 121.410140373 31.4765308097, 121.410140372 31.4756898085, 121.409584264 31.4756943668, 121.409584264 31.4754163127, 121.409301651 31.4754208703, 121.409301651 31.4751405366, 121.408752379 31.4751405367, 121.408752379 31.4748602031, 121.408469766 31.4748602033, 121.408469766 31.4743109312, 121.408203107 31.4743109316, 121.408203108 31.474030598, 121.407913658 31.4740305977, 121.407913657 31.473750264, 121.407646999 31.4737502642, 121.407646998 31.4734699305, 121.407364385 31.4734722098, 121.407364386 31.4731941548, 121.407081772 31.4731941547, 121.407081773 31.4729183794, 121.406805998 31.4729183799, 121.406805998 31.472640325, 121.406516548 31.4726403252, 121.406516547 31.4718084408, 121.406249889 31.4718084409, 121.406249888 31.471530386, 121.40598323 31.4715303865, 121.405983229 31.4712409362, 121.405693779 31.4712500522, 121.405693779 31.4706939432, 121.405976392 31.4706939435, 121.405976392 31.4704158885, 121.406516547 31.4704204473, 121.406516547 31.4701401136, 121.407081773 31.4701401135, 121.407081773 31.4698597797, 121.407364386 31.4698620592, 121.407364386 31.4695840041, 121.40819627 31.4695840045, 121.40819627 31.4693059493, 121.40875238 31.4693105075, 121.408752379 31.4690301745, 121.409028155 31.4690301741, 121.409028155 31.4687498401, 121.409577427 31.4687498407, 121.409577426 31.4684695067, 121.410140373 31.4684717856, 121.410140373 31.4681960108, 121.410698761 31.4681960105, 121.410698761 31.4679156765, 121.410972258 31.4679202352, 121.410972258 31.4676399011, 121.411797305 31.4676399011, 121.411797304 31.4673595678, 121.412360251 31.4673618468, 121.412360251 31.4670837922, 121.413750524 31.4670837928, 121.413750524 31.4673618466, 121.414017182 31.4673595681, 121.414017183 31.4676399015, 121.414306633 31.4676376225, 121.414306633 31.4681960106, 121.41402402 31.4681960108, 121.41402402 31.4684717858, 121.413750524 31.4684695069, 121.413750524 31.4687498403, 121.413481586 31.4687498409, 121.413481585 31.4690301742, 121.413750524 31.4690278949, 121.413750524 31.4695840041, 121.414306633 31.4695840042, 121.414306633 31.4698620588, 121.415412013 31.4698620584, 121.415412014 31.4695840046, 121.416253014 31.4695840041, 121.416253015 31.4693059494, 121.416526511 31.469310508, 121.416526511 31.4690301745, 121.416809124 31.4690301742, 121.416809124 31.4687498407, 121.417641008 31.4687498409, 121.417641008 31.4684695073, 121.418197117 31.4684717862, 121.418197117 31.4681960109, 121.418472893 31.4681960108, 121.418472893 31.4670837928, 121.416809124 31.4670837921, 121.416809124 31.4668057381, 121.415970402 31.4668057376, 121.415970402 31.4670837924, 121.415138518 31.4670837925, 121.415138518 31.4668057378, 121.414855905 31.4668102964, 121.414855905 31.466529962, 121.414306633 31.4665299628, 121.414306633 31.4662496284, 121.414017183 31.4662496289, 121.414017182 31.4659715741, 121.413750524 31.4659738536, 121.413750523 31.4651396893, 121.414017183 31.4651396894, 121.414017183 31.4648616355, 121.414306632 31.464861635, 121.414306633 31.4640320297, 121.414582409 31.4640320296, 121.414582408 31.463751696, 121.415703744 31.4637516961, 121.415694627 31.4634713624, 121.416809124 31.4634713628, 121.416809124 31.4637516965, 121.417075782 31.4637516964, 121.417075783 31.4640320302, 121.417358396 31.4640320301, 121.417358396 31.4643100843, 121.417641008 31.4643055262, 121.417641009 31.4645835804, 121.417914505 31.4645835801, 121.417914505 31.4648616352, 121.418197117 31.4648616352, 121.418197118 31.4654314193, 121.418472893 31.4654154654, 121.418472893 31.4656935195, 121.419029001 31.4656935194, 121.419029002 31.4659738532, 121.419585111 31.4659715746, 121.419585111 31.4662496288, 121.419860887 31.4662496283, 121.419860886 31.4676399019, 121.420143499 31.4676399013, 121.420143499 31.4679202351, 121.420416995 31.4679156768, 121.420416995 31.468471786, 121.420692771 31.468469507, 121.420692771 31.4687498407, 121.42124888 31.4687498403, 121.421248881 31.4693105079, 121.421515539 31.4693105079, 121.421515539 31.470140113, 121.421807268 31.4701401131, 121.421807269 31.4704204469, 121.421524656 31.470415889, 121.421524656 31.4706939431, 121.420134383 31.4706939429, 121.420134382 31.4709719978, 121.419860886 31.4709719979, 121.419860886 31.4712500528, 121.419304777 31.4712500523, 121.419304777 31.4731941555, 121.419860887 31.4731941554, 121.419860886 31.4729183793, 121.420416995 31.4729183794, 121.420416996 31.4726403255, 121.420692771 31.4726403253, 121.420692771 31.4723599917, 121.420966267 31.4723599918, 121.420966268 31.4720796581, 121.421524656 31.4720842163, 121.421524655 31.4718061613, 121.421807268 31.4718084406, 121.421807269 31.4715303865, 121.422080764 31.4715303859, 121.422080765 31.4712409356, 121.422363377 31.4712500524, 121.422363377 31.4709719982, 121.423195262 31.4709719978, 121.423195262 31.4712500521, 121.423468758 31.4712409355, 121.423468758 31.4715303859, 121.423751372 31.4715303858, 121.423751371 31.4718084409, 121.424027147 31.4718061611, 121.424027146 31.4720842163, 121.424309759 31.4720796575, 121.424309759 31.4723599913, 121.424576418 31.4723599915, 121.424576418 31.4726403254, 121.424859031 31.4726403255, 121.424859032 31.4729183798, 121.425139364 31.4729183793, 121.425139364 31.4731941548, 121.42541514 31.4731941554, 121.42541514 31.4734722096, 121.425688636 31.4734699309, 121.425688637 31.4737502638, 121.425399186 31.4737502643, 121.425399186 31.4740305981, 121.425139364 31.474030598, 121.425139365 31.4743109318, 121.424859031 31.474306373, 121.424859031 31.4745821485, 121.424576418 31.4745798695, 121.424576419 31.4748602032, 121.424309759 31.4748602029, 121.424309759 31.4751405367, 121.423468758 31.4751405368, 121.423468758 31.4754208704, 121.422636874 31.4754163125, 121.422636873 31.4756943665, 121.421807269 31.4756943671, 121.421807269 31.4759724219, 121.421515539 31.4759701427, 121.421515539 31.4765308097, 121.42124888 31.4765285305, 121.42124888 31.4770823604, 121.420966267 31.4770800814, 121.420966268 31.4779210826, 121.422080764 31.4779210821, 121.422080764 31.4776407486, 121.422652828 31.4776407487, 121.422652828 31.4773718104, 121.423186145 31.4773718105, 121.423186145 31.4770800816, 121.423751371 31.4770823604, 121.423751371 31.4768043064, 121.42430976 31.4768043055, 121.42430976 31.4765285302, 121.424859031 31.4765308097, 121.424859031 31.476250476, 121.425139364 31.4762504757, 121.425139365 31.475970142, 121.425697753 31.4759724218, 121.425697753 31.4756943667, 121.426247024 31.4756943669, 121.426247025 31.4754163127, 121.426796296 31.4754208706, 121.426796296 31.4745798698, 121.427361522 31.4745821486, 121.427361522 31.4740283187, 121.4295814 31.4740305978, 121.4295814 31.4723599911, 121.430137509 31.4723599913, 121.430137509 31.472079658, 121.430420122 31.4720842159, 121.430420122 31.4718061613, 121.431817232 31.4718084405, 121.431817232 31.4715303858, 121.432083891 31.4715303856, 121.432083891 31.4712409355, 121.432357387 31.4712500521, 121.432357388 31.4704158887, 121.432083892 31.4704204471, 121.432083891 31.4701401136, 121.431817232 31.470140113, 121.431817232 31.4698597795, 121.430137509 31.4698597792, 121.43013751 31.4701401135, 121.429864013 31.4701401134, 121.429864012 31.4704204467, 121.429305624 31.4704158891, 121.429305625 31.4706939437, 121.429032128 31.4706939436, 121.429032128 31.4709719981, 121.428749516 31.4709719983, 121.428749516 31.4712500528, 121.428466902 31.4712409356, 121.428466903 31.4715303864, 121.427641855 31.4715303859, 121.427641855 31.4718084403, 121.427361521 31.4718061613, 121.427361522 31.4720842158, 121.427085747 31.4720796576, 121.427085747 31.4723599916, 121.426529637 31.4723599916, 121.426529637 31.4720796576, 121.426247024 31.4720842159, 121.426247024 31.4718061615, 121.425971249 31.4718084407, 121.425971249 31.4712409357, 121.42541514 31.4712500522, 121.42541514 31.4709719979, 121.425139364 31.4709719974, 121.425139365 31.4706939431, 121.424859031 31.4706939437, 121.424859031 31.4704158885, 121.424576418 31.4704204468, 121.424576418 31.4701401137, 121.424309759 31.4701401137, 121.42430976 31.4698597798, 121.424027147 31.4698620591, 121.424027147 31.4695840038, 121.423751371 31.4695840041, 121.423751371 31.4690278955, 121.423468759 31.4690301745, 121.423468759 31.4687498406, 121.422919487 31.4687498401, 121.422919486 31.4684695071, 121.422636874 31.4684717861, 121.422636874 31.4681960105, 121.422363377 31.4681960107, 121.422363377 31.4679156768, 121.422080764 31.4679202352, 121.422080764 31.4676399013, 121.421807268 31.4676399012, 121.421807269 31.4670815131, 121.421524656 31.4670837924, 121.421524656 31.4668057381, 121.42124888 31.466810296, 121.42124888 31.4665299621, 121.420966267 31.4665299625, 121.420966267 31.4659715743, 121.420692771 31.4659738534, 121.420692771 31.4654154652, 121.420143499 31.4654314186, 121.4201435 31.4651396894, 121.419585111 31.4651396898, 121.419585112 31.4645813017, 121.419304777 31.464583581, 121.419304777 31.4643055259, 121.419029002 31.4643100845, 121.419029002 31.4637516964, 121.418746389 31.4637516956, 121.418746389 31.4612947875, 121.419022164 31.4612492054, 121.419029002 31.4609688711, 121.419785675 31.4609711504, 121.419860886 31.4606953755, 121.420416996 31.4606999334, 121.420416995 31.4604195993, 121.420692771 31.4604195993, 121.420692771 31.4601392661, 121.420966267 31.4601392663, 121.420966267 31.4598589322, 121.421524655 31.4598612111, 121.421524655 31.4595831565, 121.421807269 31.4595831572, 121.421807269 31.4593051026, 121.422080764 31.4593051022, 121.422080764 31.4590293272, 121.422652827 31.4590293267, 121.422652827 31.4587489933, 121.422919486 31.4587489934, 121.422919486 31.45846866, 121.423468759 31.4584709388, 121.423468759 31.4581951637, 121.423751371 31.4581951636, 121.423751372 31.4579171089, 121.424576418 31.4579193882, 121.424576419 31.4576390546, 121.425399186 31.4576390543, 121.425399186 31.4573609994, 121.426247024 31.4573609997, 121.426247025 31.4570806661, 121.427085747 31.4570829447, 121.427085746 31.4568071693, 121.427641856 31.456809449, 121.427641856 31.4565291153, 121.428466903 31.4565291149, 121.428466902 31.4562396653, 121.429305625 31.4562487816, 121.429305624 31.455973006, 121.430137509 31.4559730065, 121.43013751 31.4556949513, 121.430969393 31.4556949517, 121.430969394 31.4554168973, 121.430686781 31.4554191759, 121.43068678 31.4551411215, 121.430969394 31.4551411216, 121.430969394 31.4548607875, 121.431525503 31.4548607877, 121.431525503 31.4545804545, 121.432357388 31.4545850121, 121.432357388 31.4543046789, 121.435142491 31.4543092372, 121.435142491 31.4540311824, 121.435958421 31.4540311825, 121.435967538 31.4543092367, 121.436813097 31.4543092369, 121.436813097 31.4540311821, 121.436530485 31.4540311824, 121.436530485 31.4535594016, 121.436530484 31.4523628552, 121.43680626 31.4523605756, 121.43680626 31.4501384185, 121.437079756 31.4501406975, 121.437079756 31.4498603638, 121.437362369 31.4498626434, 121.437362369 31.4495823097, 121.437638145 31.4495823095, 121.437638144 31.4493065341, 121.438187416 31.4492996969, 121.438194254 31.4490284801, 121.438750363 31.4490307585, 121.438750363 31.4487504255, 121.439026138 31.448750425, 121.439026138 31.4481897581, 121.439308751 31.4481943157, 121.439308751 31.4479162614, 121.439582248 31.44792082, 121.439582247 31.447640486, 121.440140635 31.4476404864, 121.440140635 31.4473601524, 121.440407295 31.4473601524, 121.440407294 31.4468108806, 121.440689907 31.4468108803, 121.440689907 31.4465282676, 121.441246016 31.4465282677, 121.441246017 31.44596988, 121.441528629 31.4459721592, 121.441528629 31.4456941038, 121.441802126 31.4456941046, 121.441802125 31.445473028, 121.441872778 31.4454183286, 121.442093855 31.4454206075, 121.442093855 31.4452451144, 121.442093855 31.4451402742, 121.442230603 31.4451402741, 121.442360514 31.4451402745, 121.442360514 31.4450377131, 121.442360513 31.4448599403, 121.442588427 31.4448599407, 121.442916623 31.4448599402, 121.442916622 31.4446023986, 121.442916623 31.4437500011, 121.443192399 31.4437500013, 121.443192398 31.4434696678, 121.443472732 31.4434719474, 121.443472731 31.4431961713, 121.443755345 31.4432007296, 121.443755345 31.4429203961, 121.443465894 31.4429203961, 121.443465894 31.4426400626, 121.443755345 31.4426400626, 121.443755345 31.4423597291, 121.444313733 31.4423597287, 121.444313733 31.4426400622, 121.444580392 31.4426400623, 121.444580392 31.4415301233, 121.444863005 31.4415301234, 121.444863004 31.4409694561, 121.445136501 31.4409717354, 121.445136501 31.4406936805, 121.445419114 31.4406936802, 121.445419113 31.4404156261, 121.446533611 31.4404201846, 121.446533611 31.4401398508, 121.446868644 31.4401398506, 121.447082883 31.4401398502, 121.447082882 31.4398595172, 121.447365495 31.4398617958, 121.447365496 31.4395837416, 121.447638992 31.4395837413, 121.447638992 31.439305687, 121.448470876 31.4393102454, 121.448470877 31.4390299115, 121.44819738 31.4390299118, 121.448197381 31.4387518566, 121.448470876 31.4387518571, 121.448470876 31.4384715232, 121.448753489 31.4384738022, 121.44875349 31.4381934691, 121.449026986 31.4382003064, 121.449026985 31.4379199724, 121.449302761 31.4379199726, 121.449302761 31.4376419182, 121.44985887 31.4376419177, 121.44985887 31.4373615845, 121.450141483 31.4373615844, 121.450141483 31.4370812503, 121.450417258 31.4370835295, 121.450417258 31.4368054749, 121.450690754 31.4368100335, 121.450690754 31.4354220402, 121.45096653 31.4354220394, 121.450973367 31.4342209356, 121.450973368 31.4331953239, 121.451249143 31.4331998825, 121.451249143 31.4326392155, 121.451529476 31.4326392152, 121.451529476 31.4323588817, 121.451805252 31.4323611603, 121.451805252 31.4320831064, 121.452078748 31.432083106, 121.452078748 31.4318050512, 121.452637137 31.4318050511, 121.452637136 31.4315269972, 121.452887841 31.4314836936, 121.45297217 31.4312489426, 121.454581239 31.4312489422, 121.454581239 31.4318050513, 121.454863852 31.4318050513, 121.454863852 31.4323611604, 121.458474003 31.4323611609, 121.458474003 31.4320831057, 121.45819139 31.4320831065, 121.45819139 31.4318050513, 121.459030112 31.4318050519, 121.459030112 31.4320306863, 121.459030112 31.4323611606, 121.459586221 31.4323611606, 121.459586221 31.4320831062, 121.460135493 31.4320831063, 121.460135493 31.4315269966, 121.459861996 31.4315292757, 121.459861996 31.4313834114, 121.459861996 31.4298609491, 121.460135493 31.4298609485, 121.460135493 31.4295806153, 121.460693881 31.4295828946, 121.460693881 31.4293071188, 121.460983331 31.4293207938, 121.460983332 31.4290290641, 121.461509811 31.4290290638, 121.461509812 31.4287510101, 121.462081874 31.42875101, 121.462081875 31.4284706757, 121.463592941 31.4284706764, 121.463752481 31.4284706762, 121.463752481 31.4287510097, 121.464028257 31.4287510101, 121.464028257 31.4293207938, 121.463752481 31.4293207934, 121.463752481 31.4301390037, 121.464568411 31.430139003, 121.464568412 31.4298609491, 121.464860141 31.4298609486, 121.464860141 31.4295806151, 121.465416251 31.4295828943, 121.46541625 31.4293071191, 121.465972359 31.4293207941, 121.465972359 31.4287510101, 121.466530748 31.4287510095, 121.466530748 31.4284706758, 121.467086857 31.4284729548, 121.467086857 31.4281949008, 121.467360353 31.428194901, 121.467360353 31.4276387918, 121.467636128 31.4276410703, 121.467636128 31.4279191254, 121.468192237 31.4279191253, 121.468192237 31.4276410703, 121.469017284 31.4276410703, 121.469017284 31.4273607374, 121.46958251 31.4273607373, 121.46958251 31.4276410703, 121.469856006 31.4276410705, 121.469856006 31.4273607366, 121.470412116 31.4273607368, 121.470412116 31.4270804037, 121.470694729 31.427084962, 121.470694728 31.426804628, 121.470970503 31.4268114649, 121.470970503 31.4265311318, 121.471253116 31.4265311313, 121.471253117 31.426259914, 121.471526613 31.4262599142, 121.471526613 31.4259704643, 121.471802388 31.4259727436, 121.471802388 31.4254166347, 121.472075884 31.4254211929, 121.472075884 31.4251408588, 121.472923723 31.4251408588, 121.472923723 31.4248605255, 121.473472995 31.424862804, 121.473472995 31.4245824707, 121.474029104 31.424582471, 121.474029104 31.4243044164, 121.474587492 31.4243112533, 121.474587492 31.4240309199, 121.475136764 31.4240309197, 121.475136764 31.4237505862, 121.475966369 31.423750586, 121.475966369 31.4234702525, 121.476248982 31.4234725312, 121.476248982 31.4231944773, 121.47680737 31.4231944772, 121.47680737 31.4229164224, 121.477639255 31.4229164221, 121.477639254 31.423194477, 121.477912751 31.4231944769, 121.477912751 31.4226383679, 121.478202201 31.4226406469, 121.478202201 31.4218110412, 121.47846886 31.4218110416, 121.47846886 31.4215307079, 121.478751473 31.4215307077, 121.478751473 31.4212503739, 121.47957652 31.4212503742, 121.479583358 31.4210315774, 121.479859132 31.4209723201, 121.479859133 31.4204162107, 121.47957652 31.4204207691, 121.47957652 31.4201290396, 121.479309861 31.4201404354, 121.479309861 31.4198601015, 121.478751472 31.4198601012, 121.478751472 31.420140435, 121.47846886 31.4201290398, 121.47846886 31.4204207691, 121.477912751 31.4204162103, 121.477912751 31.4206942654, 121.477639255 31.4207580814, 121.477639255 31.4209723197, 121.475966369 31.4209700403, 121.475966369 31.4212503739, 121.475692873 31.4212503745, 121.475692873 31.4215307081, 121.473625696 31.4215307079, 121.472923723 31.421530708, 121.472923723 31.4212503745, 121.47264111 31.4212503745, 121.47264111 31.4209700402, 121.472358498 31.4209723201, 121.472358497 31.4206942654, 121.472085001 31.4206942655, 121.472085001 31.4204162108, 121.471802388 31.4204207692, 121.471802388 31.4201290393, 121.471526613 31.4201404355, 121.471526612 31.4184721082, 121.471802388 31.4184721083, 121.471802388 31.4181940536, 121.474587492 31.418194053, 121.474587492 31.4184721079, 121.474860988 31.418469829, 121.474860988 31.4187501627, 121.478202202 31.4187501622, 121.478195364 31.418472108, 121.478751473 31.4184721079, 121.478751473 31.4181940536, 121.480697854 31.4181940529, 121.483193508 31.4181940531, 121.483193508 31.4184721076, 121.483473842 31.4184721077, 121.483473842 31.4181940532, 121.484581502 31.4181940535, 121.484581502 31.4179182776, 121.484864115 31.4179205569, 121.484864115 31.4176402235, 121.485137611 31.417640223, 121.485137611 31.4173598896, 121.485420224 31.4173598897, 121.485420223 31.4170886722, 121.487074876 31.4170886728, 121.487074876 31.416810618, 121.487640102 31.4168106181, 121.487640101 31.417088673, 121.488196211 31.4170886722, 121.488196211 31.4168106183, 121.488738646 31.416810618, 121.48882981 31.4170886727, 121.489297033 31.4170886726, 121.489297033 31.4168106177, 121.489586484 31.416810618, 121.489586483 31.4162499505, 121.48985998 31.4162499506, 121.48985998 31.4154203449, 121.490142593 31.4154203454, 121.490142593 31.4151400115, 121.490418369 31.415140012, 121.490418368 31.4148596782, 121.490135756 31.4148619571, 121.490135755 31.4145839029, 121.48985998 31.4145839028, 121.48985998 31.4140277935, 121.490142593 31.4140300726, 121.490142593 31.4137497388, 121.490418368 31.4137497391, 121.490418368 31.413191351, 121.490691865 31.4131936297, 121.490691865 31.4126397999, 121.49096764 31.4126397999, 121.49096764 31.4120814116, 121.491247973 31.4120836902, 121.491247973 31.4118056359, 121.491530586 31.4118101942, 121.491530586 31.411529861, 121.492079858 31.4115298606, 121.492079859 31.4112518062, 121.492362471 31.411251806, 121.492362471 31.4109600772, 121.492820577 31.4109714729, 121.492957326 31.4107298845, 121.493194356 31.4106273228, 121.493278684 31.4104176428, 121.493750465 31.4104199221, 121.493750465 31.4101418675, 121.49402624 31.4101418671, 121.49402624 31.4098615338, 121.494308853 31.4098615337, 121.494308853 31.4095811994, 121.494582349 31.4095834787, 121.494582349 31.409305424, 121.494874079 31.4093099827, 121.49489687 31.4090569986, 121.495140737 31.4090319281, 121.495140737 31.4088108514, 121.495368651 31.40871057, 121.495414234 31.4084712607, 121.495696846 31.4084712602, 121.495696846 31.4081932064, 121.495972621 31.4081932062, 121.495972622 31.4079174311, 121.496246118 31.4079197094, 121.496246118 31.4076393758, 121.496567476 31.407639376, 121.496802227 31.4076393763, 121.496856927 31.4074182996, 121.49708484 31.4073613214, 121.497123586 31.4071197336, 121.497360616 31.407017172, 121.497360616 31.4068052129, 121.497604483 31.4067755837, 121.497643228 31.4065294372, 121.497848351 31.4065294374, 121.497909887 31.4062491031, 121.498176546 31.4062399872, 121.4981925 31.4059710492, 121.498475113 31.4059710488, 121.498475113 31.4056952734, 121.498810146 31.4056952738, 121.500694991 31.4056952735, 121.500694991 31.4054172192, 121.501807209 31.405419498, 121.501807209 31.4051505596, 121.502082985 31.4051505596, 121.502082984 31.4048588308, 121.502356481 31.4048611094, 121.502356481 31.404583055, 121.502639094 31.4045830552, 121.502639093 31.4043072796, 121.502905753 31.4043095592, 121.502905753 31.4040292251, 121.503197482 31.404029225, 121.503197482 31.4034708372, 121.503470978 31.4034731161, 121.503470978 31.4031950616, 121.503753591 31.4031950619, 121.503753591 31.4029170074, 121.504029367 31.4029192864, 121.504029366 31.4026389521, 121.504302863 31.4026389522, 121.504302862 31.4023608976, 121.504592313 31.4023608983, 121.504585476 31.4020828432, 121.504858972 31.4020896811, 121.504858972 31.4018093468, 121.505141584 31.4018093467, 121.505141584 31.4015312929, 121.50541736 31.4015312928, 121.50541736 31.4012509593, 121.505690857 31.401250959, 121.505690856 31.4009706255, 121.505973469 31.4009729041, 121.505973469 31.4006948494, 121.506249245 31.4006948501, 121.506249245 31.4004167953, 121.506531857 31.4004213532, 121.506531858 31.399860686, 121.506812191 31.3998606863, 121.506805354 31.3995849108, 121.507081129 31.3995849106, 121.507081129 31.3993045769, 121.507361463 31.3993114141, 121.507361462 31.3987507467, 121.507637238 31.398750747, 121.507637239 31.3984704133, 121.508193348 31.3984726925, 121.508193348 31.3979165834, 121.508469123 31.3979211412, 121.508469123 31.3975382465, 121.508469123 31.3973604745, 121.508690198 31.3973627535, 121.508751735 31.397312612, 121.508751736 31.3970824202, 121.509025232 31.3970824197, 121.509025232 31.3965285901, 121.509307844 31.3965308692, 121.509307844 31.3962505353, 121.509850279 31.3962505353, 121.509850279 31.3959588057, 121.510139729 31.3959702017, 121.510139729 31.3956944262, 121.510413225 31.3956944263, 121.510413226 31.3951405964, 121.510139729 31.3951405963, 121.510139729 31.3945799291, 121.509863954 31.3945822078, 121.509863953 31.3926381049, 121.51013973 31.3926403846, 121.51013973 31.3923600505, 121.510406387 31.3923600504, 121.510406388 31.3926403846, 121.510971613 31.3926403844, 121.510971614 31.3923600511, 121.511254226 31.3923600505, 121.511254226 31.3920797172, 121.511803498 31.3920842751, 121.511803498 31.3918062205, 121.512076994 31.3918085002, 121.512076994 31.3915304456, 121.512359607 31.3915304452, 121.512359607 31.3912501118, 121.512908879 31.3912501118, 121.512908879 31.3909697783, 121.513191492 31.3909720567, 121.513191492 31.3906940029, 121.513747601 31.3906940027, 121.5137476 31.3904182267, 121.514030214 31.3904205059, 121.514030214 31.3901492893, 121.514579486 31.3901492892, 121.514579485 31.3898598387, 121.514862098 31.389859839, 121.514862098 31.3895795054, 121.515135594 31.3895840635, 121.515135595 31.3893060086, 121.515687146 31.3893105669, 121.515687146 31.3890302332, 121.515967479 31.3890302335, 121.515967479 31.3887498998, 121.51680848 31.3887498998, 121.51680848 31.3890302335, 121.517081976 31.3890302335, 121.517081976 31.3893105673, 121.517913861 31.3893060089, 121.517913861 31.3895840631, 121.518463132 31.3895817846, 121.519301854 31.3895795056, 121.519301855 31.3898598386, 121.520972461 31.3898598385, 121.520972461 31.3901492894, 121.523750727 31.390149289, 121.523750728 31.3898598389, 121.524849271 31.3898598386, 121.52484927 31.3901492888, 121.52540538 31.3901492893, 121.52540538 31.3898598391, 121.52569483 31.3898598391, 121.52569483 31.3895795049, 121.527358599 31.3895840637, 121.527358599 31.389306009, 121.527641212 31.3893105676, 121.527641212 31.3890302332, 121.527907871 31.3890302335, 121.527907871 31.3887499, 121.528190483 31.3887498997, 121.528197321 31.3882006284, 121.528473096 31.3882006283, 121.528473097 31.3879202948, 121.528746593 31.3879202942, 121.528746593 31.3876399606, 121.529295864 31.3876399611, 121.529295865 31.3873596275, 121.52986109 31.387361906, 121.52986109 31.387083852, 121.530136866 31.387083852, 121.530136866 31.386805797, 121.530982425 31.3868012389, 121.530975587 31.3865277425, 121.532080968 31.3865300212, 121.532096922 31.3862496878, 121.532639356 31.3862496876, 121.532639356 31.3859716334, 121.533195466 31.3859739127, 121.533195466 31.3856935788, 121.53402735 31.3856935792, 121.53402735 31.385415524, 121.534576622 31.3854200827, 121.534576622 31.3851397487, 121.534859234 31.3851397488, 121.534859234 31.3845813604, 121.535141847 31.3845836396, 121.535141847 31.3843055851, 121.535415344 31.3843055854, 121.535415344 31.3845836398, 121.535697956 31.3845836399, 121.535697956 31.3843055855, 121.536247229 31.3843101432, 121.536247228 31.3840206929, 121.536529841 31.3840275311, 121.536529841 31.3834714212, 121.537635222 31.3834714212, 121.537635221 31.3831933666, 121.539305828 31.383193367, 121.539305828 31.382917592, 121.539581604 31.3829221497, 121.539581603 31.3826418161, 121.541808319 31.382641816, 121.541808319 31.3823614823, 121.542357591 31.3823614822, 121.542357591 31.3820811485, 121.544303973 31.3820834278, 121.544303973 31.3818053737, 121.544586586 31.3818122111, 121.544586586 31.3812492646, 121.544860082 31.3812492644, 121.544860082 31.3809598144, 121.54541847 31.38097121, 121.54541847 31.3806954346, 121.545691967 31.3806954343, 121.545691966 31.3809712098, 121.546250354 31.3809712096, 121.546250354 31.3806954341, 121.546439522 31.3806977131, 121.546517014 31.3804196585, 121.547362573 31.3804196588, 121.547362573 31.3801393256, 121.547920961 31.3801393253, 121.547920961 31.3804196594, 121.548194457 31.3804173795, 121.548194457 31.3809712096, 121.547920961 31.3809712097, 121.547920961 31.382083428, 121.548187619 31.3820811484, 121.54818762 31.3823614825, 121.548752846 31.3823614822, 121.548752845 31.382081149, 121.549026341 31.3820834278, 121.549026341 31.381527319, 121.549308955 31.3815318771, 121.549308955 31.3812492642, 121.549575613 31.381249264, 121.549575613 31.3809598138, 121.550696948 31.3809598143, 121.550696948 31.3812492646, 121.550972724 31.3812492645, 121.550972724 31.3818122108, 121.55264333 31.3818122109, 121.55264333 31.3815318775, 121.553466098 31.3815318772, 121.553466098 31.3818122107, 121.554307099 31.3818053732, 121.554307099 31.3821814306, 121.554307099 31.3829221495, 121.55374871 31.3829175918, 121.553748711 31.3831933667, 121.553192602 31.3831933673, 121.553192602 31.3834714219, 121.552916826 31.3834714212, 121.552916826 31.3843101434, 121.553192602 31.3843055856, 121.553192601 31.3845836401, 121.554031324 31.3845836399, 121.554031324 31.3843055853, 121.554580595 31.3843101438, 121.554580595 31.3840206934, 121.554863208 31.3840206936, 121.554863208 31.3843101431, 121.556251202 31.3843101438, 121.556251201 31.3840206933, 121.557083086 31.3840275307, 121.557083086 31.3837517557, 121.557921808 31.3837517551, 121.557914971 31.3826395373, 121.558211258 31.3826418164, 121.558211258 31.3823614828, 121.559029468 31.3823614822, 121.559029468 31.3820811485, 121.559585577 31.3820834283, 121.559585577 31.3818053733, 121.560417462 31.3818122104, 121.560417462 31.3815318775, 121.561249346 31.3815318773, 121.561249346 31.3812492646, 121.561019154 31.3812492643, 121.560959896 31.380699993, 121.559861352 31.3806954344, 121.559861352 31.3809712097, 121.559029468 31.3809598143, 121.559029468 31.3812492642, 121.558211259 31.381249264, 121.558211258 31.3815318774, 121.557358861 31.381531877, 121.557358862 31.3818122107, 121.555968589 31.3818122106, 121.555968589 31.3812492646, 121.555685976 31.3812492643, 121.555685976 31.3809598137, 121.555419317 31.3809712097, 121.555419317 31.380139325, 121.555968589 31.3801393249, 121.555968588 31.3798589912, 121.557639195 31.3798589914, 121.557639195 31.3801393252, 121.558211258 31.3801393256, 121.558211258 31.3787490523, 121.559302965 31.3787490525, 121.559302964 31.3784687185, 121.560959896 31.3784687189, 121.560959896 31.379029386, 121.561249346 31.3790293862, 121.561249346 31.379583216, 121.561531959 31.3795786581, 121.561531959 31.3806999922, 121.562081231 31.3806954346, 121.562081231 31.38097121, 121.56263734 31.3809712094, 121.562637339 31.3806954339, 121.563469224 31.380699993, 121.563469224 31.3804196589, 121.565139831 31.3804196589, 121.565139831 31.3798589914, 121.565971715 31.3798612707, 121.565971715 31.3795832161, 121.567086212 31.3795832165, 121.567086213 31.3791114347, 121.567086213 31.3787490526, 121.565971716 31.3787490524, 121.565971715 31.378468719, 121.56569594 31.3784687191, 121.56569594 31.3787490525, 121.564018496 31.3787490523, 121.564018496 31.3790293865, 121.563469225 31.379029386, 121.563469224 31.3793051616, 121.563193449 31.3793051617, 121.563193449 31.3795832161, 121.562361565 31.3795832157, 121.562361565 31.3793051614, 121.562081231 31.3793051612, 121.56208123 31.3790293865, 121.561812293 31.3790293863, 121.561812293 31.3779194468, 121.564018496 31.3779194473, 121.564018496 31.3776391131, 121.564307947 31.3776391136, 121.564307946 31.3770807248, 121.564025334 31.3770830039, 121.564025333 31.3762488402, 121.564307946 31.3762511196, 121.564307946 31.3759707862, 121.564583722 31.3759707862, 121.564583722 31.3762511197, 121.564857218 31.376248841, 121.564857218 31.3770830048, 121.565139831 31.3770830045, 121.565139831 31.3773610591, 121.565413327 31.377361059, 121.565413327 31.3770830043, 121.565971716 31.3770830044, 121.565971716 31.3765268951, 121.569306091 31.3765268951, 121.569306091 31.3773610588, 121.568747703 31.37735878, 121.568747703 31.3776391136, 121.56846509 31.3776391137, 121.56846509 31.3779194473, 121.568191594 31.3779171678, 121.568191593 31.378721703, 121.568191594 31.3793051617, 121.56930609 31.3793051617, 121.569306091 31.3790293866, 121.57015165 31.379029386, 121.57015165 31.3787490524, 121.571525969 31.3787490529, 121.571525969 31.378470998, 121.571808582 31.3784709984, 121.571808582 31.3781952223, 121.572082078 31.3781952224, 121.572082078 31.3773610593, 121.571808582 31.3773610592, 121.571808582 31.3770830042, 121.571250193 31.3770830043, 121.571250194 31.3768072291, 121.57096758 31.3768072285, 121.57096758 31.3773610587, 121.570687247 31.3773587796, 121.570687247 31.3776391134, 121.57041831 31.3776391133, 121.570418309 31.3773587796, 121.570137975 31.3773610588, 121.570137976 31.376248841, 121.570418309 31.3762511202, 121.570418309 31.3759707865, 121.571250194 31.3759730651, 121.571250194 31.375695011, 121.571808582 31.3756950107, 121.571808582 31.3743047381, 121.573752684 31.3743092959, 121.573752684 31.3748608468, 121.57402846 31.3748608473, 121.57402846 31.3754192355, 121.574301956 31.3754146768, 121.574301956 31.3756950108, 121.574860345 31.3756950107, 121.574860344 31.375973065, 121.575142958 31.3759707863, 121.575142957 31.3762511194, 121.575416453 31.3762511196, 121.575416454 31.3768095079, 121.575692229 31.3768095084, 121.575692229 31.3765291744, 121.57653095 31.3765291744, 121.576530951 31.3762511201, 121.577080223 31.3762511202, 121.577080222 31.3759707861, 121.577636332 31.3759730654, 121.577636332 31.3756950109, 121.578750829 31.3756950109, 121.578750829 31.3754146767, 121.57985621 31.3754192355, 121.57985621 31.3751411809, 121.580405482 31.3751411811, 121.580405481 31.3748608467, 121.580970707 31.374860847, 121.580970707 31.3745805135, 121.581526816 31.3745850717, 121.581526816 31.3743047382, 121.582092041 31.3743092964, 121.582092042 31.3740312416, 121.582641314 31.374031242, 121.582641314 31.3737509084, 121.583190585 31.3737509082, 121.583190585 31.3734705746, 121.583748974 31.3734728532, 121.583748974 31.3731947992, 121.584305083 31.3731947993, 121.584305082 31.3729167444, 121.584861192 31.3729213024, 121.584861192 31.3726409686, 121.585693076 31.3726409688, 121.585693076 31.3723606349, 121.586249186 31.3723606356, 121.586249185 31.3720803017, 121.586807574 31.3720825803, 121.586807573 31.3718045261, 121.587356845 31.3718113632, 121.587356845 31.3715310302, 121.587899279 31.3715310299, 121.58789928 31.3712506959, 121.588751676 31.3712506961, 121.588751676 31.3709703621, 121.589232574 31.3709726416, 121.58958356 31.3709726417, 121.589583561 31.3706945873, 121.590415445 31.3707014245, 121.590415446 31.3704210912, 121.591263284 31.3704210904, 121.591263284 31.3701407571, 121.592086052 31.3701407569, 121.592086052 31.3698604235, 121.592635323 31.3698627027, 121.592635323 31.3695823693, 121.593193711 31.369582369, 121.593193711 31.3693043143, 121.593476325 31.3693043142, 121.593476324 31.3690285392, 121.594025596 31.369028539, 121.594025596 31.3687504842, 121.594581705 31.3687504844, 121.594581706 31.3684701508, 121.595137814 31.36847243, 121.595137815 31.3681943752, 121.595696203 31.3681943756, 121.595696202 31.3679163207, 121.595969699 31.3679208793, 121.595969699 31.3676405456, 121.596528088 31.367640545, 121.596528087 31.3673602112, 121.597068242 31.3673602117, 121.597068242 31.3670798779, 121.597640305 31.3670821573, 121.597640305 31.366806382, 121.598198694 31.3668109398, 121.598198693 31.3665283271, 121.59847219 31.3665283271, 121.598472189 31.366250273, 121.599030578 31.3662502721, 121.599030578 31.3659790553, 121.599297237 31.3659790555, 121.599297237 31.3654183884, 121.598198693 31.3654183883, 121.598198694 31.3651403332, 121.597922918 31.3651403339, 121.597922918 31.3648599999, 121.597357693 31.3648599998, 121.597357693 31.3645796659, 121.597084197 31.364584224, 121.597084196 31.3642309587, 121.597084196 31.3634720063, 121.597357692 31.3634720058, 121.597357693 31.3612498485, 121.597640306 31.3612498487, 121.597640306 31.3604202432, 121.597922918 31.3604202436, 121.597922918 31.3598504592, 121.598198694 31.359861855, 121.598198693 31.3593057461, 121.598747966 31.3593103048, 121.598747965 31.3590299706, 121.599030578 31.3590299712, 121.599030578 31.3587496369, 121.602082341 31.3587496371, 121.602082341 31.3584715824, 121.602911946 31.3584715826, 121.602911947 31.3587496373, 121.603203675 31.3587496369, 121.603203675 31.3590299705, 121.602911946 31.3590299709, 121.602911946 31.3593103044, 121.602638449 31.3593057458, 121.60263845 31.3595838005, 121.602911946 31.3595838006, 121.602911946 31.3598618553, 121.603203676 31.3598504599, 121.603203675 31.3601399096, 121.603752947 31.36013991, 121.603752948 31.3604202436, 121.605423553 31.3604202432, 121.605423554 31.3593103045, 121.60569705 31.3593103041, 121.60569705 31.3590299704, 121.605972826 31.3590299707, 121.605972825 31.3584715829, 121.606246321 31.3584738619, 121.606246322 31.3581935281, 121.606528935 31.3581365496, 121.606528935 31.3579154734, 121.606795593 31.3579200315, 121.606795593 31.3576419774, 121.607085043 31.3576419767, 121.607085043 31.3573616438, 121.607643432 31.3573616433, 121.607636594 31.3570835889, 121.608192704 31.3570813094, 121.608192704 31.356805534, 121.6084662 31.3568100924, 121.6084662 31.3565320382, 121.609580697 31.3565320384, 121.609580697 31.3562517043, 121.610419419 31.3562517048, 121.610419419 31.3559713707, 121.610695194 31.3559713706, 121.610695195 31.3556933162, 121.611527078 31.3556933165, 121.611527079 31.3554175409, 121.612960655 31.3554175408, 121.613197685 31.3553696786, 121.613197685 31.3551394858, 121.613633 31.3551394862, 121.614296229 31.3551417658, 121.614296228 31.3548614314, 121.615966835 31.3548614317, 121.615966835 31.3545788193, 121.617081333 31.3545833774, 121.617081333 31.3540295475, 121.616805557 31.354040943, 121.616805557 31.3538198663, 121.616532061 31.3537492135, 121.616532061 31.3534688799, 121.616249448 31.3534711591, 121.616249448 31.3531953832, 121.615966835 31.3531999416, 121.615966835 31.3526392743, 121.61569106 31.3526392742, 121.615691059 31.3515293357, 121.615417564 31.3515293353, 121.615417564 31.3506906131, 121.61569106 31.350695172, 121.61569106 31.3504171172, 121.615966836 31.350419396, 121.615966836 31.3501390622, 121.616249448 31.3501390623, 121.616249448 31.3498610083, 121.617081333 31.3498610076, 121.617081333 31.3495806747, 121.619025436 31.3495806743, 121.619025435 31.3498610082, 121.619857319 31.3498610076, 121.61985732 31.3504193967, 121.620139932 31.3504193962, 121.620139933 31.3509800632, 121.620406591 31.3509800639, 121.620406592 31.3512490018, 121.620659575 31.351249002, 121.621527926 31.3512490021, 121.621527926 31.3509800641, 121.622093152 31.3509800635, 121.622093152 31.3506906132, 121.622359811 31.3506951719, 121.62235981 31.3504171167, 121.622918199 31.3504193963, 121.622918199 31.3501390622, 121.623467471 31.3501390628, 121.62346747 31.3498610084, 121.624030418 31.349861008, 121.624030417 31.3495806747, 121.624579689 31.3495829539, 121.624579688 31.3493071782, 121.626250295 31.349307178, 121.626250296 31.3495829537, 121.62708218 31.3495829531, 121.62708218 31.3493071782, 121.627640568 31.3493094569, 121.627640568 31.3490291234, 121.628196677 31.3490291236, 121.628196677 31.3484707353, 121.628472453 31.3484730142, 121.628472453 31.3476388508, 121.628745949 31.34764113, 121.628745948 31.3473607963, 121.629028561 31.347360796, 121.629028562 31.3470804623, 121.629302058 31.3470850209, 121.629302058 31.3468046872, 121.630143058 31.3468092456, 121.630143059 31.3465311907, 121.629860446 31.3465311908, 121.629860446 31.3462508571, 121.629584671 31.3462508574, 121.62958467 31.3456924688, 121.629302058 31.3456924686, 121.629302058 31.3451386386, 121.629028562 31.3451409184, 121.629028561 31.3447603021, 121.629028561 31.3437506456, 121.629302058 31.3437506454, 121.629302058 31.3434703115, 121.629584671 31.3434725904, 121.629584671 31.3443295456, 121.629584671 31.3448628635, 121.629860446 31.3448514679, 121.629860446 31.345140918, 121.630143059 31.345138639, 121.630143059 31.3456924691, 121.63069233 31.345692469, 121.630692331 31.3459728028, 121.630965827 31.345970523, 121.630965827 31.3462508568, 121.63124844 31.346250857, 121.631248439 31.3465311908, 121.631531053 31.3465311912, 121.631531053 31.3468092453, 121.631804549 31.3468046868, 121.631804549 31.3470850206, 121.632080324 31.3470804623, 121.632080324 31.3473607961, 121.633178868 31.3473607962, 121.633178867 31.3476411301, 121.634026706 31.3476388507, 121.634026706 31.3479169058, 121.63430704 31.3479169057, 121.634307039 31.3476388506, 121.635405583 31.3476411303, 121.635405583 31.3473607963, 121.635697313 31.3473607962, 121.635697313 31.3470804621, 121.635970809 31.3470850209, 121.635970809 31.3468046869, 121.636237467 31.3468092455, 121.636237467 31.3465311911, 121.636526918 31.346531191, 121.636526918 31.3456901895, 121.636802693 31.3456924693, 121.636802693 31.3451386393, 121.636526918 31.3451409182, 121.636526918 31.3429210403, 121.636237468 31.3429210398, 121.636237468 31.3423603722, 121.636526918 31.3423603727, 121.636526918 31.3420823182, 121.637078469 31.3420800393, 121.637078469 31.3426407061, 121.637358803 31.3426407062, 121.637358803 31.3429210396, 121.637917191 31.3429210401, 121.637917191 31.3426407067, 121.638749075 31.3426407059, 121.638749075 31.3423603725, 121.639578681 31.3423603728, 121.63957868 31.3426407063, 121.639861293 31.3426407062, 121.639861294 31.3423603727, 121.640419682 31.342360373, 121.640419681 31.3420823183, 121.640975791 31.3420823182, 121.640975791 31.3418065423, 121.6415319 31.341811101, 121.641531899 31.3415307674, 121.642081171 31.3415307676, 121.642081172 31.3412504339, 121.642639559 31.3412504335, 121.64263956 31.3409700998, 121.643195668 31.3409723787, 121.643195669 31.3406943247, 121.644027553 31.3406943246, 121.644027553 31.3404162696, 121.644576825 31.3404208284, 121.644576825 31.3401404946, 121.645142051 31.340140494, 121.645142051 31.339860161, 121.645691323 31.3398601609, 121.645691323 31.339579827, 121.646247431 31.3395821061, 121.646247432 31.3393063307, 121.646796704 31.3393086101, 121.646796703 31.3390396721, 121.647086154 31.3390396721, 121.647086154 31.3387502217, 121.647635425 31.3387502213, 121.647635425 31.3384698881, 121.648193814 31.3384721673, 121.648193813 31.3381941129, 121.648749923 31.338194113, 121.648749923 31.3379183374, 121.649308311 31.3379206162, 121.649308311 31.3376402829, 121.649581807 31.3376402824, 121.649581807 31.337359949, 121.650137916 31.3373599493, 121.650137916 31.337079615, 121.650696304 31.3370841737, 121.650696305 31.3368061192, 121.65123646 31.3368106769, 121.651236459 31.3365303434, 121.651801685 31.3365303438, 121.651801685 31.3362500102, 121.652360074 31.3362500103, 121.652360073 31.3359696758, 121.652640407 31.3359719553, 121.652640408 31.3356961804, 121.653189678 31.3357007379, 121.653198796 31.3354204044, 121.653748067 31.3354204045, 121.653748067 31.3351400708, 121.654311014 31.3351400704, 121.654311014 31.3348597376, 121.654862565 31.3348620158, 121.654862564 31.3345839618, 121.65513606 31.334583962, 121.65513606 31.3343059071, 121.65569217 31.3343059073, 121.65569217 31.3340278523, 121.656250557 31.3340301321, 121.656250558 31.3337497982, 121.656806667 31.333749798, 121.656806667 31.3334694641, 121.657362776 31.3334717431, 121.657362776 31.3331936889, 121.657638552 31.3331936886, 121.657638551 31.3329156344, 121.658194661 31.3329156344, 121.65819466 31.332639859, 121.658753048 31.3326398588, 121.658753049 31.3323618045, 121.659302321 31.332361804, 121.65930232 31.3320814708, 121.65985843 31.3320837497, 121.659858429 31.3318056953, 121.660141042 31.3318102536, 121.660141042 31.3315299195, 121.660697152 31.3315299203, 121.660697152 31.3312518658, 121.661246424 31.331251865, 121.661246424 31.3309715317, 121.661804812 31.3309738104, 121.661804811 31.330693477, 121.662360921 31.3306934768, 121.66236092 31.3304154222, 121.66291703 31.3304199809, 121.66291703 31.3301419263, 121.663176851 31.3301419263, 121.663176851 31.3298707092, 121.663748915 31.3298707094, 121.663748914 31.3295812586, 121.664307303 31.329583538, 121.664307302 31.3293054833, 121.664856574 31.3293100417, 121.664856574 31.3290319869, 121.665412684 31.3290319869, 121.665412684 31.3287516532, 121.665702133 31.3287516539, 121.665695296 31.3284713195, 121.666251405 31.3284713202, 121.666251405 31.3281932654, 121.666809793 31.3281932657, 121.666809793 31.3279174897, 121.667076453 31.3279220482, 121.667076452 31.3276394355, 121.667359065 31.3276394354, 121.667359065 31.3273591016, 121.667641678 31.3273591017, 121.667641678 31.3268098301, 121.667922012 31.3268098303, 121.667922011 31.3265294964, 121.668197787 31.3265294965, 121.668197787 31.3262491626, 121.668471283 31.3262491627, 121.668471283 31.3259688288, 121.668747059 31.3259711083, 121.668747059 31.3254172779, 121.669029672 31.3254195573, 121.669029671 31.3247198622, 121.669029672 31.3243073395, 121.669312284 31.3243005018, 121.669303168 31.3231951206, 121.66958578 31.3231951207, 121.669585781 31.3220829023, 121.669861556 31.3220829022, 121.669861556 31.3209729634, 121.670135052 31.3209729633, 121.670135053 31.3204168544, 121.669601734 31.3204214127, 121.669601735 31.3201410793, 121.669861557 31.3201410788, 121.669861556 31.3190311397, 121.670144169 31.3190311402, 121.670144169 31.3187508059, 121.670693441 31.3187508066, 121.670693441 31.3190311401, 121.671805659 31.3190311398, 121.671805659 31.3193114733, 121.673469428 31.3193114733, 121.673469428 31.3190311397, 121.674025537 31.3190311401, 121.674025537 31.3187508064, 121.674857422 31.3187508059, 121.674857421 31.3184704722, 121.67541581 31.3184727518, 121.67541581 31.3181946969, 121.675971919 31.3182015344, 121.675971919 31.317930317, 121.676528028 31.3179303172, 121.676528027 31.317640867, 121.677359912 31.3176408672, 121.677359913 31.3173605334, 121.6779183 31.3173628127, 121.677918301 31.3170824788, 121.678191797 31.3170824786, 121.678191797 31.3168044244, 121.67847441 31.3168044245, 121.67847441 31.3165286492, 121.678750185 31.3165309284, 121.678750186 31.3162505944, 121.679030519 31.3162505941, 121.679030519 31.315970261, 121.679306295 31.315970261, 121.679306294 31.3156944856, 121.679862403 31.3156944856, 121.679862404 31.3154164304, 121.680138179 31.315420989, 121.680138179 31.3151406558, 121.680420792 31.3151406551, 121.680420792 31.3148717172, 121.68068745 31.314871717, 121.680687451 31.3145799876, 121.680970064 31.3145822672, 121.680970064 31.3143064917, 121.681250396 31.3143110496, 121.681250397 31.3140307163, 121.681542126 31.3140307165, 121.681542127 31.3137503823, 121.681808785 31.3137503824, 121.681808785 31.3134700491, 121.682082281 31.3134723282, 121.682082281 31.3131942737, 121.682358057 31.3131942733, 121.682358056 31.3129162188, 121.68264067 31.3129184983, 121.682640669 31.3126404437, 121.682907329 31.3126404433, 121.682907329 31.3123601099, 121.683196779 31.3123601099, 121.683196778 31.3120797764, 121.683472555 31.3120843341, 121.683472554 31.3118062803, 121.684028663 31.3118062799, 121.684028664 31.3115282252, 121.684288484 31.3115305046, 121.684288485 31.3112501711, 121.684577935 31.3112501706, 121.684577935 31.310969837, 121.684860548 31.310972116, 121.684860548 31.3106940613, 121.685416657 31.3106940619, 121.685416657 31.310418286, 121.685692432 31.3104205653, 121.685692432 31.3101402316, 121.685965929 31.3101402316, 121.685965929 31.3098598979, 121.685692432 31.3098598979, 121.685692433 31.3095795642, 121.685416657 31.3095841225, 121.685416656 31.3090280138, 121.685143161 31.3090302925, 121.685143161 31.3087408424, 121.685692432 31.3087499592, 121.685692432 31.3084719044, 121.686248542 31.3084719048, 121.686248542 31.3081961287, 121.686813767 31.3082006875, 121.686813768 31.3070816319, 121.687080426 31.3070839104, 121.687080426 31.3068035774, 121.687636535 31.306803577, 121.687636535 31.3065278018, 121.688185807 31.3065300811, 121.688185806 31.3062497471, 121.688751032 31.3062497474, 121.688751032 31.3054201416, 121.689026808 31.3054201418, 121.689026807 31.3051398077, 121.689307141 31.3051398079, 121.689307141 31.3048617536, 121.689856413 31.3048617539, 121.689856414 31.3045814197, 121.690139026 31.3045836994, 121.690139026 31.3040275897, 121.690421639 31.3040298688, 121.690421638 31.303471481, 121.690697415 31.3034714812, 121.690697415 31.3031934258, 121.691529299 31.3031934267, 121.691529298 31.3029176511, 121.691802795 31.302931326, 121.691802795 31.3026418752, 121.692092246 31.3026418754, 121.692092245 31.3020812085, 121.692358904 31.3020834876, 121.692358904 31.301805433, 121.692641517 31.3018122702, 121.692641517 31.3015319367, 121.692917292 31.3015319365, 121.692917292 31.3012493232, 121.693190789 31.3012493235, 121.693190788 31.3009689899, 121.693473402 31.3009712693, 121.693473401 31.3006932146, 121.693749177 31.3006932144, 121.693749177 31.3001393847, 121.69403179 31.3001393848, 121.694031789 31.2998590512, 121.694305286 31.2998613297, 121.694305286 31.299583275, 121.694581062 31.2995832756, 121.694581062 31.2993052208, 121.695419784 31.2993052211, 121.695419783 31.2995832751, 121.69569328 31.2995832754, 121.69569328 31.2984710568, 121.695969055 31.2984710571, 121.695969055 31.297917227, 121.696251668 31.2979195064, 121.696251668 31.2976391726, 121.697639661 31.2976391731, 121.697639661 31.2970807841, 121.697083552 31.2970830635, 121.697083552 31.2962489002, 121.697357049 31.2962488995, 121.697357048 31.2959708453, 121.697913158 31.2959731242, 121.697913158 31.29569507, 121.698195771 31.2956950697, 121.69819577 31.2954170155, 121.698471546 31.2954192945, 121.698471546 31.2951412402, 121.698754159 31.2951412396, 121.698754159 31.2948609064, 121.69985954 31.2948609064, 121.69985954 31.2943093554, 121.700142153 31.2943093553, 121.700142152 31.293760084, 121.700415648 31.2937600839, 121.700415649 31.2934706332, 121.702079417 31.2934706332, 121.702079418 31.293760084, 121.70291814 31.2937600841, 121.702918139 31.2940313011, 121.703476528 31.2940313007, 121.703476528 31.2937600837, 121.704308413 31.2937600842, 121.704308412 31.2934706333, 121.704864521 31.2934729124, 121.704864522 31.292916804, 121.705140297 31.2929213619, 121.705140297 31.2926410283, 121.705420631 31.2926410278, 121.705420631 31.2923606941, 121.705969902 31.2923606946, 121.705969902 31.2918114226, 121.706528291 31.2918114231, 121.706528291 31.2915310894, 121.706801787 31.2915310891, 121.706801787 31.2912507554, 121.707077562 31.2912507558, 121.707077562 31.290970422, 121.707360176 31.290972701, 121.707360175 31.290694646, 121.707640509 31.2907014836, 121.707640509 31.2904211497, 121.707916284 31.2904211501, 121.707916284 31.2898604824, 121.708472394 31.2898627618, 121.708472393 31.2895824279, 121.708748169 31.2895824283, 121.708748169 31.2893043732, 121.709304278 31.2893043737, 121.709304277 31.2890285984, 121.709586891 31.2890308769, 121.709586891 31.2887505438, 121.709862666 31.2887505435, 121.709862666 31.2884702095, 121.710136162 31.288472489, 121.710136163 31.2882628088, 121.710418775 31.2881944342, 121.710418775 31.2879163799, 121.710692271 31.287920938, 121.710692271 31.2876406048, 121.710984001 31.2876406043, 121.710974884 31.2873602707, 121.71125066 31.287360271, 121.71125066 31.2870799368, 121.711524156 31.287082216, 121.711524156 31.2868064405, 121.711806769 31.2868109992, 121.711806768 31.2862503316, 121.712082544 31.2862503319, 121.712082544 31.2862438831, 121.718657036 31.2903768618, 121.744619634 31.2609856955, 121.744858975 31.2606943901, 121.745414903 31.2606943838, 121.745414903 31.2600177538, 121.745544332 31.25986022, 121.745697516 31.2598602202, 121.745697516 31.2596737737, 121.745774128 31.2595805245, 121.745971012 31.2595821652, 121.745971012 31.2593408856, 121.745999163 31.2593066219, 121.746246787 31.259308669, 121.746246788 31.2590052206, 121.746456239 31.2587502807, 121.746529401 31.2587502806, 121.746529401 31.2586612304, 121.746914272 31.2581927687, 121.74708551 31.2581941721, 121.74708551 31.2579843381, 121.747139319 31.2579188416, 121.747361285 31.257920676, 121.747361285 31.2576486614, 121.747377061 31.2576294588, 121.747641619 31.2576380628, 121.747641619 31.2573600081, 121.747917394 31.2573600084, 121.747917395 31.256971749, 121.748049671 31.256810737, 121.74819089 31.2568107367, 121.748190891 31.2566388369, 121.748279971 31.2565304028, 121.748466666 31.2565304031, 121.748466665 31.2563031477, 121.748510271 31.2562500691, 121.748749279 31.2562500695, 121.748749279 31.2559697358, 121.749031891 31.2559720142, 121.749031891 31.2556962392, 121.749312226 31.2557007972, 121.749312225 31.2552738661, 121.749422089 31.2551401296, 121.749581163 31.2551401301, 121.749581163 31.2549464883, 121.749651911 31.2548603673, 121.749863776 31.2548620755, 121.749863776 31.2546024604, 121.749878619 31.2545843914, 121.750137272 31.2545908586, 121.750137272 31.2543014081, 121.750419885 31.2543059668, 121.750419885 31.2540279118, 121.750686544 31.2540301911, 121.750686544 31.2534863103, 121.776468995 31.2126640705, 121.776251609 31.2126443076, 121.776251608 31.2126410877, 121.776527384 31.2126410876, 121.776527385 31.2123516376, 121.776803159 31.2123607536, 121.77680316 31.2120826992, 121.777083494 31.2120826989, 121.777083493 31.2118046445, 121.777359268 31.2118114821, 121.777359269 31.2115311487, 121.777641881 31.2115311481, 121.777641881 31.2109704813, 121.777915378 31.2109727598, 121.777915378 31.2106947053, 121.778191153 31.2107015429, 121.778197991 31.210140876, 121.778473765 31.2101408752, 121.778473766 31.2098605417, 121.778747262 31.2098628211, 121.778747262 31.2095824876, 121.779029875 31.2095824873, 121.779029875 31.2093044327, 121.779303371 31.2093044329, 121.779303371 31.2090286572, 121.779585984 31.2090309365, 121.779585984 31.2087506029, 121.77986176 31.2087506027, 121.77986176 31.2081899355, 121.780135256 31.2081944935, 121.780135256 31.2079164388, 121.780417869 31.2079209978, 121.780417869 31.2076406641, 121.780700482 31.2076406638, 121.780693644 31.2073603302, 121.780976257 31.2073603301, 121.780976257 31.2070822753, 121.781249753 31.2070822757, 121.781249753 31.2068064999, 121.781516412 31.2068110587, 121.781516412 31.2065421206, 121.781805862 31.2065421202, 121.781805863 31.2062503907, 121.782081638 31.2062503911, 121.782081637 31.2059700572, 121.78236425 31.2059723366, 121.78236425 31.2056942817, 121.782637747 31.2056942818, 121.782637746 31.2054162277, 121.782913522 31.2054185061, 121.782913522 31.2048601181, 121.783202972 31.2048601181, 121.783196136 31.2045843424, 121.783469631 31.2045843425, 121.783469631 31.2043062884, 121.78373629 31.2043085671, 121.78373629 31.204030513, 121.784028019 31.2040305128, 121.784028019 31.2037501788, 121.784301516 31.2037501788, 121.784301516 31.2034789624, 121.784577292 31.2034789618, 121.784577291 31.2026402403, 121.784857625 31.2026402406, 121.784857625 31.2023599065, 121.785140238 31.2023599069, 121.785140237 31.2020795727, 121.785416013 31.2020841308, 121.785416013 31.2018060765, 121.785698626 31.2018106345, 121.785698625 31.2015303012, 121.786247897 31.2015303014, 121.786247898 31.2012499672, 121.78653051 31.2012499672, 121.78653051 31.2009696339, 121.786804006 31.2009719129, 121.786804007 31.200415804, 121.78708662 31.2004158037, 121.787086619 31.2001377492, 121.787360116 31.2001400279, 121.787360115 31.1998596944, 121.788750389 31.1998619738, 121.788750388 31.1995839193, 121.789023885 31.199583919, 121.789023885 31.1976398165, 121.789306498 31.1976398166, 121.789306498 31.1973503662, 121.789862607 31.1973617619, 121.789862606 31.1970814282, 121.790138383 31.197081428, 121.790138382 31.1965275984, 121.790411878 31.1965298776, 121.790411878 31.1956911553, 121.790694492 31.1956934348, 121.790694491 31.1954176589, 121.790970266 31.1954222174, 121.790970267 31.1951418836, 121.791252879 31.1951418838, 121.791252879 31.19486155, 121.791526376 31.1948615505, 121.791526376 31.1945812166, 121.791808989 31.1945834954, 121.791808989 31.1943054414, 121.792075647 31.1943122783, 121.792075647 31.1940296655, 121.792358261 31.1940296659, 121.792358261 31.193468998, 121.792640873 31.1934712776, 121.792640873 31.1931955016, 121.792930324 31.1932000605, 121.792930324 31.1929197265, 121.793196983 31.1929197269, 121.793196983 31.1926393929, 121.793746254 31.1926393927, 121.793746254 31.1923590595, 121.794028867 31.1923613387, 121.794028867 31.1918052294, 121.794304642 31.191805229, 121.794304643 31.1912491204, 121.794587255 31.1912491203, 121.794587255 31.1906907318, 121.794860752 31.1906952902, 121.794860751 31.1904172359, 121.795143364 31.1904195144, 121.795143365 31.1901391811, 121.795692636 31.1901391814, 121.795676682 31.1898588476, 121.795975249 31.1898611264, 121.795975249 31.189583072, 121.796248745 31.1895830722, 121.796248745 31.1893072967, 121.796515404 31.1893095753, 121.796515404 31.1890292419, 121.796807133 31.1890292417, 121.796807133 31.1887511873, 121.79708063 31.1887511874, 121.79708063 31.1884708539, 121.797363242 31.1884731329, 121.797363243 31.1881950783, 121.797919351 31.1881950787, 121.797919352 31.1873609149, 121.798188289 31.1873609143, 121.79818829 31.1870805807, 121.798468623 31.1870851391, 121.798468623 31.1868048055, 121.799027012 31.186809364, 121.799027011 31.1865313093, 121.799309624 31.1865313094, 121.799309624 31.1854099746, 121.799583121 31.1854168116, 121.799583121 31.1848607029, 121.799858895 31.1848607029, 121.799858896 31.1843114311, 121.800141509 31.184311431, 121.800141508 31.1837507634, 121.800697617 31.1837507639, 121.800697617 31.1834704301, 121.800971114 31.1834727088, 121.800971114 31.1829165999, 121.801246889 31.1829211583, 121.80124689 31.1826408244, 121.801529502 31.1826408249, 121.801529503 31.1820801569, 121.801802998 31.1820824365, 121.801802999 31.1818043815, 121.802085611 31.1818043816, 121.802085611 31.1815286064, 121.802361387 31.1815308853, 121.802361387 31.1812505521, 121.80264172 31.1812505523, 121.802634883 31.1809724977, 121.802917496 31.1809724977, 121.802917496 31.1806944426, 121.803193271 31.1806944424, 121.803193272 31.1804163882, 121.803466768 31.1804209468, 121.803466768 31.1801406126, 121.80374938 31.1801406126, 121.803749381 31.1795799452, 121.804031993 31.1795822243, 121.804031993 31.179306449, 121.804305489 31.1793087285, 121.80430549 31.1790283943, 121.804581264 31.1790283947, 121.804581265 31.1787503404, 121.804870715 31.17875034, 121.804870715 31.1784700066, 121.805137374 31.1784722858, 121.805137374 31.1779184559, 121.805419987 31.1779184557, 121.805419987 31.1776404013, 121.805686646 31.1776404012, 121.805686646 31.1773600678, 121.805969259 31.1773600672, 121.805969259 31.1770797338, 121.806251871 31.1770842923, 121.806251872 31.1768062378, 121.806525368 31.176810796, 121.806525368 31.1765304616, 121.806807981 31.1765304619, 121.806807981 31.1762501283, 121.807083756 31.1762501286, 121.807083756 31.1759697951, 121.807357252 31.1759720734, 121.807357252 31.1756940197, 121.807639865 31.175694019, 121.807639864 31.1751379106, 121.807906524 31.1751401893, 121.807906524 31.1748598556, 121.808195974 31.1748621346, 121.808195974 31.1745840799, 121.80847175 31.1745840799, 121.808471749 31.1743060252, 121.808745246 31.174306026, 121.808745245 31.1740279712, 121.809027859 31.17403025, 121.809027858 31.1737499163, 121.809303634 31.1737499162, 121.809303634 31.1734695824, 121.809586247 31.1734718618, 121.809586247 31.1731938078, 121.809859744 31.1731938072, 121.809859743 31.1729157532, 121.810418131 31.1729203114, 121.810418131 31.1726399775, 121.810675674 31.1726399774, 121.810675674 31.1720815895, 121.81097424 31.1720838687, 121.810974241 31.1715277597, 121.811247736 31.1715300386, 121.811247737 31.1712497046, 121.811530349 31.1712497051, 121.81153035 31.1709716501, 121.811806125 31.1709739289, 121.811806125 31.1706935958, 121.812079621 31.1706935957, 121.812079622 31.1701397654, 121.812362234 31.1701420448, 121.812362234 31.1698617107, 121.812638009 31.1698617115, 121.812638009 31.1695813774, 121.812920622 31.1695813774, 121.812920623 31.1693056022, 121.813187281 31.1693101604, 121.813187281 31.168751772, 121.813469894 31.1687517717, 121.813469894 31.1684714384, 121.813750228 31.1684714381, 121.813750227 31.1679176084, 121.814026003 31.1679221667, 121.814026003 31.1673706159, 121.814582112 31.1673706154, 121.814582112 31.1670788862, 121.814857888 31.1670834445, 121.814857888 31.1668053901, 121.81540716 31.1668099486, 121.815407159 31.1665296151, 121.815696609 31.166529615, 121.81569661 31.1657000094, 121.815972385 31.1657000091, 121.815972385 31.1654196756, 121.816245881 31.1654196754, 121.816245882 31.1651393419, 121.816528494 31.1651393419, 121.816528494 31.1648590083, 121.81680199 31.1648612877, 121.816801991 31.1643051784, 121.817360379 31.1643097364, 121.817360378 31.1640294027, 121.817642992 31.164029403, 121.817642991 31.1637490693, 121.818192263 31.1637490694, 121.818192263 31.1634710146, 121.818474876 31.1634732938, 121.818474876 31.1629171852, 121.818748372 31.1629194639, 121.818748372 31.1626391302, 121.819030985 31.16263913, 121.819030985 31.1620807422, 121.819580257 31.1620830207, 121.819580257 31.1615269118, 121.819846916 31.1615291914, 121.819846915 31.1609708025, 121.820418979 31.1609730823, 121.820418978 31.1606950273, 121.820694754 31.1606950275, 121.820694754 31.1604169726, 121.820968251 31.1604192519, 121.820968251 31.1595805298, 121.821250863 31.159585088, 121.821250863 31.1593047548, 121.821533475 31.1593093133, 121.821533476 31.1590312582, 121.821806973 31.1590312586, 121.821806972 31.1584705913, 121.822082747 31.1584728701, 121.822082748 31.1581925369, 121.82236536 31.1581925364, 121.82236536 31.1579167611, 121.822638857 31.1579213194, 121.822638857 31.1576409852, 121.822905516 31.1576409856, 121.822905515 31.1570803181, 121.824302626 31.1570825977, 121.824302626 31.1568045433, 121.824578401 31.1568113801, 121.824578401 31.1565310467, 121.824858735 31.1565310469, 121.824858735 31.1562507125, 121.825141348 31.1562507131, 121.825141348 31.1559703796, 121.825417123 31.1559726584, 121.825417123 31.1554165494, 121.82569062 31.1554302243, 121.82569062 31.1551407739, 121.825966395 31.1551407743, 121.825966394 31.1545801063, 121.826249008 31.154582386, 121.826249008 31.1543066104, 121.826529341 31.1543111684, 121.826529341 31.1540308348, 121.826811955 31.1540308352, 121.826811954 31.1537505007, 121.827080892 31.1537505013, 121.827080892 31.1534701676, 121.827361226 31.1534724466, 121.827361226 31.153194392, 121.827637002 31.1531943923, 121.827637001 31.1526382828, 121.827919614 31.1526382825, 121.827919614 31.1523693446, 121.828475724 31.1523693449, 121.828475724 31.1520798942, 121.828751498 31.1520821738, 121.828751498 31.151806398, 121.829040949 31.1518086777, 121.829040948 31.1515306229, 121.829307607 31.1515306232, 121.829307608 31.1512502894, 121.829581104 31.1512502891, 121.829581104 31.1506896222, 121.829863717 31.15069418, 121.829863717 31.150418405, 121.830139492 31.1504206842, 121.830139492 31.1501403503, 121.830406151 31.1501403501, 121.830406151 31.1495796831, 121.830695601 31.1495842407, 121.830695602 31.1493061866, 121.830971377 31.1493061868, 121.830971376 31.1490258528, 121.831527486 31.1490304111, 121.831527486 31.1487500779, 121.831787308 31.1487500777, 121.831787307 31.1484697436, 121.832083595 31.1484720234, 121.832083594 31.147915914, 121.83235937 31.1479204719, 121.83235937 31.1473598045, 121.832641983 31.1473620843, 121.832641984 31.1470840291, 121.832915479 31.1470840296, 121.832915479 31.1468036954, 121.833191255 31.1468036953, 121.833191255 31.1465279199, 121.833473867 31.1465301997, 121.833473867 31.1462407494, 121.833747364 31.1462498659, 121.833747364 31.1459718107, 121.834029976 31.145971811, 121.834029977 31.1456937566, 121.834305752 31.1456937568, 121.834305752 31.1451399269, 121.834586086 31.145139927, 121.834586086 31.1448618717, 121.834845907 31.1448618724, 121.834845908 31.144581538, 121.835135358 31.1445838172, 121.835135358 31.1443057627, 121.83541797 31.1443103215, 121.83541797 31.1440299871, 121.835693746 31.1440299878, 121.835693746 31.1437519332, 121.835983196 31.1437519333, 121.835983196 31.1434715997, 121.836249855 31.1434715988, 121.836249855 31.1429154905, 121.836516514 31.1429200489, 121.836516514 31.1426419942, 121.836808243 31.142641994, 121.836808243 31.1423616604, 121.83708174 31.14236166, 121.83708174 31.1420813263, 121.837364352 31.1420836054, 121.837364352 31.1418055515, 121.837637848 31.1418101097, 121.837637849 31.141532055, 121.837906787 31.141532055, 121.837906786 31.1412517212, 121.838196236 31.1412517216, 121.838196236 31.140688775, 121.838469734 31.1406933332, 121.838469733 31.1404175573, 121.838752346 31.1404312319, 121.838752346 31.1401395031, 121.839028122 31.1401395027, 121.839028121 31.1398591688, 121.839301618 31.139859169, 121.839301617 31.1395788351, 121.83958423 31.1395833936, 121.83958423 31.1393053396, 121.839860006 31.1393098973, 121.839860006 31.1390295642, 121.840140339 31.1390295637, 121.84014034 31.1384688967, 121.840416114 31.1384711759, 121.840416115 31.1381953999, 121.840689611 31.1381999586, 121.840698728 31.1379196246, 121.840981341 31.1379196247, 121.84098134 31.1376392915, 121.841247999 31.1376392909, 121.841247999 31.1373589577, 121.841530612 31.1373612364, 121.841530613 31.1370831822, 121.841804108 31.1370831824, 121.841804109 31.136527073, 121.842086721 31.1365293523, 121.842086721 31.1362490181, 121.842362497 31.1362490186, 121.842362497 31.1359709634, 121.842635993 31.1359732426, 121.842635993 31.1356951882, 121.842918606 31.1356951885, 121.842918606 31.1354171341, 121.843192102 31.135419413, 121.843192102 31.1348610243, 121.843483831 31.1348610249, 121.843474715 31.1345829698, 121.84375049 31.1345829699, 121.84375049 31.1343049155, 121.844023987 31.1343049153, 121.844023986 31.1340268609, 121.844306599 31.1340291406, 121.8443066 31.1337510861, 121.844582375 31.1337510854, 121.844582375 31.1334707518, 121.844864987 31.1334730312, 121.844864988 31.1329169222, 121.845138484 31.1329214803, 121.845138484 31.1326411467, 121.84541426 31.1326411468, 121.845414259 31.1323608132, 121.84570371 31.1323608125, 121.845694593 31.1320827583, 121.845970368 31.1320827581, 121.845970369 31.1318047035, 121.846252982 31.1318115413, 121.846252981 31.1315312077, 121.846526477 31.1315312075, 121.846526477 31.1309705401, 121.846802253 31.1309728197, 121.846802253 31.130694765, 121.847075749 31.1307016024, 121.847075749 31.1304212687, 121.847358362 31.1304212689, 121.847358362 31.1301409351, 121.847640975 31.1301409346, 121.847640975 31.1298606009, 121.84791675 31.1298628799, 121.84791675 31.129582547, 121.848197084 31.1295825465, 121.848197084 31.1290287167, 121.84847286 31.1290309954, 121.848472859 31.1287506625, 121.848746355 31.1287506619, 121.848746355 31.128470328, 121.849028969 31.128470328, 121.849028968 31.128194553, 121.849295627 31.1282013904, 121.849295627 31.1279210564, 121.849587356 31.1279210563, 121.849587357 31.1276407233, 121.849860853 31.127640723, 121.849860854 31.127360389, 121.850136629 31.1273603896, 121.850136629 31.127241874, 121.850136629 31.1268065594, 121.850416962 31.1268111175, 121.850416962 31.1265307843, 121.850692737 31.1265307836, 121.850692738 31.1262504505, 121.850982188 31.1262504503, 121.850982188 31.1259701162, 121.851248846 31.1259723959, 121.851248847 31.1256943407, 121.851524623 31.125694341, 121.851524622 31.1254162867, 121.851807235 31.1254185658, 121.851807235 31.1251291156, 121.852080731 31.125138232, 121.852080731 31.1248601777, 121.852356507 31.1248601779, 121.852356506 31.1245798436, 121.852639119 31.1245844017, 121.85263912 31.1243063474, 121.852919454 31.1243086263, 121.852919453 31.1240305719, 121.853211182 31.1240305724, 121.853211182 31.1234699047, 121.853468725 31.1234721843, 121.853468725 31.1231941289, 121.853751338 31.1231941292, 121.853751338 31.1229160747, 121.854027113 31.1229206329, 121.854027113 31.1226402994, 121.854309726 31.1226402993, 121.854309726 31.1223599658, 121.854576384 31.1223599658, 121.854576385 31.1220705153, 121.854858998 31.1220841898, 121.854858998 31.1218061353, 121.85514161 31.121810694, 121.855141611 31.1215303604, 121.855421944 31.1215303601, 121.855421944 31.1212500265, 121.855697719 31.1212500266, 121.85569772 31.120969693, 121.855971216 31.120971972, 121.855971215 31.1204158627, 121.856246992 31.1204204214, 121.856246991 31.1201400877, 121.856529604 31.1201400872, 121.856529604 31.1198597534, 121.8568031 31.1198620334, 121.856803101 31.1195839786, 121.857085713 31.1195839786, 121.857085713 31.1190278691, 121.857361489 31.1190301481, 121.857361489 31.1187498143, 121.857644101 31.1187498149, 121.857634985 31.1184740394, 121.857917597 31.1184740393, 121.857917598 31.1181937055, 121.858193373 31.1181937055, 121.858193373 31.1179156516, 121.858466869 31.1179202096, 121.858466869 31.1173618208, 121.858749483 31.1173618215, 121.858749482 31.1170814876, 121.859022979 31.1170814872, 121.859022978 31.1168057122, 121.859305591 31.1168102706, 121.859305591 31.1165299366, 121.859581367 31.1165299363, 121.859581366 31.1162609983, 121.85986398 31.1162609983, 121.859863979 31.1159715482, 121.860137476 31.1159715482, 121.860137475 31.1156934941, 121.860420088 31.1156934938, 121.860420088 31.1154177186, 121.860686748 31.1154199973, 121.860686747 31.114861609, 121.86096936 31.1148616092, 121.86096936 31.114581276, 121.861251973 31.1145835546, 121.861251973 31.1143055004, 121.86152547 31.114312338, 121.861525469 31.1140297248, 121.861808082 31.114029725, 121.861808082 31.1137493908, 121.862074741 31.1137493911, 121.862074741 31.1134690577, 121.862357353 31.1134713366, 121.862357353 31.1129175069, 121.862074741 31.1129197861, 121.862074741 31.1126394518, 121.861808082 31.1126394524, 121.861808082 31.1123591182, 121.86152547 31.112361398, 121.861525469 31.1120833427, 121.861251974 31.1120833432, 121.861251973 31.1118052888, 121.860686747 31.1118098463, 121.860686747 31.1115295129, 121.85986398 31.1115295133, 121.859863979 31.111249179, 121.858749482 31.1112491793, 121.858749482 31.110968846, 121.857917598 31.1109711252, 121.857917598 31.1106953491, 121.857361488 31.110695349, 121.857361488 31.1104172947, 121.856796263 31.1104195743, 121.856796263 31.1101301241, 121.856246991 31.110136961, 121.856246991 31.1098611858, 121.85569772 31.1098611861, 121.855697719 31.109583131, 121.855415107 31.1095831311, 121.855415107 31.1090270218, 121.855697719 31.1090293016, 121.85569772 31.1087489674, 121.855980332 31.1087489675, 121.855971216 31.1084731917, 121.856246992 31.1084731918, 121.856246992 31.1081951375, 121.8568031 31.1081951377, 121.856803101 31.1076390282, 121.857085714 31.1076413075, 121.857085714 31.1070897568, 121.857361488 31.1070897563, 121.857361489 31.1065313685, 121.857644101 31.1065313685, 121.857644101 31.1059707007, 121.857917597 31.1059729803, 121.857917597 31.1056949259, 121.858473707 31.1056949258, 121.858473706 31.1054168713, 121.859016141 31.1054214294, 121.859016141 31.1043206072, 121.858749482 31.1043206073, 121.858749482 31.1037508231, 121.859016141 31.103750823, 121.859016141 31.1032015509, 121.859305591 31.1032015516, 121.859305591 31.102921218, 121.859581367 31.102921218, 121.859581367 31.1026408843, 121.85986398 31.1026408837, 121.85986398 31.10236055, 121.860137476 31.1023605503, 121.860137476 31.1004164479, 121.860420088 31.1004210058, 121.860420089 31.100140672, 121.860686747 31.1001406719, 121.860686748 31.0993087873, 121.86096936 31.0993087877, 121.86096936 31.098750399, 121.861251973 31.0987503997, 121.861251973 31.0981897318, 121.861525469 31.0981942906, 121.86152547 31.0976381816, 121.861808083 31.09764046, 121.861808082 31.0970797929, 121.862083858 31.0970843516, 121.862083858 31.0962501875, 121.862357354 31.0962501873, 121.862357354 31.0954205819, 121.862074742 31.0954205819, 121.862074741 31.0951493648, 121.862357354 31.0951493647, 121.862357354 31.0943106431, 121.862639967 31.0943106433, 121.862639966 31.0940303091, 121.86292258 31.0940303096, 121.86292258 31.0931915877, 121.863198355 31.0931938667, 121.863198355 31.0929158124, 121.863471852 31.0929203704, 121.863471851 31.092640037, 121.863747627 31.0926400366, 121.863747626 31.0923597032, 121.86402796 31.0923619818, 121.86402796 31.0920839274, 121.864586348 31.092090765, 121.864586349 31.0912497636, 121.864859845 31.0912497642, 121.864859845 31.0895814363, 121.865418233 31.0895814365, 121.865418233 31.089305661, 121.86569173 31.0893102196, 121.865691729 31.0890207689, 121.865974342 31.0890276067, 121.865974342 31.0887518311, 121.866250118 31.0887518316, 121.866250118 31.0884714979, 121.866530451 31.0884714972, 121.866530451 31.0881934434, 121.866806226 31.0881934426, 121.866806226 31.0879176678, 121.867638111 31.0879222257, 121.867638112 31.0873592791, 121.867920724 31.0873592791, 121.867920724 31.0868100075, 121.868187383 31.0868100075, 121.868187383 31.0859598898, 121.868469996 31.0859712855, 121.868469996 31.0854174557, 121.868752609 31.0854197346, 121.868752608 31.0845787337, 121.868752609 31.0834710737, 121.869026105 31.0834710741, 121.869026105 31.0831952982, 121.869308718 31.0831952985, 121.869308717 31.0826391894, 121.869582214 31.0826391895, 121.869582214 31.0823611344, 121.869857989 31.082361135, 121.86985799 31.0809708622, 121.870140602 31.0809731414, 121.870140602 31.079304814, 121.870407261 31.0793093726, 121.870407261 31.0784706506, 121.870696711 31.0784729298, 121.870696711 31.0776387658, 121.870956532 31.0776410447, 121.870956532 31.0770803778, 121.871245983 31.0770826566, 121.871245983 31.0768046021, 121.871528596 31.0768114398, 121.871528595 31.0765311054, 121.87236048 31.0765311056, 121.87236048 31.0757015001, 121.872643093 31.0757015, 121.872643093 31.0754211664, 121.872916589 31.0754211663, 121.872916589 31.074860499, 121.873192365 31.0748604993, 121.873192365 31.0743066691, 121.873465861 31.0742998316, 121.873474978 31.0731944511, 121.873748474 31.0731944509, 121.873748474 31.0709722936, 121.874031086 31.0709722939, 121.874031086 31.070694239, 121.874306862 31.0706942395, 121.874306862 31.0704184636, 121.874580358 31.0704207433, 121.874580358 31.0698600755, 121.874862971 31.0698600757, 121.874862971 31.0693108039, 121.875136467 31.0693108042, 121.875136468 31.0682099813, 121.87541908 31.0682099811, 121.87541908 31.0673598637, 121.875694855 31.0673621427, 121.875694856 31.0656938158, 121.875968352 31.065693816, 121.875968353 31.0651399857, 121.876250965 31.0651399854, 121.876250965 31.0640300463, 121.877638958 31.064030047, 121.877638959 31.0637519926, 121.877907897 31.0637519923, 121.877907896 31.0634716588, 121.878197347 31.0634716586, 121.878197347 31.0631936041, 121.878470843 31.0631936044, 121.878470843 31.0623617191, 121.878746619 31.0623617196, 121.878746618 31.0615321141, 121.879029231 31.0615321138, 121.879029232 31.0612517803, 121.879592178 31.0612517806, 121.87958534 31.0606933925, 121.88014145 31.0606933923, 121.880141449 31.0604176167, 121.880417225 31.0604198962, 121.880417225 31.0601395625, 121.880690721 31.0601395624, 121.880690721 31.0590205065, 121.880417225 31.0590296228, 121.880417224 31.058749289, 121.880690721 31.0587492898, 121.880690721 31.055419472, 121.880966497 31.0554194719, 121.880966497 31.054861084, 121.88124911 31.0548610839, 121.881249109 31.0543095331, 121.881531722 31.0543095333, 121.881531722 31.0537511452, 121.881812056 31.0537511449, 121.881812056 31.0534708108, 121.882080994 31.0534730903, 121.882080994 31.0531950362, 121.882637103 31.0531950354, 121.882637103 31.0529169813, 121.882919716 31.0529192601, 121.882919716 31.052641206, 121.883186374 31.052641206, 121.883186375 31.0523608718, 121.883468988 31.0523608722, 121.883468987 31.0520805381, 121.883751601 31.0520850968, 121.883751601 31.0512509331, 121.884307709 31.0512509331, 121.884307709 31.0509705997, 121.884583485 31.0509728789, 121.884583485 31.0506925446, 121.884863818 31.0506925447, 121.884863818 31.0504167694, 121.885139594 31.0504213273, 121.885139594 31.0498515434, 121.88541309 31.0498629397, 121.88541309 31.049304551, 121.885695703 31.0493113889, 121.885695703 31.048750721, 121.885955524 31.048750721, 121.885955525 31.0481900539, 121.886254091 31.0481946122, 121.886254091 31.0473604486, 121.886803363 31.0473604488, 121.886803363 31.0468066186, 121.887076859 31.0468111769, 121.887076859 31.0465308433, 121.887359472 31.046530843, 121.887359472 31.0459701756, 121.887642085 31.0459724546, 121.887642085 31.0448602368, 121.888191357 31.0448602371, 121.888191357 31.0443086857, 121.888747465 31.0443086858, 121.888747466 31.0429206925, 121.889030078 31.0429206921, 121.889030079 31.0412500855, 121.889319529 31.0412500854, 121.889305854 31.0409720309, 121.88957935 31.0409811481, 121.889579351 31.0407008141, 121.889861963 31.0407008143, 121.889861963 31.0401401472, 121.890137739 31.0401401466, 121.890137739 31.0398598135, 121.890693848 31.0398620919, 121.890693848 31.0395840377, 121.890418072 31.0395840376, 121.890418072 31.0393037035, 121.890693847 31.0393037036, 121.890693848 31.0390279285, 121.89097646 31.039030208, 121.890976461 31.0387498738, 121.891249957 31.0387498738, 121.891249956 31.0381914854, 121.891525733 31.0381937645, 121.891525732 31.0379157102, 121.891808345 31.0379157107, 121.891808345 31.0376399345, 121.892357617 31.0376399346, 121.892357617 31.0373618802, 121.89264023 31.0373618806, 121.89264023 31.0365299957, 121.892357616 31.0365299962, 121.892357616 31.0351420024, 121.892640229 31.0351420027, 121.89264023 31.0345813348, 121.892913726 31.0345836137, 121.892913726 31.0340275048, 121.893196338 31.0340297837, 121.893196339 31.0334691166, 121.893469835 31.0334713961, 121.893469834 31.0329175661, 121.893752448 31.0329198447, 121.893752448 31.032639511, 121.894028223 31.0326395109, 121.894028223 31.0323591772, 121.894301719 31.0323614567, 121.894301719 31.0315295719, 121.894577494 31.0315295723, 121.894577495 31.0312492386, 121.894860108 31.0312492383, 121.894860108 31.0304196332, 121.895126766 31.0304196328, 121.895126766 31.0298589661, 121.895416216 31.0298612453, 121.895416216 31.0293074148, 121.895691992 31.0293096945, 121.895691992 31.0281906389, 121.895972326 31.0281951963, 121.895972325 31.0276390875, 121.896248101 31.0276390878, 121.896248101 31.0273610329, 121.896530714 31.0273610333, 121.896530714 31.0268094824, 121.896797372 31.0268094821, 121.896797372 31.0259707608, 121.897086824 31.0259730397, 121.897086823 31.0256949847, 121.897362599 31.0256949852, 121.897362599 31.0254169302, 121.897918708 31.0254214886, 121.897918708 31.0251411554, 121.898483933 31.0251411553, 121.898483934 31.0240312156, 121.898750592 31.0240312156, 121.898750592 31.023470549, 121.899024088 31.023472828, 121.899024088 31.0229167185, 121.899582476 31.0229167182, 121.899582477 31.0220825551, 121.89986509 31.0220825553, 121.89986509 31.0218044999, 121.900138586 31.0218045002, 121.900138586 31.0215287249, 121.900405244 31.021531004, 121.900405244 31.0206900034, 121.900694695 31.0206945612, 121.900694695 31.0198603977, 121.900970471 31.0198717937, 121.90097047 31.0184701246, 121.901253083 31.0184724042, 121.901253084 31.0176382403, 121.901526579 31.0176405192, 121.901526579 31.0170798518, 121.901802355 31.0170844105, 121.901802355 31.0165283011, 121.902075851 31.0165305806, 121.902075851 31.0154206416, 121.902358464 31.0154206414, 121.902358463 31.0148599736, 121.902641076 31.0148599737, 121.902641077 31.0145796407, 121.902916852 31.0145841987, 121.902916852 31.0140280899, 121.903190348 31.0140303688, 121.903197186 31.0134697012, 121.903472962 31.0134719808, 121.903472961 31.0129158709, 121.903746458 31.0129204294, 121.903746458 31.0123597623, 121.904029071 31.0123620418, 121.90402907 31.0112498236, 121.903746457 31.0112498231, 121.903746457 31.010420218, 121.903472961 31.0104202177, 121.903472961 31.0098618294, 121.902916853 31.0098618292, 121.902916852 31.009310279, 121.902641076 31.0093102784, 121.902641077 31.0087518901, 121.903197186 31.0087518906, 121.903197186 31.0090299448, 121.904587459 31.009029945, 121.904587458 31.0087518908, 121.904860955 31.0087518904, 121.904860955 31.0084715571, 121.905692839 31.008471557, 121.905692839 31.0081935018, 121.907912718 31.0081935026, 121.907912717 31.0079177273, 121.908471106 31.0079314013, 121.908471105 31.0073593389, 121.90819533 31.0073593385, 121.90819533 31.0070835631, 121.907912718 31.007083563, 121.907912718 31.0062493995, 121.907639221 31.0062493992, 121.907639221 31.0059690657, 121.907363446 31.0059713454, 121.907363446 31.0056955699, 121.906807336 31.0057001278, 121.906807337 31.0054197942, 121.906248948 31.0054197939, 121.906248949 31.0048705221, 121.905982289 31.0048705222, 121.90598229 31.0037491879, 121.907363445 31.0037491881, 121.907356608 31.0034688543, 121.908195331 31.0034711329, 121.90819533 31.0031953574, 121.908471106 31.0031953575, 121.908471106 31.0026392482, 121.908751439 31.0026392484, 121.90875144 31.0023589146, 121.909027214 31.0023611945, 121.909027215 31.0020831398, 121.909300711 31.0020831396, 121.909300711 31.0015270301, 121.909576486 31.0015293096, 121.909576486 30.9998609826, 121.909859099 30.9998609824, 121.909859099 30.9993094317, 121.910132595 30.9993094315, 121.910132596 30.9990313766, 121.910424325 30.9990313771, 121.910424325 30.9984707091, 121.910697821 30.9984729886, 121.910697822 30.9979168795, 121.910973596 30.9979214377, 121.910973597 30.9970804364, 121.911247093 30.9970827155, 121.911247093 30.9962508312, 121.911529706 30.9962508317, 121.911529705 30.9957015596, 121.911796364 30.9957015595, 121.911796364 30.9948605586, 121.91236159 30.9948605583, 121.91236159 30.9943044497, 121.912635086 30.9943044494, 121.912635087 30.9937506198, 121.91236159 30.9937506192, 121.91236159 30.9929301307, 121.912635086 30.9929301303, 121.912635086 30.9923603471, 121.912917699 30.9923603471, 121.912917699 30.9918087964, 121.913749584 30.9918087964, 121.913749584 30.9915284619, 121.914305693 30.9915284627, 121.914305693 30.9912504082, 121.915137577 30.9912504082, 121.915137578 30.9904185236, 121.914854965 30.9904185231, 121.914854965 30.9893108629, 121.914581468 30.9893108636, 121.914581468 30.9887501963, 121.915413353 30.988750196, 121.915413353 30.9881895285, 121.915969462 30.9881940871, 121.915969462 30.9876379777, 121.916258912 30.987640257, 121.916258912 30.9868015345, 121.915969462 30.9868060934, 121.915969462 30.98597193, 121.916252075 30.9859719296, 121.916252074 30.9854158208, 121.91652785 30.9854203784, 121.916527851 30.9848597114, 121.917083959 30.9848619904, 121.917083959 30.9845839355, 121.917357456 30.9845839356, 121.917357455 30.9843058815, 121.917922681 30.9843104393, 121.917922682 30.9837611675, 121.917640069 30.9837611681, 121.917640068 30.9829201668, 121.917922681 30.9829201671, 121.917922682 30.9812518395, 121.918198456 30.9812518397, 121.918198456 30.9804199552, 121.918471953 30.9804199555, 121.918471952 30.9801396212, 121.918747728 30.980139621, 121.918747728 30.9795789543, 121.919030341 30.9795835121, 121.919030342 30.9793054578, 121.919297 30.9793100158, 121.919297 30.978749349, 121.91958645 30.9787493486, 121.91958645 30.9782000771, 121.919859946 30.978200077, 121.919859947 30.9776394091, 121.920142559 30.9776394098, 121.920142559 30.9765294702, 121.920418335 30.9765294708, 121.920418335 30.9756907482, 121.920691831 30.9756953071, 121.920691832 30.9751391981, 121.920967607 30.9751391974, 121.920967607 30.9745716928, 121.92125022 30.9745830891, 121.92125022 30.9740269798, 121.921516878 30.9740292585, 121.921516878 30.9734708701, 121.921806328 30.9734731499, 121.921806328 30.9729170405, 121.922072988 30.9729193198, 121.922072987 30.9723609313, 121.922362438 30.9723609316, 121.922362438 30.9720805978, 121.922638213 30.9720851561, 121.922638213 30.9712509927, 121.922362437 30.9712509925, 121.922362438 30.9706903248, 121.922638213 30.970692604, 121.922638213 30.9701387743, 121.922920826 30.970141053, 121.922920826 30.96986072, 121.923194322 30.9698629988, 121.923194323 30.9693046108, 121.923470097 30.9693114482, 121.923470098 30.968190113, 121.923194322 30.9681946711, 121.923194322 30.9679166171, 121.923470098 30.9679211753, 121.923470097 30.967360508, 121.92375271 30.9673605075, 121.92375271 30.9668066775, 121.924026207 30.9668112362, 121.924026207 30.9659702347, 121.924308819 30.9659725138, 121.924308819 30.9654164054, 121.924575478 30.9654186837, 121.924575478 30.9648602962, 121.924858091 30.9648602959, 121.924858091 30.9643087455, 121.924575478 30.9643087449, 121.924575478 30.9640306907, 121.924858091 30.9640306903, 121.924858091 30.9631896892, 121.925140704 30.9631942482, 121.925140704 30.9612501449, 121.924858091 30.961250145, 121.924858091 30.9587499331, 121.924575478 30.9587499328, 121.924575479 30.9584695992, 121.924858091 30.9584718786, 121.924858091 30.9579157695, 121.925696813 30.9579203276, 121.925696813 30.9576399939, 121.925972588 30.9576399944, 121.925972589 30.9570816061, 121.926246085 30.9570838852, 121.926246085 30.9568058305, 121.926528697 30.9568195053, 121.926528698 30.9562520005, 121.926795357 30.9562520001, 121.926795356 30.9556913334, 121.927084807 30.9556936123, 121.927084807 30.9554155576, 121.927360582 30.9554201159, 121.927360582 30.9551420612, 121.927084807 30.9551420619, 121.927084806 30.9529199044, 121.927360582 30.9529199045, 121.927360582 30.9512492978, 121.927643195 30.951249298, 121.927634078 30.9504174133, 121.927360582 30.9504196921, 121.927360582 30.9495809707, 121.927084807 30.9495832499, 121.927084806 30.9487490858, 121.927360582 30.9487490857, 121.927360582 30.9484710315, 121.927634078 30.9484733109, 121.927634078 30.9481952558, 121.927916691 30.9481952562, 121.927916691 30.9476391469, 121.928192467 30.9476391468, 121.928192467 30.9470830375, 121.928475079 30.9470830375, 121.928475079 30.9462511529, 121.928748575 30.9462511534, 121.928748575 30.9456904857, 121.929031188 30.9456950444, 121.929031188 30.9420826145, 121.929306964 30.9420826144, 121.929306964 30.94125073, 121.929580461 30.9412507294, 121.92958046 30.940421124, 121.929863073 30.9404211243, 121.929863073 30.9395801232, 121.92958046 30.939582402, 121.92958046 30.9384724635, 121.929863073 30.9384724637, 121.929863073 30.9379163544, 121.93013657 30.9379186334, 121.930136569 30.9373602448, 121.929031189 30.9373602448, 121.929031189 30.9368086946, 121.928748575 30.9368086944, 121.928748575 30.9354207008, 121.929031189 30.9354207009, 121.929031189 30.9343107618, 121.929306964 30.9343107615, 121.929306964 30.9332008221, 121.92958046 30.9332008221, 121.929580461 30.932920489, 121.930136569 30.9329204887, 121.930136569 30.932649272, 121.930419182 30.9326492716, 121.930419182 30.9323598212, 121.930694958 30.9323621005, 121.930694957 30.9318037124, 121.930419182 30.9318037122, 121.930419182 30.9306937731, 121.930136569 30.9306937736, 121.93013657 30.9295815553, 121.929863073 30.9295906712, 121.929863073 30.9270790644, 121.92958046 30.927083622, 121.92958046 30.9262494584, 121.928465963 30.9262494585, 121.928465962 30.9256887909, 121.928748576 30.9256933494, 121.928748575 30.9254175743, 121.929031188 30.925419853, 121.929031188 30.9251395197, 121.92958046 30.9251395195, 121.92958046 30.9248591861, 121.929863073 30.9248614648, 121.929863073 30.922639308, 121.92958046 30.9226393076, 121.92958046 30.9215293689, 121.929306963 30.9215293685, 121.929306964 30.9206906469, 121.929031188 30.9206952052, 121.929031188 30.9198610417, 121.928748576 30.9198610414, 121.928748575 30.9187511025, 121.928465963 30.9187511025, 121.928465963 30.9176502798, 121.928199304 30.9176502798, 121.928199304 30.9162508904, 121.927916691 30.9162508902, 121.927916691 30.9154212855, 121.927643195 30.9154212854, 121.927643195 30.9151409517, 121.927360582 30.9151409513, 121.927360582 30.9148606175, 121.927084806 30.9148606182, 121.927084807 30.9131945693, 121.926804473 30.9131945695, 121.926804473 30.9112481883, 121.926528697 30.911250467, 121.926528698 30.9106897993, 121.926246084 30.9106943575, 121.926246085 30.9101382488, 121.925972589 30.9101405282, 121.925972589 30.9093086429, 121.925696813 30.9093086436, 121.925696813 30.9087502548, 121.925972589 30.9087502551, 121.925972589 30.9081895881, 121.926246085 30.908194146, 121.926246085 30.9079160911, 121.926528698 30.9079206501, 121.926528697 30.9076403161, 121.926246084 30.9076403162, 121.926246084 30.9073599822, 121.925972589 30.9073622617, 121.925972588 30.9070842068, 121.925696813 30.9070842066, 121.925696813 30.9068061527, 121.925407363 30.9068107111, 121.925407363 30.9065303771, 121.925140704 30.9065303766, 121.925140703 30.9062500435, 121.924858091 30.9062500434, 121.924858091 30.9051401047, 121.924575478 30.905140104, 121.924575479 30.9009715652, 121.924308819 30.9009715651, 121.924308819 30.9004177349, 121.924026206 30.9004200141, 121.924026207 30.9001396808, 121.92375271 30.9001396804, 121.92375271 30.8998593471, 121.923470098 30.899859347, 121.923470097 30.899310075, 121.923187485 30.8993100752, 121.923187485 30.8990297418, 121.922920826 30.8990297416, 121.922920826 30.8982001363, 121.922638213 30.8982001362, 121.922638213 30.8979198028, 121.922362437 30.8979198022, 121.922362438 30.8956908081, 121.921523716 30.8956953665, 121.921523715 30.895139257, 121.920974444 30.8951392571, 121.920974444 30.8956953665, 121.920418334 30.8956953659, 121.920418334 30.8954173116, 121.92014256 30.8954195909, 121.920142559 30.8948612024, 121.919859946 30.8948612027, 121.919859946 30.8945808685, 121.919586451 30.8945831476, 121.919586451 30.8943050934, 121.919297 30.8943096514, 121.919297 30.8937512638, 121.919030342 30.8937512638, 121.919030341 30.8931905962, 121.918747728 30.8931951541, 121.918747728 30.8920852152, 121.918471953 30.8920852156, 121.918471953 30.891526827, 121.918198457 30.8915313852, 121.918198457 30.8912510517, 121.917915844 30.8912510516, 121.917915844 30.8906926629, 121.918198457 30.8906926631, 121.918198457 30.8901388335, 121.918471953 30.8901411126, 121.918471953 30.8893115067, 121.917357456 30.8893115068, 121.917357455 30.8887508399, 121.917077122 30.8887508399, 121.917077121 30.8873491714, 121.916808184 30.8873605671, 121.916808184 30.8862506276, 121.916527851 30.8862506283, 121.91652785 30.8856899603, 121.916252075 30.8856945189, 121.916252074 30.8854164645, 121.915969462 30.885418743, 121.915969462 30.884860355, 121.915695966 30.8848603553, 121.915695966 30.8845823008, 121.915413353 30.8845823009, 121.915413353 30.8843065255, 121.915137577 30.8843088046, 121.915137578 30.8837504165, 121.915413353 30.8837504157, 121.915413352 30.8831897484, 121.915137578 30.8831943066, 121.915137578 30.8826381976, 121.914854964 30.8826404774, 121.914854965 30.8820798101, 121.914581469 30.8820843682, 121.914581468 30.88152598, 121.914305693 30.8815396548, 121.914305693 30.8812502047, 121.914016243 30.881250204, 121.914016243 30.8809698704, 121.913476088 30.8809721495, 121.913476087 30.880696374, 121.913200312 30.8807009327, 121.913200312 30.8804205991, 121.912635087 30.8804205989, 121.912635087 30.8801402652, 121.91236159 30.8801402651, 121.91236159 30.8798599314, 121.912085814 30.8798622105, 121.912085815 30.8790280469, 121.911796365 30.8790303259, 121.911796364 30.8773597193, 121.911529706 30.877361999, 121.911529706 30.8765278352, 121.911247093 30.8765301146, 121.911247093 30.8759717263, 121.910973597 30.875971726, 121.910973596 30.8754156167, 121.910697821 30.8754156166, 121.910697822 30.8748617873, 121.910415208 30.8748617868, 121.910415209 30.8743056775, 121.910132595 30.8743102364, 121.910132596 30.8737495689, 121.909859099 30.8737495691, 121.909859099 30.8731889015, 121.909583324 30.87319346, 121.909583324 30.8726396297, 121.909300711 30.8726396296, 121.909300711 30.8715296907, 121.909576487 30.871529691, 121.909576486 30.8712493571, 121.9098591 30.8712493575, 121.909859099 30.8709690235, 121.910141712 30.8709713024, 121.910141712 30.8706955267, 121.910424325 30.8707000849, 121.910424325 30.8701394179, 121.910697821 30.8701394182, 121.910697821 30.8698590842, 121.910973596 30.8698613632, 121.910973597 30.8693052543, 121.911247093 30.8693098124, 121.911247093 30.8690294793, 121.911529706 30.8690294786, 121.911529705 30.8684710905, 121.911803202 30.8684733701, 121.911803202 30.8681953152, 121.912085815 30.868195315, 121.912085815 30.8676392059, 121.91236159 30.8676392063, 121.91236159 30.8673611513, 121.912635087 30.8673611518, 121.912635087 30.8668096008, 121.9129177 30.8668096006, 121.912917699 30.8665292673, 121.913200313 30.8665292674, 121.913200312 30.8659708789, 121.913476088 30.8659731578, 121.913476088 30.8656951035, 121.913749584 30.8656951034, 121.913749584 30.8651389939, 121.914016243 30.8651412737, 121.914016243 30.8648609393, 121.914305693 30.8648609401, 121.914305693 30.8643116679, 121.914581468 30.8643116679, 121.914581468 30.8640313344, 121.914854965 30.8640313346, 121.914854965 30.8634706667, 121.915137577 30.863472946, 121.915137578 30.8631926125, 121.915413352 30.8631926125, 121.915413353 30.8629168372, 121.915702803 30.8629213957, 121.915702803 30.8623607276, 121.915969462 30.8623630068, 121.915969462 30.8620826731, 121.916252075 30.8620826736, 121.916252075 30.861804619, 121.91652785 30.8618114567, 121.916527851 30.8612507893, 121.916808184 30.8612507892, 121.916808184 30.8609704555, 121.917083959 30.8609704555, 121.917083959 30.8604166254, 121.917357456 30.8604303003, 121.917357455 30.8598605162, 121.917640069 30.8598605162, 121.917640068 30.8595824615, 121.917083959 30.8595824614, 121.917083959 30.8593066858, 121.916258912 30.8593112445, 121.916258912 30.8590286316, 121.915702804 30.8590286319, 121.915702803 30.8587482981, 121.914854964 30.8587482981, 121.914854964 30.8584702435, 121.914305693 30.8584725229, 121.914305693 30.8581944682, 121.913476087 30.8581944679, 121.913476087 30.8579164133, 121.912917699 30.8579186924, 121.9129177 30.8576406378, 121.912361591 30.8576406383, 121.91236159 30.8573694211, 121.911796365 30.8573694211, 121.911796364 30.857079971, 121.911529706 30.8570845289, 121.911529706 30.8568064744, 121.910973597 30.8568087535, 121.910973597 30.856530699, 121.910424325 30.8565306991, 121.910424324 30.8562503655, 121.909859099 30.8562503652, 121.909859099 30.8559700316, 121.909300711 30.8559723106, 121.909300711 30.8556942561, 121.909027215 30.8556942563, 121.909027215 30.8554162018, 121.908471105 30.8554207602, 121.908471106 30.8551404257, 121.908202168 30.8551404265, 121.908202167 30.8548600929, 121.907639221 30.8548600924, 121.907639221 30.8545797588, 121.905975452 30.854584317, 121.905975452 30.8543062626, 121.90513673 30.8543108204, 121.90513673 30.8540304869, 121.904298008 30.8540304873, 121.904298008 30.853750153, 121.903197186 30.8537501531, 121.903190349 30.8534698195, 121.902358464 30.8534720985, 121.902358464 30.8531963233, 121.901526579 30.8532008815, 121.90152658 30.8529205481, 121.90097047 30.8529205479, 121.900970471 30.8526402146, 121.900138586 30.8526402147, 121.900138586 30.8523598805, 121.899582476 30.8523621598, 121.899582476 30.8520841056, 121.899024088 30.8520841057, 121.899024088 30.8518037715, 121.898194483 30.8518037714, 121.898194483 30.8515279964, 121.897636095 30.8515302751, 121.897636095 30.8512408253, 121.897086824 30.8512499417, 121.897086823 30.8509741658, 121.896530714 30.8509741665, 121.896530714 30.8506938324, 121.895691992 30.8506938326, 121.895691992 30.8504157776, 121.895126767 30.8504203358, 121.895126766 30.8501400026, 121.89430172 30.8501400023, 121.89430172 30.8498619482, 121.893469835 30.8498619483, 121.893469835 30.8495816142, 121.892640229 30.8495816141, 121.892640229 30.8493058392, 121.891808345 30.8493103968, 121.891808345 30.8490300637, 121.89097646 30.8490300632, 121.89097646 30.8487520092, 121.89068701 30.8487520093, 121.89068701 30.8481799458, 121.890418072 30.8481936207, 121.890418073 30.8479178449, 121.886803363 30.8479201245, 121.886803363 30.8476420698, 121.883186375 30.8476420698, 121.883186374 30.8479201244, 121.881249109 30.8479178452, 121.881249109 30.8481936206, 121.879302727 30.8481936209, 121.879302727 30.8484716752, 121.878470843 30.8484716756, 121.878470843 30.8487520091, 121.876250965 30.8487520086, 121.876250965 30.8484716753, 121.873192364 30.8484716756, 121.873192365 30.8481936207, 121.869026105 30.848193621, 121.869026104 30.8484716749, 121.867638111 30.8484716756, 121.867638111 30.8487520085, 121.866530451 30.8487520087, 121.866530451 30.8490300633, 121.865151574 30.8490300638, 121.865151574 30.8493103975, 121.863471851 30.8493103974, 121.863471851 30.8487520093, 121.862639966 30.8487520089, 121.862639967 30.849310397, 121.859863979 30.8493058386, 121.859863979 30.8495816147, 121.854027113 30.8495816139, 121.854027113 30.8498619479, 121.853751337 30.8498619476, 121.853751337 30.8495816145, 121.853468725 30.8495816144, 121.853468725 30.8493058385, 121.85263912 30.8493103974, 121.85263912 30.849861948, 121.851517784 30.8498619483, 121.851524622 30.8501400024, 121.84791675 30.850140003, 121.84791675 30.8506938322, 121.847640974 30.8506938324, 121.847640975 30.8523621598, 121.847358362 30.8523598807, 121.847358362 30.8540304868, 121.847084866 30.8540282078, 121.847084866 30.8556942561, 121.847358363 30.8556942561, 121.847358362 30.8559723105, 121.84707575 30.8559700314, 121.84707575 30.8562503649, 121.846526478 30.8562503655, 121.846526477 30.8559700311, 121.846252981 30.8559723103, 121.846252981 30.855694256, 121.845970368 30.8556942564, 121.845970368 30.8548600924, 121.845694593 30.8548600927, 121.845694593 30.8543062622, 121.844864988 30.8543062628, 121.844864988 30.8545843172, 121.844582374 30.8545797591, 121.844582375 30.8551404259, 121.844306599 30.8551404259, 121.844306599 30.8554207602, 121.843750491 30.8554207602, 121.84375049 30.855140426, 121.843483832 30.8551404263, 121.843483831 30.8540304874, 121.843192102 30.854030487, 121.843192102 30.8534698193, 121.843474715 30.8534720986, 121.843474715 30.8518037717, 121.843750491 30.8518037714, 121.84375049 30.8501400022, 121.844017149 30.8501400029, 121.84401715 30.8454313081, 121.84375049 30.8454313085, 121.84375049 30.8440296396, 121.843483832 30.8440296395, 121.843483832 30.8432000345, 121.842635993 30.8431954764, 121.842635993 30.843471252, 121.84098134 30.8434689725, 121.840981341 30.8443099735, 121.841530612 30.8443054155, 121.841530612 30.8445834701, 121.842635993 30.8445789115, 121.842635993 30.8451395789, 121.842918606 30.845139579, 121.842918606 30.8468101856, 121.842362497 30.846810185, 121.842362497 30.8479201248, 121.843192102 30.8479178449, 121.843192102 30.8481936203, 121.842918606 30.8481799455, 121.842918606 30.849310397, 121.842635993 30.8493058386, 121.842635993 30.8506938324, 121.842362497 30.8506915536, 121.842362497 30.8529205481, 121.842086721 30.8529205481, 121.842086722 30.8540304868, 121.841797271 30.8540304874, 121.841797271 30.8543108207, 121.840972224 30.8543062627, 121.84098134 30.8548600923, 121.840698728 30.854860092, 121.840698728 30.8551404262, 121.840140339 30.8551404264, 121.840140339 30.8554207595, 121.839301618 30.8554207599, 121.839301618 30.8551404259, 121.837637848 30.8551404259, 121.837637848 30.8554207599, 121.834861862 30.8554162019, 121.834861861 30.8556942557, 121.834296636 30.8556896975, 121.834296636 30.8559540777, 121.834296636 30.8562503651, 121.834029977 30.8562503648, 121.834029977 30.8570845287, 121.833473868 30.8570799704, 121.833473867 30.8573694204, 121.833191255 30.8573694208, 121.833191255 30.8570799708, 121.832915479 30.857084529, 121.832915479 30.8568064744, 121.832359371 30.8568087532, 121.83235937 30.8565306986, 121.832076758 30.8565306993, 121.832076757 30.8562503657, 121.831527486 30.8562503648, 121.831527485 30.8565306993, 121.83125399 30.8565284202, 121.831253989 30.8570845292, 121.830971377 30.8570799711, 121.830971376 30.857369421, 121.830695601 30.8573694208, 121.830695602 30.8570799709, 121.830139492 30.8570799709, 121.830139492 30.8573694208, 121.829581104 30.8573694212, 121.829581104 30.8570799705, 121.829307607 30.857084529, 121.829307607 30.8568064746, 121.828751499 30.8568087531, 121.828751498 30.8565306988, 121.827919614 30.8565306988, 121.827919614 30.8568087532, 121.827361226 30.8568087531, 121.827361226 30.8576406378, 121.827080893 30.8576383592, 121.827080892 30.8581944677, 121.826805117 30.8581944682, 121.826805116 30.8584725224, 121.822356244 30.8584702437, 121.822356243 30.8587482977, 121.821806972 30.8587482978, 121.821806972 30.8584702429, 121.819304481 30.8584725229, 121.819313598 30.8587482978, 121.815972385 30.8587482976, 121.815972385 30.8590286313, 121.814857888 30.8590286318, 121.814857888 30.8587482982, 121.814308616 30.858748298, 121.814308616 30.8584702435, 121.813750227 30.8584725223, 121.813750228 30.8581944678, 121.812920622 30.8581944679, 121.812920622 30.8579164134, 121.812638009 30.8579186921, 121.81263801 30.8576406377, 121.811247737 30.8576406376, 121.811247737 30.8573694204, 121.810142356 30.857369421, 121.810142356 30.8570799703, 121.808745246 30.857084529, 121.808745246 30.8568064747, 121.808195974 30.8568087531, 121.808195974 30.8565306988, 121.807639865 30.8565306987, 121.807639865 30.8562503654, 121.807357252 30.8562503653, 121.807357252 30.8559700319, 121.80680798 30.8559723108, 121.80680798 30.8556942566, 121.806251871 30.855694256, 121.806251871 30.8554162017, 121.805686645 30.8554207595, 121.805686645 30.8551404262, 121.805419987 30.8551404266, 121.805419987 30.8548600923, 121.804863878 30.8548600922, 121.804863878 30.8545843171, 121.80430549 30.854584317, 121.804305489 30.8543062628, 121.800697618 30.8543108208, 121.800697618 30.8540304867, 121.800421842 30.8540304872, 121.800421842 30.8543108212, 121.798751236 30.8543062621, 121.798751236 30.8545843169, 121.797919351 30.8545797584, 121.797919352 30.8548600923, 121.796807133 30.8548600929, 121.796807133 30.854579759, 121.795966132 30.8545797589, 121.795966132 30.8548600927, 121.793472758 30.8548600927, 121.793472758 30.8551404263, 121.793196983 30.8551404257, 121.793196983 30.8554207603, 121.792075647 30.8554207596, 121.792075647 30.855140426, 121.791526375 30.855140426, 121.791526376 30.8548600925, 121.789582273 30.8548600924, 121.789582273 30.8545797589, 121.789306498 30.8545797591, 121.789306498 30.8548600925, 121.789017048 30.8548600923, 121.789017047 30.8545797589, 121.788474613 30.8545843167, 121.788474614 30.8543062623, 121.787918504 30.8543108207, 121.787918504 30.8540304873, 121.78708662 30.8540304869, 121.78708662 30.8537501535, 121.78653051 30.8537501532, 121.786530511 30.8534698198, 121.785972122 30.8534720992, 121.785972123 30.8531963232, 121.785140238 30.853200882, 121.785140238 30.8529205478, 121.782355134 30.8529205482, 121.782355134 30.8532008813, 121.779861759 30.8532008815, 121.77986176 30.8529205485, 121.778197991 30.8529205482, 121.778191153 30.8526402148, 121.77680316 30.8526402147, 121.77680316 30.8523598809, 121.77513939 30.8523621594, 121.77513939 30.8520841056, 121.773475621 30.8520841054, 121.773475622 30.8518037717, 121.770141246 30.8518037713, 121.770141246 30.8515279959, 121.769858634 30.8515302753, 121.769858634 30.8509718874, 121.769026749 30.8509741659, 121.769026748 30.8506938324, 121.76847064 30.8506938324, 121.768470639 30.8509741658, 121.76791453 30.8509741661, 121.76791453 30.8506938326, 121.767082646 30.8506938323, 121.767082646 30.8504157779, 121.766806871 30.8504157782, 121.766806871 30.8506938325, 121.765694653 30.8506938326, 121.765694652 30.8504157784, 121.765136264 30.8504203365, 121.765136264 30.8501400023, 121.764304379 30.8501400022, 121.76430438 30.8518037719, 121.764030884 30.8518037717, 121.764030883 30.8520841049, 121.76264289 30.8520841055, 121.76264289 30.8518037715, 121.761801888 30.8518037717, 121.761801888 30.8515279958, 121.761573975 30.8515279959, 121.761252617 30.8515302749, 121.761252617 30.8512408246, 121.760970004 30.8512499417, 121.760970004 30.8509741659, 121.760696508 30.8509741658, 121.760696508 30.8506938328, 121.760413895 30.8506938324, 121.760413895 30.8504157775, 121.759857786 30.8504203365, 121.759857785 30.8501400026, 121.75818718 30.8501400025, 121.75818718 30.8498619477, 121.754586146 30.8498619484, 121.754586145 30.8495816147, 121.754303532 30.8495816146, 121.754303532 30.8498619483, 121.753471648 30.8498619481, 121.753471648 30.849310397, 121.745971012 30.8493058389, 121.745971012 30.8495816141, 121.741248643 30.8495816147, 121.741248643 30.8498619477, 121.73986065 30.8498619482, 121.73986065 30.8495816143, 121.739584874 30.8495816143, 121.739584875 30.8487497298, 121.739860649 30.8487520088, 121.739860649 30.8481799457, 121.739028765 30.8481936211, 121.739028765 30.848471675, 121.738472657 30.8484716753, 121.738472656 30.8487520092, 121.73790743 30.8487520093, 121.737907431 30.8490300632, 121.737358158 30.849030063, 121.737358159 30.8493103968, 121.737082383 30.8493058388, 121.737082384 30.8495816144, 121.736526274 30.8495816144, 121.736526274 30.8498619481, 121.732086518 30.849861948, 121.732086517 30.8501400024, 121.731797067 30.8501400026, 121.731797067 30.850420336, 121.731247796 30.8504157779, 121.731247796 30.8506938322, 121.730698524 30.8506915534, 121.730698524 30.8526402141, 121.729857524 30.8526402141, 121.729857523 30.8520795467, 121.729584027 30.8520841048, 121.729584027 30.8518037716, 121.729308252 30.8518037713, 121.729308251 30.8512499419, 121.729027918 30.8512499414, 121.729027918 30.8504157779, 121.729308251 30.8504203364, 121.729308252 30.8501400023, 121.728752143 30.8501400021, 121.728752143 30.8498619479, 121.725693542 30.849861948, 121.725693543 30.850140003, 121.725456512 30.8501400023, 121.724861658 30.8501400028, 121.724861658 30.8498619479, 121.723197889 30.8498619479, 121.723197888 30.8501400028, 121.717919411 30.8501400021, 121.717919411 30.8498619476, 121.717361022 30.8498619481, 121.717361023 30.8501400027, 121.716795796 30.8501400025, 121.716795797 30.849861948, 121.715141145 30.8498619483, 121.715141144 30.8495816139, 121.714302422 30.8495816144, 121.714302422 30.8498619479, 121.709862666 30.8498619478, 121.709862666 30.8495816146, 121.708472393 30.8495816139, 121.708472394 30.8493058389, 121.706252516 30.8493103974, 121.706252515 30.8490300634, 121.703187078 30.8490300637, 121.703187078 30.848752009, 121.702362031 30.8487520087, 121.702362031 30.8484716749, 121.702079417 30.8484716754, 121.702079418 30.8481936207, 121.701805921 30.8481936209, 121.701805921 30.8479178452, 121.701530147 30.8479201245, 121.701530146 30.8476420698, 121.700691424 30.8476420698, 121.700691424 30.8479201245, 121.697639661 30.847920124, 121.697639662 30.8476420695, 121.697357049 30.8476420696, 121.697357049 30.8473617361, 121.690139026 30.8473617363, 121.690139026 30.8470791232, 121.686531154 30.8470836816, 121.686531155 30.8468056267, 121.685143161 30.8468056267, 121.685143161 30.8470836814, 121.682358057 30.8470836818, 121.682358057 30.8468056272, 121.680420791 30.8468101849, 121.680420792 30.8465298513, 121.678481247 30.846529852, 121.678481248 30.8462495176, 121.675971919 30.8462495183, 121.675971919 30.845969184, 121.673195932 30.8459714638, 121.673195932 30.8456934087, 121.670693441 30.8456934089, 121.67069344 30.8454176339, 121.665139187 30.8454176332, 121.665139187 30.8456934089, 121.662636696 30.8456934084, 121.662636696 30.845971463, 121.661246423 30.845971463, 121.661246423 30.8456934085, 121.65930232 30.8456934088, 121.65930232 30.8454176335, 121.659026545 30.8454313085, 121.659026545 30.8451395791, 121.658753049 30.8451395787, 121.658753049 30.844578912, 121.658194661 30.8445834694, 121.658194661 30.844305415, 121.657914327 30.8443099731, 121.657914327 30.8440296398, 121.657638552 30.8440296396, 121.657638552 30.8437493062, 121.657362776 30.8437493064, 121.657362776 30.8434712512, 121.657082442 30.8434712513, 121.657082442 30.843195476, 121.656806667 30.8432000348, 121.656806667 30.8429197006, 121.656517216 30.8429197006, 121.656517217 30.8426393672, 121.655983899 30.8426393674, 121.655983899 30.8423704292, 121.654862564 30.8423704288, 121.654862564 30.8420809785, 121.652640408 30.8420832579, 121.652640407 30.8418074827, 121.65123646 30.8418097621, 121.651236459 30.8415294281, 121.650969801 30.8415294276, 121.650969801 30.8418097616, 121.650696304 30.8418097614, 121.650696304 30.8415294284, 121.648749922 30.8415294284, 121.648749922 30.8412490945, 121.646530045 30.8412490944, 121.646530045 30.8409710397, 121.644583662 30.8409733185, 121.644583663 30.8406952648, 121.640693178 30.8406952639, 121.640693178 30.8404149305, 121.636802694 30.8404194891, 121.636802693 30.8401391549, 121.634582815 30.8401391551, 121.634582815 30.839861101, 121.632362937 30.8398611003, 121.632362938 30.8395807672, 121.629302058 30.8395830463, 121.629302058 30.8393049916, 121.62708218 30.8393049916, 121.62708218 30.839026937, 121.625687349 30.8390314952, 121.625687349 30.8387511616, 121.624030417 30.8387511619, 121.624030417 30.8384708284, 121.623191695 30.8384731074, 121.623191695 30.838192773, 121.622086315 30.8381927737, 121.622086315 30.8379169983, 121.621261267 30.8379215559, 121.621261268 30.8376412225, 121.618751939 30.8376412223, 121.618751939 30.837360889, 121.61819355 30.8373608889, 121.618193551 30.8370828347, 121.617637442 30.837080555, 121.617637442 30.8373608891, 121.616532061 30.8373608885, 121.616532061 30.8370805554, 121.615134951 30.8370828341, 121.615134951 30.8368047801, 121.613746957 30.8368116173, 121.613746957 30.8365312834, 121.612358963 30.8365312833, 121.612358963 30.8368116171, 121.610419419 30.8368116174, 121.610419419 30.8365312837, 121.609320875 30.8365312833, 121.609320876 30.8362418337, 121.60569705 30.8362509499, 121.60569705 30.8359728954, 121.603752947 30.835972895, 121.603752947 30.8356948407, 121.60208234 30.8357016782, 121.602082341 30.835421344, 121.600692068 30.8354213443, 121.600692068 30.835141011, 121.599297237 30.8351410112, 121.599297237 30.8348606771, 121.598198694 30.8348606775, 121.598198693 30.8345826225, 121.596528087 30.8345826221, 121.596528088 30.8343045681, 121.595696202 30.8343045676, 121.595696203 30.8340287926, 121.593749821 30.834031072, 121.59374982 30.8337507382, 121.593193712 30.8337507384, 121.593193712 30.8334704047, 121.59124733 30.8334704047, 121.59124733 30.833194629, 121.589300948 30.8332014666, 121.589300948 30.8329211322, 121.587356845 30.832921133, 121.587356845 30.8326407987, 121.586807573 30.8326407992, 121.586807573 30.8323604649, 121.586249185 30.8323604655, 121.586249185 30.8320801312, 121.58512785 30.8320801316, 121.585127851 30.8323604649, 121.584587696 30.8323604654, 121.584587696 30.8326407988, 121.583473199 30.8326407989, 121.583473199 30.8323604657, 121.583190585 30.8323604655, 121.583190585 30.8320801313, 121.582358701 30.8320824111, 121.5823587 30.8315285807, 121.58125332 30.8315285806, 121.58125332 30.8318089147, 121.580405481 30.831808914, 121.580405482 30.831528581, 121.580138822 30.8315285803, 121.580138822 30.8312505262, 121.57985621 30.831250526, 121.579856209 30.8309701929, 121.579024326 30.8309724713, 121.579024326 30.8306944172, 121.578475053 30.8306944169, 121.578475054 30.8304186419, 121.577636332 30.8304300371, 121.577636332 30.8301405873, 121.575692229 30.8301405872, 121.575692229 30.830430037, 121.574584569 30.8304186414, 121.574584569 30.8306944171, 121.573752684 30.8306944169, 121.573752685 30.8304186412, 121.573189738 30.830430037, 121.573189738 30.8301405874, 121.572898009 30.8301405872, 121.572898009 30.8298602535, 121.572357854 30.8298602538, 121.572357854 30.8301405874, 121.572082078 30.830140587, 121.572082078 30.8298602535, 121.571250193 30.8298602536, 121.571250194 30.82957992, 121.570694084 30.8295844776, 121.570694084 30.8293064231, 121.570418309 30.8293087027, 121.570418309 30.8290306482, 121.57015165 30.8290306478, 121.57015165 30.8284699808, 121.5698622 30.8284722599, 121.5698622 30.8281942054, 121.569579588 30.8281942055, 121.569579587 30.8279161509, 121.569312929 30.8279207087, 121.569312928 30.8270705913, 121.569579587 30.8270842663, 121.569579587 30.8259720479, 121.569306091 30.825972048, 121.569306091 30.8256939934, 121.56709305 30.8257008306, 121.56709305 30.8254204971, 121.56569594 30.825420497, 121.56569594 30.8251401636, 121.564018496 30.8251401638, 121.564018497 30.8248598296, 121.56263734 30.8248621091, 121.56263734 30.8245840548, 121.560973571 30.8245840547, 121.560973571 30.8243059996, 121.559585577 30.824306, 121.559585577 30.824027945, 121.558211259 30.8240302247, 121.558211259 30.8237498908, 121.55680959 30.8237498907, 121.55680959 30.8234718359, 121.555419318 30.8234741149, 121.555419317 30.8231937821, 121.55374871 30.823193782, 121.55374871 30.8229157273, 121.552078104 30.8229202854, 121.552078104 30.8226399518, 121.545691967 30.8226399521, 121.545691967 30.8223618969, 121.545135858 30.8223618968, 121.545142695 30.822639952, 121.543472089 30.8226399513, 121.543472088 30.8223618971, 121.543198592 30.8223618974, 121.543198592 30.8220815634, 121.542640203 30.822081563, 121.542640204 30.822361897, 121.542074978 30.8223618975, 121.542074978 30.8220815635, 121.540137712 30.8220815635, 121.540137713 30.8218057877, 121.537361725 30.8218057884, 121.537361725 30.8220815632, 121.536789662 30.8220815636, 121.536789662 30.8223618973, 121.536247228 30.822361897, 121.536247228 30.8218103466, 121.535973732 30.821810346, 121.535973731 30.8212610746, 121.535697956 30.8212610748, 121.535697957 30.8209716244, 121.534309963 30.8209716246, 121.534300846 30.8212610746, 121.533471241 30.8212610748, 121.533471241 30.8209716246, 121.532912852 30.8209716248, 121.532912852 30.821261075, 121.532356744 30.8212610742, 121.532356744 30.8215300127, 121.531518022 30.8215300126, 121.531518021 30.8212610741, 121.531249084 30.8212610747, 121.531249084 30.8209716246, 121.52986109 30.8209716239, 121.52986109 30.8212610749, 121.530136865 30.8212610745, 121.530136865 30.8215300129, 121.52986109 30.8215277334, 121.529861089 30.8220815638, 121.530692975 30.8220815631, 121.530692975 30.8226399515, 121.530419479 30.8226399518, 121.530419478 30.8229202851, 121.52986109 30.8229202854, 121.52986109 30.8226399513, 121.529587594 30.8226399516, 121.529587593 30.8215300128, 121.529295864 30.8215300122, 121.529295865 30.8218103463, 121.527641211 30.8218057882, 121.527641212 30.8220815634, 121.527085103 30.8220815633, 121.527085103 30.8215277332, 121.52680249 30.8215300122, 121.52680249 30.8209716241, 121.526526714 30.8209716246, 121.526526714 30.8204177945, 121.526802489 30.8204223528, 121.52680249 30.8198594058, 121.526526715 30.8198594064, 121.526526715 30.8182001955, 121.52540538 30.8182001951, 121.52540538 30.817919862, 121.524017386 30.8179198618, 121.524017386 30.8176395279, 121.521530849 30.8176395279, 121.521530849 30.8173591941, 121.521248236 30.8173614739, 121.521248236 30.817083419, 121.520689848 30.8170834193, 121.520689849 30.8168053645, 121.520416352 30.8168053648, 121.520416352 30.81652731, 121.517907023 30.816529589, 121.517907024 30.8162492553, 121.517638085 30.816249255, 121.517638085 30.8159689214, 121.517364589 30.8159712005, 121.517364589 30.8156954257, 121.517081976 30.8156954256, 121.517081976 30.815417371, 121.514579485 30.8154196499, 121.514579486 30.8151301997, 121.512915716 30.8151393164, 121.512915716 30.814861262, 121.51264222 30.8148612613, 121.51264222 30.8145832069, 121.512083832 30.8145832073, 121.512083832 30.8143051529, 121.511803498 30.8143097111, 121.511803498 30.8140293768, 121.510971614 30.8140293767, 121.510971614 30.8137513224, 121.509850278 30.8137513228, 121.509850279 30.8134709886, 121.507361463 30.8134709885, 121.507361463 30.8137513225, 121.506249244 30.8137513227, 121.506249244 30.8140293768, 121.505141585 30.8140293776, 121.505141585 30.8137513227, 121.504858971 30.8137513222, 121.504858972 30.8134709892, 121.504585475 30.8134732677, 121.504585476 30.8131952137, 121.504029367 30.8131952131, 121.504029367 30.8129148802, 121.501807209 30.8129194382, 121.501807209 30.8126413835, 121.501526876 30.8126413832, 121.501526875 30.8123610495, 121.5012511 30.81236105, 121.5012511 30.8120715995, 121.50069499 30.8120852747, 121.500694991 30.8118049411, 121.500419216 30.8118094986, 121.500419216 30.8115314448, 121.499863106 30.8115314445, 121.499863107 30.8112511109, 121.499304718 30.8112511107, 121.499304718 30.8109730561, 121.498475113 30.8109730559, 121.498475112 30.8106927223, 121.497360615 30.8106927225, 121.497360615 30.8104169469, 121.496528731 30.8104215055, 121.496528731 30.810141172, 121.495972622 30.8101411714, 121.495972622 30.809860838, 121.495407396 30.8098608384, 121.495407396 30.809580504, 121.494858124 30.8095827831, 121.494858124 30.8093047287, 121.49458235 30.8093206831, 121.494582349 30.809031233, 121.49402624 30.8090312323, 121.49402624 30.808750899, 121.493470131 30.8087508986, 121.493470131 30.8084705653, 121.492920859 30.8084705654, 121.492920859 30.80819479, 121.492638247 30.8082016272, 121.492638247 30.8079212938, 121.492362471 30.8079212936, 121.49236247 30.8076409603, 121.492079858 30.8076409603, 121.492079858 30.8073606261, 121.490691865 30.8073606264, 121.490691865 30.8070825712, 121.490135756 30.8070825712, 121.490135755 30.8065287417, 121.48985998 30.8065310206, 121.48985998 30.8062620831, 121.489586484 30.8062620829, 121.489586484 30.8059703531, 121.487922714 30.8059703537, 121.487922715 30.8062620825, 121.487640102 30.8062620828, 121.487640102 30.8065310211, 121.487083993 30.8065287417, 121.487083992 30.8068045168, 121.486525604 30.8068045171, 121.486525605 30.806528742, 121.485680046 30.8065310212, 121.485680045 30.806262083, 121.485420224 30.8062620832, 121.485420223 30.8059703537, 121.485137611 30.8059726329, 121.485137611 30.8043065841, 121.484581502 30.8043088639, 121.484581501 30.80402853, 121.484305726 30.80402853, 121.484305726 30.8037504751, 121.484581502 30.8037504751, 121.484581501 30.8034701421, 121.484305726 30.8034724211, 121.484305726 30.802918591, 121.484581502 30.8029185909, 121.484581502 30.802640536, 121.484864115 30.8026405358, 121.484864115 30.8023602027, 121.485137611 30.8023602027, 121.485137611 30.8018109313, 121.485420223 30.8018109308, 121.485420223 30.8009699296, 121.485695999 30.8009722086, 121.485695999 30.8004161002, 121.485969496 30.800420658, 121.485969496 30.7995796574, 121.486252108 30.7995842156, 121.486252109 30.7987500517, 121.486525604 30.7987500521, 121.486525605 30.797920446, 121.486808217 30.7979204467, 121.486808217 30.7968105071, 121.487074876 30.7968105076, 121.487074876 30.7959717853, 121.48735749 30.7959717856, 121.487357489 30.7954156765, 121.487074877 30.7954202342, 121.487074876 30.7951421797, 121.487357489 30.7951421799, 121.48735749 30.7940322406, 121.487074876 30.7940322413, 121.487074876 30.7937496279, 121.485969496 30.7937496279, 121.485969495 30.7934692944, 121.485695999 30.7934715739, 121.485695999 30.7931935194, 121.485420224 30.7931935188, 121.485420223 30.7926396887, 121.485137611 30.7926396889, 121.485137611 30.7923593554, 121.484581502 30.792359355, 121.484581502 30.7920790216, 121.484023114 30.7920835798, 121.484023114 30.7918055254, 121.482933686 30.7918100839, 121.482933687 30.7915297496, 121.482361624 30.7915297497, 121.482361624 30.7912494164, 121.482085848 30.7912494161, 121.482085848 30.790959966, 121.481803236 30.7909713619, 121.481803235 30.7906955865, 121.480971351 30.7907001444, 121.480971351 30.7904198111, 121.480141745 30.7904198109, 121.480141746 30.7901394776, 121.479865971 30.7901394771, 121.479865971 30.7898591439, 121.479027248 30.7898614223, 121.479027248 30.7895833681, 121.478751472 30.789583368, 121.478751473 30.7893053137, 121.478195364 30.7893053134, 121.478195364 30.7890272591, 121.477912751 30.7890295381, 121.47791275 30.788749204, 121.477639254 30.788749204, 121.477639255 30.7893053133, 121.47680737 30.7893053139, 121.47680737 30.7890272588, 121.476508803 30.7890295385, 121.476508803 30.7887492045, 121.476248982 30.7887492044, 121.476248982 30.7884711503, 121.475692872 30.788473429, 121.475692872 30.7876392657, 121.474860988 30.7876392649, 121.474860988 30.7873612109, 121.473472995 30.7873612105, 121.473472995 30.7868096604, 121.47264111 30.7868096599, 121.472641111 30.7862512721, 121.471802389 30.7862512715, 121.471802389 30.7859709376, 121.470970504 30.7859732173, 121.470970504 30.7856951625, 121.470412115 30.7856951627, 121.470412116 30.7854171079, 121.470138619 30.7854193874, 121.470138619 30.7851504494, 121.469856006 30.7851504491, 121.469856006 30.7848609987, 121.46958251 30.7848609991, 121.46958251 30.7851504495, 121.468750625 30.7851504496, 121.468750626 30.7848609992, 121.467636128 30.7848609989, 121.467636128 30.7845806653, 121.466804244 30.7845829446, 121.466804244 30.78430489, 121.466248135 30.7843094483, 121.466248134 30.7840313937, 121.465972359 30.7840313938, 121.465972359 30.7837510603, 121.464860141 30.7837510597, 121.46486014 30.7834707263, 121.464301753 30.7834730055, 121.464301753 30.7831926721, 121.464028257 30.7831926717, 121.464028257 30.7829168961, 121.463752481 30.7829214549, 121.463752481 30.7826411205, 121.462637984 30.7826411211, 121.462637984 30.7829214545, 121.462081875 30.7829214543, 121.462081874 30.7826411209, 121.461799262 30.7826411204, 121.461799262 30.7823607871, 121.46124999 30.7823630659, 121.46124999 30.7820827326, 121.460418105 30.7820918488, 121.460418105 30.7815311813, 121.460135493 30.7815311818, 121.460135492 30.7812508476, 121.458474003 30.7812508481, 121.458474003 30.780970514, 121.45819139 30.7809705148, 121.45819139 30.7806947386, 121.457359505 30.7806947385, 121.457359506 30.7804166844, 121.456803397 30.7804212423, 121.456803396 30.7801409092, 121.456254124 30.7801409088, 121.456254125 30.7798605758, 121.455695737 30.7798605755, 121.455695737 30.7795825205, 121.455139627 30.7795825207, 121.455139628 30.7793067456, 121.454581239 30.7793113039, 121.454581239 30.7790195743, 121.454307743 30.7790195744, 121.454307743 30.7793113039, 121.453751634 30.7793113036, 121.453751634 30.7787483569, 121.453193245 30.7787483568, 121.453193245 30.7790286907, 121.451805252 30.7790286908, 121.444922262 30.7886648765, 121.438919019 30.7982987831, 121.42865835 30.8147609798, 121.419532691 30.8193420423, 121.409132995 30.8222912441, 121.40039023 30.8288004558, 121.379333296 30.8342772183, 121.374825166 30.8428923509, 121.376479818 30.8518288424, 121.378448992 30.8549968403, 121.381660294 30.8601613617, 121.383490439 30.869733732, 121.379668329 30.878579057, 121.376655312 30.882253024, 121.353882188 30.9100151752, 121.344086463 30.9058192864, 121.333037213 30.9040256073, 121.310478329 30.9063845124, 121.301323041 30.912029931, 121.29135866 30.9159318115, 121.281380604 30.9124219431, 121.284042634 30.9032256308, 121.281013663 30.8955335477, 121.270675504 30.8954948018, 121.260986899 30.8999824199, 121.243774866 30.9136276055, 121.2329786 30.9157494806, 121.213295985 30.9240933967, 121.202538464 30.9232341624, 121.182837615 30.9150247157, 121.172333079 30.9122304955, 121.160812049 30.9129757727, 121.139404127 30.9172696642, 121.127105911 30.9025669599, 121.116653795 30.8997043654, 121.107735537 30.9021270871, 121.110472779 30.9210598668, 121.101982998 30.926367974, 121.090812955 30.9278380166, 121.083503767 30.9216775127, 121.087325877 30.903271213, 121.054390087 30.8978104039, 121.040466848 30.8976645389, 121.03713931 30.9055343947, 121.031044901 30.9135182067, 121.026969807 30.9325808975, 121.027152137 30.9427276098, 121.035049343 30.9562770711, 121.034012336 30.9568992754, 121.018311371 30.9663098264, 120.992766818 30.9707290703, 120.987326521 30.9656055733, 120.985808617 30.9577129269, 120.985236554 30.9610700934, 120.985626286 30.9652135623, 120.986877532 30.9675428386, 120.98788491 30.969172421, 120.990485403 30.971077778, 120.992217546 30.9747973278, 120.992857984 30.9791755468, 120.992775934 30.9814706365, 120.991066583 30.9846409139, 120.990432983 30.987378156, 120.990075158 30.9898943216, 120.990508194 30.9926634718, 120.994202674 30.9939033218, 120.995109769 30.9982883789, 120.994995813 31.0015019596, 120.993835732 31.0042278065, 120.990777133 31.0075918106, 120.98886038 31.0089182672, 120.983923771 31.0126947953, 120.978485753 31.0157670702, 120.974700109 31.0170479443, 120.965583567 31.0184199836, 120.961547218 31.0192313565, 120.960419046 31.0210387105, 120.959808237 31.0230899333, 120.957572405 31.0260140642, 120.954894421 31.0261736037, 120.950919608 31.0256129362, 120.943466834 31.0249565452, 120.938452736 31.0234500365, 120.932397073 31.0212278791, 120.927398928 31.0192564263, 120.922644652 31.0179846691, 120.915214669 31.0168701719, 120.910453554 31.0158263272, 120.905418943 31.0150058385, 120.899299465 31.0143836351, 120.895119529 31.0119791468, 120.894989619 31.0156485553, 120.894638632 31.0179322493, 120.894287645 31.0202227803, 120.895705268 31.0250796181, 120.895263115 31.0298908734, 120.895415818 31.0331067341, 120.894219271 31.0365208789, 120.892548665 31.0385424727, 120.890868942 31.0405640663, 120.891226766 31.0453958333, 120.891623335 31.0490789166, 120.892256936 31.0536850497, 120.892630713 31.0580587117, 120.89332585 31.0608324196, 120.894476814 31.0659148918, 120.895142321 31.069604813, 120.895545728 31.0730599825, 120.895958251 31.0765151517, 120.895889878 31.0783475773, 120.896559944 31.0818095839, 120.895623218 31.0856886729, 120.892295681 31.0890435607, 120.889533368 31.0914959108, 120.886009825 31.0930092569, 120.882727869 31.0949875464, 120.878128573 31.0967014565, 120.873526998 31.0984176454, 120.868362477 31.10080618, 120.865600165 31.103256251, 120.861504558 31.1059023273, 120.85957413 31.1074566974, 120.858161066 31.1094851284, 120.861212828 31.1137015294, 120.867111232 31.1202859518, 120.872330452 31.123638561, 120.874885363 31.1266926024, 120.875067694 31.1289922506, 120.876289311 31.1320098262, 120.880409988 31.1362512978, 120.88731349 31.1373543996, 120.899354164 31.1367549866, 120.910294015 31.1370421571, 120.918539928 31.1379469743, 120.924116972 31.1387834172, 120.934822073 31.1383754518, 120.943915824 31.1379241829, 120.952690496 31.1390705883, 120.958771231 31.1408369183, 120.962271983 31.1400095922, 120.964195573 31.1384529423, 120.966422289 31.135984639, 120.970260353 31.1333294457, 120.975639114 31.1323197888, 120.980429857 31.1326730545, 120.985206925 31.1334867064, 120.991570272 31.1347972085, 120.996338224 31.1360689664, 120.999510781 31.1368393143, 121.003455965 31.138546387, 121.006874667 31.1402420642, 121.010307046 31.1412471631, 121.014847085 31.1413656781, 121.020729533 31.1410534367, 121.025315155 31.1397953531, 121.029618162 31.1389839815, 121.036590038 31.1382455412, 121.040596759 31.13811791, 121.044051929 31.1386649018, 121.046684331 31.1398774023, 121.046989734 31.1463136819, 121.044143093 31.1512913141, 121.043762479 31.1544980578, 121.048462056 31.1575999609, 121.056213396 31.1575657742, 121.060211 31.1576660564, 121.063941945 31.1579919721, 121.067397115 31.1585389647, 121.069746904 31.1599725416, 121.072105809 31.1614083967, 121.070107007 31.165032223, 121.068946928 31.1677580686, 121.067643261 31.1743835159, 121.064529962 31.1869324375, 121.062827448 31.1974460897, 121.060249745 31.2100086858, 121.059119295 31.2269677338, 121.057854374 31.2402482576, 121.057533016 31.2491938655, 121.057792837 31.2572369344, 121.057662927 31.2609040639, 121.05883896 31.2655261511, 121.061140888 31.2683363255, 121.065316264 31.2714268333, 121.067634145 31.274011373, 121.069899606 31.277970232, 121.071969061 31.2800875487, 121.073997492 31.2831233577, 121.076340444 31.2850195987, 121.07898652 31.2857717129, 121.082466761 31.2858605994, 121.085976629 31.2847985221, 121.08974404 31.2842059474, 121.094619112 31.2824920372, 121.0976025 31.2811860923, 121.100326067 31.2798778685, 121.105171509 31.2790824497, 121.109185067 31.2789502606, 121.111290988 31.2799234515, 121.113127972 31.2808829672, 121.116585421 31.2814299601, 121.11951411 31.2819609988, 121.121681569 31.2810972065, 121.123329383 31.2797593537, 121.128676236 31.2798915437, 121.132407181 31.2802151809, 121.136924427 31.2812430704, 121.141457628 31.281582662, 121.14372309 31.2822960311, 121.144509391 31.2821000259, 121.146300792 31.2885887248, 121.146941229 31.2931948582, 121.146544659 31.297087622, 121.146193672 31.2996060664, 121.145553235 31.3025712223, 121.138565405 31.3035489714, 121.132637374 31.3047819842, 121.127274568 31.3051079004, 121.124320807 31.305495353, 121.121604078 31.3065756639, 121.119666812 31.308592699, 121.117973414 31.3110746772, 121.117102784 31.3128911488, 121.118018997 31.3175018399, 121.120323203 31.3205444865, 121.122910023 31.3231335839, 121.125731592 31.3266457318, 121.131317754 31.3277032509, 121.132384389 31.3280998207, 121.133492049 31.3309236701, 121.132676119 31.3348528993, 121.131424873 31.3401017488, 121.13090523 31.3429597854, 121.124924778 31.3477095036, 121.122084975 31.3469847384, 121.119712395 31.3460092685, 121.117294232 31.3461779248, 121.11431996 31.347023484, 121.112786102 31.3489698662, 121.104508281 31.3538882405, 121.10477494 31.3644429182, 121.105087181 31.376768484, 121.111739978 31.37706705, 121.120498697 31.3774613412, 121.130934859 31.3804880326, 121.149186177 31.3908261922, 121.154929599 31.3990242432, 121.153885755 31.4079379427, 121.153831055 31.4084256777, 121.153769519 31.408915692, 121.159027486 31.4172550491, 121.159057114 31.424892433, 121.158774501 31.4251089512, 121.150795247 31.4312010809, 121.151396939 31.4373866551, 121.151747926 31.4409398271, 121.174065222 31.4426127124, 121.183724199 31.4467858105, 121.192405427 31.4640753335, 121.198249131 31.4706187321, 121.207251716 31.4748488076, 121.218032029 31.4741741836, 121.235389926 31.4843003832, 121.240700311 31.4837032496, 121.246758254 31.4830217878, 121.252273762 31.4781102504, 121.254753462 31.4759040477, 121.264799892 31.4767063035, 121.286337723 31.4814560222, 121.295479337 31.4871379069, 121.304962821 31.4856906562, 121.32247114 31.5021186656, 121.325974172 31.5054165747, 121.326240831 31.5054211336, 121.326240831 31.5048604657, 121.326530282 31.5048627448, 121.326530281 31.5045824113, 121.326806057 31.5045824118, 121.326806057 31.5040285815, 121.327635662 31.5040285812, 121.327635662 31.5043043567, 121.327918274 31.5043043566, 121.327918275 31.5048627453, 121.328184934 31.5048604659, 121.328184933 31.5054211332, 121.328476663 31.5054211329, 121.328476663 31.5057014666, 121.329025935 31.5056946292, 121.329025935 31.5059726842, 121.330696542 31.5059704047, 121.330696542 31.5062507385, 121.331528426 31.5062507386, 121.331528426 31.5059704048, 121.33291642 31.5059704054, 121.332916419 31.5062507384, 121.333472528 31.5062507384, 121.333472529 31.5070849024, 121.333189916 31.5070849028, 121.333189916 31.5073606776, 121.333472528 31.5073515608, 121.333472528 31.5079213449, 121.333748303 31.5079167869, 121.333748304 31.5081948413, 121.334030916 31.5081948415))"
        result = split_polygon_n(p)
        result["area"] = "ST_AREA(geom)"
        assert result["area"].sum() == pytest.approx(0.463389504549851)
