"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const reference_1 = require("../reference");
const uniqueid_1 = require("./uniqueid");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/core.CfnReference');
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target);
        this.displayName = displayName;
        this.originalDisplayName = displayName;
        this.replacementTokens = new Map();
        this.humanReadableDesc = `target = ${target.node.path}`;
        this.producingStack = stack_1.Stack.of(target);
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return CFN_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.stringValue({ produce: () => new CfnReference(...) })
     */
    static for(target, attribute) {
        return CfnReference.singletonReference(target, attribute, () => {
            const cfnIntrinsic = attribute === 'Ref' ? { Ref: target.logicalId } : { 'Fn::GetAtt': [target.logicalId, attribute] };
            return new CfnReference(cfnIntrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, () => {
            const cfnIntrinsic = { Ref: pseudoName };
            return new CfnReference(cfnIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table
     */
    static singletonReference(target, attribKey, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        let ref = attribs.get(attribKey);
        if (!ref) {
            ref = fresh();
            attribs.set(attribKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    /**
     * Register a stack this references is being consumed from.
     */
    consumeFromStack(consumingStack, consumingConstruct) {
        if (this.producingStack && consumingStack.node.root !== this.producingStack.node.root) {
            throw this.newError(`Cannot reference across apps. ` +
                `Consuming and producing stacks must be defined within the same CDK app.`);
        }
        // tslint:disable-next-line:max-line-length
        if (!this.replacementTokens.has(consumingStack) && this.isCrossStackReference(consumingStack)) {
            // We're trying to resolve a cross-stack reference
            consumingStack.addDependency(this.producingStack, `${consumingConstruct.node.path} -> ${this.target.node.path}.${this.originalDisplayName}`);
            this.replacementTokens.set(consumingStack, this.exportValue(consumingStack));
        }
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`
        });
    }
    /**
     * Export a Token value for use in another stack
     *
     * Works by mutating the producing stack in-place.
     */
    exportValue(consumingStack) {
        const producingStack = this.producingStack;
        if (producingStack.environment !== consumingStack.environment) {
            throw this.newError(`Can only reference cross stacks in the same region and account. ${this.humanReadableDesc}`);
        }
        // Ensure a singleton "Exports" scoping Construct
        // This mostly exists to trigger LogicalID munging, which would be
        // disabled if we parented constructs directly under Stack.
        // Also it nicely prevents likely construct name clashes
        const exportsName = 'Exports';
        let stackExports = producingStack.node.tryFindChild(exportsName);
        if (stackExports === undefined) {
            stackExports = new construct_1.Construct(producingStack, exportsName);
        }
        // Ensure a singleton CfnOutput for this value
        const resolved = producingStack.resolve(this);
        const id = 'Output' + JSON.stringify(resolved);
        const exportName = this.generateExportName(stackExports, id);
        let output = stackExports.node.tryFindChild(id);
        if (!output) {
            output = new cfn_output_1.CfnOutput(stackExports, id, { value: token_1.Token.asString(this), exportName });
        }
        // We want to return an actual FnImportValue Token here, but Fn.importValue() returns a 'string',
        // so construct one in-place.
        return new intrinsic_1.Intrinsic({ 'Fn::ImportValue': exportName });
    }
    generateExportName(stackExports, id) {
        const stack = stack_1.Stack.of(stackExports);
        const components = [...stackExports.node.scopes.slice(2).map(c => c.node.id), id];
        const prefix = stack.stackName ? stack.stackName + ':' : '';
        const exportName = prefix + uniqueid_1.makeUniqueId(components);
        return exportName;
    }
    isCrossStackReference(consumingStack) {
        return this.producingStack && this.producingStack !== consumingStack;
    }
}
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
exports.CfnReference = CfnReference;
const cfn_output_1 = require("../cfn-output");
const construct_1 = require("../construct");
const stack_1 = require("../stack");
const token_1 = require("../token");
const intrinsic_1 = require("./intrinsic");
//# sourceMappingURL=data:application/json;base64,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