from collections.abc import Iterable
from typing import Optional, Union

from atoti_core import HierarchyKey
from atoti_query import QueryHierarchy
from typeguard import typechecked, typeguard_ignore

from ..._exceptions import AtotiJavaException
from ..._java_api import JavaApi
from ..._local_hierarchies import CreateHierarchyFromArguments, LocalHierarchies
from ...column import Column
from ...level import Level

LevelOrColumn = Union[Level, Column]


@typeguard_ignore
class DistributedHierarchies(
    LocalHierarchies[QueryHierarchy],
):
    def __init__(
        self,
        *,
        create_hierarchy_from_arguments: CreateHierarchyFromArguments[QueryHierarchy],
        cube_name: str,
        java_api: JavaApi,
    ) -> None:
        super().__init__(
            create_hierarchy_from_arguments=create_hierarchy_from_arguments,
            java_api=java_api,
        )

        self._cube_name = cube_name

    def _get_underlying(self) -> dict[tuple[str, str], QueryHierarchy]:
        hierarchies = {
            coordinates: self._create_hierarchy_from_arguments(description)
            for coordinates, description in self._java_api.get_hierarchies(
                cube_name=self._cube_name,
            ).items()
        }
        return {  # pyright: ignore[reportGeneralTypeIssues]
            hierarchyCoordinate: hierarchies[hierarchyCoordinate]  # type: ignore[misc]
            for hierarchyCoordinate in hierarchies
        }

    @typechecked
    def __getitem__(self, key: HierarchyKey, /) -> QueryHierarchy:
        (dimension_name, hierarchy_name) = self._convert_key(key)
        try:
            hierarchy_argument = self._java_api.get_hierarchy(
                hierarchy_name,
                cube_name=self._cube_name,
                dimension_name=dimension_name,
            )
        except AtotiJavaException as exception:
            raise KeyError(str(exception)) from None
        return self._create_hierarchy_from_arguments(hierarchy_argument)

    def _delete_keys(
        self,
        keys: Optional[Iterable[tuple[str, str]]] = None,  # noqa: ARG002
        /,
    ) -> None:
        raise AssertionError("Distributed cube hierarchies cannot be changed.")
