"use strict";
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const script_editor_1 = require("@elyra/script-editor");
const ui_components_1 = require("@elyra/ui-components");
const application_1 = require("@jupyterlab/application");
const apputils_1 = require("@jupyterlab/apputils");
const codeeditor_1 = require("@jupyterlab/codeeditor");
const filebrowser_1 = require("@jupyterlab/filebrowser");
const fileeditor_1 = require("@jupyterlab/fileeditor");
const launcher_1 = require("@jupyterlab/launcher");
const mainmenu_1 = require("@jupyterlab/mainmenu");
const settingregistry_1 = require("@jupyterlab/settingregistry");
const R_FACTORY = 'R Editor';
const R = 'r';
const R_EDITOR_NAMESPACE = 'elyra-r-script-editor-extension';
const commandIDs = {
    createNewRFile: 'script-editor:create-new-r-file',
    openDocManager: 'docmanager:open',
    newDocManager: 'docmanager:new-untitled'
};
/**
 * Initialization data for the r-editor-extension extension.
 */
const extension = {
    id: R_EDITOR_NAMESPACE,
    autoStart: true,
    requires: [
        codeeditor_1.IEditorServices,
        fileeditor_1.IEditorTracker,
        apputils_1.ICommandPalette,
        settingregistry_1.ISettingRegistry,
        filebrowser_1.IFileBrowserFactory
    ],
    optional: [application_1.ILayoutRestorer, mainmenu_1.IMainMenu, launcher_1.ILauncher],
    activate: (app, editorServices, editorTracker, palette, settingRegistry, browserFactory, restorer, menu, launcher) => {
        console.log('Elyra - r-editor extension is activated!');
        const factory = new script_editor_1.ScriptEditorFactory({
            editorServices,
            factoryOptions: {
                name: R_FACTORY,
                fileTypes: [R],
                defaultFor: [R]
            }
        });
        app.docRegistry.addFileType({
            name: R,
            displayName: 'R File',
            extensions: ['.r'],
            pattern: '.*\\.r$',
            mimeTypes: ['text/x-rsrc'],
            icon: ui_components_1.rIcon
        });
        const { restored } = app;
        /**
         * Track ScriptEditor widget on page refresh
         */
        const tracker = new apputils_1.WidgetTracker({
            namespace: R_EDITOR_NAMESPACE
        });
        let config = Object.assign({}, codeeditor_1.CodeEditor.defaultConfig);
        if (restorer) {
            // Handle state restoration
            void restorer.restore(tracker, {
                command: commandIDs.openDocManager,
                args: widget => ({
                    path: widget.context.path,
                    factory: R_FACTORY
                }),
                name: widget => widget.context.path
            });
        }
        /**
         * Update the setting values. Adapted from fileeditor-extension.
         */
        const updateSettings = (settings) => {
            config = Object.assign(Object.assign({}, codeeditor_1.CodeEditor.defaultConfig), settings.get('editorConfig').composite);
            // Trigger a refresh of the rendered commands
            app.commands.notifyCommandChanged();
        };
        /**
         * Update the settings of the current tracker instances. Adapted from fileeditor-extension.
         */
        const updateTracker = () => {
            tracker.forEach(widget => {
                updateWidget(widget);
            });
        };
        /**
         * Update the settings of a widget. Adapted from fileeditor-extension.
         */
        const updateWidget = (widget) => {
            if (!editorTracker.has(widget)) {
                editorTracker.add(widget);
            }
            const editor = widget.content.editor;
            Object.keys(config).forEach((keyStr) => {
                const key = keyStr;
                editor.setOption(key, config[key]);
            });
        };
        // Fetch the initial state of the settings. Adapted from fileeditor-extension.
        Promise.all([
            settingRegistry.load('@jupyterlab/fileeditor-extension:plugin'),
            restored
        ])
            .then(([settings]) => {
            updateSettings(settings);
            updateTracker();
            settings.changed.connect(() => {
                updateSettings(settings);
                updateTracker();
            });
        })
            .catch((reason) => {
            console.error(reason.message);
            updateTracker();
        });
        app.docRegistry.addWidgetFactory(factory);
        factory.widgetCreated.connect((sender, widget) => {
            void tracker.add(widget);
            // Notify the widget tracker if restore data needs to update
            widget.context.pathChanged.connect(() => {
                void tracker.save(widget);
            });
            updateWidget(widget);
        });
        // Handle the settings of new widgets. Adapted from fileeditor-extension.
        tracker.widgetAdded.connect((sender, widget) => {
            updateWidget(widget);
        });
        /**
         * Create new r file from launcher and file menu
         */
        // Add a r launcher
        if (launcher) {
            launcher.add({
                command: commandIDs.createNewRFile,
                category: 'Elyra',
                rank: 5
            });
        }
        if (menu) {
            // Add new r file creation to the file menu
            menu.fileMenu.newMenu.addGroup([{ command: commandIDs.createNewRFile, args: { isMenu: true } }], 34);
        }
        // Function to create a new untitled r file, given the current working directory
        const createNew = (cwd) => {
            return app.commands
                .execute(commandIDs.newDocManager, {
                path: cwd,
                type: 'file',
                ext: '.r'
            })
                .then(model => {
                return app.commands.execute(commandIDs.openDocManager, {
                    path: model.path,
                    factory: R_FACTORY
                });
            });
        };
        // Add a command to create new R file
        app.commands.addCommand(commandIDs.createNewRFile, {
            label: args => args['isPalette']
                ? 'New R Editor'
                : args.isMenu
                    ? 'R File'
                    : 'R Editor',
            caption: 'Create a new R file',
            icon: args => (args['isPalette'] ? undefined : ui_components_1.rIcon),
            execute: args => {
                const cwd = args['cwd'] || browserFactory.defaultBrowser.model.path;
                return createNew(cwd);
            }
        });
        palette.addItem({
            command: commandIDs.createNewRFile,
            args: { isPalette: true },
            category: 'Elyra'
        });
    }
};
exports.default = extension;
//# sourceMappingURL=index.js.map