"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeDependable = exports.Grant = void 0;
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        if (result.success) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
//# sourceMappingURL=data:application/json;base64,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