"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const iam = require("../lib");
// tslint:disable:object-literal-key-quotes
describe('ImmutableRole', () => {
    let stack;
    let mutableRole;
    let immutableRole;
    beforeEach(() => {
        stack = new core_1.Stack();
        mutableRole = new iam.Role(stack, 'MutableRole', {
            assumedBy: new iam.AnyPrincipal(),
        });
        immutableRole = mutableRole.withoutPolicyUpdates();
    });
    test('ignores calls to attachInlinePolicy', () => {
        const user = new iam.User(stack, 'User');
        const policy = new iam.Policy(stack, 'Policy', {
            statements: [new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:*'],
                })],
            users: [user],
        });
        immutableRole.attachInlinePolicy(policy);
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 's3:*',
                        'Resource': '*',
                        'Effect': 'Allow',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'Policy23B91518',
            'Users': [
                {
                    'Ref': 'User00B015A1',
                },
            ],
        });
    });
    test('ignores calls to addManagedPolicy', () => {
        mutableRole.addManagedPolicy({ managedPolicyArn: 'Arn1' });
        immutableRole.addManagedPolicy({ managedPolicyArn: 'Arn2' });
        expect(stack).toHaveResourceLike('AWS::IAM::Role', {
            'ManagedPolicyArns': [
                'Arn1',
            ],
        });
    });
    test('ignores calls to addToPolicy', () => {
        immutableRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['iam:*'],
        }));
        mutableRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['s3:*'],
        }));
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Version': '2012-10-17',
                'Statement': [
                    {
                        'Resource': '*',
                        'Action': 's3:*',
                        'Effect': 'Allow',
                    },
                ],
            },
        });
    });
    test('ignores grants', () => {
        iam.Grant.addToPrincipal({
            grantee: immutableRole,
            actions: ['s3:*'],
            resourceArns: ['*'],
        });
        expect(stack).not.toHaveResourceLike('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Resource': '*',
                        'Action': 's3:*',
                        'Effect': 'Allow',
                    },
                ],
            },
        });
    });
    // this pattern is used here:
    // aws-codepipeline-actions/lib/cloudformation/pipeline-actions.ts#L517
    test('immutable role is a construct', () => {
        new core_1.Construct(immutableRole, 'Child');
        new core_1.Construct(mutableRole.withoutPolicyUpdates(), 'Child2');
    });
});
//# sourceMappingURL=data:application/json;base64,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