"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('managed policy', () => {
    let app;
    let stack;
    beforeEach(() => {
        app = new cdk.App();
        stack = new cdk.Stack(app, 'MyStack', { env: { account: '1234', region: 'us-east-1' } });
    });
    test('simple AWS managed policy', () => {
        const mp = lib_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/SomePolicy');
        expect(stack.resolve(mp.managedPolicyArn)).toEqual({
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':iam::aws:policy/service-role/SomePolicy',
                ]],
        });
    });
    test('simple customer managed policy', () => {
        const mp = lib_1.ManagedPolicy.fromManagedPolicyName(stack, 'MyCustomerManagedPolicy', 'SomeCustomerPolicy');
        expect(stack.resolve(mp.managedPolicyArn)).toEqual({
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':iam::1234:policy/SomeCustomerPolicy',
                ]],
        });
    });
    test('managed policy by arn', () => {
        const mp = lib_1.ManagedPolicy.fromManagedPolicyArn(stack, 'MyManagedPolicyByArn', 'arn:aws:iam::1234:policy/my-policy');
        expect(stack.resolve(mp.managedPolicyArn)).toEqual('arn:aws:iam::1234:policy/my-policy');
    });
    test('managed policy with statements', () => {
        const policy = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy', { managedPolicyName: 'MyManagedPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        ManagedPolicyName: 'MyManagedPolicyName',
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17',
                        },
                        Path: '/',
                        Description: '',
                    },
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            { Ref: 'MyManagedPolicy9F3720AE' },
                        ],
                    },
                },
            },
        });
    });
    test('managed policy from policy document alone', () => {
        new lib_1.ManagedPolicy(stack, 'MyManagedPolicy', {
            managedPolicyName: 'MyManagedPolicyName',
            document: lib_1.PolicyDocument.fromJson({
                Statement: [{
                        Action: 'sqs:SendMessage',
                        Effect: 'Allow',
                        Resource: '*',
                    }],
            }),
        });
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        ManagedPolicyName: 'MyManagedPolicyName',
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        Path: '/',
                        Description: '',
                    },
                },
            },
        });
    });
    test('managed policy from policy document with additional statements', () => {
        new lib_1.ManagedPolicy(stack, 'MyManagedPolicy', {
            managedPolicyName: 'MyManagedPolicyName',
            document: lib_1.PolicyDocument.fromJson({
                Statement: [{
                        Action: 'sqs:SendMessage',
                        Effect: 'Allow',
                        Resource: '*',
                    }],
            }),
            statements: [new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] })],
        });
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        ManagedPolicyName: 'MyManagedPolicyName',
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17',
                        },
                        Path: '/',
                        Description: '',
                    },
                },
            },
        });
    });
    test('policy name can be omitted, in which case the logical id will be used', () => {
        const policy = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17',
                        },
                        Path: '/',
                        Description: '',
                    },
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            { Ref: 'MyManagedPolicy9F3720AE' },
                        ],
                    },
                },
            },
        });
    });
    test('via props, managed policy can be attached to users, groups and roles and permissions, description and path can be added', () => {
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
        });
        new lib_1.ManagedPolicy(stack, 'MyTestManagedPolicy', {
            managedPolicyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            description: 'My Policy Description',
            path: 'tahiti/is/a/magical/place',
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        expect(stack).toMatchTemplate({
            Resources: {
                User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
                MyTestManagedPolicy6535D9F5: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        Groups: [{ Ref: 'Group1BEBD4686' }],
                        Description: 'My Policy Description',
                        Path: 'tahiti/is/a/magical/place',
                        PolicyDocument: {
                            Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }],
                    },
                },
            },
        });
    });
    test('idempotent if a principal (user/group/role) is attached twice', () => {
        const p = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        const group = new lib_1.Group(stack, 'MyGroup');
        p.attachToGroup(group);
        p.attachToGroup(group);
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
        });
        p.attachToRole(role);
        p.attachToRole(role);
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        Description: '',
                        Path: '/',
                        Users: [{ Ref: 'MyUserDC45028B' }],
                        Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                    },
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('users, groups, roles and permissions can be added using methods', () => {
        const p = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy', {
            managedPolicyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: {
                            Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyName: 'Foo',
                        Description: '',
                        Path: '/',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }],
                    },
                },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('policy can be attached to users, groups or role via methods on the principal', () => {
        const policy = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.addManagedPolicy(policy);
        group.addManagedPolicy(policy);
        role.addManagedPolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        Description: '',
                        Path: '/',
                    },
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User', Properties: { ManagedPolicyArns: [{ Ref: 'MyManagedPolicy9F3720AE' }] } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group', Properties: { ManagedPolicyArns: [{ Ref: 'MyManagedPolicy9F3720AE' }] } },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        ManagedPolicyArns: [{ Ref: 'MyManagedPolicy9F3720AE' }],
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('policy from AWS managed policy lookup can be attached to users, groups or role via methods on the principal', () => {
        const policy = lib_1.ManagedPolicy.fromAwsManagedPolicyName('AnAWSManagedPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.addManagedPolicy(policy);
        group.addManagedPolicy(policy);
        role.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyUserDC45028B: {
                    Type: 'AWS::IAM::User',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':iam::aws:policy/AnAWSManagedPolicy',
                                    ],
                                ],
                            }
                        ],
                    },
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':iam::aws:policy/AnAWSManagedPolicy',
                                    ],
                                ],
                            }
                        ],
                    },
                },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':iam::aws:policy/AnAWSManagedPolicy',
                                    ],
                                ],
                            }
                        ],
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('policy from customer managed policy lookup can be attached to users, groups or role via methods', () => {
        const policy = lib_1.ManagedPolicy.fromManagedPolicyName(stack, 'MyManagedPolicy', 'ACustomerManagedPolicyName');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.addManagedPolicy(policy);
        group.addManagedPolicy(policy);
        role.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyUserDC45028B: {
                    Type: 'AWS::IAM::User',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':iam::1234:policy/ACustomerManagedPolicyName',
                                    ],
                                ],
                            }
                        ],
                    },
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':iam::1234:policy/ACustomerManagedPolicyName',
                                    ],
                                ],
                            }
                        ],
                    },
                },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':iam::1234:policy/ACustomerManagedPolicyName',
                                    ],
                                ],
                            }
                        ],
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('fails if policy document is empty', () => {
        new lib_1.ManagedPolicy(stack, 'MyPolicy');
        expect(() => app.synth())
            .toThrow(/Managed Policy is empty. You must add statements to the policy/);
    });
    test('managed policy name is correctly calculated', () => {
        const mp = new lib_1.ManagedPolicy(stack, 'Policy');
        mp.addStatements(new lib_1.PolicyStatement({
            actions: ['a:abc'],
        }));
        expect(stack.resolve(mp.managedPolicyName)).toEqual({
            'Fn::Select': [1,
                { 'Fn::Split': ['/',
                        { 'Fn::Select': [5,
                                { 'Fn::Split': [':',
                                        { Ref: 'Policy23B91518' }] }] }] }],
        });
    });
    test('cross-stack hard-name contains the right resource type', () => {
        const mp = new lib_1.ManagedPolicy(stack, 'Policy', {
            managedPolicyName: cdk.PhysicalName.GENERATE_IF_NEEDED,
        });
        mp.addStatements(new lib_1.PolicyStatement({
            actions: ['a:abc'],
        }));
        const stack2 = new cdk.Stack(app, 'Stack2', { env: { account: '5678', region: 'us-east-1' } });
        new cdk.CfnOutput(stack2, 'Output', {
            value: mp.managedPolicyArn,
        });
        expect(stack2).toMatchTemplate({
            Outputs: {
                Output: {
                    Value: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':iam::1234:policy/mystackmystackpolicy17395e221b1b6deaf875',
                            ],
                        ],
                    },
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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