"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('IAM policy statement', () => {
    describe('from JSON', () => {
        test('parses with no principal', () => {
            // given
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action1', 'service:action2');
            s.addAllResources();
            s.addCondition('key', { equals: 'value' });
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            // when
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            // then
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses a given Principal', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action1', 'service:action2');
            s.addAllResources();
            s.addArnPrincipal('somearn');
            s.addCondition('key', { equals: 'value' });
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses a given notPrincipal', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action1', 'service:action2');
            s.addAllResources();
            s.addNotPrincipals(new lib_1.AnyPrincipal());
            s.addCondition('key', { equals: 'value' });
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notAction', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addNotActions('service:action3');
            s.addAllResources();
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notActions', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addNotActions('service:action3', 'service:action4');
            s.addAllResources();
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notResource', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action3', 'service:action4');
            s.addNotResources('resource1');
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notResources', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action3', 'service:action4');
            s.addNotResources('resource1', 'resource2');
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('the kitchen sink', () => {
            const stack = new core_1.Stack();
            /* tslint:disable */
            const policyDocument = {
                'Version': '2012-10-17',
                'Statement': [
                    {
                        'Sid': 'FirstStatement',
                        'Effect': 'Allow',
                        'Action': 'iam:ChangePassword',
                        'Resource': '*',
                    },
                    {
                        'Sid': 'SecondStatement',
                        'Effect': 'Allow',
                        'Action': 's3:ListAllMyBuckets',
                        'Resource': '*',
                    },
                    {
                        'Sid': 'ThirdStatement',
                        'Effect': 'Allow',
                        'Action': [
                            's3:List*',
                            's3:Get*',
                        ],
                        'Resource': [
                            'arn:aws:s3:::confidential-data',
                            'arn:aws:s3:::confidential-data/*',
                        ],
                        'Condition': { 'Bool': { 'aws:MultiFactorAuthPresent': 'true' } },
                    },
                ],
            };
            /* tslint:enable */
            const doc = lib_1.PolicyDocument.fromJson(policyDocument);
            expect(stack.resolve(doc)).toEqual(policyDocument);
        });
        test('throws error with field data being object', () => {
            expect(() => {
                lib_1.PolicyStatement.fromJson({
                    Action: {},
                });
            }).toThrow(/Fields must be either a string or an array of strings/);
        });
        test('throws error with field data being array of non-strings', () => {
            expect(() => {
                lib_1.PolicyStatement.fromJson({
                    Action: [{}],
                });
            }).toThrow(/Fields must be either a string or an array of strings/);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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