"""
Generates, or retrieves, brief descriptions of the various types generated by
:mod:`riggery.plugutil.mtypedesc`.

The descriptions can be accessed via the global ``DATA`` constant.
"""

import re
import os
import json

import maya.api.OpenMaya as om
import maya.cmds as m

from .mtypes import DATA as MTYPES

#--------------------------------------------|    CACHE FILE

uncap = lambda x: x[0].lower()+x[1:]

CACHE_FILE_PATH = os.path.join(
    os.path.dirname(__file__),
    os.path.splitext(os.path.basename(__file__))[0]+'.json'
)

DATA = {}

#--------------------------------------------|    REGENERATE

IS_BORING = re.compile(r"(compound|numeric|opaque|typed|tdata|tdatacompound)$", flags=re.IGNORECASE)

IS_ARRAY = re.compile(r"^(.*?)Array$") # e.g. StringArray
IS_FLOAT_TYPE = re.compile(r"^(float|double)$", flags=re.IGNORECASE)
IS_INT_TYPE = re.compile(r"^(int|short|enum|long|addr|char|byte|int32|uint64|uint|int64)$", flags=re.IGNORECASE)
IS_GEO_TYPE = re.compile(r"^(nurbsCurve|lattice|nurbsSurface|subd|subdSurface|subdiv|mesh|dataBezierCurve|bezierCurve|pluginGeometry)$",flags=re.IGNORECASE)
IS_SUBD_TYPE = re.compile(r"^(subd|subdsurface|subdiv)$", flags=re.IGNORECASE)
IS_BEZIER_TYPE = re.compile(r"^(dataBezierCurve|bezierCurve)$", flags=re.IGNORECASE)
IS_BASIC_TENSOR = re.compile(r"^(.*?)([2-4])$") # e.g. double3
IS_MATRIX_TYPE = re.compile(r"^(matrix|floatMatrix|fltMatrix)$", flags=re.IGNORECASE)
IS_3D_TENSOR = re.compile(r"^(floatVector|vector|point|reflectance|reflectanceRGB|spectrum|spectrumRGB)$", flags=re.IGNORECASE)
IS_UNIT_ATTR = re.compile(r"^.*?(Angle|Linear|Time)$", flags=re.IGNORECASE)

def _describeMType(mtype) -> dict:
    out = {}

    if re.match(IS_BORING, mtype):
        return out

    # Array
    mt = re.match(IS_ARRAY, mtype)
    if mt:
        out['isArray'] = True
        base = mt.groups()[0]
        if base:
            out.update(_describeMType(base))
            if base == 'point':
                out['tensorShape'] = 4
    else:
        mt = re.match(IS_FLOAT_TYPE, mtype)
        if mt:
            out['scalarType'] = 'float'
        else:
            mt = re.match(IS_INT_TYPE, mtype)
            if mt:
                out['scalarType'] = 'int'
                if mtype == 'enum':
                    out['otherType'] = 'enum'
            else:
                mt = re.match(IS_GEO_TYPE, mtype)
                if mt:
                    mt = re.match(IS_SUBD_TYPE, mtype)
                    if mt:
                        out['geoType'] = 'subdSurface'
                    else:
                        mt = re.match(IS_BEZIER_TYPE, mtype)
                        if mt:
                            out['geoType'] = 'bezierCurve'
                        else:
                            out['geoType'] = mtype
                else:
                    mt = re.match(IS_BASIC_TENSOR, mtype)
                    if mt:
                        scalarType, shape = mt.groups()
                        out.update(_describeMType(scalarType))
                        out['tensorShape'] = int(shape)
                    else:
                        mt = re.match(IS_MATRIX_TYPE, mtype)
                        if mt:
                            out['tensorShape'] = 16
                            out['scalarType'] = 'float'
                            out['tensorType'] = 'matrix'
                        else:
                            mt = re.match(IS_3D_TENSOR, mtype)
                            if mt:
                                out['tensorShape'] = 3
                                out['scalarType'] = 'float'

                                if mtype in ('point', 'vector'):
                                    out['tensorType'] = mtype
                                elif mtype == 'floatVector':
                                    out['tensorType'] = 'vector'
                            else:
                                mt = re.match(IS_UNIT_ATTR, mtype)
                                if mt:
                                    out['scalarType'] = 'float'
                                    if mtype == 'time':
                                        out['unitType'] = 'time'
                                    else:
                                        unitType = mt.groups()[0]
                                        unitType = {'Linear': 'Distance'}.get(unitType, unitType)
                                        out['unitType'] = uncap(unitType)
                                else:
                                    if mtype == 'bool':
                                        out['scalarType'] = 'bool'
                                    else:
                                        out['otherType'] = mtype
    return out

def _generate() -> dict[str:dict]:
    out = {}
    for mtype in MTYPES:
        entry = _describeMType(mtype)
        if entry:
            out[mtype] = entry

    return out

#--------------------------------------------|    INIT

def _start():
    global DATA

    try:
        with open(CACHE_FILE_PATH, 'r') as f:
            DATA = json.load(f)
    except FileNotFoundError:
        DATA = _generate()
        try:
            with open(CACHE_FILE_PATH, 'w') as f:
                json.dump(DATA, f, indent=4)
        except OSError:
            m.warning(f"Couldn't write cache file: {CACHE_FILE_PATH}")

_start()