"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const defaults = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App({
    postCliContext: {
        '@aws-cdk/aws-s3:keepNotificationInImportedBucket': false,
    },
});
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-s3-sqs with standard Queue';
// For S3 to SQS bucket notification a customer managed CMK must be used:
const encryptionKeyProps = {
    enableKeyRotation: true
};
const kmsKey = new kms.Key(stack, 'ImportedEncryptionKey', encryptionKeyProps);
// Configure notification filter
const filter = {
    prefix: 'the/place',
    suffix: '*.mp3'
};
// Define construct properties so that a new queue myQueue is created
const props = {
    queueProps: {
        queueName: `myQueue`,
        encryptionMasterKey: kmsKey
    },
    deadLetterQueueProps: {
        queueName: `myDLQueue`
    },
    s3EventTypes: [s3.EventType.OBJECT_REMOVED],
    s3EventFilters: [filter],
    bucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    },
    logS3AccessLogs: false
};
const construct = new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
const s3Bucket = construct.s3Bucket;
defaults.addCfnSuppressRules(s3Bucket, [
    { id: 'W35',
        reason: 'This S3 bucket is created for unit/ integration testing purposes only.' },
]);
defaults.SuppressCfnNagLambdaWarnings(stack);
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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