import os
import random
import json
import numpy as np
import pandas as pd

import torch
from torchvision import transforms
import torchvision.transforms.functional as tx

from PIL import Image, ImageOps, ImageDraw
from skimage.io import imread
from scipy.ndimage.filters import gaussian_filter

# Ignore skimage convertion warnings
import warnings
warnings.filterwarnings("ignore")

from .config import config
from .helper import image_picker, get_ctc_nuclei_array
from .distribution import DistributionPolicyDataset

"""
Input Directory Layout
data
└── train
    ├── group_1
    │   ├── sample_1
    │   │   ├── meta.json  (*1, sample-level)
    │   │   ├── 0_0 (e.g. grid index)
    │   │   │   ├── images
    │   │   │   │   ├── 0_0_DAPI.png   (*1, grid-level)
    │   │   │   │   ├── 0_0_EpCAM.png  (*1, grid-level)
    │   │   │   │   ├── 0_0_CD45.png   (*1, grid-level)
    │   │   │   │   ├── Segment.png    (*2, grid-level)
    │   │   │   │   ├── Contour.png    (*2, grid-level)
    │   │   │   │   └── Marker.png     (*2, grid-level)
    │   │   │   ├── meta.json (*1, grid-level)
    │   │   │   └── mask.csv  (*2, grid-level)
    │   │   ├── 0_1
    │   │   └── ...
    │   ├── sample_2
    │   ├── sample_3
    │   └── ...
    └── group_2

Note:
    1. (*1): Generated by individual preporcessor per partner software output
    2. (*2): Generated by nuclei_discovery.valid
    3. Training images are on-the-fly composed of image files suggested by image_picker,
       They are EpCAM/CD45/Segment for CTC's case
    4. Training labels are on-the-fly composed from grid-level meta.json and mask.csv

"""

class CTCDataset(DistributionPolicyDataset):
    def __init__(self, root, policy=None, transform=None, cache=None):
        super().__init__(root, policy)
        self.transform = transform
        self.cache = cache

    def __getitem__(self, idx):
        uid = self.samples[idx]
        if self.cache is not None and uid in self.cache:
            sample = self.cache[uid]
        else:
            fp = image_picker(self.root, uid, channel='CTC')
            # assert isinstance(fp, list)
            if not fp:
                return
            r, g, b = Image.open(fp[0]), Image.open(fp[1]), Image.open(fp[2])
            image = Image.merge('RGB', (r, g, b))
            # overlay masks to single mask, maskes generated via mask.csv & meta.json
            w, h = image.size
            json_path = os.path.join(self.root, uid, 'meta.json')
            csv_path = os.path.join(self.root, uid, 'mask.csv')
            label = get_ctc_nuclei_array(csv_path, (h, w))
            label = label.astype(np.uint8)*255
            label = Image.fromarray(label, 'L') # specify it's grayscale 8-bit
            sample = {'image': image,
                      'label': label,
                      'uid': uid,
                      'size': image.size}
            if self.cache is not None:
                self.cache[uid] = sample
        if self.transform:
            sample = self.transform(sample)
        return sample

class Compose():
    def __init__(self, augment=True, resize=False, tensor=True):
        self.tensor = tensor
        self.augment = augment
        self.resize = resize

        model_name = config['ctc_detector']['model']
        width = config.getint(model_name, 'width')
        self.size = (width, width)

        c = config['pre']
        self.mean = json.loads(c.get('mean'))
        self.std = json.loads(c.get('std'))
        self.label_binary = c.getboolean('label_to_binary')
        self.color_jitter = c.getboolean('color_jitter')
        # self.elastic_distortion = c.getboolean('elastic_distortion')
        # self.color_equalize = c.getboolean('color_equalize')
        self.min_scale = c.getfloat('min_scale')
        self.max_scale = c.getfloat('max_scale')
        self.add_noise = c.getboolean('add_noise')

    def __call__(self, sample):
        image, label = sample['image'], sample['label']

        if self.augment:
            # if self.color_equalize and random.random() > 0.5:
            #     image = clahe(image)

            # perform RandomResize() or just enlarge for image size < model input size
            if random.random() > 0.5:
                new_size = int(random.uniform(self.min_scale, self.max_scale) * np.min(image.size))
            else:
                new_size = int(np.min(image.size))
            if new_size < np.max(self.size): # make it viable for cropping
                new_size = int(np.max(self.size))
            image, label = [tx.resize(x, new_size) for x in (image, label)]

            # perform RandomCrop()
            # if no targets, random crop; if has targets, 50% probability to crop around random target
            targets = np.asarray(label) > 0
            if np.any(targets) and (random.random() > 0.5):
                i, j, h, w = self.random_target_crop(targets)
            else:
                i, j, h, w = transforms.RandomCrop.get_params(image, self.size)
            image, label = [tx.crop(x, i, j, h, w) for x in (image, label)]
            # print(f'has targets? {np.any(targets)}')

            # perform Elastic Distortion
            # if self.elastic_distortion and random.random() > 0.75:
            #     indices = ElasticDistortion.get_params(image)
            #     image, label = [ElasticDistortion.transform(x, indices) for x in (image, label)]

            # perform RandomHorizontalFlip()
            if random.random() > 0.5:
                image, label = [tx.hflip(x) for x in (image, label)]

            # perform RandomVerticalFlip()
            if random.random() > 0.5:
                image, label = [tx.vflip(x) for x in (image, label)]

            # perform Random Rotation (0, 90, 180, and 270 degrees)
            random_degree = random.randint(0, 3) * 90
            image, label = [tx.rotate(x, random_degree) for x in (image, label)]

            # perform ColorJitter()
            # if self.color_jitter and random.random() > 0.5:
            #     color = transforms.ColorJitter.get_params(0.05, 0.05, 0.05, 0.025)
            #     image = color(image)

            if self.add_noise and random.random() > 0.5:
                image = add_noise(image)

        elif self.resize:  # resize down image
            image, label = [tx.resize(x, self.size) for x in (image, label)]

        # Due to resize algorithm may introduce anti-alias edge, aka. non binary value,
        # thereafter map every pixel back to 0 and 255
        if self.label_binary:
            label.point(lambda p, threhold=100: 255 if p > threhold else 0)

        # perform ToTensor()
        if self.tensor:
            image, label = [tx.to_tensor(x) for x in (image, label)]
            # perform Normalize()
            image = tx.normalize(image, self.mean, self.std)

        # prepare a shadow copy of composed data to avoid screwup cached data
        x = sample.copy()
        x['image'], x['label'] = image, label
        return x

    def random_target_crop(self, targets_array):
        rows, cols = np.nonzero(targets_array)
        points = [(row, col) for row, col in zip(rows, cols)]
        pi, pj = random.choice(points)
        th, tw = targets_array.shape
        h, w = self.size
        i = random.randint(max(0, pi-h), min(pi, th-h))
        j = random.randint(max(0, pj-w), min(pj, tw-w))
        return i, j, h, w

    def denorm(self, tensor):
        tensor = tensor.clone()
        for t, m, s in zip(tensor, self.mean, self.std):
            t.mul_(s).add_(m)
        return tensor

    def pil(self, tensor):
        return tx.to_pil_image(tensor)

    def to_numpy(self, tensor, size=None):
        x = tx.to_pil_image(tensor)
        if size is not None:
            x = x.resize(size)
        return np.asarray(x)

    def show(self, sample):
        image, label = sample['image'], sample['label']
        for x in (image, label):
            if len(list(x.size())) == 4: # only dislay first sample
                x = x[0]
            if x.size()[0] > 1: # channel > 1
                x = self.denorm(x)
            x = self.pil(x)
            x.show()

def add_noise(x, mode='speckle'):
    from skimage.util import random_noise
    is_pil = isinstance(x, Image.Image)
    if is_pil:
        x = np.asarray(x, dtype=np.uint8)
    # input numpy array, and return [0, 1] or [-1, 1] array
    x = random_noise(x, mode=mode)
    if is_pil:
        x = (x * 255).astype(np.uint8)
        x = Image.fromarray(x)
    return x


class ElasticDistortion():
    """Elastic deformation of image as described in [Simard2003]_.
    .. [Simard2003] Simard, Steinkraus and Platt, "Best Practices for
       Convolutional Neural Networks applied to Visual Document Analysis", in
       Proc. of the International Conference on Document Analysis and
       Recognition, 2003.
    """
    def __init__(self):
        pass

    @staticmethod
    def get_params(img, alpha=2000, sigma=30):
        w, h = img.size
        dx = gaussian_filter((np.random.rand(*(h, w)) * 2 - 1),
                            sigma, mode="constant", cval=0) * alpha
        dy = gaussian_filter((np.random.rand(*(h, w)) * 2 - 1),
                            sigma, mode="constant", cval=0) * alpha
        x, y = np.meshgrid(np.arange(h), np.arange(w), indexing='ij')
        indices = [np.reshape(x + dx, (-1, 1)), np.reshape(y + dy, (-1, 1))]
        return indices

    @staticmethod
    def transform(img, indices, spline_order=1, mode='nearest'):
        x = np.asarray(img)
        if x.ndim == 2:
            x = np.expand_dims(x, -1)
        shape = x.shape[:2]
        result = np.empty_like(x)
        for i in range(x.shape[2]):
            result[:, :, i] = map_coordinates(
                x[:, :, i], indices, order=spline_order, mode=mode).reshape(shape)
        if result.shape[-1] == 1:
            result = np.squeeze(result)
        return Image.fromarray(result, mode=img.mode)

    def __call__(self, img, spline_order=1, mode='nearest'):
        """
        Args:
            img (PIL Image): Image to be transformed.
        Returns:
            PIL Image: Randomly distorted image.
        """
        indices = self.get_params(img)
        return self.transform(img, indices)

if __name__ == '__main__':
    compose = Compose()#augment=True)
    train = CTCDataset('data/train')
    idx = random.randint(0, len(train)-1)
    sample = train[idx]
    print(sample['uid'])
    # display original image
    sample['image'].show()
    sample['label'].show()
    # input("Press Enter to continue...")
    # display composed image
    sample = compose(sample)
    compose.show(sample)