"""Towerpy: an open-source toolbox for processing polarimetric radar data."""

import numpy as np
from ..utils import radutilities as tput
from ..utils import unit_conversion as tpuc


class RadarQPE:
    """
    A class to calculate rain rates from radar variables.

    Attributes
    ----------
        elev_angle : float
            Elevation angle at which the scan was taken, in deg.
        file_name : str
            Name of the file containing radar data.
        scandatetime : datetime
            Date and time of scan.
        site_name : str
            Name of the radar site.
    """

    def __init__(self, radobj):
        self.elev_angle = radobj.elev_angle
        self.file_name = radobj.file_name
        self.scandatetime = radobj.scandatetime
        self.site_name = radobj.site_name

    def z_to_r(self, zh, a=200, b=1.6, beam_height=None, mlyr=None):
        r"""
        Compute rain rates using the :math:`R(Z_h)` estimator [Eq.1]_.

        Parameters
        ----------
        zh : float or array
             Floats that corresponds to reflectivity, in dBZ.
        a, b : float
            Parameters of the :math:`R(Z_h)` relationship.
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        r_z : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math::  Z_h = aR^b
        where R in mm/hr, :math:`Z_h = 10^{0.1*Z_H}`, :math:`Z_h` in
        :math:`mm^6 \cdot m^{-3}`

        Notes
        -----
        Standard values according to [1]_.

        References
        ----------
        .. [1] Marshall, J.S., Palmer, W.M.K., 1948. "The distribution of
            raindrops with size. Journal of Meteorology 5, 165–166.
            https://doi.org/10.1175/1520-0469(1948)005<0165:TDORWS>2.0.CO;2

        """
        zh = tpuc.xdb2x(zh)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(zh))
            zh[:, mlidx:] = 0
            zh[nanidx] = np.nan
        r = {'Rainfall [mm/hr]': (zh/a)**(1/b)}
        self.r_z = r

    def z_zdr_to_r(self, zh, zdr, a=0.0058, b=0.91, c=-2.09,
                   beam_height=None, mlyr=None):
        r"""
        Compute rain rates using the :math:`R(Z_h, Z_{dr})` estimator [Eq.1]_.

        Parameters
        ----------
        zh : float or array
            Floats that corresponds to reflectivity, in dBZ.
        zdr : float or array
            Floats that corresponds to differential reflectivity, in dB.
        a, b, c : floats
            Parameters of the :math:`R(Z_h, Z_{dr})` relationship.
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        r_z_zdr : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math:: R(Z_h, Z_{dr}) = aZ_h^b Z_{dr}^c
        where R in mm/hr, :math:`Z_h = 10^{0.1*Z_H}`,
        :math:`Z_H` in dBZ, :math:`Z_h` in :math:`mm^6 m^{-3}`,
        :math:`Z_{dr} = 10^{0.1*Z_{DR}}` and :math:`Z_{DR}` in dB.

        Notes
        -----
        Standard values according to [1]_.

        References
        ----------
        .. [1] Bringi, V.N., Chandrasekar, V., 2001. Polarimetric Doppler
            Weather Radar. Cambridge University Press, Cambridge, New York,
            http://dx.doi.org/10.1017/cbo9780511541094.
        """
        zhl = tpuc.xdb2x(zh)
        zdrl = tpuc.xdb2x(zdr)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(zhl))
            zhl[:, mlidx:] = 0
            zhl[nanidx] = np.nan
        r = {'Rainfall [mm/hr]': (a*zhl**b)*(zdrl**c)}
        self.r_z_zdr = r

    def kdp_to_r(self, kdp, a=24.68, b=0.81, beam_height=None, mlyr=None):
        r"""
        Compute rain rates using the :math:`R(K_{DP})` estimator [Eq.1]_.

        Parameters
        ----------
        kdp : float or array
            Floats that corresponds to specific differential phase,
            in deg/km.
        a, b : floats
            Parameters of the :math:`R(K_{DP})` relationship
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        R : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math::  R(K_{DP}) = aK_{DP}^b

        where R in mm/hr and :math:`K_{DP}` in deg/km.

        Notes
        -----
        Standard values according to [1]_.

        References
        ----------
        .. [1] Bringi, V.N., Rico-Ramirez, M.A., Thurai, M. (2011). "Rainfall
            estimation with an operational polarimetric C-band radar in the
            United Kingdom: Comparison with a gauge network and error
            analysis" Journal of Hydrometeorology 12, 935–954.
            https://doi.org/10.1175/JHM-D-10-05013.1
        """
        kdp = np.array(kdp)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(kdp))
            kdp[:, mlidx:] = 0
            kdp[nanidx] = np.nan
        r = {'Rainfall [mm/hr]': a*abs(kdp)**b*np.sign(kdp)}
        self.r_kdp = r

    def kdp_zdr_to_r(self, kdp, zdr, a=37.9, b=0.89, c=-0.72, beam_height=None,
                     mlyr=None):
        r"""
        Compute rain rates using the :math:`R(K_{DP}, Z_{dr})` estimator [Eq.1]_.

        Parameters
        ----------
        kdp : float or array
            Floats that corresponds to specific differential phase,
            in deg/km.
        zdr : float or array
            Floats that corresponds to differential reflectivity, in dB.
        a, b, c : floats
            Parameters of the :math:`R(K_{DP}, Z_{dr})` relationship
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        R : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math::  R = aK_{DP}^b Z_{dr}^c
        where R in mm/hr, :math:`K_{DP}` in deg/km,
        :math:`Z_{dr} = 10^{0.1*Z_{DR}}` and :math:`Z_{DR}` in dB.

        Notes
        -----
        Standard values according to [1]_

        References
        ----------
        .. [1] Bringi, V.N., Chandrasekar, V., (2001). "Polarimetric Doppler
            Weather Radar" Cambridge University Press, Cambridge ; New York.
            https://doi.org/10.1017/CBO9780511541094

        """
        kdp = np.array(kdp)
        zdrl = tpuc.xdb2x(zdr)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(kdp))
            kdp[:, mlidx:] = 0
            kdp[nanidx] = np.nan
        r = {'Rainfall [mm/hr]': (a*kdp**b)*(zdrl**c)}
        self.r_kdp_zdr = r

    def ah_to_r(self, ah, a=294, b=0.89, beam_height=None, mlyr=None):
        r"""
        Compute rain rates using the :math:`R(A_{H})` estimator [Eq.1]_.

        Parameters
        ----------
        ah : float or array
            Floats that corresponds to specific attenuation, in dB/km.
        a, b : floats
            Parameters of the :math:`R(A_{H})` relationship
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        R : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math::  R = aA_H^b
        where R in mm/hr, AH in dB/km

        Notes
        -----
        Standard values according to [1]_.

        References
        ----------
        .. [1] Ryzhkov, A., Diederich, M., Zhang, P., & Simmer, C. (2014).
            "Potential Utilization of Specific Attenuation for Rainfall
            Estimation, Mitigation of Partial Beam Blockage, and Radar
            Networking" Journal of Atmospheric and Oceanic Technology, 31(3),
            599-619. https://doi.org/10.1175/JTECH-D-13-00038.1

        """
        ah = np.array(ah)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(ah))
            ah[:, mlidx:] = 0
            ah[nanidx] = np.nan
        r = {'Rainfall [mm/hr]': a*ah**b}
        self.r_ah = r

    def z_kdp_to_r(self, zh, kdp, a1=200, b1=1.6, a2=24.68, b2=0.81,
                   z_thld=40, beam_height=None, mlyr=None):
        r"""
        Compute rain rates using an hybrid estimator that combines :math:`R(Z_h)` [Eq.1]_ and :math:`R(K_{DP})` [Eq.2]_ for a given threshold in :math:`Z_H`.

        Parameters
        ----------
        zh : float or array
             Floats that corresponds to reflectivity, in dBZ.
        kdp : float or array
            Floats that corresponds to specific differential phase,
            in deg/km.
        a1, b1 : float
            Parameters of the :math:`R(Z_h)` relationship.
        a2, b2 : floats
            Parameters of the :math:`R(K_{DP})` relationship.
        z_thld : float, optional
            :math:`Z_H` threshold used for the transition to :math:`R(K_{DP})`.
            The default is 40 dBZ.
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        R : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math:: Z_H < 40 dBZ \rightarrow Z_h = aR^b
        .. [Eq.2]
        .. math:: Z_H \geq 40 dBZ \rightarrow R = aK_{DP}^b
        where R in mm/hr, :math:`Z_h = 10^{0.1*Z_H}`, :math:`Z_h` in
        :math:`mm^6 \cdot m^{-3}`, :math:`K_{DP}` in deg/km

        Notes
        -----
        Standard values according to [1]_ and [2]_.

        References
        ----------
        .. [1] Marshall, J.S., Palmer, W.M.K., 1948. "The distribution of
            raindrops with size. Journal of Meteorology 5, 165–166.
            https://doi.org/10.1175/1520-0469(1948)005<0165:TDORWS>2.0.CO;2
        .. [2] Bringi, V.N., Rico-Ramirez, M.A., Thurai, M. (2011). "Rainfall
            estimation with an operational polarimetric C-band radar in the
            United Kingdom: Comparison with a gauge network and error
            analysis" Journal of Hydrometeorology 12, 935–954.
            https://doi.org/10.1175/JHM-D-10-05013.1
        """
        zh = np.array(zh)
        zhl = tpuc.xdb2x(zh)
        kdp = np.array(kdp)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(zhl))
            zhl[:, mlidx:] = 0
            zhl[nanidx] = np.nan
        rzh = (zhl/a1)**(1/b1)
        rkdp = a2*abs(kdp)**b2*np.sign(kdp)
        rzh[(zh >= z_thld)] = rkdp[(zh >= z_thld)]
        r = {'Rainfall [mm/hr]': rzh}
        self.r_z_kdp = r

    def z_ah_to_r(self, zh, ah, a1=200, b1=1.6, a2=294, b2=0.89,
                  z_thld=40, beam_height=None, mlyr=None):
        r"""
        Compute rain rates using an hybrid estimator that combines :math:`R(Z_h)` [Eq.1]_ and :math:`R(A_H)` [Eq.2]_ for a given threshold in :math:`Z_H`.

        Parameters
        ----------
        zh : float or array
             Floats that corresponds to reflectivity, in dBZ.
        ah : float or array
            Floats that corresponds to specific attenuation, in dB/km.
        a1, b1 : float
            Parameters of the :math:`R(Z_h)` relationship.
        a2, b2 : floats
            Parameters of the :math:`R(A_{H})` relationship.
        z_thld : float, optional
            :math:`Z_H` threshold used for the transition to :math:`R(A_{H})`.
            The default is 40 dBZ.
        beam_height : array, optional
            Height of the centre of the radar beam, in km.
        mlyr : class, optional
            Melting layer class containing the top and bottom boundaries of the
            ML. Only gates below the melting layer bottom (i.e. the rain region
            below the melting layer) are included in the correction. If None,
            the default values of the melting level and the thickness of the
            melting layer are set to 5 and 0.5, respectively.

        Returns
        -------
        R : dict
            Computed rain rates, in mm/hr.

        Math
        ----
        .. [Eq.1]
        .. math:: Z_H < 40 dBZ \rightarrow Z_h = aR^b
        .. [Eq.2]
        .. math:: Z_H \geq 40 dBZ \rightarrow R = aA_{H}^b
        where R in mm/hr, :math:`Z_h = 10^{0.1*Z_H}`, :math:`Z_h` in
        :math:`mm^6 \cdot m^{-3}`, :math:`A_H` in dB/km

        Notes
        -----
        Standard values according to [1]_ and [2]_.

        References
        ----------
        .. [1] Marshall, J.S., Palmer, W.M.K., 1948. "The distribution of
            raindrops with size. Journal of Meteorology 5, 165–166.
            https://doi.org/10.1175/1520-0469(1948)005<0165:TDORWS>2.0.CO;2
        .. [2] Ryzhkov, A., Diederich, M., Zhang, P., & Simmer, C. (2014).
            "Potential Utilization of Specific Attenuation for Rainfall
            Estimation, Mitigation of Partial Beam Blockage, and Radar
            Networking" Journal of Atmospheric and Oceanic Technology, 31(3),
            599-619. https://doi.org/10.1175/JTECH-D-13-00038.1
        """
        zh = np.array(zh)
        zhl = tpuc.xdb2x(zh)
        ah = np.array(ah)
        if mlyr is None:
            mlvl = 5.
            mlyr_thickness = 0.5
            # mlyr_bottom = mlvl - mlyr_thickness
        else:
            mlvl = mlyr.ml_top
            mlyr_thickness = mlyr.ml_thickness
            # mlyr_bottom = mlyr.ml_bottom
        if beam_height is not None:
            mlidx = tput.find_nearest(beam_height, mlvl-mlyr_thickness)
            nanidx = np.where(np.isnan(zhl))
            zhl[:, mlidx:] = 0
            zhl[nanidx] = np.nan
        rzh = (zhl/a1)**(1/b1)
        rah = a2*ah**b2
        rzh[(zh >= z_thld)] = rah[(zh >= z_thld)]
        r = {'Rainfall [mm/hr]': rzh}
        self.r_z_ah = r
