"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
test('Test cross region deployment with new Lambda function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                BEDROCK_MODEL: "amazon.nova-lite-v1:0",
                BEDROCK_PROFILE: {
                    "Fn::GetAtt": [
                        assertions_1.Match.stringLikeRegexp("testlambdainferencepropstestlambdainferencepropsinference"),
                        "InferenceProfileArn"
                    ]
                }
            }
        },
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole"),
                "Arn"
            ]
        },
        Runtime: "nodejs20.x",
        Timeout: 30,
        TracingConfig: {
            Mode: "Active"
        }
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            assertions_1.Match.stringLikeRegexp("testlambdainferencepropstestlambdainferencepropsinference"),
                            "InferenceProfileArn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRoleDefaultPolicy"),
        Roles: [
            {
                "Ref": assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole")
            }
        ]
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Split": [
                            ",",
                            {
                                "Fn::FindInMap": [
                                    "testlambdainferencepropsarearegionmapping",
                                    {
                                        "Fn::Select": [
                                            0,
                                            {
                                                "Fn::Split": [
                                                    "-",
                                                    {
                                                        "Ref": "AWS::Region"
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    "regionalModels"
                                ]
                            }
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctioninlinePolicyAddedToExecutionRole"),
        Roles: [
            {
                "Ref": assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole")
            }
        ]
    });
});
test('Test single region deployment with new Lambda function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        bedrockModelId: "amazon.nova-lite-v1:0",
        deployCrossRegionProfile: false
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                BEDROCK_MODEL: "amazon.nova-lite-v1:0",
                BEDROCK_PROFILE: {
                    "Fn::GetAtt": [
                        "testlambdainferencepropstestlambdainferencepropsinference01BBD7E6",
                        "InferenceProfileArn"
                    ]
                }
            }
        },
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testlambdainferencepropsLambdaFunctionServiceRole102487FD",
                "Arn"
            ]
        },
        Runtime: "nodejs20.x",
        Timeout: 30,
        TracingConfig: {
            Mode: "Active"
        }
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            assertions_1.Match.stringLikeRegexp("testlambdainferencepropstestlambdainferencepropsinference"),
                            "InferenceProfileArn"
                        ]
                    }
                },
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    "Ref": "AWS::Partition"
                                },
                                ":bedrock:",
                                {
                                    "Ref": "AWS::Region"
                                },
                                "::foundation-model/amazon.nova-lite-v1:0"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRoleDefaultPolicy"),
        Roles: [
            {
                "Ref": assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole")
            }
        ]
    });
});
test('Test deployment with VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        bedrockModelId: "amazon.nova-lite-v1:0",
        deployVpc: true
    };
    const newConstruct = new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    expect(newConstruct.vpc).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.resourceCountIs("AWS::EC2::VPCEndpoint", 2);
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: assertions_1.Match.anyValue(),
            SubnetIds: [
                assertions_1.Match.anyValue(),
                assertions_1.Match.anyValue(),
            ]
        },
    });
});
test('Test deployment overridden Lambda props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            timeout: cdk.Duration.seconds(45)
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Timeout: 45
    });
});
test('Test deployment with overridden inference props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testName = "testName";
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        inferenceProfileProps: {
            inferenceProfileName: testName
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Bedrock::ApplicationInferenceProfile", {
        InferenceProfileName: testName
    });
});
test('Test deployment with invalid inference props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        inferenceProfileProps: {
            inferenceProfileName: "test",
            modelSource: {
                copyFrom: "test"
            }
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    const app = () => {
        new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    };
    expect(app).toThrowError('Error - The construct will create the modelSource value, it cannot be specified in the props.\n');
});
test('Test deployment with invalid Lambda props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
        }),
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    const app = () => {
        new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    };
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
//# sourceMappingURL=data:application/json;base64,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