# *******************************************************************************
# OpenStudio(R), Copyright (c) Alliance for Sustainable Energy, LLC.
# See also https://openstudio.net/license
# *******************************************************************************

# see the URL below for information on how to write OpenStudio measures
# http://openstudio.nrel.gov/openstudio-measure-writing-guide

# see the URL below for information on using life cycle cost objects in OpenStudio
# http://openstudio.nrel.gov/openstudio-life-cycle-examples

# see the URL below for access to C++ documentation on model objects (click on "model" in the main window to view model objects)
# http://openstudio.nrel.gov/sites/openstudio.nrel.gov/files/nv_data/cpp_documentation_it/model/html/namespaces.html

require 'json'
require 'time'

# start the measure
class AddMonthlyUtilityData < OpenStudio::Measure::ModelMeasure
  # define the name that a user will see, this method may be deprecated as
  # the display name in PAT comes from the name field in measure.xml
  def name
    'AddMonthlyUtilityData'
  end

  def year_month_day(str)
    result = nil
    if match_data = /(\d+)-(\d+)-(\d+)/.match(str)
      year = match_data[1].to_i
      month = match_data[2].to_i
      day = match_data[3].to_i
      result = [year, month, day]
    else
      puts "no match for '#{str}'"
    end
    result
  end

  # define the arguments that the user will input
  def arguments(_model)
    args = OpenStudio::Measure::OSArgumentVector.new

    # make an electric json argument
    electric_json = OpenStudio::Measure::OSArgument.makeStringArgument('electric_json', true)
    electric_json.setDisplayName('Path to electric JSON')
    args << electric_json

    # make a gas json argument
    gas_json = OpenStudio::Measure::OSArgument.makeStringArgument('gas_json', true)
    gas_json.setDisplayName('Path to gas JSON')
    args << gas_json

    water_json = OpenStudio::Measure::OSArgument.makeStringArgument('water_json', false)
    water_json.setDisplayName('Path to water JSON')
    args << water_json

    # make a start date argument
    start_date = OpenStudio::Measure::OSArgument.makeStringArgument('start_date', true)
    start_date.setDisplayName('Start date')
    start_date.setDescription('YYYY-MM-DD format')
    args << start_date

    # make an end date argument
    end_date = OpenStudio::Measure::OSArgument.makeStringArgument('end_date', true)
    end_date.setDisplayName('End date')
    end_date.setDescription('YYYY-MM-DD format')
    args << end_date

    args
  end

  # define what happens when the measure is run
  def run(model, runner, user_arguments)
    super(model, runner, user_arguments)

    # use the built-in error checking
    unless runner.validateUserArguments(arguments(model), user_arguments)
      return false
    end

    # assign the user inputs to variables
    electric_json = runner.getStringArgumentValue('electric_json', user_arguments)
    gas_json = runner.getStringArgumentValue('gas_json', user_arguments)
    water_json = runner.getOptionalStringArgumentValue('water_json', user_arguments)
    start_date = runner.getStringArgumentValue('start_date', user_arguments)
    end_date = runner.getStringArgumentValue('end_date', user_arguments)

    # set start date
    if date = year_month_day(start_date)

      start_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(date[1]), date[2], date[0])

      # actual year of start date
      yearDescription = model.getYearDescription
      yearDescription.setCalendarYear(date[0])

      runPeriod = model.getRunPeriod
      runPeriod.setBeginMonth(date[1])
      runPeriod.setBeginDayOfMonth(date[2])
    else
      runner.registerError("Unknown start date '#{start_date}'")
      raise "Unknown start date '#{start_date}'"
      return false
    end

    # set end date
    if date = year_month_day(end_date)

      end_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(date[1]), date[2], date[0])

      runPeriod = model.getRunPeriod
      runPeriod.setEndMonth(date[1])
      runPeriod.setEndDayOfMonth(date[2])
    else
      runner.registerError("Unknown end date '#{end_date}'")
      raise "Unknown end date '#{end_date}'"
      return false
    end

    # remove all utility bills
    model.getUtilityBills.each(&:remove)
    runner.registerInfo("electric_json is #{electric_json}")
    electric_json_path = File.expand_path(electric_json.to_s, __FILE__)
    runner.registerInfo("electric_json_path is #{electric_json_path}")
    temp = File.read(electric_json_path)
    electric_data = JSON.parse(temp)
    unless electric_data.nil?

      utilityBill = OpenStudio::Model::UtilityBill.new('Electricity'.to_FuelType, model)
      utilityBill.setName('Electric Bill')
      utilityBill.setConsumptionUnit('kWh')
      utilityBill.setPeakDemandUnit('kW')

      electric_data['data'].each do |period|
        from_date = period['from'] ? Time.iso8601(period['from']).strftime('%Y%m%dT%H%M%S') : nil
        to_date = period['to'] ? Time.iso8601(period['to']).strftime('%Y%m%dT%H%M%S') : nil

        if from_date.nil? || to_date.nil?
          runner.registerError("Unknown date format in period '#{period}'")
          raise "Unknown date format in period '#{period}'"
          return false
        end

        period_start_date = OpenStudio::DateTime.fromISO8601(from_date).get.date
        # period_start_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(from_date[1]), from_date[2], from_date[0])
        period_end_date = OpenStudio::DateTime.fromISO8601(to_date).get.date - OpenStudio::Time.new(1.0)
        # period_end_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(to_date[1]), to_date[2], to_date[0])

        if (period_start_date < start_date) || (period_end_date > end_date)
          runner.registerInfo("skipping period #{period_start_date} to #{period_end_date}")
          next
        end

        if period['tot_kwh'].nil?
          runner.registerError("Billing period missing tot_kwh '#{period}'")
          return false
        end
        tot_kwh = period['tot_kwh'].to_f

        peak_kw = nil
        peak_kw = period['peak_kw'].to_f unless period['peak_kw'].nil?

        runner.registerInfo("electric period #{period}")
        runner.registerInfo("electric period_start_date: #{period_start_date}, period_end_date: #{period_end_date}, tot_kwh: #{tot_kwh}, peak_kw: #{peak_kw}")

        bp = utilityBill.addBillingPeriod
        bp.setStartDate(period_start_date)
        bp.setEndDate(period_end_date)
        bp.setConsumption(tot_kwh)
        bp.setPeakDemand(peak_kw) if peak_kw
      end
    end
    runner.registerInfo("gas_json is #{gas_json}")
    gas_json_path = File.expand_path(gas_json.to_s, __FILE__)
    runner.registerInfo("gas_json_path is #{gas_json_path}")
    temp = File.read(gas_json_path)
    gas_data = JSON.parse(temp)
    unless gas_data.nil?

      utilityBill = OpenStudio::Model::UtilityBill.new('Gas'.to_FuelType, model)
      utilityBill.setName('Gas Bill')
      utilityBill.setConsumptionUnit('therms')

      gas_data['data'].each do |period|
        from_date = period['from'] ? Time.iso8601(period['from']).strftime('%Y%m%dT%H%M%S') : nil
        to_date = period['to'] ? Time.iso8601(period['to']).strftime('%Y%m%dT%H%M%S') : nil

        if from_date.nil? || to_date.nil?
          runner.registerError("Unknown date format in period '#{period}'")
          raise "Unknown date format in period '#{period}'"
          return false
        end

        period_start_date = OpenStudio::DateTime.fromISO8601(from_date).get.date
        # period_start_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(from_date[1]), from_date[2], from_date[0])
        period_end_date = OpenStudio::DateTime.fromISO8601(to_date).get.date - OpenStudio::Time.new(1.0)
        # period_end_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(to_date[1]), to_date[2], to_date[0])

        if (period_start_date < start_date) || (period_end_date > end_date)
          runner.registerInfo("skipping period #{period_start_date} to #{period_end_date}")
          next
        end

        if period['tot_therms'].nil?
          runner.registerError("Billing period missing tot_therms '#{period}'")
          return false
        end
        tot_therms = period['tot_therms'].to_f

        runner.registerInfo("gas period: #{period}")
        runner.registerInfo("gas period_start_date: #{period_start_date}, period_end_date: #{period_end_date}, tot_therms: #{tot_therms}")

        bp = utilityBill.addBillingPeriod
        bp.setStartDate(period_start_date)
        bp.setEndDate(period_end_date)
        bp.setConsumption(tot_therms)
      end
    end

    # water bills
    if water_json.is_initialized

      runner.registerInfo("water_json is #{water_json}")
      water_json_path = File.expand_path(water_json.to_s, __FILE__)
      runner.registerInfo("water_json_path is #{water_json_path}")
      temp = File.read(water_json_path)
      water_data = JSON.parse(temp)

      if water_data

        utilityBill = OpenStudio::Model::UtilityBill.new('Water'.to_FuelType, model)
        utilityBill.setName('Water Bill')
        utilityBill.setConsumptionUnit('gal')
        # utilityBill.setPeakDemandUnit("kW")

        water_data['data'].each do |period|
          from_date = period['from'] ? Time.iso8601(period['from']).strftime('%Y%m%dT%H%M%S') : nil
          to_date = period['to'] ? Time.iso8601(period['to']).strftime('%Y%m%dT%H%M%S') : nil

          if from_date.nil? || to_date.nil?
            runner.registerError("Unknown date format in period '#{period}'")
            raise "Unknown date format in period '#{period}'"
            return false
          end

          period_start_date = OpenStudio::DateTime.fromISO8601(from_date).get.date
          # period_start_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(from_date[1]), from_date[2], from_date[0])
          period_end_date = OpenStudio::DateTime.fromISO8601(to_date).get.date - OpenStudio::Time.new(1.0)
          # period_end_date = OpenStudio::Date.new(OpenStudio::MonthOfYear.new(to_date[1]), to_date[2], to_date[0])

          if (period_start_date < start_date) || (period_end_date > end_date)
            runner.registerInfo("skipping period #{period_start_date} to #{period_end_date}")
            next
          end

          if period['tot_gal'].nil?
            runner.registerError("Billing period missing tot_gal '#{period}'")
            return false
          end
          tot_gal = period['tot_gal'].to_f

          tot_cost = nil
          tot_cost = period['tot_cost'].to_f if period['tot_cost']

          runner.registerInfo("water period #{period}")
          runner.registerInfo("water period_start_date: #{period_start_date}, period_end_date: #{period_end_date}")

          bp = utilityBill.addBillingPeriod
          bp.setStartDate(period_start_date)
          bp.setEndDate(period_end_date)
          bp.setConsumption(tot_gal)
        end
      end
    end

    # reporting final condition of model
    runner.registerFinalCondition('Utility bill data has been added to the model.')

    true
  end
end

# this allows the measure to be use by the application
AddMonthlyUtilityData.new.registerWithApplication
