"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? true;
        this.user = props.user;
        this.group = props.group;
        this.allowPrivilegeEscalation = props.allowPrivilegeEscalation ?? false;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
            allowPrivilegeEscalation: this.allowPrivilegeEscalation,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-23.ContainerSecurityContext", version: "2.4.17" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-23.EnvValue", version: "2.4.17" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numbers = this._ports.map(p => p.number);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        if (numbers.includes(port.number)) {
            throw new Error(`Port with number ${port.number} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol ? this._portProtocolToKube(port.protocol) : undefined,
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this._imagePullPolicyToKube(this.imagePullPolicy),
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
    _imagePullPolicyToKube(imagePullPolicy) {
        switch (imagePullPolicy) {
            case ImagePullPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.ALWAYS;
            case ImagePullPolicy.IF_NOT_PRESENT:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.IF_NOT_PRESENT;
            case ImagePullPolicy.NEVER:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.NEVER;
            default:
                throw new Error(`Unsupported image pull policy: ${imagePullPolicy}`);
        }
    }
    _portProtocolToKube(protocol) {
        switch (protocol) {
            case Protocol.SCTP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.SCTP;
            case Protocol.TCP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.TCP;
            case Protocol.UDP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.UDP;
            default:
                throw new Error(`Unsupported port protocol: ${protocol}`);
        }
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-23.Container", version: "2.4.17" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-23.Cpu", version: "2.4.17" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-23.EnvFrom", version: "2.4.17" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-23.Env", version: "2.4.17" };
//# sourceMappingURL=data:application/json;base64,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