"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * (experimental) Adds "principal" functionality to the Policy Statement.
 *
 * @experimental
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        /**
         * @experimental
         */
        this.useNotPrincipals = false;
        /**
         * @experimental
         */
        this.principals = {};
    }
    /**
     * (experimental) Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     *
     * @experimental
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.principals;
        }
        return statement;
    }
    /**
     * (experimental) JSON-ify the policy statement.
     *
     * Used when JSON.stringify() is called
     *
     * @experimental
     */
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    /**
     * @experimental
     */
    cdkApplyPrincipals() { }
    /**
     * (experimental) Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     *
     * @experimental
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * (experimental) Checks weather a principal was applied to the policy.
     *
     * @experimental
     */
    hasPrincipals() {
        return Object.keys(this.principals).length > 0;
    }
    /**
     * (experimental) Adds a principal to the statement.
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**.
     * @param principal The principal string.
     * @experimental
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.principals)) {
            this.principals[prefix] = [];
        }
        this.principals[prefix].push(principal);
        return this;
    }
    /**
     * (experimental) Adds any principal to the statement.
     *
     * @param arn The ARN of the principal.
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**.
     * @experimental
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * (experimental) Adds an account principal to the statement.
     *
     * @param account ID of the AWS account.
     * @experimental
     */
    forAccount(account) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:root`);
    }
    /**
     * (experimental) Adds a [federated](https://aws.amazon.com/identity/federation/) (web identity) principal to the statement.
     *
     * @param provider ID of the AWS account.
     * @experimental
     */
    forFederated(provider) {
        return this.addPrincipal(PrincipalType.FEDERATED, provider);
    }
    /**
     * (experimental) Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement.
     *
     * @experimental
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * (experimental) Adds a federated [Amazon](https://login.amazon.com/) principal to the statement.
     *
     * @experimental
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * (experimental) Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement.
     *
     * @experimental
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * (experimental) Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement.
     *
     * @experimental
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * (experimental) Adds a canonical user principal to the statement.
     *
     * @param userID The user ID.
     * @experimental
     */
    forCanonicalUser(userID) {
        return this.addPrincipal(PrincipalType.CANONICAL_USER, userID);
    }
    /**
     * (experimental) Adds federated SAML principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param providerName Name of the SAML provider.
     * @experimental
     */
    forSaml(account, providerName) {
        return this.forFederated(`arn:aws:iam::${account}:saml-provider/${providerName}`);
    }
    /**
     * (experimental) Adds an IAM user principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param user Name of the IAM user.
     * @experimental
     */
    forUser(account, user) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:user/${user}`);
    }
    /**
     * (experimental) Adds an IAM role principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param role Name of the IAM role.
     * @experimental
     */
    forRole(account, role) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:role/${role}`);
    }
    /**
     * (experimental) Adds a specific assumed role session principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param roleName Name of the IAM role.
     * @param sessionName Name of the session.
     * @experimental
     */
    forAssumedRoleSession(account, roleName, sessionName) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:sts::${account}:assumed-role/${roleName}/${sessionName}`);
    }
    /**
     * (experimental) Adds a service principal to the statement.
     *
     * @param service Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`.
     * @experimental
     */
    forService(service) {
        return this.addPrincipal(PrincipalType.SERVICE, service);
    }
    /**
     * (experimental) Grants public asses.
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     *
     * @experimental
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
//# sourceMappingURL=data:application/json;base64,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