import requests
from .product import *
import logging
import json
import time
import prettytable
import sqlite3
import os
import re
import hashlib
import csv
from urllib3.exceptions import ProxyError, HTTPError

logging.basicConfig(level=logging.INFO)


class Subscribe:
    pass


class Page:
    def __init__(self, name=None, url_path=None, locale='cn_zh'):
        self.name = name if name is not None else " "
        self.locale = locale

        if locale == 'cn_zh':
            host = "bck.hermes.cn"
        else:
            host = 'bck.hermes.com'

        self.url = f"https://{host}/{url_path}?locale={locale}"

        if url_path == "product-page":
            self.name = f"product_{locale}_{self.name}"
            self.url += f"&productsku={name}"
        elif url_path == "products":
            self.name = f"products_{locale}_{name}"
            self.url += f"&category={name}"
        elif url_path == "categories":
            self.name = f"categories_{locale}"

        self.json = None
        self.lastmodify = 0
        self.logger = logging.getLogger(self.name)
        # self.load()
        cache_dir="cache"
        if not os.path.exists(cache_dir):
            os.mkdir(cache_dir)

    def fetch(self):
        headers = {
            "Host": "bck.hermes.cn",
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:79.0) Gecko/20100101 Firefox/79.0",
            "Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8",
            "Accept-Language": "zh-CN,zh;q=0.8,zh-TW;q=0.7,zh-HK;q=0.5,en-US;q=0.3,en;q=0.2",
            "Accept-Encoding": "gzip, deflate, br",
            "DNT": "1",
            "Connection": "keep-alive",
            "Cookie": "ECOM_SESS=q4ai07inqkiragm5781ivp6332; datadome=GP17Tfv9Xaotx1f24_tV9p0IJgEOLA0wJVH5B_M8DqWTW_61wO3tXvVakpa210CtMG9Ijm-xQODRWX9UJbmqRXABCy2g.Ooz5rYWk.2lrg",
            "Upgrade-Insecure-Requests": "1",
            "Cache-Control": "max-age=0",
            "TE": "Trailers"
        }
        try:
            resp = requests.get(self.url, headers=headers)
            if resp.status_code == 200:
                self.json = resp.json()
                self.cache()
            elif resp.status_code == 403:
                raise HTTPError("403,被机器识别了，请用代理。")
        except (ProxyError, HTTPError) as err:
            self.logger.info(f"{err}")

    def cache(self):
        file_path = os.path.join("cache", self.name)
        with open(file_path, "w", encoding="utf-8") as f:
            f.write(json.dumps(self.json, indent=4, ensure_ascii="utf-8"))
            self.logger.debug(f"将数据写入缓存 {file_path} ")

    def load(self):
        file_path = os.path.join("cache", self.name)
        if os.path.exists(file_path):
            self.lastmodify = os.path.getmtime(file_path)
            with open(file_path, "r") as f:
                self.json = json.loads(f.read().encode())
                self.logger.debug(f"从缓存中加载数据: {file_path}")
        else:
            self.fetch()


class Hermes:
    def __init__(self):
        self.host = ""
        self._products = []
        self._db = sqlite3.connect("data.db")
        self._pages = []
        self._categories = []

    def _load_products(self, categorey="", locale='cn_zh'):
        """
        加载所有得产品
        """
        page = Page(url_path="products", name=categorey)
        page.load()

        for item in page.json["products"]["items"]:
            # 判断是否存在
            if self.is_exist(item.get("sku")):
                continue

            md5 = hashlib.md5()
            md5.update(json.dumps(item).encode())
            _hash = md5.hexdigest()

            assets = []
            for asset in item.get("assets"):
                a = Asset(asset.get("url"))
                assets.append(a)
            p = Product(sku=item.get("sku").strip(),
                        title=item.get("title", ""),
                        product_code=item.get("productCode", ""),
                        more_colors=item.get("moreColors", ""),
                        size=item.get("size", ""),
                        avg_color=item.get("avgColor", ""),
                        department_code=item.get("departmentCode", ""),
                        family_code=item.get("familyCode", ""),
                        division_code=item.get("divisionCode", ""),
                        price=item.get("price", 0),
                        url=item.get("url", ""),
                        slug=item.get("slug", ""),
                        has_stock=item.get("hasStock", False),
                        has_stock_retail=item.get("hasStockRetail", False),
                        has_stock_or_has_stock_retail=item.get(
                "hasStockOrHasStockRetail", False),
                stock=item.get("stock"),
                personalize=item.get("personalize", False),
                assets=assets,
            )
            p._hash = _hash
            # p.save(self.db)
            self._products.append(p)
        logging.info(f"获取到了{len(self._products)}件商品")

    def is_exist(self, sku):
        """检查商品是否已经存在

        Args:
            sku ([type]): 商品编号

        Returns:
            Boolean: 存在返回True
        """
        for p in self._products:
            if sku == p.sku:
                return True
        return False

    def categories_list(self, locale='cn_zh'):
        """
        获取商城所有的分类
            locals  默认=cn_zh    地区名称
            返回    分类列表
        """
        page = Page(locale=locale, url_path="categories")
        page.load()
        REGX_pincode = r'pimCode": "(\w*)'
        content = json.dumps(page.json)
        m = re.finditer(REGX_pincode, content, flags=re.M)
        for i in m:
            pimCode = i.group(1)
            self._categories.append(Category(pimCode=pimCode))
        logging.info(f"加载了{len(self._categories)}个分类")
        return self._categories

    def products(self, categories=[]):
        """
        加载商品 
            参数
            categories  分类名称的列表

            返回    商品列表
        """
        self._products.clear()

        if len(categories) == 0:
            categories = self.categories_list()

        for c in categories:
            app._load_products(c.pimCode if isinstance(c, Category) else c)

    def has_stock(self, sku, locale=None):
        """
        检查库存

        sku 商品编号
        """
        
        loger=logging.getLogger(f"检查 [{locale}] [{sku}] ")
        if locale is None:
            locale = 'cn_zh'

        page = Page(name=sku, locale=locale, url_path="product-page")
        page.load()
        REGX_STOCK = r'"ecom": true'
        content = json.dumps(page.json)
        m = re.findall(REGX_STOCK, content, re.M)
        if len(m)==0:
            loger.info(f"无货")
        else:
            loger.info(f"有货")
            loger.info(f"点击购买:{page.url}")
            loger.info(f"给订阅者发送邮件..")

    def display(self):
        """
        表格的方式打印商品信息
        """
        field_title = ("#", "SKU", "商品", "Code", "价格", "库存", "颜色", "hash")
        table = prettytable.PrettyTable(field_names=field_title)
        index = 1
        for item in self._products:
            table.add_row([index, item.sku, item.title,
                           item.product_code, item.price, item.has_stock, item.avg_color, item._hash])
            index += 1
        print(table)

    def writeCSV(self, filename="list.csv"):
        """把商品信息写道CSV文件

        Args:
            filename (str, optional): csv的文件名. Defaults to "list.csv".
        """
        with open(filename, "w", newline="", encoding="utf-8") as f:
            cw = csv.writer(f)
            cw.writerow(['SKU', '名字', '产品型号', '价格', '', '',
                         '库存', '零售库存', '家族代码', '可定制', '图片', '官网'])
            for item in self._products:
                try:
                    cw.writerow([item.sku, item.title, item.product_code, item.price, item.avg_color, item.size,
                                 '是' if item.has_stock else '否', '是' if item.has_stock_retail else '否', item.family_code, '是' if item.personalize else "否", f'=HYPERLINK("http:{item.assets[0].url}","查看图片")', ''])
                except Exception as err:
                    logging.debug(f"{err}")

            logging.info(f"文件写入{filename}完成")

    def watch(self, sku, _time=30):
        index = 1
        while True:
            _t=time.strftime("%Y-%m-%d %H-%M-%S",time.localtime())
            logging.info(f"{_t}:正在检查第{index}次")
            self.has_stock(sku)
            self.has_stock(sku, 'us_en')
            self.has_stock(sku, 'hk_en')
            index += 1
            time.sleep(_time)


if __name__ == '__main__':

    logging.basicConfig(level=logging.INFO)
    app = Hermes()

    app.categories_list()

    # app.products(categories=['PRECOLLECTIONPE17'])
    # app.products()

    # 显示商品，可能显示不完。。
    # app.display()
    # app.writeCSV()

    # H079081CCAB
    # app.has_stock('H079081CCAB', locale='us_en')
    # app.has_stock('H077763CKS3')

    app.watch("H077763CKS3")
