"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainTex()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error(`secretId cannot be empty`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || ''
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error(`CloudFormation parameter must be configured with "NoEcho"`);
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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