"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    constructor() {
    }
}
exports.Token = Token;
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return resolve_1.resolve(obj, options);
    }
    constructor() {
    }
}
exports.Tokenization = Tokenization;
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9rZW4uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0b2tlbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLGlEQUFnRDtBQUNoRCxtREFBZ0Q7QUFDaEQsK0NBQTRDO0FBQzVDLG1EQUErQztBQUkvQzs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFhLEtBQUs7SUFDaEI7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBUTtRQUNqQyxPQUFPLHFCQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQVUsRUFBRSxVQUEyQixFQUFFO1FBQzlELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxFQUFFO1lBQUUsT0FBTyxLQUFLLENBQUM7U0FBRTtRQUNoRCxPQUFPLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUUsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBVTtRQUMvQixJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRTtZQUFFLE9BQU8sS0FBSyxDQUFDO1NBQUU7UUFDaEQsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFVLEVBQUUsVUFBMkIsRUFBRTtRQUM1RCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLFFBQVEsQ0FBQyxFQUFFO1lBQUUsT0FBTyxLQUFLLENBQUM7U0FBRTtRQUN0RixPQUFPLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUUsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ25GLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBVTtRQUM1QixPQUFPLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUkscUJBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7SUFDQSxDQUFDO0NBQ0Y7QUE1REQsc0JBNERDO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLFlBQVk7SUFDdkI7O09BRUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLENBQVM7UUFDbkMsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLENBQVM7UUFDbkMsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBVztRQUNuQyxPQUFPLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQVEsRUFBRSxPQUF1QjtRQUNyRCxPQUFPLGlCQUFPLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDtJQUNBLENBQUM7Q0FDRjtBQW5DRCxvQ0FtQ0M7QUFnQ0QsU0FBZ0Isa0JBQWtCLENBQUMsQ0FBTTtJQUN2QyxPQUFPLE9BQU0sQ0FBQyxDQUFDLENBQUMsS0FBSyxRQUFRLElBQUksQ0FBQyxLQUFLLElBQUksSUFBSSxPQUFPLENBQUMsQ0FBQyxPQUFPLEtBQUssVUFBVSxDQUFDO0FBQ2pGLENBQUM7QUFGRCxnREFFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElDb25zdHJ1Y3QgfSBmcm9tIFwiLi9jb25zdHJ1Y3RcIjtcbmltcG9ydCB7IHVucmVzb2x2ZWQgfSBmcm9tIFwiLi9wcml2YXRlL2VuY29kaW5nXCI7XG5pbXBvcnQgeyBJbnRyaW5zaWMgfSBmcm9tIFwiLi9wcml2YXRlL2ludHJpbnNpY1wiO1xuaW1wb3J0IHsgcmVzb2x2ZSB9IGZyb20gXCIuL3ByaXZhdGUvcmVzb2x2ZVwiO1xuaW1wb3J0IHsgVG9rZW5NYXAgfSBmcm9tIFwiLi9wcml2YXRlL3Rva2VuLW1hcFwiO1xuaW1wb3J0IHsgSVJlc29sdmFibGUsIElUb2tlblJlc29sdmVyIH0gZnJvbSBcIi4vcmVzb2x2YWJsZVwiO1xuaW1wb3J0IHsgVG9rZW5pemVkU3RyaW5nRnJhZ21lbnRzIH0gZnJvbSBcIi4vc3RyaW5nLWZyYWdtZW50c1wiO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBzcGVjaWFsIG9yIGxhemlseS1ldmFsdWF0ZWQgdmFsdWUuXG4gKlxuICogQ2FuIGJlIHVzZWQgdG8gZGVsYXkgZXZhbHVhdGlvbiBvZiBhIGNlcnRhaW4gdmFsdWUgaW4gY2FzZSwgZm9yIGV4YW1wbGUsXG4gKiB0aGF0IGl0IHJlcXVpcmVzIHNvbWUgY29udGV4dCBvciBsYXRlLWJvdW5kIGRhdGEuIENhbiBhbHNvIGJlIHVzZWQgdG9cbiAqIG1hcmsgdmFsdWVzIHRoYXQgbmVlZCBzcGVjaWFsIHByb2Nlc3NpbmcgYXQgZG9jdW1lbnQgcmVuZGVyaW5nIHRpbWUuXG4gKlxuICogVG9rZW5zIGNhbiBiZSBlbWJlZGRlZCBpbnRvIHN0cmluZ3Mgd2hpbGUgcmV0YWluaW5nIHRoZWlyIG9yaWdpbmFsXG4gKiBzZW1hbnRpY3MuXG4gKi9cbmV4cG9ydCBjbGFzcyBUb2tlbiB7XG4gIC8qKlxuICAgKiBSZXR1cm5zIHRydWUgaWYgb2JqIHJlcHJlc2VudHMgYW4gdW5yZXNvbHZlZCB2YWx1ZVxuICAgKlxuICAgKiBPbmUgb2YgdGhlc2UgbXVzdCBiZSB0cnVlOlxuICAgKlxuICAgKiAtIGBvYmpgIGlzIGFuIElSZXNvbHZhYmxlXG4gICAqIC0gYG9iamAgaXMgYSBzdHJpbmcgY29udGFpbmluZyBhdCBsZWFzdCBvbmUgZW5jb2RlZCBgSVJlc29sdmFibGVgXG4gICAqIC0gYG9iamAgaXMgZWl0aGVyIGFuIGVuY29kZWQgbnVtYmVyIG9yIGxpc3RcbiAgICpcbiAgICogVGhpcyBkb2VzIE5PVCByZWN1cnNlIGludG8gbGlzdHMgb3Igb2JqZWN0cyB0byBzZWUgaWYgdGhleVxuICAgKiBjb250YWluaW5nIHJlc29sdmFibGVzLlxuICAgKlxuICAgKiBAcGFyYW0gb2JqIFRoZSBvYmplY3QgdG8gdGVzdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaXNVbnJlc29sdmVkKG9iajogYW55KTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHVucmVzb2x2ZWQob2JqKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIHN0cmluZyByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqXG4gICAqIElmIHRoZSBUb2tlbiBpcyBpbml0aWFsaXplZCB3aXRoIGEgbGl0ZXJhbCwgdGhlIHN0cmluZ2lmaWVkIHZhbHVlIG9mIHRoZVxuICAgKiBsaXRlcmFsIGlzIHJldHVybmVkLiBPdGhlcndpc2UsIGEgc3BlY2lhbCBxdW90ZWQgc3RyaW5nIHJlcHJlc2VudGF0aW9uXG4gICAqIG9mIHRoZSBUb2tlbiBpcyByZXR1cm5lZCB0aGF0IGNhbiBiZSBlbWJlZGRlZCBpbnRvIG90aGVyIHN0cmluZ3MuXG4gICAqXG4gICAqIFN0cmluZ3Mgd2l0aCBxdW90ZWQgVG9rZW5zIGluIHRoZW0gY2FuIGJlIHJlc3RvcmVkIGJhY2sgaW50b1xuICAgKiBjb21wbGV4IHZhbHVlcyB3aXRoIHRoZSBUb2tlbnMgcmVzdG9yZWQgYnkgY2FsbGluZyBgcmVzb2x2ZSgpYFxuICAgKiBvbiB0aGUgc3RyaW5nLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc1N0cmluZyh2YWx1ZTogYW55LCBvcHRpb25zOiBFbmNvZGluZ09wdGlvbnMgPSB7fSk6IHN0cmluZyB7XG4gICAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gJ3N0cmluZycpIHsgcmV0dXJuIHZhbHVlOyB9XG4gICAgcmV0dXJuIFRva2VuTWFwLmluc3RhbmNlKCkucmVnaXN0ZXJTdHJpbmcoVG9rZW4uYXNBbnkodmFsdWUpLCBvcHRpb25zLmRpc3BsYXlIaW50KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIG51bWJlciByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFzTnVtYmVyKHZhbHVlOiBhbnkpOiBudW1iZXIge1xuICAgIGlmICh0eXBlb2YgdmFsdWUgPT09ICdudW1iZXInKSB7IHJldHVybiB2YWx1ZTsgfVxuICAgIHJldHVybiBUb2tlbk1hcC5pbnN0YW5jZSgpLnJlZ2lzdGVyTnVtYmVyKFRva2VuLmFzQW55KHZhbHVlKSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgcmV2ZXJzaWJsZSBsaXN0IHJlcHJlc2VudGF0aW9uIG9mIHRoaXMgdG9rZW5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXNMaXN0KHZhbHVlOiBhbnksIG9wdGlvbnM6IEVuY29kaW5nT3B0aW9ucyA9IHt9KTogc3RyaW5nW10ge1xuICAgIGlmIChBcnJheS5pc0FycmF5KHZhbHVlKSAmJiB2YWx1ZS5ldmVyeSh4ID0+IHR5cGVvZiB4ID09PSAnc3RyaW5nJykpIHsgcmV0dXJuIHZhbHVlOyB9XG4gICAgcmV0dXJuIFRva2VuTWFwLmluc3RhbmNlKCkucmVnaXN0ZXJMaXN0KFRva2VuLmFzQW55KHZhbHVlKSwgb3B0aW9ucy5kaXNwbGF5SGludCk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgcmVzb2x2YWJsZSByZXByZXNlbnRhdGlvbiBvZiB0aGUgZ2l2ZW4gdmFsdWVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXNBbnkodmFsdWU6IGFueSk6IElSZXNvbHZhYmxlIHtcbiAgICByZXR1cm4gaXNSZXNvbHZhYmxlT2JqZWN0KHZhbHVlKSA/IHZhbHVlIDogbmV3IEludHJpbnNpYyh2YWx1ZSk7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKCkge1xuICB9XG59XG5cbi8qKlxuICogTGVzcyBvZnQtbmVlZGVkIGZ1bmN0aW9ucyB0byBtYW5pcHVsYXRlIFRva2Vuc1xuICovXG5leHBvcnQgY2xhc3MgVG9rZW5pemF0aW9uIHtcbiAgLyoqXG4gICAqIFVuLWVuY29kZSBhIHN0cmluZyBwb3RlbnRpYWxseSBjb250YWluaW5nIGVuY29kZWQgdG9rZW5zXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJldmVyc2VTdHJpbmcoczogc3RyaW5nKTogVG9rZW5pemVkU3RyaW5nRnJhZ21lbnRzIHtcbiAgICByZXR1cm4gVG9rZW5NYXAuaW5zdGFuY2UoKS5zcGxpdFN0cmluZyhzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVbi1lbmNvZGUgYSBUb2tlbml6ZWQgdmFsdWUgZnJvbSBhIG51bWJlclxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZXZlcnNlTnVtYmVyKG46IG51bWJlcik6IElSZXNvbHZhYmxlIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gVG9rZW5NYXAuaW5zdGFuY2UoKS5sb29rdXBOdW1iZXJUb2tlbihuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVbi1lbmNvZGUgYSBUb2tlbml6ZWQgdmFsdWUgZnJvbSBhIGxpc3RcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmV2ZXJzZUxpc3QobDogc3RyaW5nW10pOiBJUmVzb2x2YWJsZSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIFRva2VuTWFwLmluc3RhbmNlKCkubG9va3VwTGlzdChsKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXNvbHZlcyBhbiBvYmplY3QgYnkgZXZhbHVhdGluZyBhbGwgdG9rZW5zIGFuZCByZW1vdmluZyBhbnkgdW5kZWZpbmVkIG9yIGVtcHR5IG9iamVjdHMgb3IgYXJyYXlzLlxuICAgKiBWYWx1ZXMgY2FuIG9ubHkgYmUgcHJpbWl0aXZlcywgYXJyYXlzIG9yIHRva2Vucy4gT3RoZXIgb2JqZWN0cyAoaS5lLiB3aXRoIG1ldGhvZHMpIHdpbGwgYmUgcmVqZWN0ZWQuXG4gICAqXG4gICAqIEBwYXJhbSBvYmogVGhlIG9iamVjdCB0byByZXNvbHZlLlxuICAgKiBAcGFyYW0gb3B0aW9ucyBQcmVmaXgga2V5IHBhdGggY29tcG9uZW50cyBmb3IgZGlhZ25vc3RpY3MuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlc29sdmUob2JqOiBhbnksIG9wdGlvbnM6IFJlc29sdmVPcHRpb25zKTogYW55IHtcbiAgICByZXR1cm4gcmVzb2x2ZShvYmosIG9wdGlvbnMpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcigpIHtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gdGhlIHJlc29sdmUoKSBvcGVyYXRpb25cbiAqXG4gKiBOT1QgdGhlIHNhbWUgYXMgdGhlIFJlc29sdmVDb250ZXh0OyBSZXNvbHZlQ29udGV4dCBpcyBleHBvc2VkIHRvIFRva2VuXG4gKiBpbXBsZW1lbnRvcnMgYW5kIHJlc29sdXRpb24gaG9va3MsIHdoZXJlYXMgdGhpcyBzdHJ1Y3QgaXMganVzdCB0byBidW5kbGVcbiAqIGEgbnVtYmVyIG9mIHRoaW5ncyB0aGF0IHdvdWxkIG90aGVyd2lzZSBiZSBhcmd1bWVudHMgdG8gcmVzb2x2ZSgpIGluIGFcbiAqIHJlYWRhYmxlIHdheS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZXNvbHZlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgc2NvcGUgZnJvbSB3aGljaCByZXNvbHV0aW9uIGlzIHBlcmZvcm1lZFxuICAgKi9cbiAgcmVhZG9ubHkgc2NvcGU6IElDb25zdHJ1Y3Q7XG5cbiAgLyoqXG4gICAqIFRoZSByZXNvbHZlciB0byBhcHBseSB0byBhbnkgcmVzb2x2YWJsZSB0b2tlbnMgZm91bmRcbiAgICovXG4gIHJlYWRvbmx5IHJlc29sdmVyOiBJVG9rZW5SZXNvbHZlcjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIHN0cmluZyBlbmNvZGluZ3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmNvZGluZ09wdGlvbnMge1xuICAvKipcbiAgICogQSBoaW50IGZvciB0aGUgVG9rZW4ncyBwdXJwb3NlIHdoZW4gc3RyaW5naWZ5aW5nIGl0XG4gICAqL1xuICByZWFkb25seSBkaXNwbGF5SGludD86IHN0cmluZztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzUmVzb2x2YWJsZU9iamVjdCh4OiBhbnkpOiB4IGlzIElSZXNvbHZhYmxlIHtcbiAgcmV0dXJuIHR5cGVvZih4KSA9PT0gJ29iamVjdCcgJiYgeCAhPT0gbnVsbCAmJiB0eXBlb2YgeC5yZXNvbHZlID09PSAnZnVuY3Rpb24nO1xufSJdfQ==