"use strict";
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const util_1 = require("./util");
module.exports = {
    'create from components with defaults'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'sqs',
            resource: 'myqueuename'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:sqs:${pseudo.region}:${pseudo.accountId}:myqueuename`));
        test.done();
    },
    'create from components with specific values for the various components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'dynamodb',
            resource: 'table',
            account: '123456789012',
            region: 'us-east-1',
            partition: 'aws-cn',
            resourceName: 'mytable/stream/label'
        });
        test.deepEqual(stack.resolve(arn), 'arn:aws-cn:dynamodb:us-east-1:123456789012:table/mytable/stream/label');
        test.done();
    },
    'allow empty string in components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 's3',
            resource: 'my-bucket',
            account: '',
            region: '',
            partition: 'aws-cn',
        });
        test.deepEqual(stack.resolve(arn), 'arn:aws-cn:s3:::my-bucket');
        test.done();
    },
    'resourcePathSep can be set to ":" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'codedeploy',
            resource: 'application',
            sep: ':',
            resourceName: 'WordPress_App'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:codedeploy:${pseudo.region}:${pseudo.accountId}:application:WordPress_App`));
        test.done();
    },
    'resourcePathSep can be set to "" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'ssm',
            resource: 'parameter',
            sep: '',
            resourceName: '/parameter-name'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:ssm:${pseudo.region}:${pseudo.accountId}:parameter/parameter-name`));
        test.done();
    },
    'fails if resourcePathSep is neither ":" nor "/"'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => stack.formatArn({
            service: 'foo',
            resource: 'bar',
            sep: 'x'
        }));
        test.done();
    },
    'Arn.parse(s)': {
        'fails': {
            'if doesn\'t start with "arn:"'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn("barn:foo:x:a:1:2"), /ARNs must start with "arn:": barn:foo/);
                test.done();
            },
            'if the ARN doesnt have enough components'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:is:too:short'), /ARNs must have at least 6 components: arn:is:too:short/);
                test.done();
            },
            'if "service" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws::4:5:6'), /The `service` component \(3rd component\) is required/);
                test.done();
            },
            'if "resource" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws:service:::'), /The `resource` component \(6th component\) is required/);
                test.done();
            }
        },
        'various successful parses'(test) {
            const stack = new lib_1.Stack();
            const tests = {
                'arn:aws:a4b:region:accountid:resourcetype/resource': {
                    partition: 'aws',
                    service: 'a4b',
                    region: 'region',
                    account: 'accountid',
                    resource: 'resourcetype',
                    resourceName: 'resource',
                    sep: '/'
                },
                'arn:aws:apigateway:us-east-1::a123456789012bc3de45678901f23a45:/test/mydemoresource/*': {
                    partition: 'aws',
                    service: 'apigateway',
                    region: 'us-east-1',
                    resource: 'a123456789012bc3de45678901f23a45',
                    sep: ':',
                    resourceName: '/test/mydemoresource/*'
                },
                'arn:aws-cn:cloud9::123456789012:environment:81e900317347585a0601e04c8d52eaEX': {
                    partition: 'aws-cn',
                    service: 'cloud9',
                    account: '123456789012',
                    resource: 'environment',
                    resourceName: '81e900317347585a0601e04c8d52eaEX',
                    sep: ':'
                },
                'arn::cognito-sync:::identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla': {
                    service: 'cognito-sync',
                    resource: 'identitypool',
                    resourceName: 'us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla',
                    sep: '/'
                },
                'arn:aws:s3:::my_corporate_bucket': {
                    partition: 'aws',
                    service: 's3',
                    resource: 'my_corporate_bucket'
                }
            };
            Object.keys(tests).forEach(arn => {
                const expected = tests[arn];
                test.deepEqual(stack.parseArn(arn), expected, arn);
            });
            test.done();
        },
        'a Token with : separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new intrinsic_1.Intrinsic(theToken).toString(), ':');
            test.deepEqual(stack.resolve(parsed.partition), { 'Fn::Select': [1, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.service), { 'Fn::Select': [2, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.region), { 'Fn::Select': [3, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.account), { 'Fn::Select': [4, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.resource), { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.resourceName), { 'Fn::Select': [6, { 'Fn::Split': [':', theToken] }] });
            test.equal(parsed.sep, ':');
            test.done();
        },
        'a Token with / separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new intrinsic_1.Intrinsic(theToken).toString());
            test.equal(parsed.sep, '/');
            // tslint:disable-next-line:max-line-length
            test.deepEqual(stack.resolve(parsed.resource), { 'Fn::Select': [0, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            // tslint:disable-next-line:max-line-length
            test.deepEqual(stack.resolve(parsed.resourceName), { 'Fn::Select': [1, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            test.done();
        }
    },
    'can use a fully specified ARN from a different stack without incurring an import'(test) {
        // GIVEN
        const stack1 = new lib_1.Stack(undefined, 'Stack1', { env: { account: '12345678', region: 'us-turbo-5' } });
        const stack2 = new lib_1.Stack(undefined, 'Stack2', { env: { account: '87654321', region: 'us-turbo-1' } });
        // WHEN
        const arn = stack1.formatArn({
            // No partition specified here
            service: 'bla',
            resource: 'thing',
            resourceName: 'thong',
        });
        new lib_1.CfnOutput(stack2, 'SomeValue', { value: arn });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack2), {
            Outputs: {
                SomeValue: {
                    Value: {
                        // Look ma, no Fn::ImportValue!
                        'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':bla:us-turbo-5:12345678:thing/thong']]
                    }
                }
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5hcm4uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmFybi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQ0EsZ0NBQW9FO0FBQ3BFLHdEQUFxRDtBQUNyRCxpQ0FBMEM7QUFFMUMsaUJBQVM7SUFDUCxzQ0FBc0MsQ0FBQyxJQUFVO1FBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUMxQixPQUFPLEVBQUUsS0FBSztZQUNkLFFBQVEsRUFBRSxhQUFhO1NBQ3hCLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLElBQUksZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXBDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFDbEIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLE1BQU0sQ0FBQyxTQUFTLFFBQVEsTUFBTSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsU0FBUyxjQUFjLENBQUMsQ0FBQyxDQUFDO1FBQzlHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3RUFBd0UsQ0FBQyxJQUFVO1FBQ2pGLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUMxQixPQUFPLEVBQUUsVUFBVTtZQUNuQixRQUFRLEVBQUUsT0FBTztZQUNqQixPQUFPLEVBQUUsY0FBYztZQUN2QixNQUFNLEVBQUUsV0FBVztZQUNuQixTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsc0JBQXNCO1NBQ3JDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFDbEIsdUVBQXVFLENBQUMsQ0FBQztRQUN4RixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0NBQWtDLENBQUMsSUFBVTtRQUMzQyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBRTFCLE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDMUIsT0FBTyxFQUFFLElBQUk7WUFDYixRQUFRLEVBQUUsV0FBVztZQUNyQixPQUFPLEVBQUUsRUFBRTtZQUNYLE1BQU0sRUFBRSxFQUFFO1lBQ1YsU0FBUyxFQUFFLFFBQVE7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUNsQiwyQkFBMkIsQ0FBQyxDQUFDO1FBRTVDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4REFBOEQsQ0FBQyxJQUFVO1FBQ3ZFLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUMxQixPQUFPLEVBQUUsWUFBWTtZQUNyQixRQUFRLEVBQUUsYUFBYTtZQUN2QixHQUFHLEVBQUUsR0FBRztZQUNSLFlBQVksRUFBRSxlQUFlO1NBQzlCLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLElBQUksZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXBDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFDbEIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLE1BQU0sQ0FBQyxTQUFTLGVBQWUsTUFBTSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsU0FBUyw0QkFBNEIsQ0FBQyxDQUFDLENBQUM7UUFDbkksSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZEQUE2RCxDQUFDLElBQVU7UUFDdEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQzFCLE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLFdBQVc7WUFDckIsR0FBRyxFQUFFLEVBQUU7WUFDUCxZQUFZLEVBQUUsaUJBQWlCO1NBQ2hDLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLElBQUksZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXBDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFDbEIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLE1BQU0sQ0FBQyxTQUFTLFFBQVEsTUFBTSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsU0FBUywyQkFBMkIsQ0FBQyxDQUFDLENBQUM7UUFDM0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlEQUFpRCxDQUFDLElBQVU7UUFDMUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDaEMsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsS0FBSztZQUNmLEdBQUcsRUFBRSxHQUFHO1NBQUUsQ0FBQyxDQUFDLENBQUM7UUFDZixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsY0FBYyxFQUFFO1FBRWQsT0FBTyxFQUFFO1lBQ1AsK0JBQStCLENBQUMsSUFBVTtnQkFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztnQkFDMUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEVBQUUsdUNBQXVDLENBQUMsQ0FBQztnQkFDL0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELDBDQUEwQyxDQUFDLElBQVU7Z0JBQ25ELE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7Z0JBQzFCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLHdEQUF3RCxDQUFDLENBQUM7Z0JBQ2hILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCwrQkFBK0IsQ0FBQyxJQUFVO2dCQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO2dCQUMxQixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsRUFBRSx1REFBdUQsQ0FBQyxDQUFDO2dCQUM3RyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1lBRUQsZ0NBQWdDLENBQUMsSUFBVTtnQkFDekMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztnQkFDMUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDLEVBQUUsd0RBQXdELENBQUMsQ0FBQztnQkFDbEgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO1FBRUQsMkJBQTJCLENBQUMsSUFBVTtZQUNwQyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sS0FBSyxHQUFxQztnQkFDOUMsb0RBQW9ELEVBQUU7b0JBQ3BELFNBQVMsRUFBRSxLQUFLO29CQUNoQixPQUFPLEVBQUUsS0FBSztvQkFDZCxNQUFNLEVBQUUsUUFBUTtvQkFDaEIsT0FBTyxFQUFFLFdBQVc7b0JBQ3BCLFFBQVEsRUFBRSxjQUFjO29CQUN4QixZQUFZLEVBQUUsVUFBVTtvQkFDeEIsR0FBRyxFQUFFLEdBQUc7aUJBQ1Q7Z0JBQ0QsdUZBQXVGLEVBQUU7b0JBQ3ZGLFNBQVMsRUFBRSxLQUFLO29CQUNoQixPQUFPLEVBQUUsWUFBWTtvQkFDckIsTUFBTSxFQUFFLFdBQVc7b0JBQ25CLFFBQVEsRUFBRSxrQ0FBa0M7b0JBQzVDLEdBQUcsRUFBRSxHQUFHO29CQUNSLFlBQVksRUFBRSx3QkFBd0I7aUJBQ3ZDO2dCQUNELDhFQUE4RSxFQUFFO29CQUM5RSxTQUFTLEVBQUUsUUFBUTtvQkFDbkIsT0FBTyxFQUFFLFFBQVE7b0JBQ2pCLE9BQU8sRUFBRSxjQUFjO29CQUN2QixRQUFRLEVBQUUsYUFBYTtvQkFDdkIsWUFBWSxFQUFFLGtDQUFrQztvQkFDaEQsR0FBRyxFQUFFLEdBQUc7aUJBQ1Q7Z0JBQ0QsaUZBQWlGLEVBQUU7b0JBQ2pGLE9BQU8sRUFBRSxjQUFjO29CQUN2QixRQUFRLEVBQUUsY0FBYztvQkFDeEIsWUFBWSxFQUFFLGdEQUFnRDtvQkFDOUQsR0FBRyxFQUFFLEdBQUc7aUJBQ1Q7Z0JBQ0Qsa0NBQWtDLEVBQUU7b0JBQ2xDLFNBQVMsRUFBRSxLQUFLO29CQUNoQixPQUFPLEVBQUUsSUFBSTtvQkFDYixRQUFRLEVBQUUscUJBQXFCO2lCQUNoQzthQUNGLENBQUM7WUFFRixNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDL0IsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUM1QixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEVBQUUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQ3JELENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDBCQUEwQixDQUFDLElBQVU7WUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLFFBQVEsR0FBRyxFQUFFLEdBQUcsRUFBRSxlQUFlLEVBQUUsQ0FBQztZQUMxQyxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUkscUJBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFBRSxHQUFHLENBQUMsQ0FBQztZQUV2RSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsV0FBVyxFQUFFLENBQUUsR0FBRyxFQUFFLFFBQVEsQ0FBRSxFQUFDLENBQUUsRUFBQyxDQUFDLENBQUM7WUFDM0csSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFFLEdBQUcsRUFBRSxRQUFRLENBQUUsRUFBQyxDQUFFLEVBQUMsQ0FBQyxDQUFDO1lBQ3pHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxXQUFXLEVBQUUsQ0FBRSxHQUFHLEVBQUUsUUFBUSxDQUFFLEVBQUMsQ0FBRSxFQUFDLENBQUMsQ0FBQztZQUN4RyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsV0FBVyxFQUFFLENBQUUsR0FBRyxFQUFFLFFBQVEsQ0FBRSxFQUFDLENBQUUsRUFBQyxDQUFDLENBQUM7WUFDekcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFFLEdBQUcsRUFBRSxRQUFRLENBQUUsRUFBQyxDQUFFLEVBQUMsQ0FBQyxDQUFDO1lBQzFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxXQUFXLEVBQUUsQ0FBRSxHQUFHLEVBQUUsUUFBUSxDQUFFLEVBQUMsQ0FBRSxFQUFDLENBQUMsQ0FBQztZQUM5RyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFFNUIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDBCQUEwQixDQUFDLElBQVU7WUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLFFBQVEsR0FBRyxFQUFFLEdBQUcsRUFBRSxlQUFlLEVBQUUsQ0FBQztZQUMxQyxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUkscUJBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1lBRWxFLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztZQUU1QiwyQ0FBMkM7WUFDM0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFFLEdBQUcsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFFLEdBQUcsRUFBRSxRQUFRLENBQUUsRUFBQyxDQUFFLEVBQUMsQ0FBRSxFQUFDLENBQUUsRUFBQyxDQUFDLENBQUM7WUFDM0osMkNBQTJDO1lBQzNDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxXQUFXLEVBQUUsQ0FBRSxHQUFHLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxXQUFXLEVBQUUsQ0FBRSxHQUFHLEVBQUUsUUFBUSxDQUFFLEVBQUMsQ0FBRSxFQUFDLENBQUUsRUFBQyxDQUFFLEVBQUMsQ0FBQyxDQUFDO1lBRS9KLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELGtGQUFrRixDQUFDLElBQVU7UUFDM0YsUUFBUTtRQUNSLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxRQUFRLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxZQUFZLEVBQUUsRUFBQyxDQUFDLENBQUM7UUFDckcsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsU0FBUyxFQUFFLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFDLENBQUMsQ0FBQztRQUVyRyxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLFNBQVMsQ0FBQztZQUMzQiw4QkFBOEI7WUFDOUIsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsT0FBTztZQUNqQixZQUFZLEVBQUUsT0FBTztTQUN0QixDQUFDLENBQUM7UUFDSCxJQUFJLGVBQVMsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFFbkQsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsdUJBQWdCLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDdkMsT0FBTyxFQUFFO2dCQUNQLFNBQVMsRUFBRTtvQkFDVCxLQUFLLEVBQUU7d0JBQ0wsK0JBQStCO3dCQUMvQixVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO3FCQUFFO2lCQUNsRzthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQXJuQ29tcG9uZW50cywgQ2ZuT3V0cHV0LCBTY29wZWRBd3MsIFN0YWNrIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IEludHJpbnNpYyB9IGZyb20gJy4uL2xpYi9wcml2YXRlL2ludHJpbnNpYyc7XG5pbXBvcnQgeyB0b0Nsb3VkRm9ybWF0aW9uIH0gZnJvbSAnLi91dGlsJztcblxuZXhwb3J0ID0ge1xuICAnY3JlYXRlIGZyb20gY29tcG9uZW50cyB3aXRoIGRlZmF1bHRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGFybiA9IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnc3FzJyxcbiAgICAgIHJlc291cmNlOiAnbXlxdWV1ZW5hbWUnXG4gICAgfSk7XG5cbiAgICBjb25zdCBwc2V1ZG8gPSBuZXcgU2NvcGVkQXdzKHN0YWNrKTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUoYXJuKSxcbiAgICAgICAgICAgICAgICAgICBzdGFjay5yZXNvbHZlKGBhcm46JHtwc2V1ZG8ucGFydGl0aW9ufTpzcXM6JHtwc2V1ZG8ucmVnaW9ufToke3BzZXVkby5hY2NvdW50SWR9Om15cXVldWVuYW1lYCkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjcmVhdGUgZnJvbSBjb21wb25lbnRzIHdpdGggc3BlY2lmaWMgdmFsdWVzIGZvciB0aGUgdmFyaW91cyBjb21wb25lbnRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGFybiA9IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnZHluYW1vZGInLFxuICAgICAgcmVzb3VyY2U6ICd0YWJsZScsXG4gICAgICBhY2NvdW50OiAnMTIzNDU2Nzg5MDEyJyxcbiAgICAgIHJlZ2lvbjogJ3VzLWVhc3QtMScsXG4gICAgICBwYXJ0aXRpb246ICdhd3MtY24nLFxuICAgICAgcmVzb3VyY2VOYW1lOiAnbXl0YWJsZS9zdHJlYW0vbGFiZWwnXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKGFybiksXG4gICAgICAgICAgICAgICAgICAgJ2Fybjphd3MtY246ZHluYW1vZGI6dXMtZWFzdC0xOjEyMzQ1Njc4OTAxMjp0YWJsZS9teXRhYmxlL3N0cmVhbS9sYWJlbCcpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhbGxvdyBlbXB0eSBzdHJpbmcgaW4gY29tcG9uZW50cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBjb25zdCBhcm4gPSBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ3MzJyxcbiAgICAgIHJlc291cmNlOiAnbXktYnVja2V0JyxcbiAgICAgIGFjY291bnQ6ICcnLFxuICAgICAgcmVnaW9uOiAnJyxcbiAgICAgIHBhcnRpdGlvbjogJ2F3cy1jbicsXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKGFybiksXG4gICAgICAgICAgICAgICAgICAgJ2Fybjphd3MtY246czM6OjpteS1idWNrZXQnKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdyZXNvdXJjZVBhdGhTZXAgY2FuIGJlIHNldCB0byBcIjpcIiBpbnN0ZWFkIG9mIHRoZSBkZWZhdWx0IFwiL1wiJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGFybiA9IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnY29kZWRlcGxveScsXG4gICAgICByZXNvdXJjZTogJ2FwcGxpY2F0aW9uJyxcbiAgICAgIHNlcDogJzonLFxuICAgICAgcmVzb3VyY2VOYW1lOiAnV29yZFByZXNzX0FwcCdcbiAgICB9KTtcblxuICAgIGNvbnN0IHBzZXVkbyA9IG5ldyBTY29wZWRBd3Moc3RhY2spO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShhcm4pLFxuICAgICAgICAgICAgICAgICAgIHN0YWNrLnJlc29sdmUoYGFybjoke3BzZXVkby5wYXJ0aXRpb259OmNvZGVkZXBsb3k6JHtwc2V1ZG8ucmVnaW9ufToke3BzZXVkby5hY2NvdW50SWR9OmFwcGxpY2F0aW9uOldvcmRQcmVzc19BcHBgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Jlc291cmNlUGF0aFNlcCBjYW4gYmUgc2V0IHRvIFwiXCIgaW5zdGVhZCBvZiB0aGUgZGVmYXVsdCBcIi9cIicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBjb25zdCBhcm4gPSBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ3NzbScsXG4gICAgICByZXNvdXJjZTogJ3BhcmFtZXRlcicsXG4gICAgICBzZXA6ICcnLFxuICAgICAgcmVzb3VyY2VOYW1lOiAnL3BhcmFtZXRlci1uYW1lJ1xuICAgIH0pO1xuXG4gICAgY29uc3QgcHNldWRvID0gbmV3IFNjb3BlZEF3cyhzdGFjayk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKGFybiksXG4gICAgICAgICAgICAgICAgICAgc3RhY2sucmVzb2x2ZShgYXJuOiR7cHNldWRvLnBhcnRpdGlvbn06c3NtOiR7cHNldWRvLnJlZ2lvbn06JHtwc2V1ZG8uYWNjb3VudElkfTpwYXJhbWV0ZXIvcGFyYW1ldGVyLW5hbWVgKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ZhaWxzIGlmIHJlc291cmNlUGF0aFNlcCBpcyBuZWl0aGVyIFwiOlwiIG5vciBcIi9cIicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2ZvbycsXG4gICAgICByZXNvdXJjZTogJ2JhcicsXG4gICAgICBzZXA6ICd4JyB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0Fybi5wYXJzZShzKSc6IHtcblxuICAgICdmYWlscyc6IHtcbiAgICAgICdpZiBkb2VzblxcJ3Qgc3RhcnQgd2l0aCBcImFybjpcIicodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgICB0ZXN0LnRocm93cygoKSA9PiBzdGFjay5wYXJzZUFybihcImJhcm46Zm9vOng6YToxOjJcIiksIC9BUk5zIG11c3Qgc3RhcnQgd2l0aCBcImFybjpcIjogYmFybjpmb28vKTtcbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuXG4gICAgICAnaWYgdGhlIEFSTiBkb2VzbnQgaGF2ZSBlbm91Z2ggY29tcG9uZW50cycodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgICB0ZXN0LnRocm93cygoKSA9PiBzdGFjay5wYXJzZUFybignYXJuOmlzOnRvbzpzaG9ydCcpLCAvQVJOcyBtdXN0IGhhdmUgYXQgbGVhc3QgNiBjb21wb25lbnRzOiBhcm46aXM6dG9vOnNob3J0Lyk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgJ2lmIFwic2VydmljZVwiIGlzIG5vdCBzcGVjaWZpZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgICAgdGVzdC50aHJvd3MoKCkgPT4gc3RhY2sucGFyc2VBcm4oJ2Fybjphd3M6OjQ6NTo2JyksIC9UaGUgYHNlcnZpY2VgIGNvbXBvbmVudCBcXCgzcmQgY29tcG9uZW50XFwpIGlzIHJlcXVpcmVkLyk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgJ2lmIFwicmVzb3VyY2VcIiBpcyBub3Qgc3BlY2lmaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICAgIHRlc3QudGhyb3dzKCgpID0+IHN0YWNrLnBhcnNlQXJuKCdhcm46YXdzOnNlcnZpY2U6OjonKSwgL1RoZSBgcmVzb3VyY2VgIGNvbXBvbmVudCBcXCg2dGggY29tcG9uZW50XFwpIGlzIHJlcXVpcmVkLyk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfVxuICAgIH0sXG5cbiAgICAndmFyaW91cyBzdWNjZXNzZnVsIHBhcnNlcycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRlc3RzOiB7IFthcm46IHN0cmluZ106IEFybkNvbXBvbmVudHMgfSA9IHtcbiAgICAgICAgJ2Fybjphd3M6YTRiOnJlZ2lvbjphY2NvdW50aWQ6cmVzb3VyY2V0eXBlL3Jlc291cmNlJzoge1xuICAgICAgICAgIHBhcnRpdGlvbjogJ2F3cycsXG4gICAgICAgICAgc2VydmljZTogJ2E0YicsXG4gICAgICAgICAgcmVnaW9uOiAncmVnaW9uJyxcbiAgICAgICAgICBhY2NvdW50OiAnYWNjb3VudGlkJyxcbiAgICAgICAgICByZXNvdXJjZTogJ3Jlc291cmNldHlwZScsXG4gICAgICAgICAgcmVzb3VyY2VOYW1lOiAncmVzb3VyY2UnLFxuICAgICAgICAgIHNlcDogJy8nXG4gICAgICAgIH0sXG4gICAgICAgICdhcm46YXdzOmFwaWdhdGV3YXk6dXMtZWFzdC0xOjphMTIzNDU2Nzg5MDEyYmMzZGU0NTY3ODkwMWYyM2E0NTovdGVzdC9teWRlbW9yZXNvdXJjZS8qJzoge1xuICAgICAgICAgIHBhcnRpdGlvbjogJ2F3cycsXG4gICAgICAgICAgc2VydmljZTogJ2FwaWdhdGV3YXknLFxuICAgICAgICAgIHJlZ2lvbjogJ3VzLWVhc3QtMScsXG4gICAgICAgICAgcmVzb3VyY2U6ICdhMTIzNDU2Nzg5MDEyYmMzZGU0NTY3ODkwMWYyM2E0NScsXG4gICAgICAgICAgc2VwOiAnOicsXG4gICAgICAgICAgcmVzb3VyY2VOYW1lOiAnL3Rlc3QvbXlkZW1vcmVzb3VyY2UvKidcbiAgICAgICAgfSxcbiAgICAgICAgJ2Fybjphd3MtY246Y2xvdWQ5OjoxMjM0NTY3ODkwMTI6ZW52aXJvbm1lbnQ6ODFlOTAwMzE3MzQ3NTg1YTA2MDFlMDRjOGQ1MmVhRVgnOiB7XG4gICAgICAgICAgcGFydGl0aW9uOiAnYXdzLWNuJyxcbiAgICAgICAgICBzZXJ2aWNlOiAnY2xvdWQ5JyxcbiAgICAgICAgICBhY2NvdW50OiAnMTIzNDU2Nzg5MDEyJyxcbiAgICAgICAgICByZXNvdXJjZTogJ2Vudmlyb25tZW50JyxcbiAgICAgICAgICByZXNvdXJjZU5hbWU6ICc4MWU5MDAzMTczNDc1ODVhMDYwMWUwNGM4ZDUyZWFFWCcsXG4gICAgICAgICAgc2VwOiAnOidcbiAgICAgICAgfSxcbiAgICAgICAgJ2Fybjo6Y29nbml0by1zeW5jOjo6aWRlbnRpdHlwb29sL3VzLWVhc3QtMToxYTFhMWExYS1mZmZmLTExMTEtOTk5OS0xMjM0NTY3ODpibGEnOiB7XG4gICAgICAgICAgc2VydmljZTogJ2NvZ25pdG8tc3luYycsXG4gICAgICAgICAgcmVzb3VyY2U6ICdpZGVudGl0eXBvb2wnLFxuICAgICAgICAgIHJlc291cmNlTmFtZTogJ3VzLWVhc3QtMToxYTFhMWExYS1mZmZmLTExMTEtOTk5OS0xMjM0NTY3ODpibGEnLFxuICAgICAgICAgIHNlcDogJy8nXG4gICAgICAgIH0sXG4gICAgICAgICdhcm46YXdzOnMzOjo6bXlfY29ycG9yYXRlX2J1Y2tldCc6IHtcbiAgICAgICAgICBwYXJ0aXRpb246ICdhd3MnLFxuICAgICAgICAgIHNlcnZpY2U6ICdzMycsXG4gICAgICAgICAgcmVzb3VyY2U6ICdteV9jb3Jwb3JhdGVfYnVja2V0J1xuICAgICAgICB9XG4gICAgICB9O1xuXG4gICAgICBPYmplY3Qua2V5cyh0ZXN0cykuZm9yRWFjaChhcm4gPT4ge1xuICAgICAgICBjb25zdCBleHBlY3RlZCA9IHRlc3RzW2Fybl07XG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnBhcnNlQXJuKGFybiksIGV4cGVjdGVkLCBhcm4pO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYSBUb2tlbiB3aXRoIDogc2VwYXJhdG9yJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgdGhlVG9rZW4gPSB7IFJlZjogJ1NvbWVQYXJhbWV0ZXInIH07XG4gICAgICBjb25zdCBwYXJzZWQgPSBzdGFjay5wYXJzZUFybihuZXcgSW50cmluc2ljKHRoZVRva2VuKS50b1N0cmluZygpLCAnOicpO1xuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHBhcnNlZC5wYXJ0aXRpb24pLCB7ICdGbjo6U2VsZWN0JzogWyAxLCB7ICdGbjo6U3BsaXQnOiBbICc6JywgdGhlVG9rZW4gXX0gXX0pO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShwYXJzZWQuc2VydmljZSksIHsgJ0ZuOjpTZWxlY3QnOiBbIDIsIHsgJ0ZuOjpTcGxpdCc6IFsgJzonLCB0aGVUb2tlbiBdfSBdfSk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHBhcnNlZC5yZWdpb24pLCB7ICdGbjo6U2VsZWN0JzogWyAzLCB7ICdGbjo6U3BsaXQnOiBbICc6JywgdGhlVG9rZW4gXX0gXX0pO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShwYXJzZWQuYWNjb3VudCksIHsgJ0ZuOjpTZWxlY3QnOiBbIDQsIHsgJ0ZuOjpTcGxpdCc6IFsgJzonLCB0aGVUb2tlbiBdfSBdfSk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHBhcnNlZC5yZXNvdXJjZSksIHsgJ0ZuOjpTZWxlY3QnOiBbIDUsIHsgJ0ZuOjpTcGxpdCc6IFsgJzonLCB0aGVUb2tlbiBdfSBdfSk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHBhcnNlZC5yZXNvdXJjZU5hbWUpLCB7ICdGbjo6U2VsZWN0JzogWyA2LCB7ICdGbjo6U3BsaXQnOiBbICc6JywgdGhlVG9rZW4gXX0gXX0pO1xuICAgICAgdGVzdC5lcXVhbChwYXJzZWQuc2VwLCAnOicpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2EgVG9rZW4gd2l0aCAvIHNlcGFyYXRvcicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHRoZVRva2VuID0geyBSZWY6ICdTb21lUGFyYW1ldGVyJyB9O1xuICAgICAgY29uc3QgcGFyc2VkID0gc3RhY2sucGFyc2VBcm4obmV3IEludHJpbnNpYyh0aGVUb2tlbikudG9TdHJpbmcoKSk7XG5cbiAgICAgIHRlc3QuZXF1YWwocGFyc2VkLnNlcCwgJy8nKTtcblxuICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShwYXJzZWQucmVzb3VyY2UpLCB7ICdGbjo6U2VsZWN0JzogWyAwLCB7ICdGbjo6U3BsaXQnOiBbICcvJywgeyAnRm46OlNlbGVjdCc6IFsgNSwgeyAnRm46OlNwbGl0JzogWyAnOicsIHRoZVRva2VuIF19IF19IF19IF19KTtcbiAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocGFyc2VkLnJlc291cmNlTmFtZSksIHsgJ0ZuOjpTZWxlY3QnOiBbIDEsIHsgJ0ZuOjpTcGxpdCc6IFsgJy8nLCB7ICdGbjo6U2VsZWN0JzogWyA1LCB7ICdGbjo6U3BsaXQnOiBbICc6JywgdGhlVG9rZW4gXX0gXX0gXX0gXX0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgJ2NhbiB1c2UgYSBmdWxseSBzcGVjaWZpZWQgQVJOIGZyb20gYSBkaWZmZXJlbnQgc3RhY2sgd2l0aG91dCBpbmN1cnJpbmcgYW4gaW1wb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCAnU3RhY2sxJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4JywgcmVnaW9uOiAndXMtdHVyYm8tNScgfX0pO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayh1bmRlZmluZWQsICdTdGFjazInLCB7IGVudjogeyBhY2NvdW50OiAnODc2NTQzMjEnLCByZWdpb246ICd1cy10dXJiby0xJyB9fSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXJuID0gc3RhY2sxLmZvcm1hdEFybih7XG4gICAgICAvLyBObyBwYXJ0aXRpb24gc3BlY2lmaWVkIGhlcmVcbiAgICAgIHNlcnZpY2U6ICdibGEnLFxuICAgICAgcmVzb3VyY2U6ICd0aGluZycsXG4gICAgICByZXNvdXJjZU5hbWU6ICd0aG9uZycsXG4gICAgfSk7XG4gICAgbmV3IENmbk91dHB1dChzdGFjazIsICdTb21lVmFsdWUnLCB7IHZhbHVlOiBhcm4gfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwodG9DbG91ZEZvcm1hdGlvbihzdGFjazIpLCB7XG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIFNvbWVWYWx1ZToge1xuICAgICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgICAvLyBMb29rIG1hLCBubyBGbjo6SW1wb3J0VmFsdWUhXG4gICAgICAgICAgICAnRm46OkpvaW4nOiBbJycsIFsnYXJuOicsIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sICc6YmxhOnVzLXR1cmJvLTU6MTIzNDU2Nzg6dGhpbmcvdGhvbmcnXV0gfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG4iXX0=