#!/usr/bin/env python3\n\"\"\"\nExample usage of the frame function for JSON-LD framing\n\"\"\"\n\nfrom cmipld.utils.extract.read import frame, view_frame\nimport json\n\n# Example frame objects for different use cases\n\n# Basic frame to extract specific properties\nbasic_frame = {\n    \"@context\": {\n        \"name\": \"http://schema.org/name\",\n        \"description\": \"http://schema.org/description\",\n        \"@vocab\": \"http://example.org/\"\n    },\n    \"@type\": \"Thing\",\n    \"name\": {},\n    \"description\": {}\n}\n\n# Frame to extract nested objects\nnested_frame = {\n    \"@context\": {\n        \"@vocab\": \"http://example.org/\",\n        \"contains\": {\"@type\": \"@id\"}\n    },\n    \"@type\": \"Container\",\n    \"contains\": {\n        \"@type\": \"Item\",\n        \"name\": {},\n        \"value\": {}\n    }\n}\n\n# CMIP-specific frame example\ncmip_experiment_frame = {\n    \"@context\": {\n        \"@vocab\": \"https://wcrp-cmip.github.io/WCRP-universe/\",\n        \"id\": \"@id\",\n        \"type\": \"@type\"\n    },\n    \"@type\": \"wcrp:experiment\",\n    \"id\": {},\n    \"description\": {},\n    \"ui-label\": {},\n    \"validation-key\": {}\n}\n\n# CMIP activity frame example  \ncmip_activity_frame = {\n    \"@context\": {\n        \"@vocab\": \"https://wcrp-cmip.github.io/WCRP-universe/\",\n        \"id\": \"@id\",\n        \"type\": \"@type\"\n    },\n    \"@type\": \"wcrp:activity\",\n    \"id\": {},\n    \"description\": {},\n    \"ui-label\": {},\n    \"contains\": {\n        \"@type\": \"wcrp:experiment\",\n        \"id\": {},\n        \"description\": {}\n    }\n}\n\ndef example_basic_framing():\n    \"\"\"\n    Example of basic JSON-LD framing\n    \"\"\"\n    print(\"🔍 Basic Framing Example\")\n    print(\"=\" * 40)\n    \n    # Example JSON-LD document URL (you can replace with actual URLs)\n    example_url = \"https://example.org/document.jsonld\"\n    \n    try:\n        # Frame the document\n        framed_data = frame(example_url, basic_frame, compact=True, depth=1)\n        \n        print(\"Framed result:\")\n        print(json.dumps(framed_data, indent=2))\n        \n    except Exception as e:\n        print(f\"Error: {e}\")\n        print(\"Note: Replace with actual JSON-LD URL for testing\")\n\ndef example_cmip_framing():\n    \"\"\"\n    Example of CMIP-specific JSON-LD framing\n    \"\"\"\n    print(\"\\n🧪 CMIP Experiment Framing Example\")\n    print(\"=\" * 40)\n    \n    # Example CMIP experiment URL (replace with actual CMIP-LD URL)\n    cmip_url = \"https://wcrp-cmip.github.io/WCRP-universe/experiment/some-experiment.json\"\n    \n    try:\n        # Frame the CMIP document\n        framed_data = frame(cmip_url, cmip_experiment_frame, compact=True, depth=2)\n        \n        print(\"CMIP Experiment framed result:\")\n        print(json.dumps(framed_data, indent=2))\n        \n    except Exception as e:\n        print(f\"Error: {e}\")\n        print(\"Note: Replace with actual CMIP-LD URL for testing\")\n\ndef example_view_framing():\n    \"\"\"\n    Example of using view_frame for rich display\n    \"\"\"\n    print(\"\\n🎨 Rich View Framing Example\")\n    print(\"=\" * 40)\n    \n    example_url = \"https://example.org/document.jsonld\"\n    \n    try:\n        # Display framed document with rich formatting\n        view_frame(example_url, basic_frame, compact=True, depth=1)\n        \n    except Exception as e:\n        print(f\"Error: {e}\")\n        print(\"Note: Replace with actual JSON-LD URL for testing\")\n\ndef create_sample_frame():\n    \"\"\"\n    Utility function to create custom frames\n    \"\"\"\n    print(\"\\n📋 Sample Frame Templates\")\n    print(\"=\" * 40)\n    \n    frames = {\n        \"basic\": basic_frame,\n        \"nested\": nested_frame,\n        \"cmip_experiment\": cmip_experiment_frame,\n        \"cmip_activity\": cmip_activity_frame\n    }\n    \n    for name, frame_obj in frames.items():\n        print(f\"\\n{name.upper()} FRAME:\")\n        print(json.dumps(frame_obj, indent=2))\n\ndef main():\n    \"\"\"\n    Main function to demonstrate framing capabilities\n    \"\"\"\n    print(\"📚 JSON-LD Framing Examples\")\n    print(\"=\" * 50)\n    \n    # Show available frame templates\n    create_sample_frame()\n    \n    # Run examples (these will show errors with placeholder URLs)\n    example_basic_framing()\n    example_cmip_framing()\n    \n    print(\"\\n💡 Usage Tips:\")\n    print(\"- Replace example URLs with actual JSON-LD document URLs\")\n    print(\"- Customize frame objects to extract specific data structures\")\n    print(\"- Use depth parameter to control link resolution\")\n    print(\"- Set compact=False to get expanded JSON-LD output\")\n    \n    print(\"\\n🔧 Function Signatures:\")\n    print(\"- frame(link, frame_obj, compact=True, depth=2)\")\n    print(\"- view_frame(link, frame_obj, compact=True, depth=1)\")\n\nif __name__ == \"__main__\":\n    main()\n