"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const core_1 = require("aws-cdk-lib/core");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, poetryWithoutUrls, commandHooks, assetExcludes = [], } = props;
        const outputPath = path.posix.join(core_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            poetryWithoutUrls,
            commandHooks,
            assetExcludes,
        });
        this.image = image ?? core_1.DockerImage.fromBuild(path.join(__dirname, '..', 'lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = props.command ?? ['bash', '-c', chain(bundlingCommands)];
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes, options.poetryWithoutUrls);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        const excludes = options.assetExcludes ?? [];
        if (packaging.dependenciesFile == packaging_1.DependenciesFile.UV && !excludes.includes('.python-version')) {
            excludes.push('.python-version');
        }
        const exclusionStr = excludes.map(item => `--exclude='${item}'`).join(' ');
        bundlingCommands.push([
            'rsync', '-rLv', exclusionStr ?? '', `${options.inputDir}/`, options.outputDir,
        ].filter(item => item).join(' '));
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile == packaging_1.DependenciesFile.UV) {
            bundlingCommands.push(`uv pip install -r ${packaging_1.DependenciesFile.PIP} --target ${options.outputDir}`);
        }
        else if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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