# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/cache.ipynb.

# %% auto 0
__all__ = ["tc", "Cache"]

# %% ../notebooks/cache.ipynb 8
import unittest

import redis
from redis import exceptions as redisExceptions

from .config import Configuration
from .logger import logger

tc = unittest.TestCase()


# %% ../notebooks/cache.ipynb 9
class Cache:
    """
    Cache management class to handle:
    - API result caching.
    - API throttling : limit the call per minutes of a user.
    """

    cache = None
    abuse_script = None
    # LUA script for API throttling
    # Thanks to https://github.com/long2ice/fastapi-limiter/blob/master/fastapi_limiter/__init__.py
    lua_abuse_script = """
        local key = KEYS[1]
        local limit = tonumber(ARGV[1])
        local expire_time = ARGV[2]
        local current = tonumber(redis.call('get', key) or "0")
        if current > 0 then
            if current + 1 > limit then
                return redis.call("PTTL",key)
            else
                redis.call("INCR", key)
                return 0
            end
        else
            redis.call("SET", key, 1,"px",expire_time)
            return 0
        end"""

    def __init__(self):
        """
        Connection to Redis
        """
        config = Configuration()
        redis_host = config.get("REDIS_HOST", fail_on_missing=False)
        redis_port = config.get("REDIS_PORT", fail_on_missing=False, default=6379)
        if type(redis_port) == str and ":" in redis_port:
            redis_port = redis_port[redis_port.rfind(":") + 1 :]
        redis_password = config.get("REDIS_PASSWORD", fail_on_missing=False)
        try:
            logger.debug(f"Connecting to Redis {redis_host}:{redis_port}")
            self.cache = redis.Redis(
                host=redis_host, password=redis_password, port=redis_port
            )
            if self.cache:
                self.cache.set("init", "OK")
                self.abuse_script = self.cache.register_script(self.lua_abuse_script)
        except redisExceptions.ConnectionError as e:
            self.cache = None
            logger.warning(f"Unable to connect to Redis cache : {str(e)}")

    def is_available(self):
        """
        Test if cache is available.
        """
        try:
            if self.get("init") == b"OK":
                return True
            else:
                return False
        except redisExceptions.ConnectionError as e:
            self.cache = None
            logger.warning(f"Unable to connect to Redis cache : {str(e)}")
            return False

    def clear_cache(self):
        """
        Flush the Redis DB.
        """
        if self.cache is None:
            logger.warning("Cache().clear_cache() : No Redis connection.")
            return False
        status = self.cache.flushdb()
        if status:
            # Set init as it can be use by API to now if cache works
            self.cache.set("init", "OK")
            logger.debug("Redis cache cleared successfully.")
            return True
        else:
            logger.warning("Unable to clear Redis cache.")
            return False

    def set(self, key, value):
        """
        Store a value in the cache.
        """
        if self.cache is None:
            logger.warning("Cache().set(key, value) : No Redis connection.")
            return False
        try:
            return self.cache.set(key, value)
        except redisExceptions.ConnectionError as e:
            logger.warning(f"Unable to use Redis cache : {str(e)}")
            return False

    def get(self, key):
        """
        Extract a value from the cache.
        """
        if self.cache is None:
            # logger.warning("Cache().get(key) : No Redis connection.")
            return None
        try:
            return self.cache.get(key)
        except redisExceptions.ConnectionError as e:
            logger.warning(f"Unable to use Redis cache : {str(e)}")
            return None

    def is_abusing(self, identifier, nb_call_per_minute=5):
        """
        Check if a user made too much call to the API.
        """
        if self.abuse_script is None:
            logger.warning("Unable to use Redis cache to check for abuse !")
            return False
        pexpire = self.abuse_script(
            keys=[identifier],
            args=[str(nb_call_per_minute), str(60 * 1000)],
            client=self.cache,
        )
        # The script return the time to wait before beeing allowed to call again
        # 0 if allowed
        if pexpire == 0:
            return False
        return True
