import json
from json import JSONDecodeError
from typing import Any, Awaitable, Callable, Dict, List, Optional, Sequence, Type, Union

from jinja2 import ChoiceLoader, FileSystemLoader, PackageLoader
from starlette.applications import Starlette
from starlette.datastructures import FormData
from starlette.exceptions import HTTPException
from starlette.middleware import Middleware
from starlette.requests import Request
from starlette.responses import JSONResponse, RedirectResponse, Response
from starlette.routing import Mount, Route
from starlette.staticfiles import StaticFiles
from starlette.status import HTTP_204_NO_CONTENT, HTTP_303_SEE_OTHER, HTTP_403_FORBIDDEN
from starlette.templating import Jinja2Templates
from starlette_admin.auth import AuthMiddleware, AuthProvider
from starlette_admin.exceptions import FormValidationError, LoginFailed
from starlette_admin.fields import BooleanField, FileField
from starlette_admin.helpers import get_file_icon
from starlette_admin.views import (
    BaseModelView,
    BaseView,
    CustomView,
    DefaultAdminIndexView,
    DropDown,
    Link,
)


class BaseAdmin:
    """Base class for implementing Admin interface."""

    def __init__(
        self,
        title: str = "Admin",
        base_url: str = "/admin",
        route_name: str = "admin",
        logo_url: Optional[str] = None,
        login_logo_url: Optional[str] = None,
        templates_dir: str = "templates",
        statics_dir: str = "statics",
        index_view: Type[CustomView] = DefaultAdminIndexView,
        auth_provider: Optional[AuthProvider] = None,
        middlewares: Optional[Sequence[Middleware]] = None,
        debug: bool = False,
    ):
        """
        Parameters:
            title: Admin title.
            base_url: Base URL for Admin interface.
            route_name: Mounted Admin name
            logo_url: URL of logo to be displayed instead of title.
            login_logo_url: If set, it will be used for login interface instead of logo_url.
            templates_dir: Templates dir for customisation
            statics_dir: Statics dir for customisation
            index_view: CustomView to use for index page.
            auth_provider: Authentication Provider
            middlewares: Starlette middlewares
        """
        self.title = title
        self.base_url = base_url
        self.route_name = route_name
        self.logo_url = logo_url
        self.login_logo_url = login_logo_url
        self.templates_dir = templates_dir
        self.statics_dir = statics_dir
        self.auth_provider = auth_provider
        self.middlewares = middlewares
        self.index_view: Type[CustomView] = index_view
        self._views: List[BaseView] = []
        self._models: List[BaseModelView] = []
        self.routes: List[Union[Route, Mount]] = []
        self.debug = debug
        self._setup_templates()
        self.init_auth()
        self.init_routes()

    def add_view(self, view: Type[BaseView]) -> None:
        """
        Add View to the Admin interface.
        """
        view_instance = view()
        self._views.append(view_instance)
        self.setup_view(view_instance)

    def init_auth(self) -> None:
        if self.auth_provider is not None:
            self.middlewares = (
                [] if self.middlewares is None else list(self.middlewares)
            )
            self.middlewares.append(
                Middleware(AuthMiddleware, provider=self.auth_provider)
            )
            self.routes.extend(
                [
                    Route(
                        self.auth_provider.login_path,
                        self._render_login,
                        methods=["GET", "POST"],
                        name="login",
                    ),
                    Route(
                        self.auth_provider.logout_path,
                        self._render_logout,
                        methods=["GET"],
                        name="logout",
                    ),
                ]
            )

    def init_routes(self) -> None:
        statics = StaticFiles(
            directory=self.statics_dir, packages=["starlette_admin"], check_dir=False
        )
        # Avoid raising error where statics directory is not Found
        statics.config_checked = True
        self.routes.extend(
            [
                Mount("/statics", app=statics, name="statics"),
                Route(
                    self.index_view.path,
                    self._render_custom_view(self.index_view()),
                    methods=self.index_view.methods,
                    name="index",
                ),
                Route(
                    "/api/{identity}",
                    self._render_api,
                    methods=["GET", "DELETE"],
                    name="api",
                ),
                Route(
                    "/{identity}/list",
                    self._render_list,
                    methods=["GET"],
                    name="list",
                ),
                Route(
                    "/{identity}/detail/{pk}",
                    self._render_detail,
                    methods=["GET"],
                    name="detail",
                ),
                Route(
                    "/{identity}/create",
                    self._render_create,
                    methods=["GET", "POST"],
                    name="create",
                ),
                Route(
                    "/{identity}/edit/{pk}",
                    self._render_edit,
                    methods=["GET", "POST"],
                    name="edit",
                ),
            ]
        )
        if self.index_view.add_to_menu:
            self._views.append(self.index_view())

    def _setup_templates(self) -> None:
        templates = Jinja2Templates(self.templates_dir)
        templates.env.loader = ChoiceLoader(
            [
                FileSystemLoader(self.templates_dir),
                PackageLoader("starlette_admin", "templates"),
            ]
        )
        templates.env.globals["views"] = self._views
        templates.env.globals["title"] = self.title
        templates.env.globals["is_auth_enabled"] = self.auth_provider is not None
        templates.env.globals["__name__"] = self.route_name
        templates.env.globals["logo_url"] = self.logo_url
        templates.env.globals["login_logo_url"] = self.login_logo_url
        templates.env.filters["is_custom_view"] = lambda res: isinstance(
            res, CustomView
        )
        templates.env.filters["is_link"] = lambda res: isinstance(res, Link)
        templates.env.filters["is_model"] = lambda res: isinstance(res, BaseModelView)
        templates.env.filters["is_dropdown"] = lambda res: isinstance(res, DropDown)
        templates.env.filters["tojson"] = lambda data: json.dumps(data, default=str)
        templates.env.filters["file_icon"] = get_file_icon
        templates.env.filters[
            "to_model"
        ] = lambda identity: self._find_model_from_identity(identity)
        templates.env.filters["is_iter"] = lambda v: isinstance(v, (list, tuple))
        self.templates = templates

    def setup_view(self, view: BaseView) -> None:
        if isinstance(view, DropDown):
            for sub_view in view._views_instance:
                self.setup_view(sub_view)
        elif isinstance(view, CustomView):
            self.routes.insert(
                0,
                Route(
                    view.path,
                    endpoint=self._render_custom_view(view),
                    methods=view.methods,
                    name=view.name,
                ),
            )
        elif isinstance(view, BaseModelView):
            view._find_foreign_model = lambda i: self._find_model_from_identity(i)  # type: ignore
            self._models.append(view)

    def _find_model_from_identity(self, identity: Optional[str]) -> BaseModelView:
        if identity is not None:
            for model in self._models:
                if model.identity == identity:
                    return model
        raise HTTPException(404, "Model with identity %s not found" % identity)

    def _render_custom_view(
        self, custom_view: CustomView
    ) -> Callable[[Request], Awaitable[Response]]:
        async def wrapper(request: Request) -> Response:
            if not custom_view.is_accessible(request):
                raise HTTPException(403)
            return await custom_view.render(request, self.templates)

        return wrapper

    async def _render_api(self, request: Request) -> Response:
        identity = request.path_params.get("identity")
        model = self._find_model_from_identity(identity)
        if request.method == "GET":
            if not model.is_accessible(request):
                return JSONResponse(None, status_code=HTTP_403_FORBIDDEN)
            skip = int(request.query_params.get("skip") or "0")
            limit = int(request.query_params.get("limit") or "100")
            order_by = request.query_params.getlist("order_by")
            where = request.query_params.get("where")
            pks = request.query_params.getlist("pks")
            select2 = "select2" in request.query_params.keys()
            if len(pks) > 0:
                items = await model.find_by_pks(request, pks)
                total = len(items)
            else:
                if where is not None:
                    try:
                        where = json.loads(where)
                    except JSONDecodeError:
                        where = str(where)
                items = await model.find_all(
                    request=request,
                    skip=skip,
                    limit=limit,
                    where=where,
                    order_by=order_by,
                )
                total = await model.count(request=request, where=where)
            return JSONResponse(
                dict(
                    items=[
                        (
                            await model.serialize(
                                item,
                                request,
                                "API",
                                include_relationships=not select2,
                                include_select2=select2,
                            )
                        )
                        for item in items
                    ],
                    total=total,
                )
            )
        else:  # "DELETE"
            if not model.can_delete(request):
                return JSONResponse(None, status_code=HTTP_403_FORBIDDEN)
            pks = request.query_params.getlist("pks")
            await model.delete(request, pks)
            return Response(status_code=HTTP_204_NO_CONTENT)

    async def _render_login(self, request: Request) -> Response:
        if request.method == "GET":
            return self.templates.TemplateResponse(
                "login.html",
                {"request": request},
            )
        else:
            form = await request.form()
            try:
                assert self.auth_provider is not None
                return await self.auth_provider.login(
                    form.get("username"),  # type: ignore
                    form.get("password"),  # type: ignore
                    form.get("remember_me") == "on",
                    request,
                    RedirectResponse(
                        request.query_params.get("next")
                        or request.url_for(self.route_name + ":index"),
                        status_code=HTTP_303_SEE_OTHER,
                    ),
                )
            except FormValidationError as errors:
                return self.templates.TemplateResponse(
                    "login.html",
                    {
                        "request": request,
                        "form_errors": errors,
                    },
                )
            except LoginFailed as error:
                return self.templates.TemplateResponse(
                    "login.html",
                    {
                        "request": request,
                        "error": error.msg,
                    },
                )

    async def _render_logout(self, request: Request) -> Response:
        assert self.auth_provider is not None
        return await self.auth_provider.logout(
            request,
            RedirectResponse(
                request.url_for(self.route_name + ":index"),
                status_code=HTTP_303_SEE_OTHER,
            ),
        )

    async def _render_list(self, request: Request) -> Response:
        identity = request.path_params.get("identity")
        model = self._find_model_from_identity(identity)
        if not model.is_accessible(request):
            raise HTTPException(403)
        return self.templates.TemplateResponse(
            model.list_template,
            {"request": request, "model": model},
        )

    async def _render_detail(self, request: Request) -> Response:
        identity = request.path_params.get("identity")
        model = self._find_model_from_identity(identity)
        if not model.is_accessible(request) or not model.can_view_details(request):
            raise HTTPException(403)
        pk = request.path_params.get("pk")
        obj = await model.find_by_pk(request, pk)
        if obj is None:
            raise HTTPException(404)
        return self.templates.TemplateResponse(
            model.detail_template,
            {
                "request": request,
                "model": model,
                "raw_obj": obj,
                "obj": await model.serialize(obj, request, "VIEW"),
            },
        )

    async def _render_create(self, request: Request) -> Response:
        identity = request.path_params.get("identity")
        model = self._find_model_from_identity(identity)
        if not model.is_accessible(request) or not model.can_create(request):
            raise HTTPException(403)
        if request.method == "GET":
            return self.templates.TemplateResponse(
                model.create_template,
                {"request": request, "model": model},
            )
        else:
            form = await request.form()
            dict_obj = await self.form_to_dict(request, form, model, "CREATE")
            try:
                obj = await model.create(request, dict_obj)
            except FormValidationError as errors:
                return self.templates.TemplateResponse(
                    model.create_template,
                    {
                        "request": request,
                        "model": model,
                        "errors": errors,
                        "obj": dict_obj,
                    },
                )
            pk = getattr(obj, model.pk_attr)  # type: ignore
            url = request.url_for(self.route_name + ":list", identity=model.identity)
            if form.get("_continue_editing", None) is not None:
                url = request.url_for(
                    self.route_name + ":edit", identity=model.identity, pk=pk
                )
            elif form.get("_add_another", None) is not None:
                url = request.url  # type: ignore
            return RedirectResponse(url, status_code=HTTP_303_SEE_OTHER)

    async def _render_edit(self, request: Request) -> Response:
        identity = request.path_params.get("identity")
        model = self._find_model_from_identity(identity)
        if not model.is_accessible(request) or not model.can_edit(request):
            raise HTTPException(403)
        pk = request.path_params.get("pk")
        obj = await model.find_by_pk(request, pk)
        if obj is None:
            raise HTTPException(404)
        if request.method == "GET":
            return self.templates.TemplateResponse(
                model.edit_template,
                {
                    "request": request,
                    "model": model,
                    "raw_obj": obj,
                    "obj": await model.serialize(obj, request, "EDIT"),
                },
            )
        else:
            form = await request.form()
            dict_obj = await self.form_to_dict(request, form, model, "EDIT")
            try:
                obj = await model.edit(request, pk, dict_obj)
            except FormValidationError as errors:
                return self.templates.TemplateResponse(
                    model.edit_template,
                    {
                        "request": request,
                        "model": model,
                        "errors": errors,
                        "obj": dict_obj,
                    },
                )
            pk = getattr(obj, model.pk_attr)  # type: ignore
            url = request.url_for(self.route_name + ":list", identity=model.identity)
            if form.get("_continue_editing", None) is not None:
                url = request.url_for(
                    self.route_name + ":edit", identity=model.identity, pk=pk
                )
            elif form.get("_add_another", None) is not None:
                url = request.url_for(
                    self.route_name + ":create", identity=model.identity
                )
            return RedirectResponse(url, status_code=HTTP_303_SEE_OTHER)

    async def _render_error(
        self,
        request: Request,
        exc: Exception = HTTPException(status_code=500),
    ) -> Response:
        assert isinstance(exc, HTTPException)
        return self.templates.TemplateResponse(
            "error.html",
            {"request": request, "exc": exc},
            status_code=exc.status_code,
        )

    async def form_to_dict(
        self,
        request: Request,
        form_data: FormData,
        model: BaseModelView,
        action: str,
    ) -> Dict[str, Any]:
        data = dict()
        for field in model.fields:
            if (action == "EDIT" and field.exclude_from_edit) or (
                action == "CREATE" and field.exclude_from_create
            ):
                continue
            if isinstance(field, FileField) and action == "EDIT":
                data[f"_{field.name}-delete"] = await BooleanField(
                    f"_{field.name}-delete"
                ).parse_form_data(request, form_data)
            data[field.name] = await field.parse_form_data(request, form_data)
        return data

    def mount_to(self, app: Starlette) -> None:
        admin_app = Starlette(
            routes=self.routes,
            middleware=self.middlewares,
            debug=self.debug,
            exception_handlers={HTTPException: self._render_error},
        )
        admin_app.state.ROUTE_NAME = self.route_name
        app.mount(
            self.base_url,
            app=admin_app,
            name=self.route_name,
        )
