# coding=utf-8
## Author: @Seon
import pandas as pd
import re
import pickle
from math import log10, sqrt
from jieba import lcut, posseg


def get_pos_lst(words_lst):  # 分词的位置列表
    return list(zip(words_lst, range(1, len(words_lst)+1)))


def get_T_pos(pos_lst, word, median_fn):  # 单个词的T_pos指标
    _lst = [i[1] for i in pos_lst if word in i[0]]
    _lst.sort()
    half = len(_lst) // 2
    median = (_lst[half] + _lst[~half]) / 2
    if median_fn is None:
        return log10(log10(median+10))+1
    if callable(median_fn):
        return median_fn(median)


def words_count(words_lst):  # 全文词频统计，返回字典
    counts_dict = {}
    for word in words_lst:
        counts_dict[word] = counts_dict.get(word, 0) + 1
    return counts_dict


def related_content(words_lst, word, size):  # 单个词的T_Rel指标
    split_lst = ' '.join(words_lst).split(word)
    left_lst = split_lst[:-1]
    right_lst = split_lst[1:]
    DL = 0
    for i in left_lst:  # 从 word 出现的每个地方往左取 size 个词，计算不重复词数
        left_words = i.split(' ')[-2: -2-size: -1]
        DL += len(set(left_words))

    DR = 0
    for i in right_lst:  # 从 word 出现的每个地方往右
        right_words = i.split(' ')[1: size+1]
        DR += len(set(right_words))

    return DL/len(left_lst) + DR/len(right_lst)


def get_pseg(x):  # 词性
    return [p for w, p in list(posseg.cut(x))][0]


def std(lst):
    ex = float(sum(lst)) / len(lst)
    s = 0
    for i in lst:
        s += (i - ex) ** 2
    return sqrt(float(s)/len(lst))


def get_S_t(content, only_cn=False, stop=None, pos_type='s', median_fn=None, tf_normal='yake', adjust=1, r_size=10):  # 各项指标原始得分表
    if only_cn:  # 纯中文分词
        clean_str = re.sub(r'[^一-龟，；。？！（）“”]+', '', content)
    else:
        clean_str = re.sub(r'[0-9]+', '', content)

    jb_lst = [w for w in lcut(clean_str) if len(w) > 1]  # 分词过滤单字

    if stop is not None:  # 停用词
        jb_lst = [w for w in jb_lst if w not in stop]

    uni_lst = list(set(jb_lst))
    uni_lst.sort(key=jb_lst.index)  # 固定顺序唯一词
    split_content = re.split(r'[，；。？！]', clean_str)  # 原文本按标点拆分句子列表

    # 位置 T_pos 得分表
    pos_lst = 0
    if pos_type == 'w':
        pos_lst = get_pos_lst(jb_lst)  # 计算全文词位置
    elif pos_type == 's':
        pos_lst = get_pos_lst(split_content)  # 计算含词的句子位置
    t_pos_scores = []
    for w in uni_lst:
        t_pos_scores.append((w, get_T_pos(pos_lst, w, median_fn)))

    # 全文词频 TF_norm 得分表
    wc_dic = words_count(jb_lst)
    mean_tf = sum(wc_dic.values())/len(uni_lst)  # 词频均值
    std_tf = std(list(wc_dic.values()))  # 词频标准差
    max_tf = max(list(wc_dic.values()))
    min_tf = min(list(wc_dic.values()))
    if max_tf - min_tf == 0:
        tf_normal = 'yake'
    tf_norm_scores = []
    TF_norm = 0
    for w in uni_lst:
        if tf_normal == 'yake':
            TF_norm = wc_dic.get(w)/(mean_tf + std_tf)  # yake版归一化
        if tf_normal == 'mm':
            TF_norm = (wc_dic.get(w)-min_tf) / (max_tf-min_tf)  # max-min归一化
        tf_norm_scores.append((w, TF_norm))

    # 上下文 T_Rel 得分表
    t_rel_scores = []
    all_words = len(uni_lst)
    for w in uni_lst:
        DL_RL = related_content(jb_lst, w, r_size)
        T_Rel = 1 + DL_RL * wc_dic.get(w)/all_words
        t_rel_scores.append((w, T_Rel))

    # 句间词频 T_sentence 得分表
    len_content = len(split_content)
    SF_dic = {}
    for w in uni_lst:
        for sentence in split_content:
            if w in sentence:
                SF_dic[w] = SF_dic.get(w, 0) + 1
    t_sentence_scores = [(i[0], i[1]/len_content) for i in list(SF_dic.items())]

    # 重要性 S_t 总分表
    df_scores = pd.DataFrame({
        'word': uni_lst,
        'fre': [wc_dic.get(i) for i in uni_lst],
        't_pos': [i[1] for i in t_pos_scores],
        'tf_norm': [i[1] for i in tf_norm_scores],
        't_rel': [i[1] for i in t_rel_scores],
        't_sentence': [i[1] for i in t_sentence_scores]
    })
    df_scores['pseg'] = df_scores['word'].apply(get_pseg)
    df_scores.eval(f's_t = t_pos*t_rel / ({adjust} + (tf_norm + t_sentence)/t_rel)', inplace=True)
    return df_scores


def get_key_words(df_scores, top=10, sort_col='s_t', ascend=True, p=None):  # 获取关键词列表，默认前10个，升序
    if p is not None:
        df_scores = df_scores[df_scores['pseg'] == p]
    result = df_scores.sort_values(sort_col, ascending=ascend)
    key_words = result['word'].to_list()
    return key_words[:top]


def get_stopwords(txt_file):
    return set([line.strip() for line in open(txt_file, 'r', encoding='utf-8').readlines()])


cn = b'\x80\x03cbuiltins\nset\nq\x00]q\x01(X\x06\x00\x00\x00\xe5\x93\xaa\xe5\xa4\xa9q\x02X\x06\x00\x00\x00\xe5\xa4\x9a\xe5\xb0\x91q\x03X\x03\x00\x00\x00\xe5\xb7\xb1q\x04X\x06\x00\x00\x00\xe5\x8d\xb3\xe8\x8b\xa5q\x05X\x06\x00\x00\x00\xe5\xb0\xb1\xe7\xae\x97q\x06X\x06\x00\x00\x00\xe6\x98\xaf\xe7\x9a\x84q\x07X\x06\x00\x00\x00\xe5\x81\x87\xe8\x8b\xa5q\x08X\x06\x00\x00\x00\xe6\xad\xa4\xe5\xa4\x96q\tX\x03\x00\x00\x00\xe6\x9c\x9bq\nX\x0c\x00\x00\x00\xe5\xb0\xa4\xe4\xb8\xba\xe9\x87\x8d\xe8\xa6\x81q\x0bX\t\x00\x00\x00\xe4\xbb\x80\xe4\xb9\x88\xe6\xa0\xb7q\x0cX\x06\x00\x00\x00\xe7\xab\x9f\xe8\x80\x8cq\rX\x06\x00\x00\x00\xe6\x9c\xac\xe6\x96\x87q\x0eX\x06\x00\x00\x00\xe5\xbd\x93\xe7\x84\xb6q\x0fX\x06\x00\x00\x00\xe6\x9d\xa5\xe8\xaf\xb4q\x10X\x06\x00\x00\x00\xe5\x81\x87\xe4\xbd\xbfq\x11X\x06\x00\x00\x00\xe4\xb8\x8a\xe9\x9d\xa2q\x12X\x06\x00\x00\x00\xe8\x80\x8c\xe5\xb7\xb2q\x13X\x03\x00\x00\x00\xe7\x9c\x8bq\x14X\x01\x00\x00\x001q\x15X\x03\x00\x00\x00\xe6\xac\xa4q\x16X\x06\x00\x00\x00\xe6\x88\x91\xe4\xba\x8eq\x17X\x06\x00\x00\x00\xe5\xa6\x82\xe4\xb8\x8aq\x18X\x06\x00\x00\x00\xe5\x90\xa6\xe5\x88\x99q\x19X\x06\x00\x00\x00\xe5\x8f\xaa\xe6\x9c\x89q\x1aX\x03\x00\x00\x00\xe5\x86\x92q\x1bX\x06\x00\x00\x00\xe6\xad\xa4\xe6\xac\xa1q\x1cX\x06\x00\x00\x00\xe5\x8f\x88\xe5\x8f\x8aq\x1dX\x06\x00\x00\x00\xe5\xa6\x82\xe6\xad\xa4q\x1eX\x06\x00\x00\x00\xe6\x9d\xa5\xe8\x87\xaaq\x1fX\x03\x00\x00\x00\xe7\x94\xb1q X\t\x00\x00\x00\xe4\xb8\x8e\xe5\x85\xb6\xe8\xaf\xb4q!X\x03\x00\x00\x00\xe5\x98\xbbq"X\x03\x00\x00\x00\xe5\x8e\xbbq#X\x06\x00\x00\x00\xe5\x8a\xa0\xe4\xb9\x8bq$X\x06\x00\x00\x00\xe5\x8d\x81\xe5\x88\x86q%X\x06\x00\x00\x00\xe5\xbc\x80\xe5\xa7\x8bq&X\x06\x00\x00\x00\xe6\x9c\x89\xe7\x9a\x84q\'X\x01\x00\x00\x007q(X\x06\x00\x00\x00\xe7\xbb\x8f\xe8\xbf\x87q)X\x03\x00\x00\x00\xe5\x8d\xb3q*X\x06\x00\x00\x00\xe7\xb1\xbb\xe5\xa6\x82q+X\x06\x00\x00\x00\xe9\x9a\x8f\xe5\x90\x8eq,X\x03\x00\x00\x00\xef\xbc\x8cq-X\x06\x00\x00\x00\xe4\xb8\x8d\xe7\xae\xa1q.X\x06\x00\x00\x00\xe4\xbd\x86\xe5\x87\xa1q/X\x06\x00\x00\x00\xe6\xad\xa3\xe5\xa6\x82q0X\x06\x00\x00\x00\xe4\xbb\x8b\xe7\xbb\x8dq1X\x06\x00\x00\x00\xe6\x9c\xac\xe4\xba\xbaq2X\x03\x00\x00\x00\xe5\x84\xbfq3X\x03\x00\x00\x00\xe4\xbb\x8eq4X\x06\x00\x00\x00\xe8\xbf\x99\xe6\xa0\xb7q5X\x03\x00\x00\x00\xe5\x85\xaeq6X\x06\x00\x00\x00\xe4\xb8\x8d\xe7\x84\xb6q7X\x03\x00\x00\x00\xe8\x80\x85q8X\x03\x00\x00\x00\xe5\xbe\x85q9X\x06\x00\x00\x00\xe8\x99\xbd\xe8\xaf\xb4q:X\x06\x00\x00\x00\xe6\x9e\x81\xe5\x85\xb6q;X\x06\x00\x00\x00\xe6\x88\x96\xe6\x98\xafq<X\x06\x00\x00\x00\xe5\x88\xab\xe4\xba\xbaq=X\x06\x00\x00\x00\xe6\xa6\x82\xe8\xbf\xb0q>X\x03\x00\x00\x00\xe6\x88\x96q?X\x06\x00\x00\x00\xe6\x89\x93\xe4\xbb\x8eq@X\x06\x00\x00\x00\xe5\xb0\x94\xe5\x90\x8eqAX\x06\x00\x00\x00\xe4\xb8\x80\xe6\x9d\xa1qBX\x06\x00\x00\x00\xe8\xbf\x98\xe6\x98\xafqCX\x06\x00\x00\x00\xe6\x9f\x90\xe4\xb8\xaaqDX\x06\x00\x00\x00\xe4\xb8\x8d\xe5\x8f\xaaqEX\x03\x00\x00\x00\xe2\x80\x9dqFX\x01\x00\x00\x002qGX\x03\x00\x00\x00\xe4\xb9\x8eqHX\x03\x00\x00\x00\xe6\x9c\x9dqIX\x06\x00\x00\x00\xe4\xbb\x8d\xe6\x97\xa7qJX\x03\x00\x00\x00\xe4\xbd\xa0qKX\x06\x00\x00\x00\xe5\x98\xbf\xe5\x98\xbfqLX\x06\x00\x00\x00\xe9\x99\xa4\xe9\x9d\x9eqMX\x06\x00\x00\x00\xe4\xb8\x8d\xe6\xaf\x94qNX\x03\x00\x00\x00\xe5\x86\xb2qOX\x03\x00\x00\x00\xe7\xae\xa1qPX\x03\x00\x00\x00\xe7\x84\x89qQX\x06\x00\x00\x00\xe5\x96\x94\xe5\x94\xb7qRX\x0c\x00\x00\x00\xe9\x99\xa4\xe6\xad\xa4\xe4\xb9\x8b\xe5\xa4\x96qSX\x06\x00\x00\x00\xe9\x81\xb5\xe5\xbe\xaaqTX\x06\x00\x00\x00\xe5\x89\x8d\xe5\x90\x8eqUX\x06\x00\x00\x00\xe8\xbf\x9b\xe8\x80\x8cqVX\x03\x00\x00\x00\xe5\x9c\xb0qWX\x03\x00\x00\x00\xe5\x90\x97qXX\x03\x00\x00\x00\xe5\x80\x9fqYX\x06\x00\x00\x00\xe5\x80\x98\xe7\x84\xb6qZX\x06\x00\x00\x00\xe6\x98\xaf\xe4\xbb\xa5q[X\x06\x00\x00\x00\xe8\xbf\x99\xe6\xac\xa1q\\X\t\x00\x00\x00\xe4\xb8\x8d\xe8\x87\xb3\xe4\xba\x8eq]X\x06\x00\x00\x00\xe5\x9f\xba\xe6\x9c\xacq^X\x03\x00\x00\x00\xe7\x94\xa8q_X\x06\x00\x00\x00\xe4\xb8\xba\xe6\xad\xa4q`X\x06\x00\x00\x00\xe8\xae\xb8\xe5\xa4\x9aqaX\x06\x00\x00\x00\xe4\xb8\x80\xe7\xbb\x84qbX\x06\x00\x00\x00\xe5\x8f\xaa\xe6\xb6\x88qcX\x03\x00\x00\x00\xe5\x93\x89qdX\x06\x00\x00\x00\xe4\xbd\xbf\xe5\xbe\x97qeX\x06\x00\x00\x00\xe4\xbd\xbf\xe7\x94\xa8qfX\x06\x00\x00\x00\xe8\xa6\x81\xe4\xb8\x8dqgX\x06\x00\x00\x00\xe5\xaf\xb9\xe4\xba\x8eqhX\t\x00\x00\x00\xe6\xb2\xa1\xe5\xa5\x88\xe4\xbd\x95qiX\x03\x00\x00\x00\xe7\xba\xb5qjX\x06\x00\x00\x00\xe7\x94\xb1\xe4\xba\x8eqkX\x03\x00\x00\x00\xe5\x90\x93qlX\x03\x00\x00\x00\xe5\xae\x81qmX\x03\x00\x00\x00\xe6\x8d\xaeqnX\x06\x00\x00\x00\xe9\x9a\x8f\xe7\x9d\x80qoX\x06\x00\x00\x00\xe4\xb8\x8d\xe5\xb0\xbdqpX\x03\x00\x00\x00\xe4\xbb\x8aqqX\x06\x00\x00\x00\xe5\x93\xaa\xe5\x84\xbfqrX\x06\x00\x00\x00\xe5\xb9\xb6\xe9\x9d\x9eqsX\x06\x00\x00\x00\xe7\x8a\xb9\xe4\xb8\x94qtX\x06\x00\x00\x00\xe5\x8f\xaf\xe4\xbb\xa5quX\x03\x00\x00\x00\xe5\x86\x8dqvX\x06\x00\x00\x00\xe5\xae\x81\xe8\x82\xafqwX\x06\x00\x00\x00\xe5\x85\xb6\xe4\xb8\xadqxX\t\x00\x00\x00\xe8\x87\xaa\xe5\x90\x84\xe5\x84\xbfqyX\x06\x00\x00\x00\xe4\xb8\x8d\xe5\xbe\x97qzX\x03\x00\x00\x00\xe5\x8f\x88q{X\x03\x00\x00\x00\xe5\x92\xb1q|X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x96\x99q}X\x0c\x00\x00\x00\xe7\xbb\xbc\xe4\xb8\x8a\xe6\x89\x80\xe8\xbf\xb0q~X\x06\x00\x00\x00\xe8\xbf\x99\xe9\x87\x8cq\x7fX\x06\x00\x00\x00\xe7\x85\xa7\xe7\x9d\x80q\x80X\x06\x00\x00\x00\xe6\x9e\x81\xe4\xba\x86q\x81X\x06\x00\x00\x00\xe5\xba\x94\xe5\xbd\x93q\x82X\x03\x00\x00\x00\xe8\xbe\x83q\x83X\x06\x00\x00\x00\xe5\x93\x88\xe5\x93\x88q\x84X\x06\x00\x00\x00\xe5\x9b\xa0\xe4\xb8\xbaq\x85X\x06\x00\x00\x00\xe8\xb0\x81\xe7\x9f\xa5q\x86X\x01\x00\x00\x006q\x87X\t\x00\x00\x00\xe9\x82\xa3\xe4\xbc\x9a\xe5\x84\xbfq\x88X\x03\x00\x00\x00\xe4\xb8\x8dq\x89X\x06\x00\x00\x00\xe5\x8f\xaa\xe5\xbd\x93q\x8aX\t\x00\x00\x00\xe8\x87\xaa\xe4\xb8\xaa\xe5\x84\xbfq\x8bX\x06\x00\x00\x00\xe6\x97\xa2\xe6\x98\xafq\x8cX\x06\x00\x00\x00\xe5\xb0\x9a\xe4\xb8\x94q\x8dX\x06\x00\x00\x00\xe5\xa4\x9a\xe4\xb9\x88q\x8eX\x03\x00\x00\x00\xe6\x89\x93q\x8fX\x03\x00\x00\x00\xe6\x9c\x80q\x90X\x06\x00\x00\x00\xe8\x8b\xa5\xe5\xa4\xabq\x91X\x06\x00\x00\x00\xe8\xb0\x81\xe6\x96\x99q\x92X\x06\x00\x00\x00\xe5\x8f\xa6\xe6\x82\x89q\x93X\x06\x00\x00\x00\xe5\xba\xb6\xe5\x87\xa0q\x94X\x03\x00\x00\x00\xe8\xb6\x81q\x95X\x03\x00\x00\x00\xe4\xb8\x8aq\x96X\x06\x00\x00\x00\xe4\xb9\x9f\xe5\xa5\xbdq\x97X\x06\x00\x00\x00\xe4\xbd\x99\xe5\xa4\x96q\x98X\x06\x00\x00\x00\xe7\x90\x86\xe8\xa7\xa3q\x99X\x06\x00\x00\x00\xe5\x88\xa9\xe4\xba\x8eq\x9aX\x06\x00\x00\x00\xe7\xba\xb5\xe4\xbb\xa4q\x9bX\x06\x00\x00\x00\xe8\x99\xbd\xe7\x84\xb6q\x9cX\x03\x00\x00\x00\xe5\xa5\xbdq\x9dX\x06\x00\x00\x00\xe7\x84\xb6\xe8\x80\x8cq\x9eX\x06\x00\x00\x00\xe5\x8f\x8a\xe8\x87\xb3q\x9fX\x06\x00\x00\x00\xe5\xb0\x94\xe5\xb0\x94q\xa0X\x03\x00\x00\x00\xe5\x91\x95q\xa1X\x06\x00\x00\x00\xe9\xa6\x96\xe5\x85\x88q\xa2X\t\x00\x00\x00\xe9\x82\xa3\xe4\xb9\x88\xe4\xba\x9bq\xa3X\x06\x00\x00\x00\xe9\x89\xb4\xe4\xba\x8eq\xa4X\x06\x00\x00\x00\xe4\xb8\x8d\xe8\xa6\x81q\xa5X\x03\x00\x00\x00\xe5\x97\xb3q\xa6X\x06\x00\x00\x00\xe6\xb2\xbf\xe7\x9d\x80q\xa7X\x06\x00\x00\x00\xe9\xa1\xba\xe7\x9d\x80q\xa8X\x06\x00\x00\x00\xe4\xbb\x8e\xe6\xad\xa4q\xa9X\x06\x00\x00\x00\xe6\x9c\x89\xe6\x97\xb6q\xaaX\x06\x00\x00\x00\xe5\x8f\xaa\xe6\x80\x95q\xabX\x06\x00\x00\x00\xe4\xb8\xa4\xe8\x80\x85q\xacX\x06\x00\x00\x00\xe6\xaf\x94\xe5\xa6\x82q\xadX\x03\x00\x00\x00\xe5\x80\x98q\xaeX\x06\x00\x00\x00\xe4\xb8\x8d\xe5\xa6\xa8q\xafX\x06\x00\x00\x00\xe5\xae\x81\xe5\x8f\xafq\xb0X\x06\x00\x00\x00\xe6\xad\xa4\xe5\x9c\xb0q\xb1X\x06\x00\x00\x00\xe4\xb8\x80\xe6\x9d\xa5q\xb2X\x03\x00\x00\x00\xe6\xaf\x8fq\xb3X\x06\x00\x00\x00\xe8\x80\x8c\xe8\xa8\x80q\xb4X\x06\x00\x00\x00\xe8\x80\x8c\xe5\xa4\x96q\xb5X\x06\x00\x00\x00\xe4\xbd\x86\xe6\x98\xafq\xb6X\x06\x00\x00\x00\xe8\x8e\xab\xe5\xa6\x82q\xb7X\x06\x00\x00\x00\xe4\xb8\x80\xe7\xa7\x8dq\xb8X\x03\x00\x00\x00\xe5\xbe\x88q\xb9X\x06\x00\x00\x00\xe4\xb9\x8c\xe4\xb9\x8eq\xbaX\x0c\x00\x00\x00\xe6\x8d\xa2\xe5\x8f\xa5\xe8\xaf\x9d\xe8\xaf\xb4q\xbbX\x06\x00\x00\x00\xe5\x85\xa8\xe9\x83\xa8q\xbcX\x06\x00\x00\x00\xe9\x99\xa4\xe5\xbc\x80q\xbdX\x03\x00\x00\x00\xe6\x98\xafq\xbeX\x06\x00\x00\x00\xe5\x90\x91\xe7\x9d\x80q\xbfX\x06\x00\x00\x00\xe5\x9c\xa8\xe4\xb8\x8bq\xc0X\x03\x00\x00\x00\xe8\xae\xa9q\xc1X\x06\x00\x00\x00\xe6\x9f\x90\xe6\x9f\x90q\xc2X\t\x00\x00\x00\xe4\xb8\x8d\xe5\xb0\xbd\xe7\x84\xb6q\xc3X\x03\x00\x00\x00\xe5\x95\xa6q\xc4X\x06\x00\x00\x00\xe4\xbd\x95\xe6\x97\xb6q\xc5X\x06\x00\x00\x00\xe5\x86\x8d\xe8\xaf\xb4q\xc6X\x06\x00\x00\x00\xe9\xa1\xb9\xe7\x9b\xaeq\xc7X\x06\x00\x00\x00\xe4\xb8\x8a\xe4\xb8\x8bq\xc8X\x06\x00\x00\x00\xe5\x85\xb6\xe4\xbb\x96q\xc9X\x03\x00\x00\x00\xe3\x80\x8bq\xcaX\x06\x00\x00\x00\xe4\xbb\x8b\xe4\xba\x8eq\xcbX\x03\x00\x00\x00\xe5\x9b\x9bq\xccX\x06\x00\x00\x00\xe5\x88\xab\xe7\x9a\x84q\xcdX\x03\x00\x00\x00\xe6\x9c\x89q\xceX\x06\x00\x00\x00\xe4\xb8\x80\xe8\x88\xacq\xcfX\x06\x00\x00\x00\xe6\x89\x8d\xe8\x83\xbdq\xd0X\x03\x00\x00\x00\xe5\x85\xadq\xd1X\x06\x00\x00\x00\xe6\x8e\xa5\xe7\x9d\x80q\xd2X\x03\x00\x00\x00\xe6\x97\xa5q\xd3X\x06\x00\x00\x00\xe8\xaf\x9a\xe7\x84\xb6q\xd4X\x06\x00\x00\x00\xe7\xbb\xa7\xe4\xb9\x8bq\xd5X\x03\x00\x00\x00\xe5\x83\x8fq\xd6X\x03\x00\x00\x00\xef\xbc\x9fq\xd7X\x03\x00\x00\x00\xe5\xb7\xb2q\xd8X\x06\x00\x00\x00\xe5\x90\x84\xe4\xbd\x8dq\xd9X\x06\x00\x00\x00\xe7\x9f\xa3\xe5\x93\x89q\xdaX\x06\x00\x00\x00\xe4\xb9\x8b\xe7\xb1\xbbq\xdbX\x06\x00\x00\x00\xe9\x82\xa3\xe6\xa0\xb7q\xdcX\x06\x00\x00\x00\xe5\xa6\x82\xe6\x98\xafq\xddX\x03\x00\x00\x00\xe4\xba\x8eq\xdeX\x03\x00\x00\x00\xe4\xbb\x8dq\xdfX\x06\x00\x00\x00\xe5\xa6\x82\xe5\x85\xb6q\xe0X\t\x00\x00\x00\xe4\xb9\x8b\xe6\x89\x80\xe4\xbb\xa5q\xe1X\x06\x00\x00\x00\xe9\x99\xa4\xe5\xa4\x96q\xe2X\x03\x00\x00\x00\xe9\x97\xaeq\xe3X\t\x00\x00\x00\xe5\xb0\xb1\xe6\x98\xaf\xe4\xba\x86q\xe4X\x06\x00\x00\x00\xe5\xbd\x93\xe5\x9c\xb0q\xe5X\x03\x00\x00\x00\xe6\x9c\xacq\xe6X\x06\x00\x00\x00\xe9\x87\x8d\xe8\xa6\x81q\xe7X\x03\x00\x00\x00\xe6\x97\xa0q\xe8X\x03\x00\x00\x00\xe5\x97\xa1q\xe9X\x03\x00\x00\x00\xe4\xba\x8cq\xeaX\x06\x00\x00\x00\xe7\x84\xb6\xe5\x90\x8eq\xebX\x03\x00\x00\x00\xef\xbc\x9aq\xecX\x03\x00\x00\x00\xef\xbc\x9bq\xedX\x06\x00\x00\x00\xe4\xbd\x95\xe4\xbb\xa5q\xeeX\x06\x00\x00\x00\xe5\x93\x8e\xe5\x93\x9fq\xefX\x06\x00\x00\x00\xe6\xbc\xab\xe8\xaf\xb4q\xf0X\x03\x00\x00\x00\xe5\xa4\x9aq\xf1X\x06\x00\x00\x00\xe4\xbb\x96\xe4\xba\xbaq\xf2X\x03\x00\x00\x00\xe6\x95\x85q\xf3X\x06\x00\x00\x00\xe5\x91\x9c\xe5\x91\xbcq\xf4X\x06\x00\x00\x00\xe5\xb2\x82\xe4\xbd\x86q\xf5X\x03\x00\x00\x00\xe8\x80\x8cq\xf6X\t\x00\x00\x00\xe8\xbf\x99\xe4\xb9\x88\xe4\xba\x9bq\xf7X\x06\x00\x00\x00\xe7\xbb\x93\xe6\x9e\x9cq\xf8X\x03\x00\x00\x00\xe5\x97\xacq\xf9X\x06\x00\x00\x00\xe6\xad\xa4\xe5\xa4\x84q\xfaX\x06\x00\x00\x00\xe9\x9d\x9e\xe5\xb8\xb8q\xfbX\x06\x00\x00\x00\xe5\x9f\xba\xe4\xba\x8eq\xfcX\x06\x00\x00\x00\xe4\xbd\x9c\xe4\xb8\xbaq\xfdX\x06\x00\x00\x00\xe4\xbb\x8e\xe8\x80\x8cq\xfeX\x06\x00\x00\x00\xe5\xb7\xb2\xe7\x9f\xa3q\xffX\x06\x00\x00\x00\xe8\xb4\xbc\xe6\xad\xbbr\x00\x01\x00\x00X\x06\x00\x00\x00\xe5\x80\x98\xe6\x88\x96r\x01\x01\x00\x00X\x03\x00\x00\x00\xe5\x87\xa1r\x02\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe8\x8b\xa5r\x03\x01\x00\x00X\x06\x00\x00\x00\xe5\xaf\xb9\xe6\xaf\x94r\x04\x01\x00\x00X\x03\x00\x00\x00\xe5\x90\xa7r\x05\x01\x00\x00X\x06\x00\x00\x00\xe4\xb9\x9f\xe7\xbd\xa2r\x06\x01\x00\x00X\x03\x00\x00\x00\xe8\x8b\xa5r\x07\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe8\x87\xb4r\x08\x01\x00\x00X\x06\x00\x00\x00\xe5\x87\xa0\xe6\x97\xb6r\t\x01\x00\x00X\x03\x00\x00\x00\xe6\x9f\x90r\n\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe8\x83\xbdr\x0b\x01\x00\x00X\x06\x00\x00\x00\xe4\xbe\x8b\xe5\xa6\x82r\x0c\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe4\xbb\x85r\r\x01\x00\x00X\x06\x00\x00\x00\xe4\xba\x91\xe5\xb0\x94r\x0e\x01\x00\x00X\x03\x00\x00\x00\xe8\xb7\x9fr\x0f\x01\x00\x00X\x06\x00\x00\x00\xe4\xbf\xba\xe4\xbb\xacr\x10\x01\x00\x00X\x06\x00\x00\x00\xe7\x8a\xb9\xe8\x87\xaar\x11\x01\x00\x00X\x03\x00\x00\x00\xe7\xbb\x8fr\x12\x01\x00\x00X\x06\x00\x00\x00\xe8\x83\xbd\xe5\x90\xa6r\x13\x01\x00\x00X\x03\x00\x00\x00\xe5\x96\x82r\x14\x01\x00\x00X\t\x00\x00\x00\xe4\xb8\xba\xe4\xbb\x80\xe4\xb9\x88r\x15\x01\x00\x00X\x06\x00\x00\x00\xe5\x93\x8e\xe5\x91\x80r\x16\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8e\xe5\x85\xb6r\x17\x01\x00\x00X\x03\x00\x00\x00\xe4\xbf\xbar\x18\x01\x00\x00X\x06\x00\x00\x00\xe6\x9c\x89\xe6\x95\x88r\x19\x01\x00\x00X\x03\x00\x00\x00\xe5\x98\x9br\x1a\x01\x00\x00X\x06\x00\x00\x00\xe9\x9d\x9e\xe7\x8b\xacr\x1b\x01\x00\x00X\x03\x00\x00\x00\xe8\xaf\xb8r\x1c\x01\x00\x00X\x06\x00\x00\x00\xe9\x80\x90\xe6\xad\xa5r\x1d\x01\x00\x00X\x03\x00\x00\x00\xe7\xbb\x99r\x1e\x01\x00\x00X\x0c\x00\x00\x00\xe7\x94\xb1\xe6\xad\xa4\xe5\x8f\xaf\xe8\xa7\x81r\x1f\x01\x00\x00X\x06\x00\x00\x00\xe5\x87\xba\xe7\x8e\xb0r \x01\x00\x00X\x06\x00\x00\x00\xe7\x94\x9a\xe6\x88\x96r!\x01\x00\x00X\x03\x00\x00\x00\xe5\x97\xafr"\x01\x00\x00X\x03\x00\x00\x00\xe6\x80\x8er#\x01\x00\x00X\x06\x00\x00\x00\xe5\xb0\xb1\xe6\x98\xafr$\x01\x00\x00X\x01\x00\x00\x003r%\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe7\x89\xb9r&\x01\x00\x00X\x06\x00\x00\x00\xe5\x8d\xb3\xe4\xbd\xbfr\'\x01\x00\x00X\x03\x00\x00\x00\xe4\xb9\x88r(\x01\x00\x00X\x06\x00\x00\x00\xe7\x9c\xa8\xe7\x9c\xbcr)\x01\x00\x00X\x03\x00\x00\x00\xe8\xaf\xb4r*\x01\x00\x00X\x06\x00\x00\x00\xe5\xaf\xb9\xe5\xbe\x85r+\x01\x00\x00X\x03\x00\x00\x00\xe4\xb8\xbar,\x01\x00\x00X\x06\x00\x00\x00\xe6\x9f\x90\xe4\xba\x9br-\x01\x00\x00X\x06\x00\x00\x00\xe6\x89\x80\xe5\x9c\xa8r.\x01\x00\x00X\x06\x00\x00\x00\xe5\x85\xb6\xe6\xac\xa1r/\x01\x00\x00X\x03\x00\x00\x00\xe9\x82\xa3r0\x01\x00\x00X\x01\x00\x00\x008r1\x01\x00\x00X\x03\x00\x00\x00\xe7\x9f\xa3r2\x01\x00\x00X\x06\x00\x00\x00\xe6\x9e\x9c\xe7\x9c\x9fr3\x01\x00\x00X\x06\x00\x00\x00\xe5\x8d\xb3\xe4\xbe\xbfr4\x01\x00\x00X\x06\x00\x00\x00\xe5\x8a\xa0\xe4\xbb\xa5r5\x01\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe8\xbe\xb9r6\x01\x00\x00X\x06\x00\x00\x00\xe6\xaf\x8f\xe4\xb8\xaar7\x01\x00\x00X\x06\x00\x00\x00\xe8\x80\x8c\xe6\x98\xafr8\x01\x00\x00X\x06\x00\x00\x00\xe4\xb9\x8b\xe4\xb8\x80r9\x01\x00\x00X\x03\x00\x00\x00\xe5\x90\x8cr:\x01\x00\x00X\x06\x00\x00\x00\xe5\x89\x8d\xe6\xad\xa4r;\x01\x00\x00X\x03\x00\x00\x00\xe6\x8b\xbfr<\x01\x00\x00X\x0c\x00\x00\x00\xe8\xbf\x99\xe4\xb9\x88\xe7\x82\xb9\xe5\x84\xbfr=\x01\x00\x00X\x03\x00\x00\x00\xe8\xae\xbar>\x01\x00\x00X\x03\x00\x00\x00\xe5\x92\x9ar?\x01\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe8\xbe\xb9r@\x01\x00\x00X\t\x00\x00\x00\xe6\x9c\x89\xe5\x8a\xa9\xe4\xba\x8erA\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xbb\xe5\x87\xadrB\x01\x00\x00X\x06\x00\x00\x00\xe9\x9d\x9e\xe4\xbd\x86rC\x01\x00\x00X\x06\x00\x00\x00\xe6\x9e\x9c\xe7\x84\xb6rD\x01\x00\x00X\x03\x00\x00\x00\xe5\x8f\xaarE\x01\x00\x00X\x03\x00\x00\x00\xe5\x8f\xabrF\x01\x00\x00X\x03\x00\x00\x00\xe8\x85\xberG\x01\x00\x00X\x03\x00\x00\x00\xe6\x8c\x89rH\x01\x00\x00X\x03\x00\x00\x00\xe5\xb9\xb6rI\x01\x00\x00X\x03\x00\x00\x00\xe4\xbb\xa5rJ\x01\x00\x00X\x01\x00\x00\x00?rK\x01\x00\x00X\x06\x00\x00\x00\xe8\xaf\xb8\xe5\xa6\x82rL\x01\x00\x00X\t\x00\x00\x00\xe6\x80\x8e\xe4\xb9\x88\xe5\x8a\x9erM\x01\x00\x00X\x06\x00\x00\x00\xe8\xaf\x9a\xe5\xa6\x82rN\x01\x00\x00X\x03\x00\x00\x00\xe5\x91\xb5rO\x01\x00\x00X\x03\x00\x00\x00\xe4\xba\xa6rP\x01\x00\x00X\x06\x00\x00\x00\xe6\x9c\xac\xe8\xba\xabrQ\x01\x00\x00X\x06\x00\x00\x00\xe6\x8a\x80\xe6\x9c\xafrR\x01\x00\x00X\x06\x00\x00\x00\xe6\x8d\xae\xe6\xad\xa4rS\x01\x00\x00X\x06\x00\x00\x00\xe8\xb5\x96\xe4\xbb\xa5rT\x01\x00\x00X\x0c\x00\x00\x00\xe5\x8f\x8d\xe8\xbf\x87\xe6\x9d\xa5\xe8\xaf\xb4rU\x01\x00\x00X\x03\x00\x00\x00\xe5\xaf\xb9rV\x01\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe9\xa1\xb9rW\x01\x00\x00X\x0c\x00\x00\x00\xe5\xb0\xbd\xe7\xae\xa1\xe5\xa6\x82\xe6\xad\xa4rX\x01\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe4\xb8\xaarY\x01\x00\x00X\x06\x00\x00\x00\xe5\xa7\x8b\xe8\x80\x8crZ\x01\x00\x00X\x03\x00\x00\x00\xe4\xbe\x9dr[\x01\x00\x00X\x06\x00\x00\x00\xe8\x88\xac\xe7\x9a\x84r\\\x01\x00\x00X\x03\x00\x00\x00\xe5\xae\x83r]\x01\x00\x00X\x03\x00\x00\x00\xe5\xbe\x80r^\x01\x00\x00X\x06\x00\x00\x00\xe5\x94\xaf\xe6\x9c\x89r_\x01\x00\x00X\x06\x00\x00\x00\xe5\xbe\x97\xe4\xba\x86r`\x01\x00\x00X\x06\x00\x00\x00\xe6\x88\x91\xe4\xbb\xacra\x01\x00\x00X\x06\x00\x00\x00\xe7\xbb\xa7\xe8\x80\x8crb\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe6\x95\x85rc\x01\x00\x00X\x03\x00\x00\x00\xe6\xad\xa4rd\x01\x00\x00X\x03\x00\x00\x00\xe7\x9a\x84re\x01\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe6\x97\xb6rf\x01\x00\x00X\x03\x00\x00\x00\xe4\xb9\x8brg\x01\x00\x00X\x06\x00\x00\x00\xe5\x95\xb7\xe5\xbd\x93rh\x01\x00\x00X\x03\x00\x00\x00\xe5\x9c\xa8ri\x01\x00\x00X\x06\x00\x00\x00\xe4\xbc\xbc\xe7\x9a\x84rj\x01\x00\x00X\x06\x00\x00\x00\xe6\x97\xb6\xe5\x80\x99rk\x01\x00\x00X\x03\x00\x00\x00\xe5\x93\xa6rl\x01\x00\x00X\x06\x00\x00\x00\xe8\x87\xb3\xe4\xba\x8erm\x01\x00\x00X\x06\x00\x00\x00\xe5\xad\xb0\xe7\x9f\xa5rn\x01\x00\x00X\x06\x00\x00\x00\xe9\x9d\x9e\xe7\x89\xb9ro\x01\x00\x00X\t\x00\x00\x00\xe4\xbb\xa5\xe8\x87\xb3\xe4\xba\x8erp\x01\x00\x00X\x03\x00\x00\x00\xe5\x8f\x8arq\x01\x00\x00X\x06\x00\x00\x00\xe5\x86\x8d\xe6\x9c\x89rr\x01\x00\x00X\x06\x00\x00\x00\xe5\xb8\xb8\xe8\xa7\x81rs\x01\x00\x00X\x03\x00\x00\x00\xe9\x9d\xa0rt\x01\x00\x00X\x01\x00\x00\x005ru\x01\x00\x00X\x06\x00\x00\x00\xe5\x90\x84\xe8\x87\xaarv\x01\x00\x00X\x06\x00\x00\x00\xe5\x93\xbc\xe5\x94\xb7rw\x01\x00\x00X\x03\x00\x00\x00\xe6\x9c\x88rx\x01\x00\x00X\x06\x00\x00\x00\xe5\xa6\x82\xe4\xb8\x8bry\x01\x00\x00X\x06\x00\x00\x00\xe5\x9b\xa0\xe6\xad\xa4rz\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe5\x85\x89r{\x01\x00\x00X\x06\x00\x00\x00\xe4\xba\x8c\xe6\x9d\xa5r|\x01\x00\x00X\x06\x00\x00\x00\xe4\xba\x8e\xe6\x98\xafr}\x01\x00\x00X\x06\x00\x00\x00\xe5\x8d\xb3\xe6\x88\x96r~\x01\x00\x00X\x03\x00\x00\x00\xe5\x96\xbdr\x7f\x01\x00\x00X\x03\x00\x00\x00\xe4\xb8\xb4r\x80\x01\x00\x00X\x06\x00\x00\x00\xe8\x80\x8c\xe5\x90\x8er\x81\x01\x00\x00X\x03\x00\x00\x00\xe8\xb0\x81r\x82\x01\x00\x00X\x03\x00\x00\x00\xe5\x91\xa2r\x83\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe4\xbe\xbfr\x84\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe8\xae\xbar\x85\x01\x00\x00X\x03\x00\x00\x00\xe5\x95\x8ar\x86\x01\x00\x00X\x06\x00\x00\x00\xe7\x94\x9a\xe4\xb9\x88r\x87\x01\x00\x00X\x0c\x00\x00\x00\xe6\x80\xbb\xe7\x9a\x84\xe6\x9d\xa5\xe8\xaf\xb4r\x88\x01\x00\x00X\x06\x00\x00\x00\xe8\x87\xaa\xe5\xb7\xb1r\x89\x01\x00\x00X\x06\x00\x00\x00\xe7\xad\x89\xe5\x88\xb0r\x8a\x01\x00\x00X\x0c\x00\x00\x00\xe7\x9b\xb8\xe5\xaf\xb9\xe8\x80\x8c\xe8\xa8\x80r\x8b\x01\x00\x00X\x06\x00\x00\x00\xe5\x88\xab\xe6\x98\xafr\x8c\x01\x00\x00X\x03\x00\x00\x00\xe4\xb8\x83r\x8d\x01\x00\x00X\x03\x00\x00\x00\xe4\xb9\x83r\x8e\x01\x00\x00X\x03\x00\x00\x00\xe5\x91\xb8r\x8f\x01\x00\x00X\x06\x00\x00\x00\xe5\xa4\x84\xe5\x9c\xa8r\x90\x01\x00\x00X\x06\x00\x00\x00\xe5\x91\xbc\xe5\x93\xa7r\x91\x01\x00\x00X\x06\x00\x00\x00\xe5\x88\x99\xe7\x94\x9ar\x92\x01\x00\x00X\x03\x00\x00\x00\xe3\x80\x8ar\x93\x01\x00\x00X\x06\x00\x00\x00\xe8\xb6\x81\xe7\x9d\x80r\x94\x01\x00\x00X\x06\x00\x00\x00\xe7\x94\x9a\xe8\x80\x8cr\x95\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x80\x95r\x96\x01\x00\x00X\x03\x00\x00\x00\xe8\xb7\x9dr\x97\x01\x00\x00X\x03\x00\x00\x00\xe4\xbd\x95r\x98\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe9\xa1\xb9r\x99\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x88\x90r\x9a\x01\x00\x00X\x06\x00\x00\x00\xe5\x8d\xb3\xe4\xbb\xa4r\x9b\x01\x00\x00X\x03\x00\x00\x00\xe9\xa1\xbar\x9c\x01\x00\x00X\x06\x00\x00\x00\xe4\xb9\x8b\xe5\x89\x8dr\x9d\x01\x00\x00X\x06\x00\x00\x00\xe5\x87\xa1\xe6\x98\xafr\x9e\x01\x00\x00X\x06\x00\x00\x00\xe8\x8b\xa5\xe6\x9e\x9cr\x9f\x01\x00\x00X\x06\x00\x00\x00\xe6\x96\xb9\xe6\xb3\x95r\xa0\x01\x00\x00X\x03\x00\x00\x00\xe4\xb8\x80r\xa1\x01\x00\x00X\x06\x00\x00\x00\xe5\x9b\xa0\xe7\x9d\x80r\xa2\x01\x00\x00X\x03\x00\x00\x00\xe7\x9d\x80r\xa3\x01\x00\x00X\x03\x00\x00\x00\xe5\x94\x89r\xa4\x01\x00\x00X\x03\x00\x00\x00\xe5\x93\x88r\xa5\x01\x00\x00X\x06\x00\x00\x00\xe5\x90\x91\xe4\xbd\xbfr\xa6\x01\x00\x00X\x03\x00\x00\x00\xe4\xba\x9br\xa7\x01\x00\x00X\x06\x00\x00\x00\xe7\x9b\xb4\xe5\x88\xb0r\xa8\x01\x00\x00X\x01\x00\x00\x004r\xa9\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe4\xb8\x8ar\xaa\x01\x00\x00X\x06\x00\x00\x00\xe5\xbd\xbc\xe6\xad\xa4r\xab\x01\x00\x00X\x06\x00\x00\x00\xe8\x80\x8c\xe5\x86\xb5r\xac\x01\x00\x00X\x06\x00\x00\x00\xe5\x82\xa5\xe7\x84\xb6r\xad\x01\x00\x00X\x06\x00\x00\x00\xe6\x9c\x9d\xe7\x9d\x80r\xae\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe6\xa0\xb7r\xaf\x01\x00\x00X\x03\x00\x00\x00\xe5\x93\x9fr\xb0\x01\x00\x00X\x06\x00\x00\x00\xe8\xb0\x81\xe4\xba\xbar\xb1\x01\x00\x00X\x06\x00\x00\x00\xe6\x89\x80\xe4\xbb\xa5r\xb2\x01\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe4\xb9\x88r\xb3\x01\x00\x00X\x03\x00\x00\x00\xe5\xb0\xb1r\xb4\x01\x00\x00X\x06\x00\x00\x00\xe4\xb9\x83\xe8\x87\xb3r\xb5\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe8\x87\xb3r\xb6\x01\x00\x00X\x06\x00\x00\x00\xe7\x94\x9a\xe4\xb8\x94r\xb7\x01\x00\x00X\x03\x00\x00\x00\xe5\xbe\x97r\xb8\x01\x00\x00X\x03\x00\x00\x00\xe5\x92\xa6r\xb9\x01\x00\x00X\t\x00\x00\x00\xe4\xb8\x8d\xe5\xa4\x96\xe4\xb9\x8er\xba\x01\x00\x00X\t\x00\x00\x00\xe4\xba\x8e\xe6\x98\xaf\xe4\xb9\x8er\xbb\x01\x00\x00X\x03\x00\x00\x00\xe5\x8f\xafr\xbc\x01\x00\x00X\x06\x00\x00\x00\xe6\x97\xa2\xe5\xbe\x80r\xbd\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\xba\xe7\x9d\x80r\xbe\x01\x00\x00X\x06\x00\x00\x00\xe5\xb7\xb4\xe5\xb7\xb4r\xbf\x01\x00\x00X\x06\x00\x00\x00\xe4\xba\xba\xe5\xae\xb6r\xc0\x01\x00\x00X\x06\x00\x00\x00\xe6\x9d\xa5\xe7\x9d\x80r\xc1\x01\x00\x00X\x06\x00\x00\x00\xe6\x80\x8e\xe5\xa5\x88r\xc2\x01\x00\x00X\x06\x00\x00\x00\xe4\xba\xba\xe4\xbb\xacr\xc3\x01\x00\x00X\x06\x00\x00\x00\xe8\xb5\xb7\xe8\xa7\x81r\xc4\x01\x00\x00X\x03\x00\x00\x00\xe4\xb8\xaar\xc5\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe5\x88\x87r\xc6\x01\x00\x00X\x06\x00\x00\x00\xe6\x9c\x89\xe5\x85\xb3r\xc7\x01\x00\x00X\x03\x00\x00\x00\xe5\x96\x8fr\xc8\x01\x00\x00X\x06\x00\x00\x00\xe5\x98\x8e\xe7\x99\xbbr\xc9\x01\x00\x00X\x06\x00\x00\x00\xe5\x85\xbc\xe4\xb9\x8br\xca\x01\x00\x00X\x06\x00\x00\x00\xe4\xbd\xa0\xe4\xbb\xacr\xcb\x01\x00\x00X\x06\x00\x00\x00\xe8\xbf\x98\xe8\xa6\x81r\xcc\x01\x00\x00X\x03\x00\x00\x00\xe5\x92\x8cr\xcd\x01\x00\x00X\x03\x00\x00\x00\xe5\xb0\xbdr\xce\x01\x00\x00X\x06\x00\x00\x00\xe5\xbc\x80\xe5\xa4\x96r\xcf\x01\x00\x00X\x06\x00\x00\x00\xe5\x9b\xba\xe7\x84\xb6r\xd0\x01\x00\x00X\x03\x00\x00\x00\xe5\x87\xa0r\xd1\x01\x00\x00X\x06\x00\x00\x00\xe6\x97\xa0\xe5\xae\x81r\xd2\x01\x00\x00X\x03\x00\x00\x00\xef\xbc\x81r\xd3\x01\x00\x00X\x03\x00\x00\x00\xe6\x97\xa2r\xd4\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe4\xbd\x86r\xd5\x01\x00\x00X\x06\x00\x00\x00\xe6\xad\xa4\xe9\x97\xb4r\xd6\x01\x00\x00X\x06\x00\x00\x00\xe7\xad\x89\xe7\xad\x89r\xd7\x01\x00\x00X\x06\x00\x00\x00\xe5\xae\x8c\xe6\x88\x90r\xd8\x01\x00\x00X\x03\x00\x00\x00\xe5\xbd\x92r\xd9\x01\x00\x00X\x06\x00\x00\x00\xe7\x9f\xa3\xe4\xb9\x8er\xda\x01\x00\x00X\x06\x00\x00\x00\xe6\x8c\x89\xe7\x85\xa7r\xdb\x01\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe5\x84\xbfr\xdc\x01\x00\x00X\x06\x00\x00\x00\xe8\xae\xbe\xe4\xbd\xbfr\xdd\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\xba\xe6\xad\xa2r\xde\x01\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe9\x87\x8cr\xdf\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe5\x88\xb0r\xe0\x01\x00\x00X\x03\x00\x00\x00\xe5\x90\xb1r\xe1\x01\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe6\xa0\xb7r\xe2\x01\x00\x00X\x06\x00\x00\x00\xe9\x99\xa4\xe4\xba\x86r\xe3\x01\x00\x00X\x06\x00\x00\x00\xe6\xad\xa4\xe6\x97\xb6r\xe4\x01\x00\x00X\x06\x00\x00\x00\xe5\x80\x98\xe8\x8b\xa5r\xe5\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe5\xa6\x82r\xe6\x01\x00\x00X\x0c\x00\x00\x00\xe6\x80\xbb\xe7\x9a\x84\xe6\x9d\xa5\xe7\x9c\x8br\xe7\x01\x00\x00X\t\x00\x00\x00\xe6\x80\x8e\xe4\xb9\x88\xe6\xa0\xb7r\xe8\x01\x00\x00X\x06\x00\x00\x00\xe6\x89\x80\xe5\xb9\xb8r\xe9\x01\x00\x00X\t\x00\x00\x00\xe8\xbf\x99\xe4\xb9\x88\xe6\xa0\xb7r\xea\x01\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe4\xba\x9br\xeb\x01\x00\x00X\x06\x00\x00\x00\xe5\x8d\x95\xe4\xb8\xaar\xec\x01\x00\x00X\x06\x00\x00\x00\xe5\x85\x89\xe6\x98\xafr\xed\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x87\xe4\xb8\x80r\xee\x01\x00\x00X\x03\x00\x00\x00\xe8\xb5\xb6r\xef\x01\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe7\x8b\xacr\xf0\x01\x00\x00X\x01\x00\x00\x00_r\xf1\x01\x00\x00X\x06\x00\x00\x00\xe5\x8f\x97\xe5\x88\xb0r\xf2\x01\x00\x00X\x03\x00\x00\x00\xe5\x98\x8er\xf3\x01\x00\x00X\x06\x00\x00\x00\xe5\x8f\xaa\xe6\x98\xafr\xf4\x01\x00\x00X\x06\x00\x00\x00\xe5\xb9\xb6\xe4\xb8\x94r\xf5\x01\x00\x00X\t\x00\x00\x00\xe6\x8d\xa2\xe8\xa8\x80\xe4\xb9\x8br\xf6\x01\x00\x00X\x06\x00\x00\x00\xe4\xbb\xbb\xe4\xbd\x95r\xf7\x01\x00\x00X\x03\x00\x00\x00\xe8\xb5\xb7r\xf8\x01\x00\x00X\x06\x00\x00\x00\xe4\xbe\xbf\xe4\xba\x8er\xf9\x01\x00\x00X\x03\x00\x00\x00\xe5\x91\x80r\xfa\x01\x00\x00X\x03\x00\x00\x00\xe4\xb9\x98r\xfb\x01\x00\x00X\x0c\x00\x00\x00\xe8\xbf\x99\xe5\xb0\xb1\xe6\x98\xaf\xe8\xaf\xb4r\xfc\x01\x00\x00X\x06\x00\x00\x00\xe5\x85\xb3\xe4\xba\x8er\xfd\x01\x00\x00X\t\x00\x00\x00\xe5\xb0\xb1\xe6\x98\xaf\xe8\xaf\xb4r\xfe\x01\x00\x00X\x06\x00\x00\x00\xe6\xaf\x94\xe5\x8f\x8ar\xff\x01\x00\x00X\x06\x00\x00\x00\xe5\x8f\x8d\xe4\xb9\x8br\x00\x02\x00\x00X\x03\x00\x00\x00\xe4\xb8\x8br\x01\x02\x00\x00X\x06\x00\x00\x00\xe6\x88\x96\xe8\x80\x85r\x02\x02\x00\x00X\x06\x00\x00\x00\xe5\x9b\xa0\xe4\xba\x86r\x03\x02\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe8\xbe\xb9r\x04\x02\x00\x00X\x0c\x00\x00\x00\xe5\x8f\xa6\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2r\x05\x02\x00\x00X\x06\x00\x00\x00\xe7\xba\xb5\xe7\x84\xb6r\x06\x02\x00\x00X\t\x00\x00\x00\xe8\x8e\xab\xe4\xb8\x8d\xe7\x84\xb6r\x07\x02\x00\x00X\t\x00\x00\x00\xe9\x9a\xbe\xe9\x81\x93\xe8\xaf\xb4r\x08\x02\x00\x00X\x06\x00\x00\x00\xe5\x89\x8d\xe8\x80\x85r\t\x02\x00\x00X\x06\x00\x00\x00\xe8\x87\xaa\xe4\xbb\x8er\n\x02\x00\x00X\x06\x00\x00\x00\xe5\x90\x8e\xe8\x80\x85r\x0b\x02\x00\x00X\x06\x00\x00\x00\xe9\x81\xb5\xe7\x85\xa7r\x0c\x02\x00\x00X\x06\x00\x00\x00\xe5\x8d\xb3\xe5\xa6\x82r\r\x02\x00\x00X\x03\x00\x00\x00\xe5\x98\x98r\x0e\x02\x00\x00X\x03\x00\x00\x00\xe6\x82\xa8r\x0f\x02\x00\x00X\x06\x00\x00\x00\xe7\x84\xb6\xe5\x88\x99r\x10\x02\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe6\x80\x95r\x11\x02\x00\x00X\x03\x00\x00\x00\xe8\x83\xbdr\x12\x02\x00\x00X\x06\x00\x00\x00\xe4\xbd\x95\xe5\xa4\x84r\x13\x02\x00\x00X\x06\x00\x00\x00\xe5\x9b\xa0\xe8\x80\x8cr\x14\x02\x00\x00X\x06\x00\x00\x00\xe4\xba\x91\xe4\xba\x91r\x15\x02\x00\x00X\x03\x00\x00\x00\xe6\xb2\xbfr\x16\x02\x00\x00X\x06\x00\x00\x00\xe6\xaf\x94\xe6\x96\xb9r\x17\x02\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe4\xb8\xaar\x18\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x9b\xe8\xa1\x8cr\x19\x02\x00\x00X\x03\x00\x00\x00\xe6\x9d\xa5r\x1a\x02\x00\x00X\t\x00\x00\x00\xe9\x82\xa3\xe4\xb9\x88\xe6\xa0\xb7r\x1b\x02\x00\x00X\x06\x00\x00\x00\xe5\x86\xb5\xe4\xb8\x94r\x1c\x02\x00\x00X\x03\x00\x00\x00\xe9\x83\xbdr\x1d\x02\x00\x00X\t\x00\x00\x00\xe4\xb9\x83\xe8\x87\xb3\xe4\xba\x8er\x1e\x02\x00\x00X\x06\x00\x00\x00\xe6\x83\x9f\xe5\x85\xb6r\x1f\x02\x00\x00X\x03\x00\x00\x00\xe6\x89\x80r \x02\x00\x00X\x03\x00\x00\x00\xe8\xaf\xa5r!\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe4\xbd\x95r"\x02\x00\x00X\x03\x00\x00\x00\xe5\x95\xa5r#\x02\x00\x00X\x06\x00\x00\x00\xe8\xa6\x81\xe4\xb9\x88r$\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x98\xe6\x9c\x89r%\x02\x00\x00X\t\x00\x00\x00\xe5\x86\x8d\xe8\x80\x85\xe8\xaf\xb4r&\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\xaf\xe8\xa7\x81r\'\x02\x00\x00X\x06\x00\x00\x00\xe6\x95\x85\xe8\x80\x8cr(\x02\x00\x00X\x03\x00\x00\x00\xe3\x80\x81r)\x02\x00\x00X\t\x00\x00\x00\xe5\x85\x88\xe4\xb8\x8d\xe5\x85\x88r*\x02\x00\x00X\x03\x00\x00\x00\xe5\xbd\x93r+\x02\x00\x00X\x03\x00\x00\x00\xe5\x98\xbfr,\x02\x00\x00X\t\x00\x00\x00\xe8\xa6\x81\xe4\xb8\x8d\xe7\x84\xb6r-\x02\x00\x00X\x03\x00\x00\x00\xe5\x90\x91r.\x02\x00\x00X\t\x00\x00\x00\xe5\x86\x8d\xe5\x85\xb6\xe6\xac\xa1r/\x02\x00\x00X\x06\x00\x00\x00\xe5\xa6\x82\xe5\xaf\xb9r0\x02\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe5\xb9\xb4r1\x02\x00\x00X\x06\x00\x00\x00\xe5\x88\xab\xe8\xaf\xb4r2\x02\x00\x00X\x06\x00\x00\x00\xe6\x89\x80\xe6\x9c\x89r3\x02\x00\x00X\x03\x00\x00\x00\xe4\xbb\x96r4\x02\x00\x00X\t\x00\x00\x00\xe5\x8f\x8d\xe8\xbf\x87\xe6\x9d\xa5r5\x02\x00\x00X\x06\x00\x00\x00\xe5\x97\xa1\xe5\x97\xa1r6\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x98\xafr7\x02\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe6\x9c\x9fr8\x02\x00\x00X\x03\x00\x00\x00\xe5\x91\x97r9\x02\x00\x00X\x06\x00\x00\x00\xe5\x90\x8c\xe6\x97\xb6r:\x02\x00\x00X\x03\x00\x00\x00\xe5\x92\xb3r;\x02\x00\x00X\x03\x00\x00\x00\xe5\x93\xa9r<\x02\x00\x00X\x06\x00\x00\x00\xe6\x80\xbb\xe4\xb9\x8br=\x02\x00\x00X\x06\x00\x00\x00\xe5\x87\xba\xe6\x9d\xa5r>\x02\x00\x00X\x06\x00\x00\x00\xe6\x97\x81\xe4\xba\xbar?\x02\x00\x00X\x03\x00\x00\x00\xe5\x93\x87r@\x02\x00\x00X\x03\x00\x00\x00\xe5\x95\x90rA\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe4\xba\x9brB\x02\x00\x00X\t\x00\x00\x00\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2rC\x02\x00\x00X\x03\x00\x00\x00\xe7\x85\xa7rD\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\xa6\xe5\xa4\x96rE\x02\x00\x00X\x06\x00\x00\x00\xe5\xae\x81\xe6\x84\xbfrF\x02\x00\x00X\x03\x00\x00\x00\xe8\xa6\x81rG\x02\x00\x00X\x06\x00\x00\x00\xe5\xbd\x93\xe7\x9d\x80rH\x02\x00\x00X\x06\x00\x00\x00\xe5\xa6\x82\xe8\x8b\xa5rI\x02\x00\x00X\x06\x00\x00\x00\xe8\xaf\xb8\xe4\xbd\x8drJ\x02\x00\x00X\x06\x00\x00\x00\xe8\xaf\xb4\xe6\x9d\xa5rK\x02\x00\x00X\t\x00\x00\x00\xe4\xb8\x80\xe8\xbd\xac\xe7\x9c\xbcrL\x02\x00\x00X\x03\x00\x00\x00\xe4\xbd\x86rM\x02\x00\x00X\x03\x00\x00\x00\xe9\x98\xbfrN\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x83\xb3rO\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\xae\xe5\x92\x9arP\x02\x00\x00X\x06\x00\x00\x00\xe5\x80\x98\xe4\xbd\xbfrQ\x02\x00\x00X\x0c\x00\x00\x00\xe6\x80\xbb\xe7\x9a\x84\xe8\xaf\xb4\xe6\x9d\xa5rR\x02\x00\x00X\x03\x00\x00\x00\xe7\xac\xacrS\x02\x00\x00X\x06\x00\x00\x00\xe8\x87\xaa\xe6\x89\x93rT\x02\x00\x00X\x06\x00\x00\x00\xe5\x85\xb6\xe4\xba\x8crU\x02\x00\x00X\x06\x00\x00\x00\xe7\x9a\x84\xe7\xa1\xaerV\x02\x00\x00X\x03\x00\x00\x00\xe4\xb9\x9drW\x02\x00\x00X\x06\x00\x00\x00\xe6\x9c\xac\xe5\x9c\xb0rX\x02\x00\x00X\x03\x00\x00\x00\xe5\x93\xaarY\x02\x00\x00X\x06\x00\x00\x00\xe8\x87\xaa\xe5\xae\xb6rZ\x02\x00\x00X\x01\x00\x00\x009r[\x02\x00\x00X\x03\x00\x00\x00\xe5\xa6\x82r\\\x02\x00\x00X\x06\x00\x00\x00\xe6\xad\xa3\xe5\xb7\xa7r]\x02\x00\x00X\x06\x00\x00\x00\xe5\x95\xaa\xe8\xbe\xber^\x02\x00\x00X\x06\x00\x00\x00\xe5\x90\xa7\xe5\x93\x92r_\x02\x00\x00X\x06\x00\x00\x00\xe5\xad\xb0\xe6\x96\x99r`\x02\x00\x00X\x03\x00\x00\x00\xe5\xb0\x94ra\x02\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe9\x87\x8crb\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x94\xe8\xaf\xb4rc\x02\x00\x00X\x06\x00\x00\x00\xe9\x82\xa3\xe8\x88\xacrd\x02\x00\x00X\x03\x00\x00\x00\xe5\xb0\x8fre\x02\x00\x00X\x06\x00\x00\x00\xe4\xbe\x9d\xe7\x84\xb6rf\x02\x00\x00X\x06\x00\x00\x00\xe5\xb0\xbd\xe7\xae\xa1rg\x02\x00\x00X\x06\x00\x00\x00\xe5\xbd\x92\xe9\xbd\x90rh\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\x8a\xe5\x85\xb6ri\x02\x00\x00X\x06\x00\x00\x00\xe8\xae\xbe\xe6\x88\x96rj\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe5\x8f\xaark\x02\x00\x00X\x06\x00\x00\x00\xe4\xbb\x80\xe4\xb9\x88rl\x02\x00\x00X\x06\x00\x00\x00\xe6\x97\xa0\xe8\xae\xbarm\x02\x00\x00X\x06\x00\x00\x00\xe6\x9b\xbf\xe4\xbb\xa3rn\x02\x00\x00X\x03\x00\x00\x00\xe4\xb9\x9fro\x02\x00\x00X\x03\x00\x00\x00\xe5\x9b\xa0rp\x02\x00\x00X\x03\x00\x00\x00\xe3\x80\x82rq\x02\x00\x00X\x03\x00\x00\x00\xe5\x8d\xb4rr\x02\x00\x00X\x06\x00\x00\x00\xe6\x95\x85\xe6\xad\xa4rs\x02\x00\x00X\x06\x00\x00\x00\xe5\xb8\xae\xe5\x8a\xa9rt\x02\x00\x00X\x06\x00\x00\x00\xe5\xba\xb6\xe4\xb9\x8eru\x02\x00\x00X\x03\x00\x00\x00\xe6\x89\x8drv\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\xba\xe4\xbd\x95rw\x02\x00\x00X\x06\x00\x00\x00\xe5\x87\xba\xe4\xba\x8erx\x02\x00\x00X\t\x00\x00\x00\xe5\xb9\xb4\xe6\x9c\x88\xe6\x97\xa5ry\x02\x00\x00X\x06\x00\x00\x00\xe5\xa4\xa7\xe5\xae\xb6rz\x02\x00\x00X\x06\x00\x00\x00\xe6\x88\xaa\xe8\x87\xb3r{\x02\x00\x00X\x06\x00\x00\x00\xe6\x9c\xac\xe7\x9d\x80r|\x02\x00\x00X\x0c\x00\x00\x00\xe5\x85\xb7\xe4\xbd\x93\xe8\xaf\xb4\xe6\x9d\xa5r}\x02\x00\x00X\x03\x00\x00\x00\xe5\x85\xb6r~\x02\x00\x00X\x06\x00\x00\x00\xe5\xb0\xb1\xe8\xa6\x81r\x7f\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8e\xe5\x90\xa6r\x80\x02\x00\x00X\x06\x00\x00\x00\xe7\x94\xb1\xe6\x98\xafr\x81\x02\x00\x00X\x06\x00\x00\x00\xe8\xae\xbe\xe8\xae\xa1r\x82\x02\x00\x00X\x06\x00\x00\x00\xe6\x80\x8e\xe4\xb9\x88r\x83\x02\x00\x00X\x03\x00\x00\x00\xe5\x91\x83r\x84\x02\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe5\x85\x8dr\x85\x02\x00\x00X\x0c\x00\x00\x00\xe4\xb8\x8e\xe6\xad\xa4\xe5\x90\x8c\xe6\x97\xb6r\x86\x02\x00\x00X\x06\x00\x00\x00\xe5\xae\x9e\xe7\x8e\xb0r\x87\x02\x00\x00X\x06\x00\x00\x00\xe5\x91\xb5\xe5\x91\xb5r\x88\x02\x00\x00X\t\x00\x00\x00\xe4\xb8\x94\xe4\xb8\x8d\xe8\xaf\xb4r\x89\x02\x00\x00X\x06\x00\x00\x00\xe5\x85\xb6\xe4\xb8\x80r\x8a\x02\x00\x00X\x03\x00\x00\x00\xe9\x99\xa4r\x8b\x02\x00\x00X\x0c\x00\x00\x00\xe5\xa6\x82\xe4\xb8\x8a\xe6\x89\x80\xe8\xbf\xb0r\x8c\x02\x00\x00X\x06\x00\x00\x00\xe5\x88\xab\xe5\xa4\x84r\x8d\x02\x00\x00X\x03\x00\x00\x00\xe4\xb8\x89r\x8e\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe4\xba\x9br\x8f\x02\x00\x00X\x06\x00\x00\x00\xe8\xa1\xa8\xe7\xa4\xbar\x90\x02\x00\x00X\x03\x00\x00\x00\xe5\x8d\x81r\x91\x02\x00\x00X\x03\x00\x00\x00\xe6\x9b\xbfr\x92\x02\x00\x00X\x06\x00\x00\x00\xe5\x90\x84\xe7\xa7\x8dr\x93\x02\x00\x00X\x06\x00\x00\x00\xe8\x80\x8c\xe4\xb8\x94r\x94\x02\x00\x00X\x03\x00\x00\x00\xe4\xba\x86r\x95\x02\x00\x00X\x06\x00\x00\x00\xe9\x9d\x9e\xe5\xbe\x92r\x96\x02\x00\x00X\x03\x00\x00\x00\xe8\x87\xb4r\x97\x02\x00\x00X\x06\x00\x00\x00\xe8\x8b\xa5\xe6\x98\xafr\x98\x02\x00\x00X\x0c\x00\x00\x00\xe5\x85\xb7\xe4\xbd\x93\xe5\x9c\xb0\xe8\xaf\xb4r\x99\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe4\xba\x86r\x9a\x02\x00\x00X\x03\x00\x00\x00\xe5\x90\x8er\x9b\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe6\x97\xa6r\x9c\x02\x00\x00X\x06\x00\x00\x00\xe6\x8a\x91\xe6\x88\x96r\x9d\x02\x00\x00X\x03\x00\x00\x00\xe7\xa6\xbbr\x9e\x02\x00\x00X\x06\x00\x00\x00\xe8\x8e\xab\xe8\x8b\xa5r\x9f\x02\x00\x00X\x03\x00\x00\x00\xe5\x88\xb0r\xa0\x02\x00\x00X\x0c\x00\x00\x00\xe6\x80\xbb\xe8\x80\x8c\xe8\xa8\x80\xe4\xb9\x8br\xa1\x02\x00\x00X\t\x00\x00\x00\xe8\xbf\x99\xe4\xbc\x9a\xe5\x84\xbfr\xa2\x02\x00\x00X\x06\x00\x00\x00\xe8\xb6\x8a\xe6\x98\xafr\xa3\x02\x00\x00X\x06\x00\x00\x00\xe5\x87\xad\xe5\x80\x9fr\xa4\x02\x00\x00X\x03\x00\x00\x00\xe4\xba\x94r\xa5\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe6\x97\xb6r\xa6\x02\x00\x00X\x06\x00\x00\x00\xe7\xbb\xa7\xe5\x90\x8er\xa7\x02\x00\x00X\x06\x00\x00\x00\xe6\xa0\xb9\xe6\x8d\xaer\xa8\x02\x00\x00X\x06\x00\x00\x00\xe5\xae\x83\xe4\xbb\xacr\xa9\x02\x00\x00X\x06\x00\x00\x00\xe8\xae\xbe\xe8\x8b\xa5r\xaa\x02\x00\x00X\x03\x00\x00\x00\xe5\x93\xbcr\xab\x02\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe5\x8f\x8ar\xac\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\xaa\xe8\xa6\x81r\xad\x02\x00\x00X\x06\x00\x00\x00\xe4\xbd\x95\xe5\x86\xb5r\xae\x02\x00\x00X\x06\x00\x00\x00\xe9\x92\x88\xe5\xaf\xb9r\xaf\x02\x00\x00X\x06\x00\x00\x00\xe6\xaf\x8f\xe5\xbd\x93r\xb0\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x9e\xe5\x90\x8cr\xb1\x02\x00\x00X\x06\x00\x00\x00\xe9\x9a\x8f\xe6\x97\xb6r\xb2\x02\x00\x00X\x03\x00\x00\x00\xe5\x88\x99r\xb3\x02\x00\x00X\t\x00\x00\x00\xe7\xae\x80\xe8\xa8\x80\xe4\xb9\x8br\xb4\x02\x00\x00X\x06\x00\x00\x00\xe6\x80\x8e\xe6\xa0\xb7r\xb5\x02\x00\x00X\x06\x00\x00\x00\xe7\x9d\x80\xe5\x91\xa2r\xb6\x02\x00\x00X\x06\x00\x00\x00\xe7\xba\xb5\xe4\xbd\xbfr\xb7\x02\x00\x00X\x06\x00\x00\x00\xe8\x87\xaa\xe8\xba\xabr\xb8\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\xaa\xe9\x99\x90r\xb9\x02\x00\x00X\x03\x00\x00\x00\xe5\x92\xa7r\xba\x02\x00\x00X\x06\x00\x00\x00\xe5\x92\xb1\xe4\xbb\xacr\xbb\x02\x00\x00X\x01\x00\x00\x000r\xbc\x02\x00\x00X\x03\x00\x00\x00\xe5\x85\xabr\xbd\x02\x00\x00X\x06\x00\x00\x00\xe6\x97\xa2\xe7\x84\xb6r\xbe\x02\x00\x00X\x03\x00\x00\x00\xe8\xa2\xabr\xbf\x02\x00\x00X\x03\x00\x00\x00\xe5\xb0\x86r\xc0\x02\x00\x00X\x06\x00\x00\x00\xe6\xad\xa3\xe5\x80\xbcr\xc1\x02\x00\x00X\x03\x00\x00\x00\xe7\xad\x89r\xc2\x02\x00\x00X\x0c\x00\x00\x00\xe6\x81\xb0\xe6\x81\xb0\xe7\x9b\xb8\xe5\x8f\x8dr\xc3\x02\x00\x00X\x06\x00\x00\x00\xe7\x9c\x81\xe5\xbe\x97r\xc4\x02\x00\x00X\x06\x00\x00\x00\xe5\x88\x9d\xe5\xa7\x8br\xc5\x02\x00\x00X\x0c\x00\x00\x00\xe8\x87\xb3\xe5\x85\xb3\xe9\x87\x8d\xe8\xa6\x81r\xc6\x02\x00\x00X\x06\x00\x00\x00\xe7\x94\xb1\xe6\xad\xa4r\xc7\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe8\x88\xacr\xc8\x02\x00\x00X\x06\x00\x00\x00\xe8\xad\xac\xe5\xa6\x82r\xc9\x02\x00\x00X\x06\x00\x00\x00\xe5\x85\xb6\xe4\xbd\x99r\xca\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe9\x97\xaer\xcb\x02\x00\x00X\x06\x00\x00\x00\xe5\x88\x86\xe5\x88\xabr\xcc\x02\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe4\xb8\xaar\xcd\x02\x00\x00X\x03\x00\x00\x00\xe5\xb7\xb4r\xce\x02\x00\x00X\x06\x00\x00\x00\xe6\x88\x96\xe5\x88\x99r\xcf\x02\x00\x00X\x03\x00\x00\x00\xe5\xb9\xb4r\xd0\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe5\x84\xbfr\xd1\x02\x00\x00X\x03\x00\x00\x00\xe5\xbd\xbcr\xd2\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe8\xbf\x87r\xd3\x02\x00\x00X\x03\x00\x00\x00\xe8\x87\xb3r\xd4\x02\x00\x00X\x03\x00\x00\x00\xe8\xbf\x9er\xd5\x02\x00\x00X\x06\x00\x00\x00\xe5\x93\xaa\xe4\xba\x9br\xd6\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\xba\xe4\xba\x86r\xd7\x02\x00\x00X\x03\x00\x00\x00\xe5\x86\x85r\xd8\x02\x00\x00X\x03\x00\x00\x00\xe6\xaf\x94r\xd9\x02\x00\x00X\x06\x00\x00\x00\xe6\x88\x96\xe6\x9b\xb0r\xda\x02\x00\x00X\x03\x00\x00\x00\xe4\xbb\xacr\xdb\x02\x00\x00X\x06\x00\x00\x00\xe8\x87\xb3\xe4\xbb\x8ar\xdc\x02\x00\x00X\x06\x00\x00\x00\xe5\xa6\x82\xe5\x90\x8cr\xdd\x02\x00\x00X\x06\x00\x00\x00\xe8\xbe\x83\xe4\xb9\x8br\xde\x02\x00\x00X\x03\x00\x00\x00\xe8\xbf\x98r\xdf\x02\x00\x00X\x03\x00\x00\x00\xe2\x80\x9cr\xe0\x02\x00\x00X\x03\x00\x00\x00\xe4\xb8\x94r\xe1\x02\x00\x00X\x06\x00\x00\x00\xe5\x88\xab\xe7\xae\xa1r\xe2\x02\x00\x00X\x06\x00\x00\x00\xe7\x94\x9a\xe8\x87\xb3r\xe3\x02\x00\x00X\x06\x00\x00\x00\xe6\x85\xa2\xe8\xaf\xb4r\xe4\x02\x00\x00X\x06\x00\x00\x00\xe7\x94\xa8\xe6\x9d\xa5r\xe5\x02\x00\x00X\x03\x00\x00\x00\xe5\x8f\xa6r\xe6\x02\x00\x00X\x03\x00\x00\x00\xe4\xb8\x8er\xe7\x02\x00\x00X\x06\x00\x00\x00\xe8\xbf\x99\xe4\xb9\x88r\xe8\x02\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe4\xb8\xbar\xe9\x02\x00\x00X\x03\x00\x00\x00\xe8\x87\xaar\xea\x02\x00\x00X\t\x00\x00\x00\xe8\xbf\x99\xe4\xb8\x80\xe6\x9d\xa5r\xeb\x02\x00\x00X\x06\x00\x00\x00\xe5\x85\xa8\xe4\xbd\x93r\xec\x02\x00\x00X\x06\x00\x00\x00\xe8\x87\xaa\xe5\x90\x8er\xed\x02\x00\x00X\x06\x00\x00\x00\xe5\xa5\xb9\xe4\xbb\xacr\xee\x02\x00\x00X\x03\x00\x00\x00\xe5\x91\x9cr\xef\x02\x00\x00X\x06\x00\x00\x00\xe5\x8f\xaf\xe6\x98\xafr\xf0\x02\x00\x00X\x03\x00\x00\x00\xe8\xbf\x87r\xf1\x02\x00\x00X\x06\x00\x00\x00\xe8\xa6\x81\xe6\x98\xafr\xf2\x02\x00\x00X\x06\x00\x00\x00\xe5\x9c\xa8\xe4\xba\x8er\xf3\x02\x00\x00X\x03\x00\x00\x00\xe6\x9b\xber\xf4\x02\x00\x00X\x03\x00\x00\x00\xe5\x8f\xb7r\xf5\x02\x00\x00X\x06\x00\x00\x00\xe6\xa0\xb9\xe6\x9c\xacr\xf6\x02\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe6\xac\xber\xf7\x02\x00\x00X\x06\x00\x00\x00\xe5\xaf\xb9\xe6\x96\xb9r\xf8\x02\x00\x00X\x06\x00\x00\x00\xe5\x90\x84\xe4\xb8\xaar\xf9\x02\x00\x00X\x06\x00\x00\x00\xe5\x81\x87\xe5\xa6\x82r\xfa\x02\x00\x00X\x03\x00\x00\x00\xe6\x8a\x8ar\xfb\x02\x00\x00X\x06\x00\x00\x00\xe9\x80\x9a\xe8\xbf\x87r\xfc\x02\x00\x00X\x03\x00\x00\x00\xe9\x9a\x8fr\xfd\x02\x00\x00X\x03\x00\x00\x00\xe5\x93\x97r\xfe\x02\x00\x00X\x06\x00\x00\x00\xe6\xaf\x8b\xe5\xae\x81r\xff\x02\x00\x00X\x06\x00\x00\x00\xe9\x99\x84\xe5\x8a\xa0r\x00\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x83\x9fr\x01\x03\x00\x00X\x03\x00\x00\x00\xe5\xa5\xb9r\x02\x03\x00\x00X\x06\x00\x00\x00\xe5\x8f\x8d\xe8\x80\x8cr\x03\x03\x00\x00X\x06\x00\x00\x00\xe6\x9c\x89\xe4\xba\x9br\x04\x03\x00\x00X\x06\x00\x00\x00\xe7\xbd\xa2\xe4\xba\x86r\x05\x03\x00\x00X\x06\x00\x00\x00\xe6\x9c\x89\xe5\x8f\x8ar\x06\x03\x00\x00X\x06\x00\x00\x00\xe8\x87\xb3\xe8\x8b\xa5r\x07\x03\x00\x00X\x03\x00\x00\x00\xe5\xa4\xa7r\x08\x03\x00\x00X\x03\x00\x00\x00\xe6\x88\x91r\t\x03\x00\x00X\x06\x00\x00\x00\xe5\x86\x8d\xe8\x80\x85r\n\x03\x00\x00X\x06\x00\x00\x00\xe4\xbe\x9d\xe7\x85\xa7r\x0b\x03\x00\x00X\t\x00\x00\x00\xe8\xa6\x81\xe4\xb8\x8d\xe6\x98\xafr\x0c\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\xaa\xe5\x88\xabr\r\x03\x00\x00X\x06\x00\x00\x00\xe8\xad\xac\xe5\x96\xbbr\x0e\x03\x00\x00X\x04\x00\x00\x00\xef\xbb\xbf$r\x0f\x03\x00\x00X\x03\x00\x00\x00\xe8\x99\xbdr\x10\x03\x00\x00X\t\x00\x00\x00\xe7\x94\x9a\xe8\x87\xb3\xe4\xba\x8er\x11\x03\x00\x00X\x03\x00\x00\x00\xe8\xbf\x99r\x12\x03\x00\x00X\x06\x00\x00\x00\xe5\xa6\x82\xe6\x9e\x9cr\x13\x03\x00\x00X\x03\x00\x00\x00\xe8\xbe\xb9r\x14\x03\x00\x00X\x03\x00\x00\x00\xe5\x93\x8er\x15\x03\x00\x00X\x03\x00\x00\x00\xe4\xbd\xbfr\x16\x03\x00\x00X\x06\x00\x00\x00\xe4\xbe\x9d\xe6\x8d\xaer\x17\x03\x00\x00X\x03\x00\x00\x00\xe5\x88\xabr\x18\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe5\x8d\x95r\x19\x03\x00\x00X\x03\x00\x00\x00\xe5\x92\x8br\x1a\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe5\xa5\xbdr\x1b\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x8d\xe6\x8b\x98r\x1c\x03\x00\x00X\x03\x00\x00\x00\xe4\xbb\xbbr\x1d\x03\x00\x00X\x03\x00\x00\x00\xe5\x90\x84r\x1e\x03\x00\x00X\x06\x00\x00\x00\xe5\x85\xb6\xe5\xae\x83r\x1f\x03\x00\x00X\x06\x00\x00\x00\xe6\xad\xa3\xe6\x98\xafr \x03\x00\x00X\x06\x00\x00\x00\xe4\xbb\xa5\xe6\x9d\xa5r!\x03\x00\x00X\x06\x00\x00\x00\xe7\x9a\x84\xe8\xaf\x9dr"\x03\x00\x00X\x06\x00\x00\x00\xe8\x99\xbd\xe5\x88\x99r#\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe5\x88\x99r$\x03\x00\x00X\x03\x00\x00\x00\xe5\x87\xadr%\x03\x00\x00X\x06\x00\x00\x00\xe4\xbb\x96\xe4\xbb\xacr&\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe4\xb8\xaar\'\x03\x00\x00X\x06\x00\x00\x00\xe4\xb8\x80\xe7\x9b\xb4r(\x03\x00\x00X\x06\x00\x00\x00\xe8\x8b\xa5\xe9\x9d\x9er)\x03\x00\x00X\x06\x00\x00\x00\xe9\x84\x99\xe4\xba\xbar*\x03\x00\x00X\x06\x00\x00\x00\xe5\xa6\x82\xe4\xbd\x95r+\x03\x00\x00X\x06\x00\x00\x00\xe5\xbd\xbc\xe6\x97\xb6r,\x03\x00\x00X\t\x00\x00\x00\xe7\xb4\xa7\xe6\x8e\xa5\xe7\x9d\x80r-\x03\x00\x00X\x06\x00\x00\x00\xe5\x86\x8d\xe5\x88\x99r.\x03\x00\x00X\x03\x00\x00\x00\xe4\xba\xbar/\x03\x00\x00e\x85r0\x03\x00\x00Rr1\x03\x00\x00.'
en = b'\x80\x03cbuiltins\nset\nq\x00]q\x01(X\x07\x00\x00\x00thoughhq\x02X\t\x00\x00\x00somethingq\x03X\t\x00\x00\x00beginningq\x04X\x05\x00\x00\x00groupq\x05X\x04\x00\x00\x00fromq\x06X\x04\x00\x00\x00thanq\x07X\x06\x00\x00\x00becameq\x08X\x04\x00\x00\x00abstq\tX\n\x00\x00\x00throughoutq\nX\x04\x00\x00\x00doneq\x0bX\x05\x00\x00\x00we\'llq\x0cX\x04\x00\x00\x00elseq\rX\x02\x00\x00\x00inq\x0eX\x06\x00\x00\x00latelyq\x0fX\x03\x00\x00\x00nowq\x10X\x08\x00\x00\x00wouldn\'tq\x11X\x03\x00\x00\x00etcq\x12X\x07\x00\x00\x00similarq\x13X\t\x00\x00\x00necessaryq\x14X\x04\x00\x00\x00turnq\x15X\x07\x00\x00\x00turningq\x16X\x07\x00\x00\x00doesn\'tq\x17X\x01\x00\x00\x00oq\x18X\n\x00\x00\x00everythingq\x19X\x07\x00\x00\x00readilyq\x1aX\x08\x00\x00\x00obtainedq\x1bX\x03\x00\x00\x00hisq\x1cX\x04\x00\x00\x00hersq\x1dX\x08\x00\x00\x00possibleq\x1eX\x01\x00\x00\x00qq\x1fX\x06\x00\x00\x00towardq X\x05\x00\x00\x00showsq!X\t\x00\x00\x00interestsq"X\x04\x00\x00\x00suchq#X\x07\x00\x00\x00whereinq$X\n\x00\x00\x00specifyingq%X\x07\x00\x00\x00welcomeq&X\x03\x00\x00\x00mayq\'X\x06\x00\x00\x00anyoneq(X\x06\x00\x00\x00thinksq)X\x04\x00\x00\x00pastq*X\x05\x00\x00\x00thirdq+X\x02\x00\x00\x00weq,X\n\x00\x00\x00whereafterq-X\t\x00\x00\x00indicatesq.X\x07\x00\x00\x00haven\'tq/X\x04\x00\x00\x00i\'llq0X\r\x00\x00\x00predominantlyq1X\x06\x00\x00\x00wasn\'tq2X\x05\x00\x00\x00whatsq3X\x04\x00\x00\x00letsq4X\x02\x00\x00\x00doq5X\x07\x00\x00\x00whoeverq6X\x05\x00\x00\x00you\'dq7X\x0b\x00\x00\x00necessarilyq8X\x03\x00\x00\x00tilq9X\x06\x00\x00\x00memberq:X\x04\x00\x00\x00goneq;X\x05\x00\x00\x00beginq<X\x07\x00\x00\x00omittedq=X\x07\x00\x00\x00showingq>X\x04\x00\x00\x00longq?X\x08\x00\x00\x00whateverq@X\x06\x00\x00\x00latterqAX\x04\x00\x00\x00whodqBX\x0b\x00\x00\x00furthermoreqCX\x04\x00\x00\x00overqDX\x03\x00\x00\x00oneqEX\x05\x00\x00\x00underqFX\x06\x00\x00\x00askingqGX\x04\x00\x00\x00arenqHX\x07\x00\x00\x00includeqIX\x05\x00\x00\x00valueqJX\x05\x00\x00\x00sidesqKX\t\x00\x00\x00thereuponqLX\x05\x00\x00\x00novelqMX\x03\x00\x00\x00herqNX\n\x00\x00\x00beginningsqOX\x04\x00\x00\x00showqPX\x07\x00\x00\x00variousqQX\x07\x00\x00\x00awfullyqRX\x0b\x00\x00\x00differentlyqSX\x03\x00\x00\x00itsqTX\x04\x00\x00\x00keptqUX\x05\x00\x00\x00causeqVX\x01\x00\x00\x00pqWX\x05\x00\x00\x00notedqXX\x05\x00\x00\x00otherqYX\x06\x00\x00\x00unlikeqZX\x04\x00\x00\x00makeq[X\x07\x00\x00\x00insofarq\\X\x03\x00\x00\x00i\'mq]X\x02\x00\x00\x00mlq^X\x03\x00\x00\x00seeq_X\r\x00\x00\x00unfortunatelyq`X\x07\x00\x00\x00couldntqaX\x05\x00\x00\x00wouldqbX\x07\x00\x00\x00howeverqcX\x03\x00\x00\x00itdqdX\t\x00\x00\x00certainlyqeX\x04\x00\x00\x00madeqfX\x07\x00\x00\x00suggestqgX\x07\x00\x00\x00brieflyqhX\x05\x00\x00\x00c\'monqiX\t\x00\x00\x00importantqjX\x06\x00\x00\x00eightyqkX\x08\x00\x00\x00resultedqlX\n\x00\x00\x00yourselvesqmX\x05\x00\x00\x00couldqnX\x06\x00\x00\x00namelyqoX\x05\x00\x00\x00theydqpX\x04\x00\x00\x00cameqqX\x05\x00\x00\x00wordsqrX\x04\x00\x00\x00keysqsX\x03\x00\x00\x00runqtX\x06\x00\x00\x00herebyquX\x07\x00\x00\x00betweenqvX\r\x00\x00\x00approximatelyqwX\x04\x00\x00\x00we\'dqxX\x03\x00\x00\x00\'veqyX\x04\x00\x00\x00wellqzX\x02\x00\x00\x00kmq{X\x0c\x00\x00\x00specificallyq|X\x03\x00\x00\x00oldq}X\x07\x00\x00\x00anotherq~X\x01\x00\x00\x00wq\x7fX\r\x00\x00\x00correspondingq\x80X\x07\x00\x00\x00wantingq\x81X\x04\x00\x00\x00youdq\x82X\x03\x00\x00\x00whyq\x83X\x02\x00\x00\x00noq\x84X\x07\x00\x00\x00usuallyq\x85X\n\x00\x00\x00afterwardsq\x86X\x07\x00\x00\x00anymoreq\x87X\x07\x00\x00\x00sectionq\x88X\x04\x00\x00\x00plusq\x89X\x06\x00\x00\x00newestq\x8aX\t\x00\x00\x00elsewhereq\x8bX\x02\x00\x00\x00ahq\x8cX\x06\x00\x00\x00tryingq\x8dX\x03\x00\x00\x00upsq\x8eX\x05\x00\x00\x00todayq\x8fX\x01\x00\x00\x00lq\x90X\n\x00\x00\x00apparentlyq\x91X\x05\x00\x00\x00findsq\x92X\x04\x00\x00\x00endsq\x93X\x06\x00\x00\x00givingq\x94X\x04\x00\x00\x00faceq\x95X\x05\x00\x00\x00beganq\x96X\t\x00\x00\x00downwardsq\x97X\x03\x00\x00\x00hasq\x98X\x04\x00\x00\x00knowq\x99X\x06\x00\x00\x00showedq\x9aX\x07\x00\x00\x00againstq\x9bX\x05\x00\x00\x00aboveq\x9cX\x07\x00\x00\x00thoughtq\x9dX\x06\x00\x00\x00you\'req\x9eX\x04\x00\x00\x00seesq\x9fX\x04\x00\x00\x00backq\xa0X\x05\x00\x00\x00oughtq\xa1X\x05\x00\x00\x00triedq\xa2X\x08\x00\x00\x00actuallyq\xa3X\x02\x00\x00\x00req\xa4X\x04\x00\x00\x00usesq\xa5X\x03\x00\x00\x00gotq\xa6X\x06\x00\x00\x00she\'llq\xa7X\x04\x00\x00\x00thusq\xa8X\x07\x00\x00\x00exactlyq\xa9X\t\x00\x00\x00otherwiseq\xaaX\x06\x00\x00\x00theredq\xabX\x08\x00\x00\x00togetherq\xacX\x02\x00\x00\x00ifq\xadX\x05\x00\x00\x00oftenq\xaeX\x03\x00\x00\x00offq\xafX\x03\x00\x00\x00nosq\xb0X\x05\x00\x00\x00proudq\xb1X\x04\x00\x00\x00intoq\xb2X\x05\x00\x00\x00allowq\xb3X\x08\x00\x00\x00there\'llq\xb4X\t\x00\x00\x00indicatedq\xb5X\x05\x00\x00\x00againq\xb6X\x04\x00\x00\x00authq\xb7X\x05\x00\x00\x00et-alq\xb8X\x07\x00\x00\x00downingq\xb9X\x03\x00\x00\x00eduq\xbaX\x07\x00\x00\x00exampleq\xbbX\x05\x00\x00\x00we\'req\xbcX\x08\x00\x00\x00furthersq\xbdX\x04\x00\x00\x00saysq\xbeX\x0b\x00\x00\x00informationq\xbfX\x05\x00\x00\x00rightq\xc0X\x07\x00\x00\x00they\'llq\xc1X\x04\x00\x00\x00he\'sq\xc2X\x06\x00\x00\x00thanksq\xc3X\x06\x00\x00\x00theresq\xc4X\x03\x00\x00\x00getq\xc5X\x04\x00\x00\x00it\'sq\xc6X\x07\x00\x00\x00seriousq\xc7X\x05\x00\x00\x00apartq\xc8X\n\x00\x00\x00regardlessq\xc9X\x07\x00\x00\x00despiteq\xcaX\x07\x00\x00\x00neitherq\xcbX\x03\x00\x00\x00ordq\xccX\x04\x00\x00\x00openq\xcdX\x07\x00\x00\x00they\'req\xceX\x02\x00\x00\x00myq\xcfX\t\x00\x00\x00availableq\xd0X\x06\x00\x00\x00evenlyq\xd1X\x06\x00\x00\x00partedq\xd2X\x07\x00\x00\x00regardsq\xd3X\x07\x00\x00\x00theretoq\xd4X\x06\x00\x00\x00anyhowq\xd5X\x06\x00\x00\x00seemedq\xd6X\x05\x00\x00\x00everyq\xd7X\x04\x00\x00\x00areaq\xd8X\n\x00\x00\x00usefulnessq\xd9X\x08\x00\x00\x00everyoneq\xdaX\x05\x00\x00\x00doingq\xdbX\x08\x00\x00\x00wheneverq\xdcX\t\x00\x00\x00describedq\xddX\x04\x00\x00\x00waysq\xdeX\x05\x00\x00\x00eightq\xdfX\x06\x00\x00\x00betterq\xe0X\x06\x00\x00\x00ordersq\xe1X\x07\x00\x00\x00greaterq\xe2X\x05\x00\x00\x00helloq\xe3X\x04\x00\x00\x00manyq\xe4X\x08\x00\x00\x00anywhereq\xe5X\x04\x00\x00\x00wentq\xe6X\t\x00\x00\x00similarlyq\xe7X\x07\x00\x00\x00somehowq\xe8X\t\x00\x00\x00currentlyq\xe9X\x05\x00\x00\x00comesq\xeaX\x05\x00\x00\x00theseq\xebX\x05\x00\x00\x00let\'sq\xecX\x06\x00\x00\x00workedq\xedX\x04\x00\x00\x00everq\xeeX\x08\x00\x00\x00indicateq\xefX\x04\x00\x00\x00nextq\xf0X\x0c\x00\x00\x00neverthelessq\xf1X\x08\x00\x00\x00orderingq\xf2X\x04\x00\x00\x00lastq\xf3X\x04\x00\x00\x00whenq\xf4X\n\x00\x00\x00containingq\xf5X\x03\x00\x00\x00viaq\xf6X\x02\x00\x00\x00amq\xf7X\x06\x00\x00\x00thenceq\xf8X\x06\x00\x00\x00almostq\xf9X\x04\x00\x00\x00someq\xfaX\x02\x00\x00\x00ppq\xfbX\x04\x00\x00\x00whomq\xfcX\t\x00\x00\x00hereafterq\xfdX\x05\x00\x00\x00factsq\xfeX\x07\x00\x00\x00presentq\xffX\x05\x00\x00\x00casesr\x00\x01\x00\x00X\t\x00\x00\x00obviouslyr\x01\x01\x00\x00X\x01\x00\x00\x00yr\x02\x01\x00\x00X\x05\x00\x00\x00knowsr\x03\x01\x00\x00X\x07\x00\x00\x00herselfr\x04\x01\x00\x00X\x07\x00\x00\x00smallerr\x05\x01\x00\x00X\x04\x00\x00\x00fullr\x06\x01\x00\x00X\x03\x00\x00\x00outr\x07\x01\x00\x00X\x04\x00\x00\x00meanr\x08\x01\x00\x00X\x06\x00\x00\x00exceptr\t\x01\x00\x00X\x01\x00\x00\x00cr\n\x01\x00\x00X\x05\x00\x00\x00wantsr\x0b\x01\x00\x00X\x02\x00\x00\x00mrr\x0c\x01\x00\x00X\x01\x00\x00\x00jr\r\x01\x00\x00X\x06\x00\x00\x00placedr\x0e\x01\x00\x00X\x08\x00\x00\x00announcer\x0f\x01\x00\x00X\x01\x00\x00\x00dr\x10\x01\x00\x00X\x03\x00\x00\x00putr\x11\x01\x00\x00X\x04\x00\x00\x00biolr\x12\x01\x00\x00X\x03\x00\x00\x00subr\x13\x01\x00\x00X\x05\x00\x00\x00triesr\x14\x01\x00\x00X\x06\x00\x00\x00withinr\x15\x01\x00\x00X\x04\x00\x00\x00kindr\x16\x01\x00\x00X\t\x00\x00\x00generallyr\x17\x01\x00\x00X\x06\x00\x00\x00gottenr\x18\x01\x00\x00X\x06\x00\x00\x00wonderr\x19\x01\x00\x00X\x03\x00\x00\x00fixr\x1a\x01\x00\x00X\x06\x00\x00\x00unlessr\x1b\x01\x00\x00X\x03\x00\x00\x00your\x1c\x01\x00\x00X\x02\x00\x00\x00isr\x1d\x01\x00\x00X\x06\x00\x00\x00mainlyr\x1e\x01\x00\x00X\x04\x00\x00\x00giver\x1f\x01\x00\x00X\x03\x00\x00\x00volr \x01\x00\x00X\x04\x00\x00\x00haver!\x01\x00\x00X\x08\x00\x00\x00greatestr"\x01\x00\x00X\x03\x00\x00\x00adjr#\x01\x00\x00X\x04\x00\x00\x00seenr$\x01\x00\x00X\x06\x00\x00\x00obtainr%\x01\x00\x00X\x02\x00\x00\x00cor&\x01\x00\x00X\x0b\x00\x00\x00appropriater\'\x01\x00\x00X\x03\x00\x00\x00refr(\x01\x00\x00X\x07\x00\x00\x00throughr)\x01\x00\x00X\x05\x00\x00\x00endedr*\x01\x00\x00X\x04\x00\x00\x00lessr+\x01\x00\x00X\x03\x00\x00\x00vizr,\x01\x00\x00X\x04\x00\x00\x00helpr-\x01\x00\x00X\x07\x00\x00\x00believer.\x01\x00\x00X\x05\x00\x00\x00thanxr/\x01\x00\x00X\x07\x00\x00\x00thereinr0\x01\x00\x00X\t\x00\x00\x00greetingsr1\x01\x00\x00X\x07\x00\x00\x00there\'sr2\x01\x00\x00X\x01\x00\x00\x00hr3\x01\x00\x00X\x05\x00\x00\x00trulyr4\x01\x00\x00X\x07\x00\x00\x00furtherr5\x01\x00\x00X\x08\x00\x00\x00inasmuchr6\x01\x00\x00X\x06\x00\x00\x00oldestr7\x01\x00\x00X\x06\x00\x00\x00beyondr8\x01\x00\x00X\x04\x00\x00\x00i\'ver9\x01\x00\x00X\x08\x00\x00\x00moreoverr:\x01\x00\x00X\x08\x00\x00\x00somewhatr;\x01\x00\x00X\x05\x00\x00\x00amongr<\x01\x00\x00X\x05\x00\x00\x00theirr=\x01\x00\x00X\x02\x00\x00\x00tor>\x01\x00\x00X\x02\x00\x00\x00byr?\x01\x00\x00X\x05\x00\x00\x00mayber@\x01\x00\x00X\x07\x00\x00\x00anywaysrA\x01\x00\x00X\x05\x00\x00\x00yourerB\x01\x00\x00X\x06\x00\x00\x00thingsrC\x01\x00\x00X\x06\x00\x00\x00pleaserD\x01\x00\x00X\x04\x00\x00\x00caserE\x01\x00\x00X\t\x00\x00\x00primarilyrF\x01\x00\x00X\n\x00\x00\x00furtheringrG\x01\x00\x00X\x0c\x00\x00\x00particularlyrH\x01\x00\x00X\x02\x00\x00\x00ZZrI\x01\x00\x00X\x05\x00\x00\x00seemsrJ\x01\x00\x00X\x01\x00\x00\x00krK\x01\x00\x00X\x05\x00\x00\x00ariserL\x01\x00\x00X\x06\x00\x00\x00beforerM\x01\x00\x00X\x06\x00\x00\x00widelyrN\x01\x00\x00X\x06\x00\x00\x00theirsrO\x01\x00\x00X\x04\x00\x00\x00whatrP\x01\x00\x00X\x07\x00\x00\x00outsiderQ\x01\x00\x00X\x06\x00\x00\x00selvesrR\x01\x00\x00X\x05\x00\x00\x00won\'trS\x01\x00\x00X\x01\x00\x00\x00rrT\x01\x00\x00X\x02\x00\x00\x00berU\x01\x00\x00X\t\x00\x00\x00resultingrV\x01\x00\x00X\x05\x00\x00\x00quiterW\x01\x00\x00X\x06\x00\x00\x00you\'llrX\x01\x00\x00X\x05\x00\x00\x00foundrY\x01\x00\x00X\x05\x00\x00\x00earlyrZ\x01\x00\x00X\x04\x00\x00\x00morer[\x01\x00\x00X\x06\x00\x00\x00longerr\\\x01\x00\x00X\x06\x00\x00\x00placesr]\x01\x00\x00X\x03\x00\x00\x00quer^\x01\x00\x00X\n\x00\x00\x00everywherer_\x01\x00\x00X\x06\x00\x00\x00beginsr`\x01\x00\x00X\x04\x00\x00\x00veryra\x01\x00\x00X\x05\x00\x00\x00untilrb\x01\x00\x00X\x05\x00\x00\x00opensrc\x01\x00\x00X\x07\x00\x00\x00longestrd\x01\x00\x00X\x06\x00\x00\x00througre\x01\x00\x00X\x07\x00\x00\x00workingrf\x01\x00\x00X\x06\x00\x00\x00pointsrg\x01\x00\x00X\x04\x00\x00\x00factrh\x01\x00\x00X\x03\x00\x00\x00andri\x01\x00\x00X\x02\x00\x00\x00\'srj\x01\x00\x00X\n\x00\x00\x00concerningrk\x01\x00\x00X\n\x00\x00\x00presumablyrl\x01\x00\x00X\x07\x00\x00\x00weren\'trm\x01\x00\x00X\t\x00\x00\x00differentrn\x01\x00\x00X\x06\x00\x00\x00recentro\x01\x00\x00X\x03\x00\x00\x00norrp\x01\x00\x00X\x07\x00\x00\x00seemingrq\x01\x00\x00X\x08\x00\x00\x00slightlyrr\x01\x00\x00X\x04\x00\x00\x00sentrs\x01\x00\x00X\x04\x00\x00\x00onlyrt\x01\x00\x00X\x04\x00\x00\x00mostru\x01\x00\x00X\x05\x00\x00\x00twicerv\x01\x00\x00X\x07\x00\x00\x00nothingrw\x01\x00\x00X\x04\x00\x00\x00uucprx\x01\x00\x00X\x05\x00\x00\x00mightry\x01\x00\x00X\x07\x00\x00\x00affectsrz\x01\x00\x00X\x05\x00\x00\x00knownr{\x01\x00\x00X\x03\x00\x00\x00ltdr|\x01\x00\x00X\x02\x00\x00\x00mer}\x01\x00\x00X\x03\x00\x00\x00letr~\x01\x00\x00X\x05\x00\x00\x00alongr\x7f\x01\x00\x00X\x05\x00\x00\x00heresr\x80\x01\x00\x00X\x05\x00\x00\x00sincer\x81\x01\x00\x00X\x05\x00\x00\x00addedr\x82\x01\x00\x00X\x05\x00\x00\x00smallr\x83\x01\x00\x00X\x02\x00\x00\x00ier\x84\x01\x00\x00X\x08\x00\x00\x00formerlyr\x85\x01\x00\x00X\x07\x00\x00\x00that\'llr\x86\x01\x00\x00X\x02\x00\x00\x00mgr\x87\x01\x00\x00X\x03\x00\x00\x00sayr\x88\x01\x00\x00X\x03\x00\x00\x00hedr\x89\x01\x00\x00X\x04\x00\x00\x00bestr\x8a\x01\x00\x00X\x07\x00\x00\x00lookingr\x8b\x01\x00\x00X\x06\x00\x00\x00effectr\x8c\x01\x00\x00X\x07\x00\x00\x00followsr\x8d\x01\x00\x00X\x05\x00\x00\x00thankr\x8e\x01\x00\x00X\x06\x00\x00\x00endingr\x8f\x01\x00\x00X\x08\x00\x00\x00there\'ver\x90\x01\x00\x00X\x03\x00\x00\x00nayr\x91\x01\x00\x00X\x02\x00\x00\x00car\x92\x01\x00\x00X\x03\x00\x00\x00comr\x93\x01\x00\x00X\x04\x00\x00\x00homer\x94\x01\x00\x00X\x05\x00\x00\x00who\'sr\x95\x01\x00\x00X\x07\x00\x00\x00nowherer\x96\x01\x00\x00X\x03\x00\x00\x00n\'tr\x97\x01\x00\x00X\x03\x00\x00\x00\'rer\x98\x01\x00\x00X\x06\x00\x00\x00hardlyr\x99\x01\x00\x00X\x04\x00\x00\x00findr\x9a\x01\x00\x00X\x07\x00\x00\x00withoutr\x9b\x01\x00\x00X\x04\x00\x00\x00awayr\x9c\x01\x00\x00X\x05\x00\x00\x00needsr\x9d\x01\x00\x00X\x04\x00\x00\x00pager\x9e\x01\x00\x00X\x08\x00\x00\x00whereverr\x9f\x01\x00\x00X\x01\x00\x00\x00nr\xa0\x01\x00\x00X\x04\x00\x00\x00downr\xa1\x01\x00\x00X\x0b\x00\x00\x00potentiallyr\xa2\x01\x00\x00X\x03\x00\x00\x00tryr\xa3\x01\x00\x00X\x08\x00\x00\x00presentsr\xa4\x01\x00\x00X\x05\x00\x00\x00goingr\xa5\x01\x00\x00X\x06\x00\x00\x00besider\xa6\x01\x00\x00X\x08\x00\x00\x00researchr\xa7\x01\x00\x00X\x03\x00\x00\x00manr\xa8\x01\x00\x00X\x04\x00\x00\x00taker\xa9\x01\x00\x00X\x06\x00\x00\x00mostlyr\xaa\x01\x00\x00X\x04\x00\x00\x00werer\xab\x01\x00\x00X\x02\x00\x00\x00ffr\xac\x01\x00\x00X\x04\x00\x00\x00niner\xad\x01\x00\x00X\x07\x00\x00\x00what\'llr\xae\x01\x00\x00X\x04\x00\x00\x00eachr\xaf\x01\x00\x00X\x01\x00\x00\x00tr\xb0\x01\x00\x00X\x05\x00\x00\x00makesr\xb1\x01\x00\x00X\x05\x00\x00\x00whoser\xb2\x01\x00\x00X\x07\x00\x00\x00discussr\xb3\x01\x00\x00X\x05\x00\x00\x00arentr\xb4\x01\x00\x00X\x07\x00\x00\x00hundredr\xb5\x01\x00\x00X\x03\x00\x00\x00ourr\xb6\x01\x00\x00X\x06\x00\x00\x00neededr\xb7\x01\x00\x00X\x04\x00\x00\x00liner\xb8\x01\x00\x00X\x08\x00\x00\x00problemsr\xb9\x01\x00\x00X\x07\x00\x00\x00anybodyr\xba\x01\x00\x00X\x04\x00\x00\x00doesr\xbb\x01\x00\x00X\x06\x00\x00\x00poorlyr\xbc\x01\x00\x00X\x03\x00\x00\x00ranr\xbd\x01\x00\x00X\x03\x00\x00\x00mugr\xbe\x01\x00\x00X\x05\x00\x00\x00whiler\xbf\x01\x00\x00X\x05\x00\x00\x00indexr\xc0\x01\x00\x00X\n\x00\x00\x00reasonablyr\xc1\x01\x00\x00X\x04\x00\x00\x00zeror\xc2\x01\x00\x00X\x03\x00\x00\x00endr\xc3\x01\x00\x00X\x01\x00\x00\x00gr\xc4\x01\x00\x00X\x08\x00\x00\x00groupingr\xc5\x01\x00\x00X\x04\x00\x00\x00liker\xc6\x01\x00\x00X\x05\x00\x00\x00stillr\xc7\x01\x00\x00X\x03\x00\x00\x00twor\xc8\x01\x00\x00X\x07\x00\x00\x00youngerr\xc9\x01\x00\x00X\t\x00\x00\x00sometimesr\xca\x01\x00\x00X\x06\x00\x00\x00didn\'tr\xcb\x01\x00\x00X\n\x00\x00\x00themselvesr\xcc\x01\x00\x00X\x04\x00\x00\x00withr\xcd\x01\x00\x00X\x03\x00\x00\x00\'llr\xce\x01\x00\x00X\x04\x00\x00\x00herer\xcf\x01\x00\x00X\x06\x00\x00\x00myselfr\xd0\x01\x00\x00X\x04\x00\x00\x00stopr\xd1\x01\x00\x00X\x07\x00\x00\x00himselfr\xd2\x01\x00\x00X\x04\x00\x00\x00cantr\xd3\x01\x00\x00X\x03\x00\x00\x00yesr\xd4\x01\x00\x00X\x05\x00\x00\x00hencer\xd5\x01\x00\x00X\x03\x00\x00\x00didr\xd6\x01\x00\x00X\x07\x00\x00\x00openingr\xd7\x01\x00\x00X\x02\x00\x00\x00unr\xd8\x01\x00\x00X\x05\x00\x00\x00wherer\xd9\x01\x00\x00X\x02\x00\x00\x00hir\xda\x01\x00\x00X\x04\x00\x00\x00alsor\xdb\x01\x00\x00X\x05\x00\x00\x00meansr\xdc\x01\x00\x00X\x04\x00\x00\x00abler\xdd\x01\x00\x00X\x07\x00\x00\x00secondsr\xde\x01\x00\x00X\x04\x00\x00\x00justr\xdf\x01\x00\x00X\x0b\x00\x00\x00significantr\xe0\x01\x00\x00X\t\x00\x00\x00regardingr\xe1\x01\x00\x00X\x07\x00\x00\x00where\'sr\xe2\x01\x00\x00X\x04\x00\x00\x00volsr\xe3\x01\x00\x00X\x05\x00\x00\x00likedr\xe4\x01\x00\x00X\x02\x00\x00\x00\'tr\xe5\x01\x00\x00X\x01\x00\x00\x00br\xe6\x01\x00\x00X\x05\x00\x00\x00leastr\xe7\x01\x00\x00X\x04\x00\x00\x00goodr\xe8\x01\x00\x00X\x08\x00\x00\x00recentlyr\xe9\x01\x00\x00X\x03\x00\x00\x00mrsr\xea\x01\x00\x00X\x03\x00\x00\x00forr\xeb\x01\x00\x00X\x0c\x00\x00\x00consequentlyr\xec\x01\x00\x00X\x07\x00\x00\x00happensr\xed\x01\x00\x00X\t\x00\x00\x00accordingr\xee\x01\x00\x00X\x07\x00\x00\x00perhapsr\xef\x01\x00\x00X\x02\x00\x00\x00nar\xf0\x01\x00\x00X\x08\x00\x00\x00secondlyr\xf1\x01\x00\x00X\x05\x00\x00\x00takenr\xf2\x01\x00\x00X\x07\x00\x00\x00certainr\xf3\x01\x00\x00X\x05\x00\x00\x00nooner\xf4\x01\x00\x00X\x05\x00\x00\x00backsr\xf5\x01\x00\x00X\x07\x00\x00\x00relatedr\xf6\x01\x00\x00X\x03\x00\x00\x00hidr\xf7\x01\x00\x00X\x07\x00\x00\x00adoptedr\xf8\x01\x00\x00X\x06\x00\x00\x00nearlyr\xf9\x01\x00\x00X\x05\x00\x00\x00afterr\xfa\x01\x00\x00X\x03\x00\x00\x00butr\xfb\x01\x00\x00X\x03\x00\x00\x00ther\xfc\x01\x00\x00X\x06\x00\x00\x00cannotr\xfd\x01\x00\x00X\x02\x00\x00\x00vsr\xfe\x01\x00\x00X\x05\x00\x00\x00placer\xff\x01\x00\x00X\t\x00\x00\x00specifiedr\x00\x02\x00\x00X\x02\x00\x00\x00edr\x01\x02\x00\x00X\x06\x00\x00\x00indeedr\x02\x02\x00\x00X\x07\x00\x00\x00resultsr\x03\x02\x00\x00X\x06\x00\x00\x00reallyr\x04\x02\x00\x00X\x05\x00\x00\x00roomsr\x05\x02\x00\x00X\x05\x00\x00\x00asider\x06\x02\x00\x00X\x02\x00\x00\x00ndr\x07\x02\x00\x00X\x03\x00\x00\x00anyr\x08\x02\x00\x00X\x06\x00\x00\x00takingr\t\x02\x00\x00X\x07\x00\x00\x00whetherr\n\x02\x00\x00X\x07\x00\x00\x00groupedr\x0b\x02\x00\x00X\x06\x00\x00\x00backedr\x0c\x02\x00\x00X\x04\x00\x00\x00namer\r\x02\x00\x00X\x04\x00\x00\x00thisr\x0e\x02\x00\x00X\x04\x00\x00\x00sider\x0f\x02\x00\x00X\x04\x00\x00\x00wantr\x10\x02\x00\x00X\x05\x00\x00\x00aboutr\x11\x02\x00\x00X\n\x00\x00\x00particularr\x12\x02\x00\x00X\x03\x00\x00\x00himr\x13\x02\x00\x00X\x03\x00\x00\x00farr\x14\x02\x00\x00X\x03\x00\x00\x00t\'sr\x15\x02\x00\x00X\x06\x00\x00\x00who\'llr\x16\x02\x00\x00X\x05\x00\x00\x00thoser\x17\x02\x00\x00X\x06\x00\x00\x00eitherr\x18\x02\x00\x00X\x04\x00\x00\x00getsr\x19\x02\x00\x00X\x04\x00\x00\x00beenr\x1a\x02\x00\x00X\x05\x00\x00\x00thinkr\x1b\x02\x00\x00X\x0b\x00\x00\x00consideringr\x1c\x02\x00\x00X\x08\x00\x00\x00followedr\x1d\x02\x00\x00X\t\x00\x00\x00furtheredr\x1e\x02\x00\x00X\x01\x00\x00\x00ar\x1f\x02\x00\x00X\x08\x00\x00\x00whomeverr \x02\x00\x00X\x02\x00\x00\x00itr!\x02\x00\x00X\x04\x00\x00\x00yourr"\x02\x00\x00X\x04\x00\x00\x00fiver#\x02\x00\x00X\x03\x00\x00\x00a\'sr$\x02\x00\x00X\x0c\x00\x00\x00sufficientlyr%\x02\x00\x00X\x06\x00\x00\x00shownsr&\x02\x00\x00X\t\x00\x00\x00presentedr\'\x02\x00\x00X\x05\x00\x00\x00youngr(\x02\x00\x00X\x01\x00\x00\x00fr)\x02\x00\x00X\x07\x00\x00\x00generalr*\x02\x00\x00X\x02\x00\x00\x00asr+\x02\x00\x00X\x07\x00\x00\x00orderedr,\x02\x00\x00X\x03\x00\x00\x00bigr-\x02\x00\x00X\x04\x00\x00\x00whimr.\x02\x00\x00X\x07\x00\x00\x00millionr/\x02\x00\x00X\x02\x00\x00\x00okr0\x02\x00\x00X\t\x00\x00\x00affectingr1\x02\x00\x00X\x05\x00\x00\x00givenr2\x02\x00\x00X\x06\x00\x00\x00here\'sr3\x02\x00\x00X\x08\x00\x00\x00usefullyr4\x02\x00\x00X\x07\x00\x00\x00wherebyr5\x02\x00\x00X\x05\x00\x00\x00partsr6\x02\x00\x00X\x04\x00\x00\x00nearr7\x02\x00\x00X\x06\x00\x00\x00hasn\'tr8\x02\x00\x00X\x03\x00\x00\x00nonr9\x02\x00\x00X\x03\x00\x00\x00yetr:\x02\x00\x00X\x03\x00\x00\x00wasr;\x02\x00\x00X\x05\x00\x00\x00pagesr<\x02\x00\x00X\x05\x00\x00\x00tendsr=\x02\x00\x00X\x06\x00\x00\x00sayingr>\x02\x00\x00X\x03\x00\x00\x00hesr?\x02\x00\x00X\x06\x00\x00\x00they\'dr@\x02\x00\x00X\n\x00\x00\x00interestedrA\x02\x00\x00X\x08\x00\x00\x00promptlyrB\x02\x00\x00X\x06\x00\x00\x00downedrC\x02\x00\x00X\r\x00\x00\x00significantlyrD\x02\x00\x00X\x08\x00\x00\x00entirelyrE\x02\x00\x00X\x05\x00\x00\x00worksrF\x02\x00\x00X\x06\x00\x00\x00nobodyrG\x02\x00\x00X\x05\x00\x00\x00olderrH\x02\x00\x00X\x02\x00\x00\x00idrI\x02\x00\x00X\x08\x00\x00\x00becomingrJ\x02\x00\x00X\x02\x00\x00\x00onrK\x02\x00\x00X\x08\x00\x00\x00thoroughrL\x02\x00\x00X\x06\x00\x00\x00causesrM\x02\x00\x00X\x04\x00\x00\x00keeprN\x02\x00\x00X\x05\x00\x00\x00downsrO\x02\x00\x00X\x06\x00\x00\x00numberrP\x02\x00\x00X\x02\x00\x00\x00\'drQ\x02\x00\x00X\x07\x00\x00\x00becomesrR\x02\x00\x00X\x03\x00\x00\x00sawrS\x02\x00\x00X\n\x00\x00\x00previouslyrT\x02\x00\x00X\x04\x00\x00\x00shedrU\x02\x00\x00X\x07\x00\x00\x00becauserV\x02\x00\x00X\t\x00\x00\x00ourselvesrW\x02\x00\x00X\x08\x00\x00\x00anythingrX\x02\x00\x00X\x03\x00\x00\x00allrY\x02\x00\x00X\n\x00\x00\x00beforehandrZ\x02\x00\x00X\x07\x00\x00\x00besidesr[\x02\x00\x00X\x07\x00\x00\x00problemr\\\x02\x00\x00X\x02\x00\x00\x00ohr]\x02\x00\x00X\x02\x00\x00\x00egr^\x02\x00\x00X\x0b\x00\x00\x00accordinglyr_\x02\x00\x00X\x04\x00\x00\x00saidr`\x02\x00\x00X\x06\x00\x00\x00formerra\x02\x00\x00X\x04\x00\x00\x00thrurb\x02\x00\x00X\x04\x00\x00\x00usedrc\x02\x00\x00X\x06\x00\x00\x00usefulrd\x02\x00\x00X\x03\x00\x00\x00newre\x02\x00\x00X\x06\x00\x00\x00seeingrf\x02\x00\x00X\t\x00\x00\x00shouldn\'trg\x02\x00\x00X\x05\x00\x00\x00greatrh\x02\x00\x00X\x07\x00\x00\x00insteadri\x02\x00\x00X\x05\x00\x00\x00thatsrj\x02\x00\x00X\x07\x00\x00\x00ignoredrk\x02\x00\x00X\x07\x00\x00\x00membersrl\x02\x00\x00X\x06\x00\x00\x00othersrm\x02\x00\x00X\x06\x00\x00\x00becomern\x02\x00\x00X\x02\x00\x00\x00zzro\x02\x00\x00X\x08\x00\x00\x00unlikelyrp\x02\x00\x00X\x07\x00\x00\x00largelyrq\x02\x00\x00X\x06\x00\x00\x00wheresrr\x02\x00\x00X\x02\x00\x00\x00orrs\x02\x00\x00X\x07\x00\x00\x00alreadyrt\x02\x00\x00X\x06\x00\x00\x00alwaysru\x02\x00\x00X\x06\x00\x00\x00havingrv\x02\x00\x00X\x03\x00\x00\x00whorw\x02\x00\x00X\x06\x00\x00\x00latestrx\x02\x00\x00X\x05\x00\x00\x00turnsry\x02\x00\x00X\x04\x00\x00\x00workrz\x02\x00\x00X\x02\x00\x00\x00her{\x02\x00\x00X\x07\x00\x00\x00thereofr|\x02\x00\x00X\x04\x00\x00\x00lookr}\x02\x00\x00X\x06\x00\x00\x00ratherr~\x02\x00\x00X\x02\x00\x00\x00exr\x7f\x02\x00\x00X\x05\x00\x00\x00goodsr\x80\x02\x00\x00X\x0b\x00\x00\x00nonethelessr\x81\x02\x00\x00X\x05\x00\x00\x00clearr\x82\x02\x00\x00X\x06\x00\x00\x00merelyr\x83\x02\x00\x00X\x08\x00\x00\x00somebodyr\x84\x02\x00\x00X\x08\x00\x00\x00pointingr\x85\x02\x00\x00X\x04\x00\x00\x00seemr\x86\x02\x00\x00X\r\x00\x00\x00substantiallyr\x87\x02\x00\x00X\x07\x00\x00\x00overallr\x88\x02\x00\x00X\x01\x00\x00\x00sr\x89\x02\x00\x00X\x06\x00\x00\x00allowsr\x8a\x02\x00\x00X\x05\x00\x00\x00shownr\x8b\x02\x00\x00X\x07\x00\x00\x00willingr\x8c\x02\x00\x00X\n\x00\x00\x00relativelyr\x8d\x02\x00\x00X\x05\x00\x00\x00belowr\x8e\x02\x00\x00X\x07\x00\x00\x00numbersr\x8f\x02\x00\x00X\x02\x00\x00\x00qvr\x90\x02\x00\x00X\x02\x00\x00\x00sor\x91\x02\x00\x00X\x05\x00\x00\x00shallr\x92\x02\x00\x00X\x07\x00\x00\x00specifyr\x93\x02\x00\x00X\x04\x00\x00\x00missr\x94\x02\x00\x00X\x04\x00\x00\x00ontor\x95\x02\x00\x00X\x05\x00\x00\x00innerr\x96\x02\x00\x00X\x03\x00\x00\x00toor\x97\x02\x00\x00X\x06\x00\x00\x00hitherr\x98\x02\x00\x00X\x03\x00\x00\x00user\x99\x02\x00\x00X\n\x00\x00\x00importancer\x9a\x02\x00\x00X\n\x00\x00\x00appreciater\x9b\x02\x00\x00X\t\x00\x00\x00seriouslyr\x9c\x02\x00\x00X\x08\x00\x00\x00normallyr\x9d\x02\x00\x00X\x05\x00\x00\x00larger\x9e\x02\x00\x00X\x06\x00\x00\x00beingsr\x9f\x02\x00\x00X\x08\x00\x00\x00providesr\xa0\x02\x00\x00X\x03\x00\x00\x00sixr\xa1\x02\x00\x00X\x07\x00\x00\x00therebyr\xa2\x02\x00\x00X\x04\x00\x00\x00bothr\xa3\x02\x00\x00X\x04\x00\x00\x00muchr\xa4\x02\x00\x00X\x05\x00\x00\x00orderr\xa5\x02\x00\x00X\t\x00\x00\x00thereforer\xa6\x02\x00\x00X\x03\x00\x00\x00duer\xa7\x02\x00\x00X\x08\x00\x00\x00yourselfr\xa8\x02\x00\x00X\x01\x00\x00\x00ir\xa9\x02\x00\x00X\x01\x00\x00\x00mr\xaa\x02\x00\x00X\x03\x00\x00\x00wayr\xab\x02\x00\x00X\x05\x00\x00\x00firstr\xac\x02\x00\x00X\x07\x00\x00\x00partingr\xad\x02\x00\x00X\x04\x00\x00\x00willr\xae\x02\x00\x00X\x05\x00\x00\x00yearsr\xaf\x02\x00\x00X\x07\x00\x00\x00someoner\xb0\x02\x00\x00X\x04\x00\x00\x00dater\xb1\x02\x00\x00X\x05\x00\x00\x00sevenr\xb2\x02\x00\x00X\x08\x00\x00\x00youngestr\xb3\x02\x00\x00X\x04\x00\x00\x00selfr\xb4\x02\x00\x00X\x04\x00\x00\x00thenr\xb5\x02\x00\x00X\x06\x00\x00\x00aroundr\xb6\x02\x00\x00X\x05\x00\x00\x00worldr\xb7\x02\x00\x00X\x06\x00\x00\x00turnedr\xb8\x02\x00\x00X\x08\x00\x00\x00somethanr\xb9\x02\x00\x00X\x04\x00\x00\x00uponr\xba\x02\x00\x00X\x08\x00\x00\x00sometimer\xbb\x02\x00\x00X\x02\x00\x00\x00rdr\xbc\x02\x00\x00X\x06\x00\x00\x00statesr\xbd\x02\x00\x00X\x06\x00\x00\x00aren\'tr\xbe\x02\x00\x00X\x06\x00\x00\x00inwardr\xbf\x02\x00\x00X\x04\x00\x00\x00onesr\xc0\x02\x00\x00X\x06\x00\x00\x00shouldr\xc1\x02\x00\x00X\x06\x00\x00\x00higherr\xc2\x02\x00\x00X\x06\x00\x00\x00what\'sr\xc3\x02\x00\x00X\x07\x00\x00\x00highestr\xc4\x02\x00\x00X\x08\x00\x00\x00thousandr\xc5\x02\x00\x00X\x04\x00\x00\x00yearr\xc6\x02\x00\x00X\x05\x00\x00\x00ain\'tr\xc7\x02\x00\x00X\x04\x00\x00\x00it\'dr\xc8\x02\x00\x00X\t\x00\x00\x00hopefullyr\xc9\x02\x00\x00X\x05\x00\x00\x00it\'llr\xca\x02\x00\x00X\x02\x00\x00\x00usr\xcb\x02\x00\x00X\x02\x00\x00\x00kgr\xcc\x02\x00\x00X\x08\x00\x00\x00affectedr\xcd\x02\x00\x00X\x04\x00\x00\x00lestr\xce\x02\x00\x00X\x06\x00\x00\x00likelyr\xcf\x02\x00\x00X\t\x00\x00\x00meanwhiler\xd0\x02\x00\x00X\x05\x00\x00\x00areasr\xd1\x02\x00\x00X\x03\x00\x00\x00canr\xd2\x02\x00\x00X\x08\x00\x00\x00possiblyr\xd3\x02\x00\x00X\x03\x00\x00\x00notr\xd4\x02\x00\x00X\x06\x00\x00\x00that\'sr\xd5\x02\x00\x00X\x02\x00\x00\x00imr\xd6\x02\x00\x00X\x07\x00\x00\x00gettingr\xd7\x02\x00\x00X\t\x00\x00\x00followingr\xd8\x02\x00\x00X\x08\x00\x00\x00smallestr\xd9\x02\x00\x00X\x02\x00\x00\x00anr\xda\x02\x00\x00X\x04\x00\x00\x00untor\xdb\x02\x00\x00X\x06\x00\x00\x00duringr\xdc\x02\x00\x00X\x05\x00\x00\x00looksr\xdd\x02\x00\x00X\x05\x00\x00\x00owingr\xde\x02\x00\x00X\x04\x00\x00\x00comer\xdf\x02\x00\x00X\x05\x00\x00\x00whichr\xe0\x02\x00\x00X\x04\x00\x00\x00samer\xe1\x02\x00\x00X\x08\x00\x00\x00containsr\xe2\x02\x00\x00X\x04\x00\x00\x00theyr\xe3\x02\x00\x00X\x03\x00\x00\x00actr\xe4\x02\x00\x00X\x06\x00\x00\x00hereinr\xe5\x02\x00\x00X\x06\x00\x00\x00courser\xe6\x02\x00\x00X\n\x00\x00\x00thoroughlyr\xe7\x02\x00\x00X\x05\x00\x00\x00fullyr\xe8\x02\x00\x00X\x01\x00\x00\x00er\xe9\x02\x00\x00X\x05\x00\x00\x00laterr\xea\x02\x00\x00X\x04\x00\x00\x00soonr\xeb\x02\x00\x00X\x07\x00\x00\x00thererer\xec\x02\x00\x00X\x05\x00\x00\x00pointr\xed\x02\x00\x00X\x03\x00\x00\x00secr\xee\x02\x00\x00X\t\x00\x00\x00somewherer\xef\x02\x00\x00X\x06\x00\x00\x00whencer\xf0\x02\x00\x00X\x03\x00\x00\x00c\'sr\xf1\x02\x00\x00X\x06\x00\x00\x00groupsr\xf2\x02\x00\x00X\x07\x00\x00\x00whereasr\xf3\x02\x00\x00X\x02\x00\x00\x00ofr\xf4\x02\x00\x00X\x04\x00\x00\x00partr\xf5\x02\x00\x00X\n\x00\x00\x00definitelyr\xf6\x02\x00\x00X\x06\x00\x00\x00secondr\xf7\x02\x00\x00X\t\x00\x00\x00inventionr\xf8\x02\x00\x00X\x06\x00\x00\x00you\'ver\xf9\x02\x00\x00X\x06\x00\x00\x00thoughr\xfa\x02\x00\x00X\x04\x00\x00\x00whosr\xfb\x02\x00\x00X\x04\x00\x00\x00shesr\xfc\x02\x00\x00X\x07\x00\x00\x00they\'ver\xfd\x02\x00\x00X\x03\x00\x00\x00howr\xfe\x02\x00\x00X\x03\x00\x00\x00ownr\xff\x02\x00\x00X\n\x00\x00\x00presentingr\x00\x03\x00\x00X\x0c\x00\x00\x00successfullyr\x01\x03\x00\x00X\x06\x00\x00\x00hadn\'tr\x02\x03\x00\x00X\x02\x00\x00\x00etr\x03\x03\x00\x00X\x04\x00\x00\x00mustr\x04\x03\x00\x00X\x07\x00\x00\x00towardsr\x05\x03\x00\x00X\x04\x00\x00\x00feltr\x06\x03\x00\x00X\x03\x00\x00\x00arer\x07\x03\x00\x00X\x03\x00\x00\x00incr\x08\x03\x00\x00X\x05\x00\x00\x00facesr\t\x03\x00\x00X\x04\x00\x00\x00gaver\n\x03\x00\x00X\x03\x00\x00\x00askr\x0b\x03\x00\x00X\x08\x00\x00\x00describer\x0c\x03\x00\x00X\x06\x00\x00\x00differr\r\x03\x00\x00X\x07\x00\x00\x00severalr\x0e\x03\x00\x00X\x02\x00\x00\x00upr\x0f\x03\x00\x00X\x07\x00\x00\x00pointedr\x10\x03\x00\x00X\x04\x00\x00\x00knewr\x11\x03\x00\x00X\x05\x00\x00\x00newerr\x12\x03\x00\x00X\x06\x00\x00\x00wantedr\x13\x03\x00\x00X\x07\x00\x00\x00whitherr\x14\x03\x00\x00X\x07\x00\x00\x00amongstr\x15\x03\x00\x00X\x04\x00\x00\x00roomr\x16\x03\x00\x00X\x01\x00\x00\x00ur\x17\x03\x00\x00X\x07\x00\x00\x00howbeitr\x18\x03\x00\x00X\x04\x00\x00\x00tookr\x19\x03\x00\x00X\x04\x00\x00\x00wishr\x1a\x03\x00\x00X\t\x00\x00\x00everybodyr\x1b\x03\x00\x00X\x05\x00\x00\x00forthr\x1c\x03\x00\x00X\x02\x00\x00\x00ZTr\x1d\x03\x00\x00X\x08\x00\x00\x00probablyr\x1e\x03\x00\x00X\x05\x00\x00\x00don\'tr\x1f\x03\x00\x00X\x06\x00\x00\x00littler \x03\x00\x00X\x02\x00\x00\x00tsr!\x03\x00\x00X\x06\x00\x00\x00behindr"\x03\x00\x00X\x06\x00\x00\x00ninetyr#\x03\x00\x00X\n\x00\x00\x00accordancer$\x03\x00\x00X\x05\x00\x00\x00fifthr%\x03\x00\x00X\x05\x00\x00\x00neverr&\x03\x00\x00X\x01\x00\x00\x00zr\'\x03\x00\x00X\x03\x00\x00\x00fewr(\x03\x00\x00X\x08\x00\x00\x00latterlyr)\x03\x00\x00X\x06\x00\x00\x00theyrer*\x03\x00\x00X\x04\x00\x00\x00goesr+\x03\x00\x00X\x08\x00\x00\x00sensibler,\x03\x00\x00X\n\x00\x00\x00especiallyr-\x03\x00\x00X\x05\x00\x00\x00stater.\x03\x00\x00X\t\x00\x00\x00immediater/\x03\x00\x00X\x03\x00\x00\x00tipr0\x03\x00\x00X\x03\x00\x00\x00wwwr1\x03\x00\x00X\x06\x00\x00\x00itselfr2\x03\x00\x00X\x02\x00\x00\x00\'mr3\x03\x00\x00X\x03\x00\x00\x00wedr4\x03\x00\x00X\x02\x00\x00\x00ztr5\x03\x00\x00X\x07\x00\x00\x00that\'ver6\x03\x00\x00X\x05\x00\x00\x00we\'ver7\x03\x00\x00X\x01\x00\x00\x00xr8\x03\x00\x00X\x06\x00\x00\x00anywayr9\x03\x00\x00X\n\x00\x00\x00associatedr:\x03\x00\x00X\x04\x00\x00\x00tellr;\x03\x00\x00X\x05\x00\x00\x00isn\'tr<\x03\x00\x00X\x03\x00\x00\x00perr=\x03\x00\x00X\x01\x00\x00\x00vr>\x03\x00\x00X\x08\x00\x00\x00interestr?\x03\x00\x00X\x02\x00\x00\x00atr@\x03\x00\x00X\x08\x00\x00\x00considerrA\x03\x00\x00X\x04\x00\x00\x00thourB\x03\x00\x00X\x05\x00\x00\x00wholerC\x03\x00\x00X\x05\x00\x00\x00thererD\x03\x00\x00X\x04\x00\x00\x00needrE\x03\x00\x00X\x06\x00\x00\x00acrossrF\x03\x00\x00X\x0c\x00\x00\x00respectivelyrG\x03\x00\x00X\x07\x00\x00\x00changesrH\x03\x00\x00X\x05\x00\x00\x00sorryrI\x03\x00\x00X\x03\x00\x00\x00hadrJ\x03\x00\x00X\x04\x00\x00\x00putsrK\x03\x00\x00X\x05\x00\x00\x00usingrL\x03\x00\x00X\x03\x00\x00\x00menrM\x03\x00\x00X\x08\x00\x00\x00thoughtsrN\x03\x00\x00X\x02\x00\x00\x00gorO\x03\x00\x00X\x04\x00\x00\x00thatrP\x03\x00\x00X\n\x00\x00\x00thereafterrQ\x03\x00\x00X\x06\x00\x00\x00openedrR\x03\x00\x00X\x06\x00\x00\x00enoughrS\x03\x00\x00X\x05\x00\x00\x00threerT\x03\x00\x00X\x04\x00\x00\x00fourrU\x03\x00\x00X\x04\x00\x00\x00themrV\x03\x00\x00X\x0b\x00\x00\x00interestingrW\x03\x00\x00X\x04\x00\x00\x00refsrX\x03\x00\x00X\x06\x00\x00\x00appearrY\x03\x00\x00X\x07\x00\x00\x00needingrZ\x03\x00\x00X\x04\x00\x00\x00noner[\x03\x00\x00X\x04\x00\x00\x00oursr\\\x03\x00\x00X\x04\x00\x00\x00oncer]\x03\x00\x00X\x03\x00\x00\x00supr^\x03\x00\x00X\x08\x00\x00\x00stronglyr_\x03\x00\x00X\t\x00\x00\x00whereuponr`\x03\x00\x00X\x0b\x00\x00\x00immediatelyra\x03\x00\x00X\x07\x00\x00\x00clearlyrb\x03\x00\x00X\x04\x00\x00\x00okayrc\x03\x00\x00X\x05\x00\x00\x00can\'trd\x03\x00\x00X\x03\x00\x00\x00i\'dre\x03\x00\x00X\x08\x00\x00\x00meantimerf\x03\x00\x00X\x07\x00\x00\x00backingrg\x03\x00\x00X\x08\x00\x00\x00couldn\'trh\x03\x00\x00X\x04\x00\x00\x00sureri\x03\x00\x00X\x05\x00\x00\x00thingrj\x03\x00\x00X\x05\x00\x00\x00yoursrk\x03\x00\x00X\x02\x00\x00\x00thrl\x03\x00\x00X\x05\x00\x00\x00briefrm\x03\x00\x00X\x08\x00\x00\x00althoughrn\x03\x00\x00X\x08\x00\x00\x00hereuponro\x03\x00\x00X\x05\x00\x00\x00wellsrp\x03\x00\x00X\x05\x00\x00\x00alonerq\x03\x00\x00X\x03\x00\x00\x00sherr\x03\x00\x00X\x04\x00\x00\x00highrs\x03\x00\x00X\x07\x00\x00\x00quicklyrt\x03\x00\x00X\x05\x00\x00\x00beingru\x03\x00\x00X\x07\x00\x00\x00containrv\x03\x00\x00X\x05\x00\x00\x00askedrw\x03\x00\x00X\x05\x00\x00\x00keepsrx\x03\x00\x00X\x06\x00\x00\x00makingry\x03\x00\x00X\x05\x00\x00\x00givesrz\x03\x00\x00X\x04\x00\x00\x00asksr{\x03\x00\x00X\x04\x00\x00\x00evenr|\x03\x00\x00e\x85r}\x03\x00\x00Rr~\x03\x00\x00.'
cn_stopwords = pickle.loads(cn)
en_stopwords = pickle.loads(en)