"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'AWS';
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 */
class EncryptionKeyAlias extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.alias.startsWith(REQUIRED_ALIAS_PREFIX)) {
            throw new Error(`Alias must start with the prefix "${REQUIRED_ALIAS_PREFIX}": ${props.alias}`);
        }
        if (props.alias === REQUIRED_ALIAS_PREFIX) {
            throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${props.alias}`);
        }
        if (props.alias.startsWith(DISALLOWED_PREFIX)) {
            throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${props.alias}`);
        }
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: props.alias,
            targetKeyId: props.key.keyArn
        });
        this.aliasName = resource.aliasName;
    }
}
exports.EncryptionKeyAlias = EncryptionKeyAlias;
//# sourceMappingURL=data:application/json;base64,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