"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'default key'(test) {
        const stack = new cdk_1.Stack();
        new lib_1.Key(stack, 'MyKey');
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'default with no retention'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'TestStack');
        new lib_1.Key(stack, 'MyKey', { retain: false });
        assert_1.expect(app.synthesizeStack(stack.name)).to(assert_1.haveResource('AWS::KMS::Key', { DeletionPolicy: "Delete" }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default with some permission'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey');
        const p = new aws_iam_1.PolicyStatement().addAllResources().addAction('kms:encrypt');
        p.addAwsPrincipal('arn');
        key.addToResourcePolicy(p);
        assert_1.expect(app.synthesizeStack(stack.name)).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'key with some options'(test) {
        const stack = new cdk_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const p = new aws_iam_1.PolicyStatement().addAllResources().addAction('kms:encrypt');
        p.addAwsPrincipal('arn');
        key.addToResourcePolicy(p);
        key.node.apply(new cdk_1.Tag('tag1', 'value1'));
        key.node.apply(new cdk_1.Tag('tag2', 'value2'));
        key.node.apply(new cdk_1.Tag('tag3', ''));
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        Enabled: false,
                        EnableKeyRotation: true,
                        Tags: [
                            {
                                Key: "tag1",
                                Value: "value1"
                            },
                            {
                                Key: "tag2",
                                Value: "value2"
                            },
                            {
                                Key: "tag3",
                                Value: ""
                            }
                        ]
                    },
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'addAlias creates an alias'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        const alias = key.addAlias('alias/xoo');
        test.ok(alias.aliasName);
        test.deepEqual(app.synthesizeStack(stack.name).template, {
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        EnableKeyRotation: true,
                        Enabled: false,
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain"
                },
                MyKeyAlias1B45D9DA: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/xoo",
                        TargetKeyId: {
                            "Fn::GetAtt": [
                                "MyKey6AB29FA6",
                                "Arn"
                            ]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'grant decrypt on a key'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const key = new lib_1.Key(stack, 'Key');
        const user = new aws_iam_1.User(stack, 'User');
        // WHEN
        key.grantDecrypt(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    // This one is there by default
                    {
                        // tslint:disable-next-line:max-line-length
                        Action: ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*", "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion"],
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::", { Ref: "AWS::AccountId" }, ":root"]] } },
                        Resource: "*"
                    },
                    // This is the interesting one
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::GetAtt": ["User00B015A1", "Arn"] } },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Key961B73FD", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'import/export can be used to bring in an existing key'(test) {
        const stack2 = new cdk_1.Stack();
        const myKeyImported = lib_1.Key.fromKeyArn(stack2, 'MyKeyImported', 'arn:of:key');
        // addAlias can be called on imported keys.
        myKeyImported.addAlias('alias/hello');
        assert_1.expect(stack2).toMatch({
            Resources: {
                MyKeyImportedAliasB1C5269F: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/hello",
                        TargetKeyId: 'arn:of:key'
                    }
                }
            }
        });
        test.done();
    },
    'addToResourcePolicy allowNoOp and there is no policy': {
        'succeed if set to true (default)'(test) {
            const stack = new cdk_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            key.addToResourcePolicy(new aws_iam_1.PolicyStatement().addAllResources().addAction('*'));
            test.done();
        },
        'fails if set to false'(test) {
            const stack = new cdk_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            test.throws(() => key.addToResourcePolicy(new aws_iam_1.PolicyStatement().addAllResources().addAction('*'), /* allowNoOp */ false), 'Unable to add statement to IAM resource policy for KMS key: "foo/bar"');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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