# ocean_crow_mizukage/mizukage.py (Version 0.2.1)
import click
import json
import os
import subprocess
import sys
from pathlib import Path
import shutil
import platform
import importlib.metadata
import asyncio
from rich.console import Console
from rich.panel import Panel
from rich.prompt import Prompt
from rich.progress import track, Progress
from rich.traceback import install as install_rich_traceback
from time import sleep, time

# --- Set event loop policy for Windows ---
if platform.system() == "Windows":
    asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

# --- Rich Traceback for Better Debugging ---
install_rich_traceback()

# --- Dependency Checks ---
try:
    subprocess.run(["python", "-m", "venv", "--help"], capture_output=True)
    import pytest
    import flask
    import pandas
    import numpy
except (importlib.metadata.PackageNotFoundError, subprocess.CalledProcessError, ImportError) as e:
    console = Console()
    console.print(f"[red]Error: {str(e)}. Install missing tools (e.g., 'pip install pytest flask pandas numpy') before proceeding.[/red]")
    exit(1)

console = Console()
VERSION = "0.2.1"

# --- Signature ---
CREATOR = "Created by Sheldon Kenny Salmon of OceanCrow with assistance from Grok, built by xAI"

# --- Visual Helpers ---
def show_wave_animation():
    waves = ["~ ", " ~ ", " ~"]
    for _ in range(2):
        for wave in waves:
            print(f"\r{wave} OceanCrow_Mizukage Loading...", end="")
            sleep(0.2)
    print("\r" + " " * 40, end="\r")

def show_progress(steps=5, delay=0.2):
    try:
        console.print("🚀 Initializing...", style="white bold")
        for i in track(range(steps), description="Progress..."):
            sleep(delay)
        console.print(" " * (steps + 2), end="\r")
    except Exception as e:
        console.print(f"[red]Error in progress display: {str(e)}[/red]")

def show_spinner(task_name):
    try:
        console.print(f"⏳ {task_name}...", style="white")
        spinners = ["|", "/", "-", "\\"]
        for _ in range(4):
            for spinner in spinners:
                print(f"\r{spinner} Working...", end="")
                sleep(0.1)
        print("\r" + " " * 20, end="\r")
    except Exception as e:
        console.print(f"[red]Error in spinner: {str(e)}[/red]")

def show_success_banner(project_name, success=True):
    try:
        border = "═" * (len(project_name) + 20)
        console.print(f"╔{border}╗", style="white bold")
        if success:
            console.print(f"║ 🎉 Project {project_name} is ready! Happy coding! 💻 ║", style="green bold")
            console.print(f"║ Tip: Optimize I/O for faster runs! - {CREATOR} ║", style="yellow")
        else:
            console.print(f"║ ⚠️ Project {project_name} creation failed. Check logs! ║", style="red bold")
        console.print(f"╚{border}╝", style="white bold")
    except Exception as e:
        console.print(f"[red]Error in success banner: {str(e)}[/red]")

# --- Config Handling ---
def generate_key():
    return b"" # Placeholder, no encryption needed

def load_or_create_config():
    config_path = Path.home() / ".ocean_crow_config.json"
    try:
        config_path.parent.mkdir(parents=True, exist_ok=True)
        backup_path = Path.home() / f".ocean_crow_config_backup_{int(time())}.json"
        if config_path.exists():
            shutil.copy(config_path, backup_path)
            with open(config_path, 'r', encoding='utf-8') as f:
                config = json.load(f)
            # Migrate old base_path to MizuKage World if it’s still "desktop"
            if config.get("base_path", "").endswith("desktop"):
                console.print("[yellow]Migrating base path from 'desktop' to 'MizuKage World'...[/yellow]")
                config["base_path"] = str(Path.home() / "MizuKage World")
                save_config(config)
            return config
        default_config = {
            "name": "", "type": "basic", "env": {}, "tasks": [], "features": {},
            "key": "", "projects_created": 0, "base_path": str(Path.home() / "MizuKage World"),
            "version": VERSION, "display_mode": "progress"
        }
        with open(config_path, 'w', encoding='utf-8') as f:
            json.dump(default_config, f, indent=4)
        return default_config
    except PermissionError:
        console.print("[red]Permission denied for global config. Using local config.json[/red]")
        return {"name": "", "type": "basic", "env": {}, "tasks": [], "features": {}, "key": "", "projects_created": 0, "base_path": str(Path.cwd()), "version": VERSION, "display_mode": "progress"}
    except Exception as e:
        console.print(f"[red]Error loading config: {str(e)}. Using default config.[/red]")
        return {"name": "", "type": "basic", "env": {}, "tasks": [], "features": {}, "key": "", "projects_created": 0, "base_path": str(Path.cwd()), "version": VERSION, "display_mode": "progress"}

def save_config(config):
    config_path = Path.home() / ".ocean_crow_config.json"
    try:
        json.dumps(config)
        with open(config_path, 'w', encoding='utf-8') as f:
            json.dump(config, f, indent=4)
    except json.JSONDecodeError as e:
        console.print(f"[red]Config corruption: {str(e)}. Reverting to backup.[/red]")
        backup = sorted((Path.home() / ".ocean_crow_config_backup_*").glob("*"), reverse=True)
        if backup:
            shutil.copy(backup[0], config_path)
            console.print("[yellow]Reverted to latest backup.[/yellow]")
        else:
            console.print("[red]No backup available. Config lost.[/red]")
    except PermissionError:
        console.print("[red]Permission denied saving config. Using in-memory config.[/red]")
    except Exception as e:
        console.print(f"[red]Error saving config: {str(e)}. Using in-memory config.[/red]")

# --- Project Features ---
async def create_project(config, progress=None):
    console.print(f"[yellow]Starting create_project for {config['name']}[/yellow]")
    if config["features"].get("scaffold", "n") == "y":
        console.print(f"[green]Creating {config['name']} (type: {config['type']})[/green]")
        project_dir = Path(config["name"].replace(" ", "_"))
        base_path = Path(config["base_path"]).resolve()
        full_path = base_path / project_dir
        console.print(f"[cyan]Creating project at: {full_path}[/cyan]")
        try:
            if full_path.exists():
                console.print(f"[yellow]⚠️ {full_path} exists. Overwrite? [y/N][/yellow]")
                if Prompt.ask("", default="n") not in ['y', 'yes']:
                    console.print("[yellow]Skipping creation due to overwrite refusal.[/yellow]")
                    return
                shutil.rmtree(full_path, ignore_errors=True)
            if progress:
                task1 = progress.add_task("[green]Creating directories...", total=3)
                full_path.mkdir(parents=True, exist_ok=True)
                for folder in ["src", "tests", "docs"]:
                    (full_path / folder).mkdir(exist_ok=True)
                    progress.update(task1, advance=1)
                task2 = progress.add_task("[green]Writing files...", total=1)
                template_files = {
                    'basic': ('main.py', f"# {config['name']} - Main Script\n# {CREATOR}\ndef main():\n print('Hello from {config['name']}!')\nif __name__ == '__main__':\n main()"),
                    'flask': ('app.py', f"# {config['name']} - Flask App\n# {CREATOR}\nfrom flask import Flask\napp = Flask(__name__)\n@app.route('/')\ndef hello():\n return 'Hello from {config['name']}!'\nif __name__ == '__main__':\n app.run(debug=True)"),
                    'datascience': ('notebook.py', f"# {config['name']} - Data Notebook\n# {CREATOR}\nimport pandas as pd\nimport numpy as np\ndata = pd.DataFrame({{'col1': np.random.randn(100)}})\nprint(data.describe())")
                }
                template_file, content = template_files.get(config["type"].lower(), template_files['basic'])
                (full_path / "src" / template_file).write_text(content, encoding='utf-8')
                progress.update(task2, advance=1)
            else:
                full_path.mkdir(parents=True, exist_ok=True)
                for folder in ["src", "tests", "docs"]:
                    (full_path / folder).mkdir(exist_ok=True)
                template_files = {
                    'basic': ('main.py', f"# {config['name']} - Main Script\n# {CREATOR}\ndef main():\n print('Hello from {config['name']}!')\nif __name__ == '__main__':\n main()"),
                    'flask': ('app.py', f"# {config['name']} - Flask App\n# {CREATOR}\nfrom flask import Flask\napp = Flask(__name__)\n@app.route('/')\ndef hello():\n return 'Hello from {config['name']}!'\nif __name__ == '__main__':\n app.run(debug=True)"),
                    'datascience': ('notebook.py', f"# {config['name']} - Data Notebook\n# {CREATOR}\nimport pandas as pd\nimport numpy as np\ndata = pd.DataFrame({{'col1': np.random.randn(100)}})\nprint(data.describe())")
                }
                template_file, content = template_files.get(config["type"].lower(), template_files['basic'])
                (full_path / "src" / template_file).write_text(content, encoding='utf-8')
            create_start_project_script(full_path, config["name"], config["type"])
            create_gitignore(full_path)
            create_commands_txt(full_path, config["name"], config["type"])
            create_docs(full_path, config["name"], config["type"])
            config["projects_created"] += 1
        except PermissionError:
            console.print(f"[red]Permission denied creating {full_path}. Check folder permissions.[/red]")
        except Exception as e:
            console.print(f"[red]Error creating project: {str(e)}[/red]")
            log_error(base_path, f"Error in create_project - {str(e)}")
    console.print(f"[green]Completed create_project for {config['name']}[/green]")

async def update_project(config, progress=None):
    console.print(f"[yellow]Starting update_project for {config['name']}[/yellow]")
    if config["features"].get("scaffold", "n") == "y":
        project_dir = Path(config["name"].replace(" ", "_"))
        base_path = Path(config["base_path"]).resolve()
        full_path = base_path / project_dir
        try:
            if not full_path.exists():
                console.print(f"[red]❌ {full_path} does not exist.[/red]")
                return
            console.print(f"[green]Updating {config['name']}...[/green]")
            if progress:
                task = progress.add_task("[green]Updating directories...", total=3)
                for folder in ["src", "tests", "docs"]:
                    (full_path / folder).mkdir(exist_ok=True)
                    progress.update(task, advance=1)
            else:
                for folder in ["src", "tests", "docs"]:
                    (full_path / folder).mkdir(exist_ok=True)
            create_docs(full_path, config["name"], config["type"])
            create_gitignore(full_path)
            create_commands_txt(full_path, config["name"], config["type"])
        except PermissionError:
            console.print(f"[red]Permission denied updating {full_path}. Check folder permissions.[/red]")
        except Exception as e:
            console.print(f"[red]Error updating project: {str(e)}[/red]")
            log_error(base_path, f"Error in update_project - {str(e)}")
    console.print(f"[green]Completed update_project for {config['name']}[/green]")

async def configure_env(config, progress=None):
    console.print(f"[yellow]Starting configure_env for {config['name']}[/yellow]")
    if config["features"].get("autoenv", "n") == "y":
        console.print(f"[blue]Configuring env for {config['name']}[/blue]")
        project_dir = Path(config["name"].replace(" ", "_"))
        base_path = Path(config["base_path"]).resolve()
        env_dir = base_path / project_dir / "env"
        try:
            if env_dir.exists():
                console.print(f"[yellow]⚠️ Env for {config['name']} exists. Overwrite? [y/N][/yellow]")
                if Prompt.ask("", default="n") not in ['y', 'yes']:
                    console.print("[yellow]Skipping env configuration due to overwrite refusal.[/yellow]")
                    return
                shutil.rmtree(env_dir, ignore_errors=True)
            if progress:
                task = progress.add_task("[green]Setting up virtual env...", total=1)
                await asyncio.to_thread(run_command, ["python", "-m", "venv", str(env_dir)])
                progress.update(task, advance=1)
            else:
                await asyncio.to_thread(run_command, ["python", "-m", "venv", str(env_dir)])
            config["env"] = {"path": str(env_dir)}
        except subprocess.CalledProcessError as e:
            console.print(f"[red]Error creating virtual env: {str(e)}. Ensure Python is installed.[/red]")
            log_error(base_path, f"Error in configure_env - {str(e)}")
        except PermissionError:
            console.print(f"[red]Permission denied creating env at {env_dir}. Check permissions.[/red]")
        except Exception as e:
            console.print(f"[red]Error configuring env: {str(e)}[/red]")
            log_error(base_path, f"Error in configure_env - {str(e)}")
    console.print(f"[green]Completed configure_env for {config['name']}[/green]")

def automate_workflow(config, progress=None):
    console.print(f"[yellow]Starting automate_workflow for {config['name']}[/yellow]")
    if config["features"].get("codeflow", "n") == "y":
        console.print(f"[yellow]Automating workflow for {config['name']}[/yellow]")
        try:
            if progress:
                task = progress.add_task("[green]Checking pytest...", total=1)
                subprocess.run(["pytest", "--version"], check=True, capture_output=True)
                progress.update(task, advance=1)
            else:
                subprocess.run(["pytest", "--version"], check=True, capture_output=True)
            config["tasks"].append("tests_run")
        except subprocess.CalledProcessError:
            console.print("[red]Warning: pytest not found. Install with 'pip install pytest' for codeflow.[/red]")
        except Exception as e:
            console.print(f"[red]Error in automate_workflow: {str(e)}[/red]")
            log_error(Path(config["base_path"]).resolve(), f"Error in automate_workflow - {str(e)}")
    console.print(f"[green]Completed automate_workflow for {config['name']}[/green]")

def process_data(config, progress=None):
    console.print(f"[yellow]Starting process_data for {config['name']}[/yellow]")
    if config["features"].get("databridge", "n") == "y":
        console.print(f"[cyan]Processing data for {config['name']}[/cyan]")
        try:
            if progress:
                task = progress.add_task("[green]Generating dummy data...", total=1)
                (Path.cwd() / "dummy.csv").write_text("data,1\n", encoding='utf-8')
                progress.update(task, advance=1)
            else:
                (Path.cwd() / "dummy.csv").write_text("data,1\n", encoding='utf-8')
            config["tasks"].append("data_processed")
        except PermissionError:
            console.print("[red]Permission denied writing dummy data.[/red]")
        except Exception as e:
            console.print(f"[red]Error in process_data: {str(e)}[/red]")
            log_error(Path(config["base_path"]).resolve(), f"Error in process_data - {str(e)}")
    console.print(f"[green]Completed process_data for {config['name']}[/green]")

def map_project(config, progress=None):
    console.print(f"[yellow]Starting map_project for {config['name']}[/yellow]")
    if config["features"].get("mindmap", "n") == "y":
        console.print(f"[magenta]Mapping {config['name']}[/magenta]")
        try:
            base_path = Path(config["base_path"]).resolve()
            if progress:
                task = progress.add_task("[green]Creating diagram...", total=1)
                (base_path / Path(config["name"].replace(" ", "_")) / "diagram.txt").write_text("UML-like structure - " + CREATOR, encoding='utf-8')
                progress.update(task, advance=1)
            else:
                (base_path / Path(config["name"].replace(" ", "_")) / "diagram.txt").write_text("UML-like structure - " + CREATOR, encoding='utf-8')
        except PermissionError:
            console.print("[red]Permission denied creating diagram.[/red]")
        except Exception as e:
            console.print(f"[red]Error in map_project: {str(e)}[/red]")
            log_error(base_path, f"Error in map_project - {str(e)}")
    console.print(f"[green]Completed map_project for {config['name']}[/green]")

def optimize_code(config, progress=None):
    console.print(f"[yellow]Starting optimize_code for {config['name']}[/yellow]")
    if config["features"].get("quantum", "n") == "y":
        console.print(f"[red]Optimizing {config['name']} with quantum boost[/red]")
        try:
            base_path = Path(config["base_path"]).resolve()
            if progress:
                task = progress.add_task("[green]Optimizing code...", total=1)
                (base_path / Path(config["name"].replace(" ", "_")) / "optimized.py").write_text("def optimized(): pass # Quantum-inspired", encoding='utf-8')
                progress.update(task, advance=1)
            else:
                (base_path / Path(config["name"].replace(" ", "_")) / "optimized.py").write_text("def optimized(): pass # Quantum-inspired", encoding='utf-8')
            config["tasks"].append("optimized")
        except PermissionError:
            console.print("[red]Permission denied optimizing code.[/red]")
        except Exception as e:
            console.print(f"[red]Error in optimize_code: {str(e)}[/red]")
            log_error(base_path, f"Error in optimize_code - {str(e)}")
    console.print(f"[green]Completed optimize_code for {config['name']}[/green]")

def run_all(config, start_time):
    console.print(f"[yellow]Starting run_all for {config['name']}[/yellow]")
    if config["features"].get("orchestrate", "n") == "y":
        console.print(f"[white]Orchestrating {config['name']} - {CREATOR}[/white]")
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        try:
            if config["display_mode"] == "progress":
                with Progress(console=console) as progress:
                    task = progress.add_task("[cyan]Running tasks...", total=len([create_project, configure_env, update_project, automate_workflow, process_data, map_project, optimize_code]))
                    for func in [create_project, configure_env, update_project, automate_workflow, process_data, map_project, optimize_code]:
                        console.print(f"[cyan]Executing {func.__name__}[/cyan]")
                        try:
                            if func in [create_project, configure_env, update_project]:
                                loop.run_until_complete(func(config, progress))
                            else:
                                func(config, progress)
                        except Exception as e:
                            console.print(f"[red]Error in {func.__name__}: {str(e)}[/red]")
                            log_error(Path(config["base_path"]).resolve(), f"Error in {func.__name__} - {str(e)}")
                            continue
                        console.print(f"[cyan]Finished {func.__name__}[/cyan]")
                        progress.update(task, advance=1)
            else: # silent mode
                for func in [create_project, configure_env, update_project, automate_workflow, process_data, map_project, optimize_code]:
                    console.print(f"[cyan]Executing {func.__name__}[/cyan]")
                    try:
                        if func in [create_project, configure_env, update_project]:
                            loop.run_until_complete(func(config))
                        else:
                            func(config)
                    except Exception as e:
                        console.print(f"[red]Error in {func.__name__}: {str(e)}[/red]")
                        log_error(Path(config["base_path"]).resolve(), f"Error in {func.__name__} - {str(e)}")
                        continue
                    console.print(f"[cyan]Finished {func.__name__}[/cyan]")
        except Exception as e:
            console.print(f"[red]Critical error in run_all: {str(e)}[/red]")
            log_error(Path(config["base_path"]).resolve(), f"Critical error in run_all - {str(e)}")
        finally:
            loop.close()
    console.print(f"[gray]Time for orchestration: {time() - start_time:.2f}s[/gray]")
    console.print(f"[green]Completed run_all for {config['name']}[/green]")

# --- Utility Functions ---
def run_command(cmd):
    console.print(f"[yellow]Starting run_command: {cmd}[/yellow]")
    try:
        shell_flag = platform.system() == "Windows"
        result = subprocess.run(cmd, shell=shell_flag, check=True, capture_output=True, text=True, encoding='utf-8')
        console.print(f"[green]Completed run_command: {cmd}[/green]")
        return result
    except subprocess.CalledProcessError as e:
        console.print(f"[red]Oops! {cmd[0]} failed: {e.stderr}[/red]")
        if Prompt.ask("[green]Yes[/green]/[red]No[/red] - More help?", default="n") == "y":
            console.print("[cyan]Fix: Run 'pip install {cmd[0]}', retry, or check PATH.[/cyan]")
        raise
    except Exception as e:
        console.print(f"[red]Unexpected error in run_command: {str(e)}[/red]")
        raise

def create_start_project_script(project_path, project_name, project_type):
    try:
        start_file = 'start_project.bat' if platform.system() == 'Windows' else 'start_project.sh'
        main_file = 'app.py' if project_type.lower() == 'flask' else 'notebook.py' if project_type.lower() == 'datascience' else 'main.py'
        script_content = (
            f"@echo off\ncd /d {project_path}\ncall env\\Scripts\\activate\n{platform.python_version()} src\\{main_file}\ncmd /k"
            if platform.system() == 'Windows' else
            f"#!/bin/bash\ncd {project_path}\nsource env/bin/activate\n{platform.python_version()} src/{main_file}\nbash"
        )
        (project_path / start_file).write_text(script_content, encoding='utf-8')
        if platform.system() != 'Windows':
            os.chmod(project_path / start_file, 0o755)
    except PermissionError:
        console.print(f"[red]Permission denied creating {start_file}.[/red]")
    except Exception as e:
        console.print(f"[red]Error creating start script: {str(e)}[/red]")

def create_gitignore(project_path):
    try:
        gitignore_content = (
            "# OceanCrow_Mizukage .gitignore\n"
            "env/\nvenv/\n__pycache__/\n*.pyc\n.env\n*.log\n*.db\n"
        )
        (project_path / '.gitignore').write_text(gitignore_content, encoding='utf-8')
    except PermissionError:
        console.print("[red]Permission denied creating .gitignore.[/red]")
    except Exception as e:
        console.print(f"[red]Error creating .gitignore: {str(e)}[/red]")

def create_commands_txt(project_path, project_name, project_type):
    try:
        python_cmd = platform.python_version()
        activate_cmd = f"{project_path}\\env\\Scripts\\activate" if platform.system() == 'Windows' else f"source {project_path}/env/bin/activate"
        start_script = f"{project_path}/start_project.{'bat' if platform.system() == 'Windows' else 'sh'}"
        main_file = 'app.py' if project_type.lower() == 'flask' else 'notebook.py' if project_type.lower() == 'datascience' else 'main.py'
        (project_path / 'commands.txt').write_text(
            f"# OceanCrow_Mizukage Commands for {project_name}\n# {CREATOR}\n## 🚀 Start\n{start_script}\nOR\n{activate_cmd}\n## 🛠️ Run\n{python_cmd} src/{main_file}\n",
            encoding='utf-8'
        )
    except PermissionError:
        console.print("[red]Permission denied creating commands.txt.[/red]")
    except Exception as e:
        console.print(f"[red]Error creating commands.txt: {str(e)}[/red]")

def create_docs(project_path, project_name, project_type):
    try:
        docs_dir = project_path / 'docs'
        docs_dir.mkdir(exist_ok=True)
        readme_path = docs_dir / 'README.md'
        contrib_path = docs_dir / 'contribution.md'
        readme_path.write_text(
            f"# {project_name}\n# {CREATOR}\n## Installation\npip install ocean_crow_mizukage\n## Usage\nmizukage start\n## 🚀 Start\nRun {project_path / 'start_project.bat'}\n## 📂 Structure\n- src/: Code\n- tests/: Tests\n- docs/: Docs\n",
            encoding='utf-8'
        )
        contrib_path.write_text(
            f"# Contribution to {project_name}\n# {CREATOR}\n1. Fork repo\n2. Branch: git checkout -b feature/your-feature\n3. Commit: git commit -m 'Add feature'\n4. Push: git push\n5. PR\n",
            encoding='utf-8'
        )
        (docs_dir / 'LICENSE').write_text(
            f"# MIT License\nCopyright (c) 2025 {CREATOR}\n\nPermission is hereby granted, free of charge, to any person obtaining a copy\n"
            "of this software and associated documentation files (the 'Software'), to deal\n"
            "in the Software without restriction, including without limitation the rights\n"
            "to use, copy, modify, merge, publish, distribute, sublicense, and/or sell\n"
            "copies of the Software, and to permit persons to whom the Software is\n"
            "furnished to do so, subject to the following conditions:\n\n"
            "The above copyright notice and this permission notice shall be included in all\n"
            "copies or substantial portions of the Software.\n\n"
            "THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR\n"
            "IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\n"
            "FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE\n"
            "AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER\n"
            "LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,\n"
            "OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE\n"
            "SOFTWARE.",
            encoding='utf-8'
        )
    except PermissionError:
        console.print("[red]Permission denied creating docs.[/red]")
    except Exception as e:
        console.print(f"[red]Error creating docs: {str(e)}[/red]")

def visualize_project(config):
    console.print(f"[yellow]Starting visualize_project for {config['name']}[/yellow]")
    if config["features"].get("mindmap", "n") == "y":
        try:
            base_path = Path(config["base_path"]).resolve()
            console.print(Panel(f"4D Map - {config['name']}\n[green]src: {len(list((base_path / Path(config['name'].replace(' ', '_')) / 'src').glob('*')))} files[/]\n[blue]tests: {len(list((base_path / Path(config['name'].replace(' ', '_')) / 'tests').glob('*')))} files[/]\n[magenta]docs: {len(list((base_path / Path(config['name'].replace(' ', '_')) / 'docs').glob('*')))} files[/]", style="on black", width=50))
        except Exception as e:
            console.print(f"[red]Error in visualize_project: {str(e)}[/red]")
            log_error(base_path, f"Error in visualize_project - {str(e)}")
    console.print(f"[green]Completed visualize_project for {config['name']}[/green]")

def log_error(base_path, message):
    log_path = base_path / "mizukage_log.txt"
    try:
        log_path.parent.mkdir(parents=True, exist_ok=True)
        with open(log_path, "a", encoding='utf-8') as f:
            f.write(f"{time()}: {message}\n")
    except Exception as e:
        console.print(f"[red]Error logging: {str(e)}[/red]")

# --- CLI Commands ---
@click.group()
@click.option("--quick", is_flag=True, help="Use last config")
@click.option("--undo", is_flag=True, help="Revert to backup")
@click.option("--feedback", help="Submit feedback")
@click.option("--theme", type=click.Choice(["default", "dark"]), default="default", help="Output theme")
@click.option("--set-path", help="Set custom base path for projects")
@click.option("--display-mode", type=click.Choice(["progress", "silent"]), default=None, help="Choose display mode (progress or silent)")
@click.pass_context
def cli(ctx, quick, undo, feedback, theme, set_path, display_mode):
    ctx.obj = {'quick': quick, 'theme': theme, 'set_path': set_path, 'display_mode': display_mode}
    try:
        show_wave_animation()
        console.print(Panel(f"~ OceanCrow_Mizukage Igniting... - {CREATOR} (v{VERSION})", style=f"bold {'white' if theme == 'default' else 'white'} on black", width=80))
        console.print("[██████] Loading...", style=f"{'white' if theme == 'default' else 'white'}")
    except Exception as e:
        console.print(f"[red]Error initializing UI: {str(e)}[/red]")
        return

    if feedback:
        try:
            with open("feedback.txt", "a", encoding='utf-8') as f:
                f.write(f"{time()}(): {feedback} - Tag @OceanCrowtt on X!\n")
            console.print("[green]Feedback saved! Share on X with @OceanCrowtt[/green]")
        except PermissionError:
            console.print("[red]Permission denied saving feedback.[/red]")
        except Exception as e:
            console.print(f"[red]Error saving feedback: {str(e)}[/red]")
        return

    config_path = Path.home() / ".ocean_crow_config.json"
    try:
        if undo and (Path.home() / ".ocean_crow_config_backup_*").exists():
            backup = sorted((Path.home() / ".ocean_crow_config_backup_*").glob("*"))[-1]
            shutil.copy(backup, config_path)
            console.print("[green]Reverted to backup[/green]")
        if set_path:
            config = load_or_create_config()
            config["base_path"] = set_path
            save_config(config)
            console.print(f"[green]Base path set to {set_path}[/green]")
        if display_mode:
            config = load_or_create_config()
            config["display_mode"] = display_mode
            save_config(config)
            console.print(f"[green]Display mode set to {display_mode}[/green]")
    except Exception as e:
        console.print(f"[red]Error handling config options: {str(e)}[/red]")

@cli.command()
@click.pass_context
def start(ctx):
    console.print("[yellow]Starting start command[/yellow]")
    try:
        config = load_or_create_config()
        quick = ctx.obj.get('quick', False)
        theme = ctx.obj.get('theme', 'default')
        set_path = ctx.obj.get('set_path')
        display_mode = ctx.obj.get('display_mode', config.get('display_mode', 'progress'))
        if set_path:
            config["base_path"] = set_path
            save_config(config)
        config["display_mode"] = display_mode

        # Add choice for new or continue project
        with console.screen() as screen:
            choice = Prompt.ask("Choose action", choices=["new", "continue"], default="new")
            if choice == "new":
                if not quick:
                    screen.update(Panel(f"[bold green]Setup - {CREATOR} (v{VERSION})[/bold green]", style=f"on black", height=7))
                    config["name"] = Prompt.ask("Project Name")
                    config["type"] = Prompt.ask("Project Type", choices=["basic", "flask", "datascience"], default=config.get("type", "basic"))
                    screen.update(Panel("[bold yellow]Features[/bold yellow]", style=f"on black", height=10))
                    for feature in ["scaffold", "autoenv", "codeflow", "databridge", "mindmap", "quantum", "orchestrate"]:
                        response = Prompt.ask(f"{feature.capitalize()} [green]Yes[/green]/[red]No[/red]", default=config["features"].get(feature, "n"))
                        config["features"][feature] = "y" if response.lower() == "y" else "n"
                    config["theme"] = Prompt.ask("Theme", choices=["default", "dark"], default=config.get("theme", "default"))
            else: # Continue project
                base_path = Path(config["base_path"]).resolve()
                existing_projects = [d for d in base_path.iterdir() if d.is_dir() and not d.name.startswith('.')]
                if not existing_projects:
                    console.print("[red]No existing projects found. Starting a new project instead.[/red]")
                    choice = "new"
                    screen.update(Panel(f"[bold green]Setup - {CREATOR} (v{VERSION})[/bold green]", style=f"on black", height=7))
                    config["name"] = Prompt.ask("Project Name")
                    config["type"] = Prompt.ask("Project Type", choices=["basic", "flask", "datascience"], default=config.get("type", "basic"))
                    screen.update(Panel("[bold yellow]Features[/bold yellow]", style=f"on black", height=10))
                    for feature in ["scaffold", "autoenv", "codeflow", "databridge", "mindmap", "quantum", "orchestrate"]:
                        response = Prompt.ask(f"{feature.capitalize()} [green]Yes[/green]/[red]No[/red]", default=config["features"].get(feature, "n"))
                        config["features"][feature] = "y" if response.lower() == "y" else "n"
                    config["theme"] = Prompt.ask("Theme", choices=["default", "dark"], default=config.get("theme", "default"))
                else:
                    console.print("[cyan]Existing projects:[/cyan]")
                    for i, project in enumerate(existing_projects, 1):
                        console.print(f"{i}. {project.name}")
                    selection = Prompt.ask("Select project number to continue", choices=[str(i) for i in range(1, len(existing_projects) + 1)], default="1")
                    config["name"] = existing_projects[int(selection) - 1].name
                    config["type"] = config.get("type", "basic") # Reuse last type or default
                    screen.update(Panel(f"[bold green]Continuing {config['name']} - {CREATOR} (v{VERSION})[/bold green]", style=f"on black", height=7))

        if not quick:
            console.print(f"[bold green]Ninja Level: {config['projects_created']} - {['Apprentice', 'Ninja', 'Master'][min(config['projects_created'] // 5, 2)]}![/bold green]")
        start_time = time()
        if Prompt.ask("Start Workflow [green]Yes[/green]/[red]No[/red]", default="y") == "y":
            console.print(f"[cyan]Starting workflow for {config['name']}[/cyan]")
            save_config(config)
            run_all(config, start_time)
            visualize_project(config)
            project_dir = Path(config["base_path"]).resolve() / Path(config["name"].replace(" ", "_"))
            show_success_banner(config["name"], project_dir.exists())
            if not project_dir.exists() and choice == "new":
                console.print(f"[yellow]Project folder not found at {project_dir}. Check mizukage_log.txt for details.[/yellow]")
            console.print(Panel(f"[bold green]Complete for {config['name']} - {CREATOR} (v{VERSION}) in {time() - start_time:.2f}s[/bold green]", style=f"on black", width=80))
    except Exception as e:
        console.print(f"[red]Critical error in start: {str(e)}[/red]")
        log_error(Path(config["base_path"]).resolve() if 'config' in locals() else Path.cwd(), f"Critical error in start - {str(e)}")
    console.print("[green]Completed start command[/green]")

if __name__ == "__main__":
    cli()