"use strict";
/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const ui_components_1 = require("@elyra/ui-components");
const application_1 = require("@jupyterlab/application");
const apputils_1 = require("@jupyterlab/apputils");
const filebrowser_1 = require("@jupyterlab/filebrowser");
const launcher_1 = require("@jupyterlab/launcher");
const mainmenu_1 = require("@jupyterlab/mainmenu");
const ui_components_2 = require("@jupyterlab/ui-components");
const PipelineEditorWidget_1 = require("./PipelineEditorWidget");
const PipelineService_1 = require("./PipelineService");
const RuntimesWidget_1 = require("./RuntimesWidget");
const SubmitNotebookButtonExtension_1 = require("./SubmitNotebookButtonExtension");
require("../style/index.css");
const PIPELINE_FACTORY = 'Pipeline Editor';
const PIPELINE = 'pipeline';
const PIPELINE_EDITOR_NAMESPACE = 'elyra-pipeline-editor-extension';
/**
 * Initialization data for the pipeline-editor-extension extension.
 */
const extension = {
    id: PIPELINE,
    autoStart: true,
    requires: [
        apputils_1.ICommandPalette,
        launcher_1.ILauncher,
        filebrowser_1.IFileBrowserFactory,
        application_1.ILayoutRestorer,
        mainmenu_1.IMainMenu
    ],
    activate: (app, palette, launcher, browserFactory, restorer, menu) => {
        console.log('Elyra - pipeline-editor extension is activated!');
        // Set up new widget Factory for .pipeline files
        const pipelineEditorFactory = new PipelineEditorWidget_1.PipelineEditorFactory({
            name: PIPELINE_FACTORY,
            fileTypes: [PIPELINE],
            defaultFor: [PIPELINE],
            shell: app.shell,
            commands: app.commands,
            browserFactory: browserFactory,
            serviceManager: app.serviceManager
        });
        // Add the default behavior of opening the widget for .pipeline files
        app.docRegistry.addFileType({
            name: PIPELINE,
            extensions: ['.pipeline'],
            icon: ui_components_1.pipelineIcon
        });
        app.docRegistry.addWidgetFactory(pipelineEditorFactory);
        const tracker = new apputils_1.WidgetTracker({
            namespace: PIPELINE_EDITOR_NAMESPACE
        });
        pipelineEditorFactory.widgetCreated.connect((sender, widget) => {
            void tracker.add(widget);
            // Notify the widget tracker if restore data needs to update
            widget.context.pathChanged.connect(() => {
                void tracker.save(widget);
            });
        });
        // Handle state restoration
        void restorer.restore(tracker, {
            command: PipelineEditorWidget_1.commandIDs.openDocManager,
            args: widget => ({
                path: widget.context.path,
                factory: PIPELINE_FACTORY
            }),
            name: widget => widget.context.path
        });
        // Add command to add file to pipeline
        const addFileToPipelineCommand = PipelineEditorWidget_1.commandIDs.addFileToPipeline;
        app.commands.addCommand(addFileToPipelineCommand, {
            label: 'Add File to Pipeline',
            icon: ui_components_2.addIcon,
            execute: args => {
                pipelineEditorFactory.addFileToPipelineSignal.emit(args);
            }
        });
        app.contextMenu.addItem({
            selector: '[data-file-type="notebook"]',
            command: addFileToPipelineCommand
        });
        // Add an application command
        const openPipelineEditorCommand = PipelineEditorWidget_1.commandIDs.openPipelineEditor;
        app.commands.addCommand(openPipelineEditorCommand, {
            label: args => args['isPalette'] ? 'New Pipeline Editor' : 'Pipeline Editor',
            icon: args => (args['isPalette'] ? undefined : ui_components_1.pipelineIcon),
            execute: () => {
                // Creates blank file, then opens it in a new window
                app.commands
                    .execute(PipelineEditorWidget_1.commandIDs.newDocManager, {
                    type: 'file',
                    path: browserFactory.defaultBrowser.model.path,
                    ext: '.pipeline'
                })
                    .then(model => {
                    return app.commands.execute(PipelineEditorWidget_1.commandIDs.openDocManager, {
                        path: model.path,
                        factory: PIPELINE_FACTORY
                    });
                });
            }
        });
        // Add the command to the palette.
        palette.addItem({
            command: openPipelineEditorCommand,
            args: { isPalette: true },
            category: 'Elyra'
        });
        // Add the command to the launcher
        if (launcher) {
            launcher.add({
                command: openPipelineEditorCommand,
                category: 'Elyra',
                rank: 1
            });
        }
        // Add new pipeline to the file menu
        menu.fileMenu.newMenu.addGroup([{ command: openPipelineEditorCommand }], 30);
        // SubmitNotebookButtonExtension initialization code
        const buttonExtension = new SubmitNotebookButtonExtension_1.SubmitNotebookButtonExtension();
        app.docRegistry.addWidgetExtension('Notebook', buttonExtension);
        app.contextMenu.addItem({
            selector: '.jp-Notebook',
            command: PipelineEditorWidget_1.commandIDs.submitNotebook,
            rank: -0.5
        });
        const runtimesWidget = new RuntimesWidget_1.RuntimesWidget({
            app,
            display_name: 'Runtimes',
            namespace: PipelineService_1.RUNTIMES_NAMESPACE,
            schema: PipelineService_1.KFP_SCHEMA,
            icon: ui_components_1.runtimesIcon
        });
        const runtimesWidgetID = `elyra-metadata:${PipelineService_1.RUNTIMES_NAMESPACE}:${PipelineService_1.KFP_SCHEMA}`;
        runtimesWidget.id = runtimesWidgetID;
        runtimesWidget.title.icon = ui_components_1.runtimesIcon;
        runtimesWidget.title.caption = 'Runtimes';
        restorer.add(runtimesWidget, runtimesWidgetID);
        app.shell.add(runtimesWidget, 'left', { rank: 950 });
    }
};
exports.default = extension;
//# sourceMappingURL=index.js.map