/*
 * Decompiled with CFR 0.152.
 */
package fr.gael.streams;

import fr.gael.streams.CacheUtils;
import fr.gael.streams.CachedFile;
import fr.gael.streams.ObservableInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import java.util.function.Function;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class RandomAccessInputStreamChannel
extends ObservableInputStream {
    private InputStream origin;
    private long originCursor;
    private File fileCache;
    private FileChannel buffer;
    private String keyLock;
    private static final int CHUNK = 0x200000;
    public static final String BUFFER_SUFFIX = ".dat";
    public static final String TMP_DIR = System.getProperty("fr.gael.streams.tmpdir", System.getProperty("java.io.tmpdir"));
    private static final int MAX_WAIT = Integer.valueOf(System.getProperty("fr.gael.streams.lock.waiting", "900"));
    private static final int LOOP_INTERVAL = 3;
    private static Logger LOGGER = LogManager.getLogger(RandomAccessInputStreamChannel.class);
    private static final String REG_EXP_BEG_URL_QUERY = "^(http|https)://[^/]+/v[^/]+/[^/]+/[^/]+/";
    private static final String REG_EXP_END_URL_QUERY = "\\?temp_url_sig=[^&]+&temp_url_expires=\\d+";
    private boolean cache_active;
    private Function<Void, InputStream> constructor;
    private boolean isXmlFile;

    public RandomAccessInputStreamChannel(Function<Void, InputStream> constructor, String fileName) throws IOException {
        fileName = this.cleanFileName(fileName);
        String name = fileName.toLowerCase();
        if (name.endsWith(".xml") || name.endsWith(".safe") || name.endsWith(".xsl") || name.endsWith(".xsd")) {
            this.isXmlFile = true;
        }
        LOGGER.debug("RAIS : " + fileName);
        this.cache_active = !Boolean.getBoolean("fr.gael.streams.cache.inactive");
        this.keyLock = DigestUtils.md5Hex((String)fileName);
        if (this.cache_active) {
            this.createCacheFile(this.keyLock);
        }
        this.constructor = constructor;
        this.origin = constructor.apply(null);
    }

    private String cleanFileName(String fileName) {
        Objects.requireNonNull(fileName);
        fileName = fileName.replaceAll(REG_EXP_BEG_URL_QUERY, "");
        fileName = fileName.replaceAll(REG_EXP_END_URL_QUERY, "");
        fileName = fileName.replace("/.value", "");
        return fileName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createCacheFile(String fileName) throws IOException {
        String string = BUFFER_SUFFIX;
        synchronized (BUFFER_SUFFIX) {
            String completeFileName = CacheUtils.getCompleteFileName(fileName);
            this.fileCache = new File(completeFileName);
            if (!this.fileCache.exists()) {
                CacheUtils.putInCache(fileName);
            }
            String directories = completeFileName.replace(this.fileCache.toPath().getFileName().toString(), "");
            Files.createDirectories(Paths.get(directories, new String[0]), new FileAttribute[0]);
            this.buffer = new RandomAccessFile(this.fileCache, "rw").getChannel();
            // ** MonitorExit[var2_2] (shouldn't be in output)
            return;
        }
    }

    @Override
    public int read() throws IOException {
        if (!this.cache_active) {
            int read = this.getOrigin().read();
            if (read != -1) {
                ++this.originCursor;
            }
            return read;
        }
        int read = this.readFromBuffer();
        if (read != -1) {
            return read;
        }
        long bufferCursor = this.buffer.position();
        this.readFromOrigin(this.originCursor + 1L);
        this.buffer.position(bufferCursor);
        return this.readFromBuffer();
    }

    private int readFromBuffer() throws IOException {
        this.waitForLock();
        ByteBuffer dst = ByteBuffer.allocate(1);
        int read = this.buffer.read(dst);
        if (read != -1) {
            this.updateLastAccess();
            return dst.get();
        }
        return -1;
    }

    private int readFromBuffer(byte[] b, int off, int len) throws IOException {
        this.waitForLock();
        int read = this.buffer.read(ByteBuffer.wrap(b, off, len));
        this.updateLastAccess();
        return read;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        long bufferCursor;
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || len < 0 || len > b.length - off) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return 0;
        }
        if (!this.cache_active) {
            int read = this.getOrigin().read(b, off, len);
            if (read != -1) {
                this.originCursor += (long)read;
            }
            RandomAccessInputStreamChannel.informObservers(String.format("%d|0", read));
            return read;
        }
        long readFromOrigin = 0L;
        long bufferLength = this.buffer.size();
        if (bufferLength - (bufferCursor = this.buffer.position()) < (long)len) {
            readFromOrigin = this.readFromOrigin((long)len - (bufferLength - bufferCursor));
            this.buffer.position(bufferCursor);
        }
        int readFromBuffer = this.readFromBuffer(b, off, len);
        RandomAccessInputStreamChannel.informObservers(String.format("%d|%d", readFromOrigin, readFromBuffer));
        return readFromBuffer;
    }

    private long skipBytes(long n) throws IOException {
        int numRead;
        byte[] skipBuf = new byte[8192];
        for (long num = n; num > 0L; num -= (long)numRead) {
            numRead = this.getOrigin().read(skipBuf, 0, num > (long)skipBuf.length ? skipBuf.length : (int)num);
            if (numRead == -1) {
                n -= num;
                break;
            }
            this.originCursor += (long)numRead;
        }
        return n;
    }

    public void seek(long n) throws IOException {
        if (!this.cache_active) {
            if (this.originCursor == n) {
                return;
            }
            if (this.originCursor > n) {
                if (this.origin != null) {
                    this.origin.close();
                }
                this.origin = this.constructor.apply(null);
                this.originCursor = 0L;
            }
            this.skipBytes(n - this.originCursor);
            return;
        }
        long bufferLength = this.buffer.size();
        if (n > bufferLength) {
            this.readFromOrigin(n - this.originCursor);
        }
        this.buffer.position(n);
    }

    @Override
    public long skip(long n) throws IOException {
        this.seek(this.buffer.position() + n);
        return n;
    }

    private long readFromOrigin(long bytesToRead) throws IOException {
        int read;
        this.waitForLock();
        this.lockBuffer();
        long totalRead = 0L;
        long bufferLength = this.buffer.size();
        if (this.originCursor < bufferLength) {
            if (this.isXmlFile) {
                this.getOrigin().skip(bufferLength - this.originCursor + 1L);
            } else {
                this.getOrigin().skip(bufferLength - this.originCursor);
            }
            this.originCursor = bufferLength;
        }
        while (totalRead < bytesToRead && (read = this.readOneChunk(0x200000)) != -1) {
            totalRead += (long)read;
        }
        this.unlockBuffer(this.buffer.size());
        return totalRead;
    }

    private int readOneChunk(int chunkSize) throws IOException {
        int offset;
        int read;
        byte[] b = new byte[chunkSize];
        for (offset = 0; offset < chunkSize; offset += read) {
            read = this.getOrigin().read(b, offset, chunkSize - offset);
            if (read != -1) continue;
            if (offset != 0) break;
            return -1;
        }
        this.buffer.position(this.buffer.size());
        if (offset > chunkSize) {
            this.buffer.write(ByteBuffer.wrap(b, 0, chunkSize));
        } else {
            this.buffer.write(ByteBuffer.wrap(b));
        }
        this.originCursor += (long)offset;
        return offset;
    }

    private void updateLastAccess() {
        CachedFile f = CacheUtils.getFromCache(this.keyLock);
        if (f != null) {
            f.setLastAccess(System.currentTimeMillis());
            CacheUtils.updateInCache(this.keyLock, f);
        }
    }

    private void lockBuffer() {
        CachedFile f = CacheUtils.getFromCache(this.keyLock);
        if (f != null) {
            f.lock();
            f.setLastAccess(System.currentTimeMillis());
            CacheUtils.updateInCache(this.keyLock, f);
        }
    }

    private void unlockBuffer() {
        CachedFile f = CacheUtils.getFromCache(this.keyLock);
        if (f != null) {
            f.unlock();
            f.setLastAccess(System.currentTimeMillis());
            CacheUtils.updateInCache(this.keyLock, f);
        }
    }

    private void unlockBuffer(long bufferLength) {
        CachedFile f = CacheUtils.getFromCache(this.keyLock);
        if (f != null) {
            f.unlock();
            f.setSize(bufferLength);
            f.setLastAccess(System.currentTimeMillis());
            CacheUtils.updateInCache(this.keyLock, f);
        }
    }

    private boolean isBufferLocked() {
        CachedFile f = CacheUtils.getFromCache(this.keyLock);
        if (f != null) {
            return f.isLock();
        }
        return false;
    }

    private void waitForLock() throws IOException {
        boolean locked = this.isBufferLocked();
        int current = 0;
        if (locked) {
            while (locked) {
                try {
                    LOGGER.debug(Thread.currentThread() + " waiting for " + this.keyLock);
                    Thread.sleep(42L);
                }
                catch (InterruptedException e) {
                    LOGGER.warn(String.format("Lock '%s' interrupted by user call.", this.keyLock));
                    this.unlockBuffer();
                    break;
                }
                if ((current += 3) >= MAX_WAIT) {
                    LOGGER.warn("Potential deadlock detected. Releasing lock '" + this.keyLock + "'.");
                    this.unlockBuffer();
                    break;
                }
                locked = this.isBufferLocked();
            }
            this.fileCache = new File(this.fileCache.getAbsolutePath());
            LOGGER.debug("refresh buffer");
        }
    }

    private InputStream getOrigin() {
        if (this.origin != null) {
            return this.origin;
        }
        this.origin = this.constructor.apply(null);
        this.originCursor = 0L;
        return this.origin;
    }

    public String getKeyLock() {
        return this.keyLock;
    }

    @Override
    public void close() throws IOException {
        this.unlockBuffer();
        if (this.buffer != null) {
            this.buffer.close();
        }
        if (this.origin != null) {
            this.origin.close();
        }
        CacheUtils.deregisterOpenedStream(this.keyLock);
    }
}

