#! /usr/bin/env python3

import os
import sys
import numpy as np
from numpy import linalg
from scipy import stats
from scipy.spatial import ConvexHull
import random
import matplotlib
import matplotlib.pyplot as plt
import matplotlib.cm as cm
from matplotlib.widgets import Slider, Button, RadioButtons, TextBox, CheckButtons, Cursor, LassoSelector, SpanSelector
from matplotlib.path import Path
import seaborn as sns
import nmrglue as ng
import lmfit as l
from datetime import datetime
import warnings

from . import fit, misc, sim, figures, processing
#from .__init__ import CM
from .config import CM, COLORS, cron

# Suppress warnings that create alarmism
warnings.filterwarnings(action='ignore', message='Error reading the pulse program')
warnings.filterwarnings(action='ignore', message=r'[0-9]*cannot')

# Declare interesting things
proc_keys_1D = ['wf', 'zf', 'fcor', 'tdeff']
wf0 = {
        'mode':None,
        'ssb':2,
        'lb':5,
        'gb':10,
        'gc':0,
        'sw':None
        }
"""
Classes for the management of NMR data.
"""

class Spectrum_1D:
    """
    Class: 1D NMR spectrum
    ---------
    Attributes:
    - datadir: str
        Path to the input file/dataset directory
    - filename: str
        Base of the name of the file, without extensions
    - fid: 1darray
        FID
    - acqus: dict
        Dictionary of acqusition parameters
    - ngdic: dict
        Created only if it is an experimental spectrum. Generated by nmrglue.bruker.read, contains all the information on the spectrometer and on the spectrum.
    - procs: dict
        Dictionary of processing parameters
    - S: 1darray
        Complex spectrum
    - r: 1darray
        Real part of the spectrum
    - i: 1darray
        Imaginary part of the spectrum
    - freq: 1darray
        Frequency scale of the spectrum, in Hz
    - ppm: 1darray
        ppm scale of the spectrum
    - F: fit.Voigt_Fit object
        Used for deconvolution. See fit.Voigt_fit.
    - baseline: 1darray
        Baseline of the spectrum.
    - integrals: dict
        Dictionary where to save the regions and values of the integrals.
    """
    def __str__(self):
        """ Prints info on the object """
        doc = '-'*64
        doc += '\nSpectrum_1D object.\n'
        if 'ngdic' in self.__dict__.keys():
            doc += f'Read from "{self.datadir}"\n'
        else:
            doc += f'Simulated from "{self.filename}" in {self.datadir}\n'
        doc += f'It is a {self.acqus["nuc"]} spectrum recorded over a\nsweep width of {self.acqus["SWp"]} ppm, centered at {self.acqus["o1p"]} ppm.\n'
        if 'fid' in self.__dict__.keys():
            N = self.fid.shape[-1]
            doc += f'The FID is {N} points long.\n'
        elif 'S' in self.__dict__.keys():
            N = self.S.shape[-1]
            doc += f'The spectrum is {N} points long.\n'
        else:
            pass
        doc += '-'*64

        return doc

    def __init__(self, in_file, pv=False, isexp=True, spect='bruker'):
        """
        Initialize the class. 
        Simulation of the dataset (i.e. isexp=False) employs sim.sim_1D.
        -------
        Parameters:
        - in_file: str
            path to file to read, or to the folder of the spectrum
        - pv: bool
            True if you want to use pseudo-voigt lineshapes for simulation, False for Voigt
        - isexp: bool
            True if this is an experimental dataset, False if it is simulated
        - spect: str
            Data file format. Allowed: 'bruker', 'varian', 'magritek', 'oxford'
        """
        ## Set up the filenames
        if isinstance(in_file, dict):   # Simulated data with already given acqus dictionary
            self.datadir = os.getcwd()      # Current directory
            self.filename = 'dict'  # Filename is just "dict"
        else:                           # You need to actually read a file
            self.datadir = os.path.abspath(in_file) # Get the file position
            if not os.path.isdir(self.datadir):
                self.datadir = os.path.dirname(self.datadir)
            self.filename = os.path.basename(in_file).rsplit('.', 1)[0] # Get the filename
            # If filename is a directory, write things inside it
            if os.path.isdir(f'{os.sep}'.join([self.datadir, self.filename])) and isexp:
                self.datadir = os.path.join(self.datadir, self.filename) # i.e. add filename to datadir
        
        ## Read the data
        if isexp is False:  # Simulate the dataset 
            if isinstance(in_file, dict):   # using the provided dictionary
                self.acqus = dict(in_file)
            else:   # or building it from the file
                self.acqus = sim.load_sim_1D(in_file)
            # Generate the FID
            self.fid = sim.sim_1D(in_file, pv=pv)
            # The acqus dictionary will know your dataset is simulated
            self.acqus['spect'] = 'simulated'
        else:   # Read the data from the directory using nmrglue
            with warnings.catch_warnings():   # Suppress errors due to CONVDTA in TopSpin
                warnings.simplefilter("ignore")
                # Discriminate between different spectrometer formats
                if spect == 'bruker':   
                    dic, data = ng.bruker.read(in_file, cplex=True)
                    self.acqus = misc.makeacqus_1D(dic)     
                    # Set the data format keys in acqus
                    self.acqus['BYTORDA'] = dic['acqus']['BYTORDA']
                    self.acqus['DTYPA'] = dic['acqus']['DTYPA']

                elif spect == 'varian':
                    dic, data = ng.varian.read(in_file)
                    self.acqus = misc.makeacqus_1D_varian(dic)

                elif spect == 'magritek':
                    dic1, data = ng.spinsolve.read(in_file, specfile='data.1d')         # Actual FID
                    dic2, _, = ng.spinsolve.read(in_file, specfile='spectrum.1d')       # for config
                    dic3, _, = ng.spinsolve.read(in_file, specfile='nmr_fid.dx')        # for config
                    # Join the dictionary together
                    dic = dict(dic1)
                    dic.update(dic3)
                    dic.update(dic2)        # Important because it contains the ppm scale!
                    self.fid = data
                    self.acqus = misc.makeacqus_1D_spinsolve(dic)

                elif spect == 'oxford':
                    if '.jdx' in in_file:
                        jdx_file = in_file
                    else:
                        dirlist = os.listdir(self.datadir)
                        all_jdx = [w for w in dirlist if '.jdx' in w]
                        if len(all_jdx) == 0:
                            raise NameError(f'No .jdx file were found in {self.datadir}.')
                        elif len(all_jdx) > 1:
                            raise ValueError(f'There are more than one .jdx file in {self.datadir}.')
                        jdx_file = os.path.join(self.datadir, all_jdx[-1])
                    dic, cplx = ng.jcampdx.read(jdx_file)
                    self.acqus = misc.makeacqus_1D_oxford(dic)
                    data = cplx[0] + 1j*cplx[1]

                else:
                    raise NotImplementedError('Unknown dataset format.')

            # Store the local variables as class attributes
            self.acqus['spect'] = spect
            self.fid = data
            self.ngdic = dic        # NMRGLUE dictionary of parameters
            del dic
            del data
        # Look for group delay points: if there are not, put it to 0
        try:
            self.acqus['GRPDLY'] = int(self.ngdic['acqus']['GRPDLY'])
        except:
            self.acqus['GRPDLY'] = 0

        ## Initalize the procs dictionary 
        # If there already is a procs dictionary saved as file, load it
        if os.path.exists(os.path.join(self.datadir, f'{self.filename}.procs')):
            self.procs = self.read_procs()
        # Otherwise, initialize it with default values
        else:
            self.procs = {} 
            proc_init_1D = (dict(wf0), None, 0.5, 0)    # Make a shallow copy
            for k, key in enumerate(proc_keys_1D):  # Loop in this way to keep the order
                self.procs[key] = proc_init_1D[k]         
            self.procs['wf']['sw'] = round(self.acqus['SW'], 4)
            #   phases
            self.procs['p0'] = 0
            self.procs['p1'] = 0
            self.procs['pv'] = round(self.acqus['o1p'], 2)
            #   baseline
            self.procs['basl_c'] = None
            #   calibration
            self.procs['cal'] = 0

            # Write this in datadir
            self.write_procs()
        
    def add_noise(self, s_n=1):
        """
        Adds noise to the FID, using the function sim.noisegen.
        ------------------
        Parameters:
        - s_n: float
            Standard deviation of the noise
        """
        self.fid += sim.noisegen(self.fid.shape, self.acqus['o1'], self.acqus['t1'], s_n=s_n)

    def scan(self, ns=1, s_n=1):
        """
        Simulates the acquisition of ns scans, by adding a different realization of noise at each iteration.
        The function is supposed to start with the FID without noise at all. If not, the results will be biased.
        --------------------
        Parameters:
        - ns: int
            Number of scans to accumulate
        - s_n: float
            Standard deviation of the noise
        """
        clean_fid = np.copy(self.fid)   # Save a shallow copy of the FID without noise
        for k in range(ns):
            # Each scan is: clean FID + noise
            self.fid += clean_fid + sim.noisegen(self.fid.shape, self.acqus['o1'], self.acqus['t1'], s_n=s_n)

    def convdta(self, scaling=1):
        """ Call processing.convdta using self.acqus['GRPDLY'] """
        self.fid = processing.convdta(self.fid, self.acqus['GRPDLY'], scaling)

    def pknl(self):
        """
        Reverses the effect of the digital filter by applying a first order phase correction.
        To be called after having processed the data by 'self.process()'
        """
        self.adjph(p1=-360 * self.acqus['GRPDLY'], update=False)

    def blp(self, pred=8, order=8, N=2048):
        """
        Call processing.blp on self.fid for the application of backward linear prediction to the data. Important for Oxford benchtop data, where you have to predict 8 points to have a usable spectrum.
        ----------
        Parameters:
        - pred: int
            Number of points to be predicted
        - order: int
            Number of coefficients to be used for the prediction
        - N: int
            Number of FID points to be used for calculation; used to decrease computation time
        """
        self.fid = processing.blp(self.fid, pred, order=order, N=N)

    def process(self, interactive=False):
        """
        Performs the processing of the FID. The parameters are read from self.procs.
        Calls processing.interactive_fp or processing.fp using self.acqus and self.procs
        Writes the result is self.S, then unpacks it in self.r and self.i
        Calculates frequency and ppm scales.
        Also initializes self.F with fit.Voigt_Fit class using the current parameters
        --------
        Parameters:
        - interactive: bool
            True if you want to open the interactive panel, False to read the parameters from self.procs.
        """
        if interactive is True:
            self.S, self.procs = processing.interactive_fp(self.fid, self.acqus, self.procs)
        else:
            self.S = processing.fp(self.fid, wf=self.procs['wf'], zf=self.procs['zf'], fcor=self.procs['fcor'], tdeff=self.procs['tdeff'])
        if self.acqus['spect'] not in ['bruker', 'simulated']: # Bruker data are meant to be ordered already
            self.S = self.S[::-1]
        if self.acqus['SFO1'] < 0:  # Correct for negative gamma nuclei
            self.S = self.S[::-1]
        # Unpack the complex spectrum into real and imaginary part
        self.r = self.S.real
        self.i = self.S.imag

        # Calculate frequency and ppm scales
        self.freq = processing.make_scale(self.r.shape[0], dw=self.acqus['dw'])
        if self.acqus['SFO1'] < 0:  # Correct for negative gamma nuclei
            self.freq = self.freq[::-1]
        self.ppm = misc.freq2ppm(self.freq, B0=self.acqus['SFO1'], o1p=self.acqus['o1p'])


        # Initializes the F attribute
        self.F = fit.Voigt_Fit(self.ppm, self.S, self.acqus['t1'], self.acqus['SFO1'], self.acqus['o1p'], self.acqus['nuc'], filename=os.path.join(self.datadir, self.filename))

        # Compute the baseline, if there is
        if self.procs['basl_c'] is None:    # Initialize it to zero
            self.baseline = np.zeros_like(self.ppm)
        else:   # Baseline was computed before!
            x_basl = np.linspace(-1, 1, self.S.shape[-1])
            self.baseline = misc.polyn(x_basl, self.procs['basl_c'])

        # Apply phase correction according to procs
        self.adjph(p0=self.procs['p0'], p1=self.procs['p1'], pv=self.procs['pv'], update=False) # Do not update because otherwise it accumulates the phase
        # Calibrate
        self.cal(self.procs['cal'], update=False)

        # Initialize the integrals attribute
        self.integrals = {}


    def inv_process(self):
        """
        Performs the inverse processing of the spectrum according to the given parameters.
        Overwrites the S attribute!!
        Calls processing.inv_fp
        """
        if self.acqus['SFO1'] < 0:  # Correction for negative gamma nuclei
            self.S = self.S[::-1]
        # Make the inverse processing
        self.S = processing.inv_fp(self.S, wf=self.procs['wf'], size=self.acqus['TD'], fcor=self.procs['fcor'])

    def mc(self):
        """
        Calculates the magnitude of the spectrum and overwrites self.S, self.r, self.i
        """
        # Basically, || self.S ||_2 
        self.S = (self.S.real**2 + self.S.imag**2)**0.5
        self.r = self.S.real
        self.i = self.S.imag

        self.F.S = self.r

    def adjph(self, p0=None, p1=None, pv=None, update=True):
        """
        Adjusts the phases of the spectrum according to the given parameters, or interactively if they are left as default.
        Calls for processing.ps
        -------
        Parameters:
        - p0: float or None
            0-th order phase correction /°
        - p1: float or None
            1-st order phase correction /°
        - pv: float or None
            1-st order pivot /ppm
        - update: bool
            Choose if you want to update the procs dictionary or not
        """
        # Adjust the phases
        self.S, values = processing.ps(self.S, self.ppm, p0=p0, p1=p1, pivot=pv)
        self.r = self.S.real
        self.i = self.S.imag

        if update:      # Add the new phases to the previous ones
            self.procs['p0'] += round(values[0], 2)
            self.procs['p1'] += round(values[1], 2)
            if values[2] is not None:
                self.procs['pv'] = round(values[2], 5)

        # Update the .procs file
        self.write_procs()

        # Put the phased spectrum in S
        self.F.S = self.S

    def acme(self, **method_kws):
        """
        Automatic phase correction based on entropy minimization
        It calculates the phase angles using the algorithm specified in method, then calls self.adjph with those values.
        --------
        Parameters:
        - method_kws: keyworded arguments
            Additional parameters for the chosen method.
        """
        p0, p1 = processing.acme(np.copy(self.S), **method_kws)
        self.adjph(p0=p0, p1=p1)


    def rpbc(self, **rpbc_kws):
        """
        Computes the phase angles and the baseline using processing.RPBC on self.S.
        Then applies the phase correction and subtracts the baseline, automatically.
        The procs dictionary is then updated and saved.
        The polynomial baseline is computed according to the given coefficients and stored in self.baseline
        ---------------
        Parameters:
        - rpbc_kws: keyworded arguments
            See processing.RPBC for details.
        """
        self.S, p0, p1, c = processing.rpbc(self.S, **rpbc_kws)
        self.r = self.S.real
        self.i = self.S.imag
        # Update the procs dictionary
        self.procs['p0'] += p0
        self.procs['p1'] += p1
        self.procs['basl_c'] = c
        # Make the new baseline
        self.baseline = misc.polyn(np.linspace(-1, 1, self.S.shape[-1]), c)

        # Update the .procs file
        self.write_procs()

    def cal(self, offset=None, isHz=False, update=True):
        """
        Calibrates the ppm and frequency scale according to a given value, or interactively.
        Calls processing.calibration
        -------
        Parameters:
        - offset: float or None
            scale shift value
        - isHz: bool
            True if offset is in frequency units, False if offset is in ppm
        - update: bool
            Choose if to update the procs dictionary or not
        """
        # Make shallow copies
        in_ppm = np.copy(self.ppm)
        in_S = np.copy(self.r)
        if offset is None:  # Get the values interactively
            offppm = processing.calibration(in_ppm, in_S)
            offhz = misc.ppm2freq(offppm, self.acqus['SFO1'], self.acqus['o1p'])
        else:               # Calculate the missing one
            if isHz:    # offppm is missing
                offhz = offset
                offppm = misc.freq2ppm(offhz, self.acqus['SFO1'])
            else:       # offhz is missing
                offppm = offset
                offhz = misc.ppm2freq(offppm, self.acqus['SFO1'])

        # Apply the calibration
        #   to the scales
        self.freq += offhz
        self.ppm += offppm
        #   to the offset, so that it stays in the center of the SW
        self.acqus['o1p'] += offppm
        self.acqus['o1'] += offhz
        if update:  # update the procs dictionary
            self.procs['cal'] += offppm
        # Update the .procs file
        self.write_procs()

    def write_acqus(self, other_dir=None):
        """
        Write the acqus dictionary in a file named "filename.acqus".
        Calls misc.write_acqus_1D
        --------
        Parameters:
        - other_dir: str or None
            Different location for the acqus dictionary to write into. If None, self.datadir is used instead.
        """
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.acqus')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.acqus')
        misc.write_acqus_1D(self.acqus, path=path)

    def write_procs(self, other_dir=None):
        """
        Writes the actual procs dictionary in a file named "filename.procs" in the same directory of the input file.
        ---------
        Parameters:
        - other_dir: str or None
            Different location for the procs dictionary to write into. If None, self.datadir is used instead. W! Do not put the trailing slash!
        """
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.procs')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.procs')
        with open(path, 'w') as f:
            f.write(f'{self.procs}')

    def read_procs(self, other_dir=None):
        """
        Reads the procs dictionary from a file named "filename.procs" in the same directory of the input file.
        ---------
        Parameters:
        - other_dir: str or None
            Different location for the procs dictionary to look into. If None, self.datadir is used instead. W! Do not put the trailing slash!
        ---------
        Returns:
        - procs: dict
            Dictionary of processing parameters
        """
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.procs')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.procs')
        with open(path, 'r') as f:
            procs = eval(f.read().replace('array', 'np.array'))
        # Check if it was read correctly
        if isinstance(procs, dict):
            return procs
        else:
            raise ValueError(f'{self.filename}.procs cannot be interpreted as a dictionary')

    @staticmethod
    def write_ser(ser, acqus, path=None):
        """
        Writes a real/complex array in binary format.
        Calls misc.write_ser. Be sure that acqus contains the BYTORDA and DTYPA keys.
        See misc.write_ser to understand the meaning of these values.
        --------
        Parameters:
        - ser: ndarray
            Array that you want to convert in binary format.
        - acqus: dict
            Dictionary of acquisition parameters. It must contain BYTORDA and DTYPA.
        - path: str 
            Path where to save the binary file.
        """
        if path is None:
            raise NameError('You must specify a filename!')
        misc.write_ser(ser, path, acqus['BYTORDA'], acqus['DTYPA'])

    def plot(self, name=None, ext='png', dpi=600):
        """
        Plots the real part of the spectrum.
        ---------
        Parameters:
        - name: str
            Filename for the figure. If None, it is shown instead.
        - ext: str
            Format of the image
        - dpi: int
            Resolution of the image in dots per inches
        """
        def onselect(xmin, xmax):
            """ Moves the tracker """
            # Set the bars visible
            bar1.set_visible(True)
            bar2.set_visible(True)
            # Update the bars positions
            bar1.set_xdata((xmin,))
            bar2.set_xdata((xmax,))
            # Compute distance
            d_ppm = np.abs(xmin-xmax)
            # Convert from ppm to Hz
            d_hz = misc.ppm2freq(d_ppm, self.acqus['SFO1'])
            # Update the distance text
            text = f'{d_ppm:12.3f} ppm | {d_hz:12.3f} Hz'
            text_measure.set_text(text)
            plt.draw()

        def onclick(event):
            """ Correct appearance of the tracker with mouse buttons """
            if not event.inaxes:    # Do things only if click is inside the panel
                return
            # Left button interacts also with the span selector, right button only resets
            if event.button == 1 or event.button == 3:
                # Erase the text
                text_measure.set_text('')
                # Make the bars invisible
                bar1.set_visible(False)
                bar2.set_visible(False)
                plt.draw()

        # Default of 10 ticks on the ppm axis
        n_xticks = 10

        # Make the figure
        fig = plt.figure(f'{self.filename}')
        fig.set_size_inches(figures.figsize_large)
        plt.subplots_adjust(left=0.10, bottom=0.15, right=0.95, top=0.90)
        ax = fig.add_subplot(1,1,1)

        # Plot the spectrum
        spect, = ax.plot(self.ppm, self.r, lw=0.8)
        # Bars of the spanselector
        bar1 = ax.axvline(self.ppm[0], c='r', lw=0.4, visible=False)
        bar2 = ax.axvline(self.ppm[-1], c='r', lw=0.4, visible=False)

        # Placeholder for distance measurement
        text_measure = plt.text(0.75, 0.05, '', ha='left', va='bottom', transform=fig.transFigure, fontsize=12, color='r')

        # Make the label of the x-axis
        X_label = r'$\delta\ $'+misc.nuc_format(self.acqus['nuc'])+' /ppm'
        ax.set_xlabel(X_label)

        # Fancy figure adjustments
        #   Make pretty x-scale
        xsx, xdx = max(self.ppm), min(self.ppm) # Order it as a ppm scale
        misc.pretty_scale(ax, (xsx, xdx), axis='x', n_major_ticks=n_xticks)
        #   Auto-adjusts the limits for the y-axis
        misc.set_ylim(ax, self.r)
        #   Make pretty y-scale
        misc.pretty_scale(ax, ax.get_ylim(), axis='y', n_major_ticks=n_xticks)
        misc.mathformat(ax)
        #   Make the fontsizes bigger
        misc.set_fontsizes(ax, 14)

        # Set a vertical line for inspection
        cursor = Cursor(ax, useblit=True, c='tab:red', lw=0.8, horizOn=False)

        # Widget for distance measurement
        tracker = SpanSelector(ax, onselect, direction='horizontal', useblit=False, button=1, 
                               props={'alpha':1, 'fill':False, 'color':'r', 'lw':0.4, 'edgecolor':'r'})
        # Connect mouse buttons
        fig.canvas.mpl_connect('button_press_event', onclick)

        if name:    # Save the figure
            plt.savefig(f'{name}.{ext}', format=f'{ext}', dpi=dpi)
        else:       # Show it
            plt.show()
        plt.close()

    def qfil(self, u=None, s=None):
        """
        Gaussian filter to suppress signals.
        Tries to read self.procs['qfil'], which is
            { 'u': u, 's': s }
        Otherwise, these are set interactively by processing.interactive_qfil and then added to self.procs.
        Calls processing.qfil
        ---------
        Parameters:
        - u: float
            Position of the filter /ppm
        - s: float
            Width (standard deviation) of the filter /ppm
        """
        if 'qfil' not in self.procs.keys():     # Then add it
            self.procs['qfil'] = {'u': u, 's': s}
        for key, value in self.procs['qfil'].items():
            if value is None:   # At the first non-set value, do the interactive correction
                self.procs['qfil']['u'], self.procs['qfil']['s'] = processing.interactive_qfil(self.ppm, self.r)
                break
        # Apply it
        self.S = processing.qfil(self.ppm, self.S, self.procs['qfil']['u'], self.procs['qfil']['s'])
        self.r = self.S.real
        self.i = self.S.imag

        # Update the .procs file
        self.write_procs()

    def baseline_correction(self, basl_file='spectrum.basl', winlim=None):
        """
        Correct the baseline of the spectrum, according to a pre-existing file or interactively.
        Calls processing.baseline_correction or processing.load_baseline
        -------
        Parameters:
        - basl_file: str
            Path to the baseline file. If it already exists, the baseline will be built according to this file; otherwise this will be the destination file of the baseline.
        - winlim: tuple or None
            Limits of the baseline. If it is None, it will be interactively set. If basl_file exists, it will be read from there. Else, (ppm1, ppm2).
        """
        if not os.path.exists(basl_file):
            processing.baseline_correction(self.ppm, self.r, basl_file=basl_file, winlim=winlim)
        self.baseline = processing.load_baseline(basl_file, self.ppm, self.r)

    def basl(self, from_procs=False, phase=True):
        """
        Apply the baseline correction by subtracting self.baseline from self.S. Then, self.S is unpacked in self.r and self.i
        --------
        Parameters:
        - from_procs: bool
            If True, computes the baseline using the polynomion model reading self.procs['basl_c'] as coefficients
        - phase: bool
            Choose if to apply the same phase correction of the spectrum to the baseline. This should be done if the baseline was computed before the phase adjustment!
        """
        if from_procs:  # Compute the baseline with a polynomial model
            x_basl = np.linspace(-1, 1, self.S.shape[-1])
            self.baseline = misc.polyn(x_basl, self.procs['basl_c'])
        if phase and np.iscomplexobj(self.baseline):       # Phase the baseline as the spectrum, only if it is complex
            self.baseline, *_ = processing.ps(self.baseline, self.ppm, p0=self.procs['p0'], p1=self.procs['p1'], pivot=self.procs['pv'])
        # Subtract the baseline
        self.S -= self.baseline
        # Unpack S
        self.r = self.S.real
        self.i = self.S.imag

    def integrate(self, lims=None):
        """
        Integrate the spectrum with a dedicated GUI.
        Calls fit.integrate and writes in self.integrals with keys [ppm1:ppm2]
        --------
        Parameters:
        - lims: tuple
            Integrates from lims[0] to lims[1]. If it is None, calls for interactive integration.
        """
        X_label = r'$\delta\,$' + misc.nuc_format(self.acqus['nuc']) + ' /ppm'
        if lims is None:
            integrals = fit.integrate(self.ppm, self.r, X_label=X_label)
            for key, value in integrals.items():
                self.integrals[key] = value
        else:
            self.integrals[f'{lims[0]:.2f}:{lims[1]:.2f}'] = processing.integrate(self.r, self.ppm, lims)

    def write_integrals(self, other_dir=None):
        """
        Write the integrals in a file named "{self.filename}.int".
        -------
        Parameters:
        - other_dir: str or None
            Different location for the integrals file to write into. If None, self.datadir is used instead.
            
        """
        # Detect the file position
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.int')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.int')

        f = open(path, 'w')
        for key, value in self.integrals.items():
            if 'total' in key:  # first entry
                f.write('{:12}\t\t{:.4e}\n'.format(key, value))
            elif 'ref' in key:  # second entry
                if 'pos' in key:
                    f.write('{:12}\t\t{}\n'.format(key, value))
                elif 'int' in key:
                    f.write('{:12}\t\t{:.4e}\n'.format(key, value))
                elif 'val' in key:
                    f.write('{:12}\t\t{:.3f}\n'.format(key, value))
            else:   # All the rest
                f.write('{:12}\t{:.8f}\n'.format(key, value))
        f.close()

    def to_vf(self, filename=None, Hs=None, fvf=True):
        """
        Transform a simulated spectrum in a .ivf or .fvf file to be used in a deconvolution procedure.
        To do this, it reads the peak parameters saved in acqus.
        The number of signals is determined by acqus['amplitudes'].
        Multiplets are splitted according to their Js, and saved as components.
        ---------
        Parameters:
        - filename: str or None
            Path to the filename to be saved, without extension. If None, self.filename is used
        - Hs: int or None
            Number of nuclei the spectrum integrates for. If None, the sum of the amplitudes is used.
        - fvf: bool
            If True, adds the '.fvf' extension to the filename, if False, adds '.ivf'
        """

        # Create default filename
        if filename is None:
            filename = os.path.join(self.datadir, self.filename)
        if fvf:
            filename += '.fvf'
        else:
            filename += '.ivf'

        # If Hs is not given, sum the intensities
        if Hs is None:
            Hs = np.sum(self.acqus['amplitudes'])
        # Correct the intensities to get the relative ones
        A, _ = misc.molfrac(self.acqus['amplitudes'])

        dic = {}    # Placeholder
        mult_counter = 0    # Multiplets counter, to set "group"
        peak_counter = 0    # Peak counter, for the dictionary keys

        # Make the fit.Peak objects
        for k, _ in enumerate(A):
            # Singlets
            if self.acqus['mult'][k] == 's':
                peak_counter += 1       # Increase peak counter
                # Make the fit.Peak object straightforwardly
                dic[peak_counter] = fit.Peak(self.acqus,
                        u = self.acqus['shifts'][k],
                        fwhm = self.acqus['fwhm'][k],
                        k = A[k],
                        x_g = self.acqus['x_g'][k],
                        N = self.procs['zf'],
                        group = 0,      # Identifier for singlets
                        )

            else:
                mult_counter += 1       # Increase multiplets counter
                # Get the chemical shifts and intensities of the components of the multiplet
                u_in, I_in = sim.multiplet(
                        # CS must be given in Hz
                        misc.ppm2freq(self.acqus['shifts'][k], self.acqus['SFO1'], self.acqus['o1p']),
                        A[k],
                        self.acqus['mult'][k],
                        self.acqus['Jconst'][k],
                        )
                # Make the components one by one
                for u, I in zip(u_in, I_in):
                    peak_counter += 1   # Increase peak counter FOR EACH COMPONENT
                    dic[peak_counter] = fit.Peak(self.acqus,
                            # Convert the chemical shift given by sim.multiplet back to ppm
                            u = misc.freq2ppm(u, self.acqus['SFO1'], self.acqus['o1p']),
                            fwhm = self.acqus['fwhm'][k],
                            k = I,  # Relative intensity of the component, A[k] is already accounted for
                            x_g = self.acqus['x_g'][k],
                            N = self.procs['zf'],
                            group = mult_counter,   # Identify all the components as part of the same multiplet
                            )

        # Compute limits to save the spectrum
        limits = []     # Placeholder
        i_umax = np.argmax(self.acqus['shifts'])        # Index of the most deshielded peak
        umax = self.acqus['shifts'][i_umax]             # Chemical shift of the most deshielded peak
        # FWHM of the most deshielded peak converted to ppm
        f_umax = misc.freq2ppm(self.acqus['fwhm'][i_umax], self.acqus['SFO1'])
        i_umin = np.argmin(self.acqus['shifts'])        # Index of the most shielded peak
        umin = self.acqus['shifts'][i_umin]             # Chemical shift of the most shielded peak
        # FWHM of the most shielded peak converted to ppm
        f_umin = misc.freq2ppm(self.acqus['fwhm'][i_umin], self.acqus['SFO1'])
        # Limits are taken with an extra space of 64 FWHM, to allow them to go to zero
        limits = umax + 64*f_umax, umin - 64*f_umin
                
        # Write the file
        with open(filename, 'w') as f:
            f.write('!\n\n')
        fit.write_vf(filename, dic, lims=limits, I=Hs)
        print(f'File {filename} generated successfully.')

    def to_wav(self, filename=None, cutoff=None, rate=44100):
        """
        Converts the FID in an audio file by using misc.data2wav.
        ---------
        Parameters:
        - filename: str
            Path where to save the file. If None, self.filename is used
        - cutoff: float
            Clipping limits for the FID
        - rate: int
            Sampling rate in samples/sec
        """
        if filename is None:
            cwd = os.getcwd()
            filename = os.path.join(cwd, self.filename)
        # Make a shallow copy of the FID
        data = np.copy(self.fid)
        misc.data2wav(data, filename, cutoff, rate)


class pSpectrum_1D(Spectrum_1D):
    """
    Subclass of Spectrum_1D that allows to handle processed 1D NMR spectra.
    Useful when dealing with traces of 2D spectra.
    Shares the same attributes with Spectrum_1D.
    -------------
    Attributes:
    - datadir: str
        Path to the input file/dataset directory
    - filename: str
        Base of the name of the file, without extensions
    - acqus: dict
        Dictionary of acqusition parameters
    - ngdic: dict
        Created only if it is an experimental spectrum. Generated by nmrglue.bruker.read, contains all the information on the spectrometer and on the spectrum.
    - procs: dict
        Dictionary of processing parameters
    - S: 1darray
        Complex spectrum
    - r: 1darray
        Real part of the spectrum
    - i: 1darray
        Imaginary part of the spectrum
    - freq: 1darray
        Frequency scale of the spectrum, in Hz
    - ppm: 1darray
        ppm scale of the spectrum
    - F: fit.Voigt_Fit object
        Used for deconvolution. See fit.Voigt_fit.
    - baseline: 1darray
        Baseline of the spectrum.
    - integrals: dict
        Dictionary where to save the regions and values of the integrals.
    """
    def __init__(self, in_file, acqus=None, procs=None, istrace=False, filename='T'):
        """
        Initialize the class. 
        -------
        Parameters:
        - in_file: str or 1darray
            If istrace is True, in_file is the NMR spectrum. Else, it is the directory of the processed data.
        - acqus: dict or None
            If istrace is True, you must supply the associated 'acqus' dictionary. Else, it is not necessary as it is read from the input directory
        - procs: dict or None
            You can pass the dictionary of processing parameters, if you want. Otherwise, it is initialized with standard values.
        - istrace: bool
            Declare the object as trace extracted from a 2D (True) or as true experimental spectrum (False)
        - filename: str
            If istrace is True, this will be the filename of self.acqus and self.procs
        """
        if istrace is True:     # it comes from a 2D
            assert isinstance(in_file, np.ndarray), 'The first parameter must be an array if istrace=True!'
            self.datadir = os.getcwd()
            self.filename = filename
            if np.iscomplexobj(in_file):    # There is the whole information
                self.r = in_file.real
                self.i = in_file.imag
            else:                           # There is the real part only
                self.r = in_file        
                self.i = np.zeros_like(in_file)
            self.S = self.r + 1j * self.i   # Make the complex spectrum
            self.acqus = acqus

        else:
            with warnings.catch_warnings():   # Suppress errors due to CONVDTA in TopSpin
                warnings.simplefilter("ignore")
                self.datadir = os.path.abspath(in_file) # Get the file position
                if not os.path.isdir(self.datadir):
                    self.datadir = os.path.dirname(self.datadir)
                self.filename = os.path.basename(in_file).rsplit('.', 1)[0] # Get the filename
                # If filename is a directory, write things inside it
                if os.path.isdir('/'.join([self.datadir, self.filename])):
                    self.datadir = os.path.join(self.datadir, self.filename) # i.e. add filename to datadir
                # Read the dictionary
                dic, _ = ng.bruker.read_pdata(in_file)
                # Read the real and imaginary part, separately
                _, self.r = ng.bruker.read_pdata(in_file, bin_files=['1r'])
                _, self.i = ng.bruker.read_pdata(in_file, bin_files=['1i'])
                # Mount them together to make the complex spectrum
                self.S = self.r + 1j * self.i
                # Make acqus
                self.acqus = misc.makeacqus_1D(dic)
                self.acqus['BYTORDA'] = dic['acqus']['BYTORDA']
                self.acqus['DTYPA'] = dic['acqus']['DTYPA']
                self.ngdic = dic
                del dic
        # Set the group delay, if there is
        try:
            self.acqus['GRPDLY'] = int(self.ngdic['acqus']['GRPDLY'])
        except:
            self.acqus['GRPDLY'] = 0

        if procs is None:   # Make it
            if os.path.exists(os.path.join(self.datadir, f'{self.filename}.procs')):
                self.procs = self.read_procs()
            else:
                self.procs = {}
                proc_init_1D = (dict(wf0), None, 0.5, 0)
                for k, key in enumerate(proc_keys_1D):
                    self.procs[key] = proc_init_1D[k]         # Processing parameters
                self.procs['wf']['sw'] = round(self.acqus['SW'], 4)
                #   Then, phases
                self.procs['p0'] = 0
                self.procs['p1'] = 0
                self.procs['pv'] = round(self.acqus['o1p'], 2)
                #   Then, baseline
                self.procs['basl_c'] = None
                #   calibration
                self.procs['cal'] = 0

        else:   # Copy it
            self.procs = dict(procs)

        # Write the .procs file
        self.write_procs()
        
        # Calculate frequency and ppm scales
        self.freq = processing.make_scale(self.r.shape[0], dw=self.acqus['dw'])
        if self.acqus['SFO1'] < 0:      # Correct for negative gamma nuclei
            self.freq = self.freq[::-1]
        self.ppm = misc.freq2ppm(self.freq, B0=self.acqus['SFO1'], o1p=self.acqus['o1p'])

        # Initialize fit object
        self.F = fit.Voigt_Fit(self.ppm, self.S, self.acqus['t1'], self.acqus['SFO1'], self.acqus['o1p'], self.acqus['nuc'])


class Spectrum_2D:
    """
    Class: 2D NMR spectrum
    ---------
    Attributes:
    - datadir: str
        Path to the input file/dataset directory
    - filename: str
        Base of the name of the file, without extensions
    - fid: 2darray
        FID
    - acqus: dict
        Dictionary of acqusition parameters
    - ngdic: dict
        Created only if it is an experimental spectrum. Generated by nmrglue.bruker.read, contains all the information on the spectrometer and on the spectrum.
    - procs: dict
        Dictionary of processing parameters
    - eaeflag: int
        If FnMODE is Echo-Antiecho, keeps track of the manipulation of the data so to not repeat the same process twice
    - S: 2darray
        Complex (or hypercomplex, depending on FnMODE) spectrum
    - rr: 2darray
        Real part F2, real part F1
    - ii: 2darray
        Imaginary part F2, imaginary part F1
    - ir: 2darray
        Real part F2, imaginary part F1. Only exist if F1 is acquired in phase-sensitive mode
    - ri: 2darray
        Imaginary part F2, real part F1. Only exist if F1 is acquired in phase-sensitive mode
    - freq_f1: 1darray
        Frequency scale of the indirect dimension, in Hz
    - freq_f2: 1darray
        Frequency scale of the direct dimension, in Hz
    - ppm_f1: 1darray
        ppm scale of the indirect dimension
    - ppm_f2: 1darray
        ppm scale of the direct dimension
    - trf1: dict
        Projections of the indirect dimension, as 1darrays. Keys: 'ppm_f2' where they were taken
    - trf2: dict
        Projections of the direct dimension, as 1darrays. Keys: 'ppm_f1' where they were taken
    - Trf1: dict
        Projections of the indirect dimension, as pSpectrum_1D objects. Keys: 'ppm_f2' where they were taken
    - Trf2: dict
        Projections of the direct dimension, as pSpectrum_1D objects. Keys: 'ppm_f1' where they were taken
    - integrals: dict
        Dictionary where to save the regions and values of the integrals.
    """
    def __str__(self):
        """ Prints info on the object """
        doc = '-'*64
        doc += '\nSpectrum_2D object.\n'
        if 'ngdic' in self.__dict__.keys():
            doc += f'Read from "{self.datadir}"\n'
        else:
            doc += f'Simulated from "{self.filename}" in {self.datadir}\n'
        N = self.fid.shape
        doc += f'It is a {self.acqus["nuc1"]}-{self.acqus["nuc2"]} spectrum recorded over a \nsweep width of \n{self.acqus["SW1p"]} ppm centered at {self.acqus["o1p"]} ppm in F1, and\n{self.acqus["SW2p"]} ppm centered at {self.acqus["o2p"]} ppm in F2.\n'
        doc += f'The FID is {N[0]}x{N[1]} points long.\n'
        doc += '-'*64

        return doc

    def __init__(self, in_file, pv=False, isexp=True, is_pseudo=False):
        """
        Initialize the class. 
        -------
        Parameters:
        - in_file: str
            path to file to read, or to the folder of the spectrum
        - pv: bool
            True if you want to use pseudo-voigt lineshapes for simulation, False for Voigt
        - isexp: bool
            True if this is an experimental dataset, False if it is simulated
        - is_pseudo: bool
            True if it is a pseudo-2D. Legacy option
        """
        if isinstance(in_file, dict):   # Simulated data with already given acqus dictionary
            self.datadir = os.getcwd()      # Current directory
            self.filename = 'dict'  # Filename is just "dict"
        else:                           # You need to actually read a file
            self.datadir = os.path.abspath(in_file) # Get the file position
            if not os.path.isdir(self.datadir):
                self.datadir = os.path.dirname(self.datadir)
            self.filename = os.path.basename(in_file).rsplit('.', 1)[0] # Get the filename
            # If filename is a directory, write things inside it
            if os.path.isdir('/'.join([self.datadir, self.filename])) and isexp:
                self.datadir = os.path.join(self.datadir, self.filename) # i.e. add filename to datadir
        
        if isexp is False: # Simulate the data
            self.acqus = sim.load_sim_2D(in_file)   # Read the acqus dictionary from the file
            if is_pseudo:   # Tell not to do FT in f1
                self.acqus['FnMODE'] = 'No'
            else:   # Use States-TPPI as f1 acquisition because it is the simulation standard
                self.acqus['FnMODE'] = 'States-TPPI'
            self.fid = sim.sim_2D(in_file, pv=pv)   # Read the FID
        else:   # Read from nmrglue
            with warnings.catch_warnings():   # Suppress errors due to CONVDTA in TopSpin
                warnings.simplefilter("ignore")
                dic, data = ng.bruker.read(in_file, cplex=True) 
                self.ngdic = dic
                self.fid = data
                self.acqus = misc.makeacqus_2D(dic)
                self.fid = np.reshape(self.fid, (self.acqus['TD1'], -1))
                self.acqus['BYTORDA'] = dic['acqus']['BYTORDA']
                self.acqus['DTYPA'] = dic['acqus']['DTYPA']
                FnMODE_flag = dic['acqu2s']['FnMODE']       # Get f1 acquisition mode
                # List of possible modes
                FnMODEs = ['Undefined', 'QF', 'QSEC', 'TPPI', 'States', 'States-TPPI', 'Echo-Antiecho', 'QF-nofreq']
                self.acqus['FnMODE'] = FnMODEs[FnMODE_flag] # Add to acqus

                del dic
                del data
        # put a flag for shuffling EAE data
        if self.acqus['FnMODE'] == 'Echo-Antiecho':
            self.eaeflag = 1    # i.e. to be shuffled
        else:
            self.eaeflag = 0    # no need of shuffling

        # Get group delay points
        try:
            self.acqus['GRPDLY'] = int(self.ngdic['acqus']['GRPDLY'])
        except:
            self.acqus['GRPDLY'] = 0

        # initialize the procs dictionary with default values
        if os.path.exists(os.path.join(self.datadir, f'{self.filename}.procs')):
            self.procs = self.read_procs()
        else:
            proc_init_2D = (
                    [dict(wf0), dict(wf0)],     # window function
                    [None, None],   # zero-fill
                    [0.5, 0.5],     # fcor
                    [0,0]           # tdeff
                    )

            self.procs = {}
            for k, key in enumerate(proc_keys_1D):
                self.procs[key] = proc_init_2D[k]         # Processing parameters
            self.procs['wf'][0]['sw'] = round(self.acqus['SW1'], 4)
            self.procs['wf'][1]['sw'] = round(self.acqus['SW2'], 4)

            # Then, phases
            self.procs['p0_1'] = 0
            self.procs['p1_1'] = 0
            self.procs['pv_1'] = round(self.acqus['o1p'], 2)
            self.procs['p0_2'] = 0
            self.procs['p1_2'] = 0
            self.procs['pv_2'] = round(self.acqus['o2p'], 2)
            # Calibration
            self.procs['cal_1'] = 0
            self.procs['cal_2'] = 0
            # Write the .procs file
            self.write_procs()

        # Create empty dictionary where to save the projections
        self.trf1 = {}
        self.trf2 = {}
        self.Trf1 = {}
        self.Trf2 = {}

    def convdta(self, scaling=1):
        """
        Calls processing.convdta to compensate for the group delay. 
        It does not always work, depends on TopSpin version and planets alignment.
        --------
        Parameters:
        - scaling: float
            Scaling factor for processingconvdta.
        """
        self.fid = processing.convdta(self.fid, self.acqus['GRPDLY'], scaling)

    def eae(self):
        """
        Calls processing.EAE to shuffle the data and make a States-like FID.
        Sets self.eaeflag to 0.
        """
        if self.eaeflag:    # Do it only if it was not done before
            self.fid = processing.eae(self.fid)
            self.eaeflag = 0

    def add_noise(self, s_n=1):
        """
        Adds noise to the FID, using the function sim.noisegen.
        ------------------
        Parameters:
        - s_n: float
            Standard deviation of the noise
        """
        self.fid += sim.noisegen(self.fid.shape, self.acqus['o2'], self.acqus['t2'], s_n=s_n)

    def scan(self, ns=1, s_n=1):
        """
        Simulates the acquisition of ns scans, by adding a different realization of noise at each iteration.
        The function is supposed to start with the FID without noise at all. If not, the results will be biased.
        --------------------
        Parameters:
        - ns: int
            Number of scans to accumulate
        - s_n: float
            Standard deviation of the noise
        """
        clean_fid = np.copy(self.fid)   # Save a shallow copy of the FID without noise
        for k in range(ns):
            # Each scan is: clean FID + noise
            self.fid += clean_fid + sim.noisegen(self.fid.shape, self.acqus['o2'], self.acqus['t2'], s_n=s_n)

    def xf2(self):
        """
        Process only the direct dimension.
        Calls processing.fp using procs[keys][1]
        The result is stored in self.S, then self.rr and self.ii are written.
        freq_f1 and ppm_f1 are assigned with the indexes of the transients.
        """
        # Make empty self.S according to the zero-filling option in F2
        if self.procs['zf'][1] is None:
            self.S = np.zeros_like(self.fid).astype(self.fid.dtype)
        else:
            self.S = np.zeros((self.fid.shape[0], self.procs['zf'][1])).astype(self.fid.dtype)

        # Apply 1D processing on F2 only, row-by-row
        for k in range(self.fid.shape[0]):
            self.S[k] = processing.fp(self.fid[k], wf=self.procs['wf'][1], zf=self.procs['zf'][1], fcor=self.procs['fcor'][1], tdeff=self.procs['tdeff'][1])

        # Make frequency and ppm scales
        self.freq_f2 = processing.make_scale(self.S.shape[1], dw=self.acqus['dw2'])
        if self.acqus['SFO2'] < 0:  # Correct for negative gamma nuclei
            self.freq_f2 = self.freq_f2[::-1]
        self.ppm_f2 = misc.freq2ppm(self.freq_f2, B0=self.acqus['SFO2'], o1p=self.acqus['o2p']) 

        # Correct also the spectrum for negative gamma nuclei
        if self.acqus['SFO2'] < 0:
            self.S = self.S[:,::-1]

        # Unpack 
        self.rr = self.S.real
        self.ii = self.S.imag

        # Use the spectrum index as fake scale
        self.freq_f1 = np.arange(self.S.shape[0])
        self.ppm_f1 = np.arange(self.S.shape[0])

    def xf1(self):
        """
        Process only the indirect dimension. 
        Transposes the spectrum in hypermode or normally if FnMODE != QF, 
        then calls for processing.fp using self.procs[keys][0], finally transposes it back.
        The result is stored in self.S, then self.rr and self.ii are written.
        freq_f1 and ppm_f1 are assigned with the indexes of the transients.
        """
        # Transpose
        if self.acqus['FnMODE'] in ['QF', 'No', 'QF-nofreq']:    # Just complex spectrum
            self.fid = self.fid.T
        else:   # Hypercomplex spectrum
            self.fid = processing.tp_hyper(self.fid)

        # Make empty self.S according to the zero-filling option in f1
        if self.procs['zf'][0] is None:
            self.S = np.zeros_like(self.fid).astype(self.fid.dtype)
        else:
            self.S = np.zeros((self.fid.shape[0], self.procs['zf'][0])).astype(self.fid.dtype)

        # Apply 1D processing on F1 only, row by row because fid is transposed
        for k in range(self.fid.shape[0]):
            self.S[k] = processing.fp(self.fid[k], wf=self.procs['wf'][0], zf=self.procs['zf'][0], fcor=self.procs['fcor'][0], tdeff=self.procs['tdeff'][0])

        # Transpose it back
        if self.acqus['FnMODE'] in ['QF', 'No', 'QF-nofreq']:
            self.fid = self.fid.T
            self.S = self.S.T
        else:
            self.fid = processing.tp_hyper(self.fid)
            self.S = processing.tp_hyper(self.S)
        # Unpack S
        self.rr = np.copy(self.S.real)
        self.ii = np.copy(self.S.imag)

        # Make frequency and ppm scales
        self.freq_f1 = processing.make_scale(self.S.shape[0], dw=self.acqus['dw1'])
        if self.acqus['SFO1'] < 0:  # Correct for negative gamma nuclei
            self.freq_f1 = self.freq_f1[::-1]
        self.ppm_f1 = misc.freq2ppm(self.freq_f1, B0=self.acqus['SFO1'], o1p=self.acqus['o1p'])

        # Use evolution index as fake scale
        self.freq_f2 = np.arange(self.S.shape[1])
        self.ppm_f2 = np.arange(self.S.shape[1])

    def process(self, interactive=False, **int_kwargs):
        """
        Performs the full processing of the FID on both dimensions. The parameters are read from self.procs.
        If FnMODE is Echo-Antiecho and you did not call self.eae before, the FID is converted to States with processing.EAE before to start.
        If interactive is True, calls processing.interactive_xfb with int_kwargs, else calls processing.xfb.
        The complex/hypercomplex spectrum is stored in self.S, then unpacked into self.rr, self.ri, self.ir, self.ii.
        If FnMODE is Echo-Antiecho, a phase correction of -90 degrees is applied on the indirect dimension.
        --------
        Parameters:
        - interactive: bool
            True if you want to open the interactive panel, False to read the parameters from self.procs.
        - int_kwargs: keyworded arguments
            Additional parameters for processing.interactive_xfb, if interactive=True.
        """
        # If Echo-Antiecho, pre-process the FID to get the correct spectral arrangement
        self.eae()

        # Call for the interacrive processing
        if interactive is True:
            self.S, self.procs = processing.interactive_xfb(self.fid, self.acqus, self.procs, **int_kwargs)
        else:
            self.S = processing.xfb(self.fid, wf=self.procs['wf'], zf=self.procs['zf'], fcor=self.procs['fcor'], tdeff=self.procs['tdeff'], FnMODE=self.acqus['FnMODE'], u=False)

        # For EAE, correct the 90° phase shift in F1
        if self.acqus['FnMODE'] == 'Echo-Antiecho':
            self.S = processing.tp_hyper(self.S)
            self.S = processing.ps(self.S, p0=-90)[0]
            self.S = processing.tp_hyper(self.S)

        # Correct for negative gamma nuclei
        if self.acqus['SFO2'] < 0:
            self.S = self.S[:,::-1]
        if self.acqus['SFO1'] < 0:
            # Reversing the spectrum in the indirect dimension causes a 90° dephasing
            self.S = self.S[::-1,:]
            if self.acqus['FnMODE'] in ['QF', 'QF-nofreq']:
                self.S = self.S.T
            else:
                self.S = processing.tp_hyper(self.S)
            self.S = processing.ps(self.S, p0=-90)[0]   #...that has to be corrected
            if self.acqus['FnMODE'] in ['QF', 'QF-nofreq']:
                self.S = self.S.T
            else:
                self.S = processing.tp_hyper(self.S)

        # Unpack according to FnMODE
        if self.acqus['FnMODE'] in ['QF', 'No', 'QF-nofreq']:
            self.rr = self.S.real
            self.ii = self.S.imag
        else:
            rr, ir, ri, ii = processing.unpack_2D(self.S)
            self.rr = rr
            self.ri = ri
            self.ir = ir
            self.ii = ii


        # Calculates the frequency and ppm scales
        self.freq_f1 = processing.make_scale(self.rr.shape[0], dw=self.acqus['dw1'])
        if self.acqus['SFO1'] < 0:  # Correct for negative gamma nuclei
            self.freq_f1 = self.freq_f1[::-1]
        self.ppm_f1 = misc.freq2ppm(self.freq_f1, B0=self.acqus['SFO1'], o1p=self.acqus['o1p'])
        self.freq_f2 = processing.make_scale(self.rr.shape[1], dw=self.acqus['dw2'])
        if self.acqus['SFO2'] < 0:  # Correct for negative gamma nuclei
            self.freq_f2 = self.freq_f2[::-1]
        self.ppm_f2 = misc.freq2ppm(self.freq_f2, B0=self.acqus['SFO2'], o1p=self.acqus['o2p']) 

        # Apply phase correction according to procs
        self.adjph(p02=self.procs['p0_2'], p12=self.procs['p1_2'], pv2=self.procs['pv_2'], p01=self.procs['p0_1'], p11=self.procs['p1_1'], pv1=self.procs['pv_1'], update=False)

        # Apply calibration according to procs
        self.cal([self.procs['cal_1'], self.procs['cal_2']], update=False)


    def inv_process(self):
        """
        Performs the inverse processing of the spectrum according to the given parameters.
        Overwrites the S attribute!!
        Calls inv_xfb.
        """

        # For EAE, correct the 90° phase shift in F1
        if self.acqus['FnMODE'] == 'Echo-Antiecho':
            self.S = processing.tp_hyper(self.S)
            self.S = processing.ps(self.S, p0=90)[0]
            self.S = processing.tp_hyper(self.S)

        # Correction for negative gamma nuclei
        if self.acqus['SFO2'] < 0:
            self.S = self.S[:,::-1]
        if self.acqus['SFO1'] < 0:
            self.S = self.S[::-1,:]
            self.S = processing.tp_hyper(self.S)
            self.S = processing.ps(self.S, p0=-90)[0]
            self.S = processing.tp_hyper(self.S)

        # Compute
        self.S = processing.inv_xfb(self.S, wf=self.procs['wf'], size=(self.acqus['TD1'], self.acqus['TD2']), fcor=self.procs['fcor'], FnMODE=self.acqus['FnMODE'])


    def mc(self):
        """
        Computes the magnitude of the spectrum on self.S.
        Then, updates rr, ri, ir, ii.
        """
        self.S = (self.S.real**2 + self.S.imag**2 )**0.5
        if self.acqus['FnMODE'] in ['QF', 'QF-nofreq']:
            self.rr = self.S.real
            self.ii = self.S.imag
        else:
            rr, ir, ri, ii = processing.unpack_2D(self.S)
            self.rr = rr
            self.ri = ri
            self.ir = ir
            self.ii = ii

    def adjph(self, p01=None, p11=None, pv1=None, p02=None, p12=None, pv2=None, update=True):
        """
        Adjusts the phases of the spectrum according to the given parameters, or interactively if they are left as default.
        The non-interactive workflow is to apply processing.ps on F2, transpose according to FnMODE, apply processing.ps on F1, transpose back.
        If FnMODE is 'No', the phase correction is applied only on F2, as it should be done in a pseudo-2D experiment.
        Once self.S was updated and unpacked, the phase values are added to the procs dictionary to keep track of multiple phase adjustments.
        -------
        Parameters:
        - p01: float or None
            0-th order phase correction /° of the indirect dimension
        - p11: float or None
            1-st order phase correction /° of the indirect dimension
        - pv1: float or None
            1-st order pivot /ppm of the indirect dimension
        - p02: float or None
            0-th order phase correction /° of the direct dimension
        - p12: float or None
            1-st order phase correction /° of the direct dimension
        - pv2: float or None
            1-st order pivot /ppm of the direct dimension
        - update: bool
            Choose if to update the procs dictionary or not
        """
        interactive = True      # by default
        # Set pivot to carrier if not specified
        if pv1 is None:
            pv1 = self.acqus['o1p']
        if pv2 is None:
            pv2 = self.acqus['o2p']
        ph = [p01, p11, p02, p12]   # for easier handling
        for p in ph:
            # If a phase is specified, interactive is set to False...
            if p is not None:
                interactive = False
        if interactive is False:
            # ... and the not-set phases are put to 0
            for i, p in enumerate(ph):
                if p is None:
                    ph[i] = 0
            # Adjust the phases according to the given values

            # Phase F2
            self.S, values_f2 = processing.ps(self.S, self.ppm_f2, p0=ph[2], p1=ph[3], pivot=pv2)
            # Phase F1
            if self.acqus['FnMODE'] in ['No', 'QF-nofreq']:    # Skip it
                pass
            # else, transpose according to FnMODE, phase, transpose back
            elif self.acqus['FnMODE'] == 'QF':
                self.S = self.S.T
                self.S, values_f1 = processing.ps(self.S, self.ppm_f1, p0=ph[0], p1=ph[1], pivot=pv1)
                self.S = self.S.T
            else:
                self.S = processing.tp_hyper(self.S)
                self.S, values_f1 = processing.ps(self.S, self.ppm_f1, p0=ph[0], p1=ph[1], pivot=pv1)
                self.S = processing.tp_hyper(self.S)

        else:
            # Call interactive phase correction
            if self.acqus['FnMODE'] in ['QF', 'No', 'QF-nofreq']: 
                self.S, values_f1, values_f2 = processing.interactive_phase_2D(self.ppm_f1, self.ppm_f2, self.S, False)
            else:
                self.S, values_f1, values_f2 = processing.interactive_phase_2D(self.ppm_f1, self.ppm_f2, self.S)

        # Unpack the phased spectrum
        if self.acqus['FnMODE'] in ['QF', 'No', 'QF-nofreq']:
            self.rr = self.S.real
            self.ii = self.S.imag
        else:
            rr, ir, ri, ii = processing.unpack_2D(self.S)
            self.rr = rr
            self.ri = ri
            self.ir = ir
            self.ii = ii

        # update the procs dictionary
        if update:
            self.procs['p0_2'] += round(values_f2[0], 2)
            self.procs['p1_2'] += round(values_f2[1], 2)
            if values_f2[2] is not None:
                self.procs['pv_2'] = round(values_f2[2], 5)
            self.procs['p0_1'] += round(values_f1[0], 2)
            self.procs['p1_1'] += round(values_f1[1], 2)
            if values_f1[2] is not None:
                self.procs['pv_1'] = round(values_f1[2], 5)
        
        # Update the .procs file
        self.write_procs()

    def pknl(self):
        """
        Reverses the effect of the digital filter by applying a first order phase correction.
        To be called after having processed the data by 'self.process()'
        """
        self.adjph(p12=-360 * self.acqus['GRPDLY'], update=False)


    def qfil(self, which=None, u=None, s=None):
        """ 
        Gaussian filter to suppress signals.
        Tries to read self.procs['qfil'], which is
            { 'u': u, 's': s }
        Otherwise, these are set interactively by processing.interactive_qfil and then added to self.procs.
        Calls processing.qfil
        ---------
        Parameters:
        - which: int or None
            Index of the F2 trace to be used for interactive_qfil. If None, a suitable trace can be selected using misc.select_traces.
        - u: float
            Position /ppm
        - s: float
            Width (standard deviation) /ppm
        """
        if 'qfil' not in self.procs.keys(): # Then add it
            self.procs['qfil'] = {'u': u, 's': s}

        for key, value in self.procs['qfil'].items():
            if value is None:   # missing value --> call for interaction
                if which is None:   # select a spectrum to be used
                    which_list = misc.select_traces(self.ppm_f1, self.ppm_f2, self.rr, Neg=False, grid=False)
                    which, _ = misc.ppmfind(self.ppm_f1, which_list[0][1])
                # Now get the values
                self.procs['qfil']['u'], self.procs['qfil']['s'] = processing.interactive_qfil(self.ppm_f2, self.rr[which])
                break

        # Apply the filter
        self.S = processing.qfil(self.ppm_f2, self.S, self.procs['qfil']['u'], self.procs['qfil']['s'])
        # Unpack according to procs
        if self.acqus['FnMODE'] in ['QF', 'No', 'QF-nofreq']:
            self.rr = self.S.real
            self.ii = self.S.imag
        else:
            self.rr, self.ir, self.ri, self.ii = processing.unpack_2D(self.S)

    def cal(self, offset=[None,None], isHz=False, update=True):
        """
        Calibration of the ppm and frequency scales according to a given value, or interactively. In this latter case, a reference peak must be chosen.
        Calls processing.calibration
        --------
        Parameters:
        - offset: tuple
            (scale shift F1, scale shift F2)
        - isHz: tuple of bool
            True if offset is in frequency units, False if offset is in ppm
        - update: bool
            Choose if to update the procs dictionary or not
        """
        def _calibrate(ppm, trace, SFO1, o1p):
            """ Main function that calls the real calibration """
            offppm = processing.calibration(ppm, trace)
            offhz = misc.ppm2freq(offppm, SFO1, o1p)
            return offppm, offhz

        # Get the missing entries
        if offset[0] is None or offset[1] is None:  # Select the reference traces
            coord = misc.select_traces(self.ppm_f1, self.ppm_f2, self.rr, Neg=False, grid=False)
            ix, iy = coord[0][0], coord[0][1]   # Position of the first crosshair
            # F2 reference spectrum
            X = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, iy, column=False)
            # F1 reference spectrum
            Y = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, ix, column=True)

        if offset[1] is None:   # Get it
            ppm_f2 = np.copy(self.ppm_f2)
            offp2, offh2 = _calibrate(ppm_f2, X, self.acqus['SFO2'], self.acqus['o2p'])
        else:   # Calculate offh2 from offp2 or viceversa
            if isHz:    # offp2 is missing
                offh2 = offset[1]
                offp2 = misc.freq2ppm(offh2, self.acqus['SFO2'], self.acqus['o2p']) 
            else:       # offh2 is missing
                offp2 = offset[1]
                offh2 = misc.ppm2freq(offp2, self.acqus['SFO2'], self.acqus['o2p']) 
            
        if offset[0] is None:   # Get it
            ppm_f1 = np.copy(self.ppm_f1)
            offp1, offh1 = _calibrate(ppm_f1, Y, self.acqus['SFO1'], self.acqus['o1p'])
        else:   # Calculate offh1 from offp1 or viceversa
            if isHz:    # offp1 is missing
                offh1 = offset[0]
                offp1 = misc.freq2ppm(offh1, self.acqus['SFO1'], self.acqus['o1p']) 
            else:       # offh1 is missing
                offp1 = offset[0]
                offh1 = misc.ppm2freq(offp1, self.acqus['SFO1'], self.acqus['o1p']) 

        # Apply the calibration
        self.freq_f2 += offh2
        self.ppm_f2 += offp2
        self.freq_f1 += offh1
        self.ppm_f1 += offp1
        # Move the offsets to the center of the SW
        self.acqus['o1p'] += offp1
        self.acqus['o1'] += offh1
        self.acqus['o2p'] += offp2
        self.acqus['o2'] += offh2
        # Update the procs dictionary
        if update:  
            self.procs['cal_1'] += offp1
            self.procs['cal_2'] += offp2
        # Update the .procs file
        self.write_procs()


    def calf2(self, value=None, isHz=False):
        """
        Calibrates the ppm and frequency scale of the direct dimension according to a given value, or interactively.
        Calls self.cal on F2 only
        -------
        Parameters:
        - value: float or None
            scale shift value
        - isHz: bool
            True if offset is in frequency units, False if offset is in ppm
        """
        offset = [0, value]
        self.cal(offset, isHz)

    def calf1(self, value=None, isHz=False):
        """
        Calibrates the ppm and frequency scale of the indirect dimension according to a given value, or interactively.
        Calls self.cal on F1 only.
        -------
        Parameters:
        - value: float or None
            scale shift value
        - isHz: bool
            True if offset is in frequency units, False if offset is in ppm
        """
        offset = [value, 0]
        self.cal(offset, isHz)

    def write_acqus(self, other_dir=None):
        """
        Write the acqus dictionary in a file named "filename.acqus".
        Calls misc.write_acqus_1D
        --------
        Parameters:
        - other_dir: str or None
            Different location for the acqus dictionary to write into. If None, self.datadir is used instead.
        """
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.acqus')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.acqus')
        misc.write_acqus_2D(self.acqus, path=path)

    def write_procs(self, other_dir=None):
        """
        Writes the actual procs dictionary in a file named "filename.procs" in the same directory of the input file.
        ---------
        Parameters:
        - other_dir: str or None
            Different location for the procs dictionary to write into. If None, self.datadir is used instead. W! Do not put the trailing slash!
        """
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.procs')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.procs')
        with open(path, 'w') as f:
            f.write(f'{self.procs}')

    def read_procs(self, other_dir=None):
        """
        Reads the procs dictionary from a file named "filename.procs" in the same directory of the input file.
        ---------
        Parameters:
        - other_dir: str or None
            Different location for the procs dictionary to look into. If None, self.datadir is used instead. W! Do not put the trailing slash!
        ---------
        Returns:
        - procs: dict
            Dictionary of processing parameters
        """
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.procs')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.procs')
        with open(path, 'r') as f:
            procs = eval(f.read().replace('array', 'np.array'))
        # Check if it was read correctly
        if isinstance(procs, dict):
            return procs
        else:
            raise ValueError(f'{self.filename}.procs cannot be interpreted as a dictionary')

    @staticmethod
    def write_ser(ser, acqus, path=None):
        """
        Writes a real/complex array in binary format.
        Calls misc.write_ser. Be sure that acqus contains the BYTORDA and DTYPA keys.
        See misc.write_ser to understand the meaning of these values.
        --------
        Parameters:
        - ser: ndarray
            Array that you want to convert in binary format.
        - acqus: dict
            Dictionary of acquisition parameters. It must contain BYTORDA and DTYPA.
        - path: str 
            Path where to save the binary file.
        """
        if path is None:
            raise NameError('You must specify a filename!')
        misc.write_ser(ser, path, acqus['BYTORDA'], acqus['DTYPA'])

    def projf1(self, a, b=None):
        """
        Calculates the sum trace of the indirect dimension, from a ppm to b ppm in F2.
        Store the trace in the dictionary trf1 and as 1D spectrum in Trf1. The key is 'a' or 'a:b'
        Calls misc.get_trace on self.rr with column=True
        -------
        Parameters:
        - a: float
            ppm F2 value where to extract the trace.
        - b: float or None.
            If it is None, extract the trace in a. Else, sum from a to b in F2.
        """
        # make dictionary label
        if b is None:
            label = f'{a:.2f}'
        else:
            label = f'{a:.2f}:{b:.2f}'
        # Compute the trace
        f1 = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, a, b, column=True)
        # Store it 
        self.trf1[label] = f1
        self.Trf1[label] = pSpectrum_1D(f1, acqus=misc.split_acqus_2D(self.acqus)[0], procs=misc.split_procs_2D(self.procs)[0], istrace=True, filename=f'T1_{label}')

    def projf2(self, a, b=None):
        """
        Calculates the sum trace of the direct dimension, from a ppm to b ppm in F1.
        Store the trace in the dictionary trf2 and as 1D spectrum in Trf2. The key is 'a' or 'a:b'
        Calls misc.get_trace on self.rr with column=False
        -------
        Parameters:
        - a: float
            ppm F1 value where to extract the trace.
        - b: float or None.
            If it is None, extract the trace in a. Else, sum from a to b in F1.
        """
        # make dictionary label
        if b is None:
            label = f'{a:.2f}'
        else:
            label = f'{a:.2f}:{b:.2f}'
        # Compute the trace
        f2 = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, a, b, column=False)
        # Store it
        self.trf2[label] = f2
        self.Trf2[label] = pSpectrum_1D(f2, acqus=misc.split_acqus_2D(self.acqus)[1], procs=misc.split_procs_2D(self.procs)[1], istrace=True, filename=f'T2_{label}')

    def integrate(self, **kwargs):
        """
        Integrates the spectrum with a dedicated GUI.
        Calls fit.integrate_2D 
        --------
        Parameters:
        - kwargs: keyworded arguments
            Additional parameters for fit.integrate_2D
        """
        self.integrals = fit.integrate_2D(self.ppm_f1, self.ppm_f2, self.rr, self.acqus['SFO1'], self.acqus['SFO2'], **kwargs)

    def write_integrals(self, other_dir=None):
        """
        Write the integrals in a file named "{self.filename}.int".
        -------
        Parameters:
        - other_dir: str or None
            Different location for the integrals file to write into. If None, self.datadir is used instead.
            
        """
        # Detect the file position
        if other_dir:
            path = os.path.join(other_dir, f'{self.filename}.int')
        else:
            path = os.path.join(self.datadir, f'{self.filename}.int')

        f = open(path, 'w')
        f.write('{:12}\t{:12}\t\t{:20}\n'.format('ppm F2', 'ppm F1', 'Value'))
        f.write('-'*60+'\n')
        for key, value in self.integrals.items():
            ppm2, ppm1 = tuple(key.split(':'))
            f.write('{:12}\t{:12}\t\t{:20.5e}\n'.format(ppm2, ppm1, value))
        f.close()

    def plot(self, Neg=True, lvl0=0.2):
        """
        Plots the real part of the spectrum. Use the mouse scroll to adjust the contour starting level.
        -------
        Parameters:
        - Neg: bool
            Plot (True) or not (False) the negative contours.
        - lvl0: float
            Starting contour value with respect to the maximum of the spectrum
        """
        class FakeSpanSelector:
            """ 
            Line selector that stores (x1, y1) when you press the mouse left button,
            and (x2, y2) when you release it. Then, draws the triangle 
            [(x1, y1), (x2, y1), (x2, y2)]
            and writes the distance (x2-x1), (y2-y1) in ppm and in Hz.
            """
            def __init__(self, ppm_f2, ppm_f1, acqus):
                """
                Set initial values for x1, x2, y1, y2
                -----------
                Parameters:
                - ppm_f2: 1darray
                    x-axis scale, in ppm
                - ppm_f1: 1darray
                    y-axis scale, in ppm
                - acqus: dict
                    Dictionary of acquisition parameters of the spectrum. Must contain SFO1 and SFO2.
                """
                self.x1 = ppm_f2[0]
                self.x2 = ppm_f2[-1]
                self.y1 = ppm_f1[0]
                self.y2 = ppm_f1[-1]
                self.acqus = acqus

            def draw_lines(self): #onselect(self, *null):
                """ Moves the tracker """
                # Makes the lines visible
                dots.set_visible(True)
                lx.set_visible(True)
                ly.set_visible(True)

                # Updates the values for the lines
                dots.set_data((self.x1, self.x2), (self.y1, self.y2))
                lx.set_data((self.x1, self.x2), (self.y1, self.y1))
                ly.set_data((self.x2, self.x2), (self.y1, self.y2))

                # Compute x-distance
                xd_ppm = np.abs(self.x1-self.x2)
                #   convert it in Hz
                xd_hz = np.abs(misc.ppm2freq(xd_ppm, self.acqus['SFO2']))
                # Compute y-distance
                yd_ppm = np.abs(self.y1-self.y2)
                #   convert it in Hz
                yd_hz = np.abs(misc.ppm2freq(yd_ppm, self.acqus['SFO1']))

                # Update the measure text
                text = '\n'.join([
                    r'$\Delta$'+f'F2: {xd_ppm:12.3f} ppm | {xd_hz:12.3f} Hz',
                    r'$\Delta$'+f'F1: {yd_ppm:12.3f} ppm | {yd_hz:12.3f} Hz',
                    ])
                text_measure.set_text(text)
                plt.draw()

            def onclick(self, event):
                """
                If left click, saves x1 and y1 in the click positions.
                Right click clears the selection
                """
                if not event.inaxes:    # Do things only if click is inside the panel
                    return
                if event.button == 1 or event.button == 3:
                    # Erase the text
                    text_measure.set_text(_text)
                    # Make the bars invisible
                    lx.set_visible(False)
                    ly.set_visible(False)
                if event.button == 1:
                    # Store position of first point
                    self.x1 = event.xdata
                    self.y1 = event.ydata
                    # Draw it
                    dots.set_data((event.xdata,), (event.ydata,))
                elif event.button == 3:
                    dots.set_visible(False)
                plt.draw()

            def onrelease(self, event):
                """ If left click, draws the position of the second point, then calls draw_lines. """
                if not event.inaxes:    # Do things only if click is inside the panel
                    return
                if event.button == 1:
                    # Store position of the second point
                    self.x2 = event.xdata
                    self.y2 = event.ydata
                    # Draw triangle and stuff
                    self.draw_lines()

        warnings.filterwarnings("ignore", message="No contour levels were found within the data range.")
        # Functions connected to the sliders
        def increase_zoom(event):
            nonlocal lvlstep
            lvlstep *= 2

        def decrease_zoom(event):
            nonlocal lvlstep
            lvlstep /= 2

        def on_scroll(event):
            nonlocal lvl, cnt
            if Neg:
                nonlocal Ncnt

            # Get window limits to reset them after redrawing
            act_xlim = ax.get_xlim()
            act_ylim = ax.get_ylim()
                
            # Update level threshold
            if event.button == 'up':
                lvl += lvlstep 
            elif event.button == 'down':
                lvl += -lvlstep
            # Correct if lvl goes out of bounds
            if lvl <= 0:
                lvl = 1e-6
            elif lvl > 1:
                lvl = 1

            # Redraw the contours
            if Neg:
                cnt, Ncnt = figures.redraw_contours(ax, self.ppm_f2, self.ppm_f1, S, lvl=lvl, cnt=cnt, Neg=Neg, Ncnt=Ncnt, lw=0.5, cmap=[cmaps[0], cmaps[1]])
            else:
                cnt, _ = figures.redraw_contours(ax, self.ppm_f2, self.ppm_f1, S, lvl=lvl, cnt=cnt, Neg=Neg, Ncnt=None, lw=0.5, cmap=[cmaps[0], cmaps[1]])

            # Set the window limits as it was before
            misc.pretty_scale(ax, act_xlim, axis='x', n_major_ticks=n_xticks)
            misc.pretty_scale(ax, act_ylim, axis='y', n_major_ticks=n_yticks)
            # Correct the labels of the axes
            ax.set_xlabel(X_label)
            ax.set_ylabel(Y_label)
            # Correct the fontsize
            misc.set_fontsizes(ax, 14)
            # Update level threshold value
            lvl_text.set_text(f'{lvl:.5g}')
            fig.canvas.draw()

        # copy stuff
        S = np.copy(self.rr)
        n_xticks, n_yticks = 10, 10

        # Initialize the custom span selector
        span = FakeSpanSelector(self.ppm_f2, self.ppm_f1, self.acqus)

        # Compute labels for the axes
        X_label = r'$\delta\ $'+misc.nuc_format(self.acqus['nuc2'])+' /ppm'
        Y_label = r'$\delta\ $'+misc.nuc_format(self.acqus['nuc1'])+' /ppm'

        # Cmaps for positive and negative contours
        cmaps = ['Blues_r', 'Reds_r']

        # flags for the activation of scroll zoom
        lvlstep = 0.02

        # Make the figure
        fig = plt.figure(f'{self.filename}')
        fig.set_size_inches(15,8)
        plt.subplots_adjust(left = 0.10, bottom=0.10, right=0.90, top=0.95)
        ax = fig.add_subplot(1,1,1)

        # Default values for initial plot
        contour_num = 16
        contour_factor = 1.40
        lvl = lvl0

        # Placeholder for levels threshold text
        lvl_text = ax.text(0.925, 0.60, f'{lvl:.5g}', ha='left', va='center', transform=fig.transFigure, fontsize=12)

        # Plot the spectrum
        #   positive contours
        cnt = figures.ax2D(ax, self.ppm_f2, self.ppm_f1, S, lvl=lvl, cmap=cmaps[0])
        if Neg:
            # Negative contours
            Ncnt = figures.ax2D(ax, self.ppm_f2, self.ppm_f1, -S, lvl=lvl, cmap=cmaps[1])

        # Placeholders for tracker
        dots, = ax.plot((self.ppm_f2[0], self.ppm_f2[-1]), (self.ppm_f1[0], self.ppm_f1[-1]), '--.', c='r',
                        lw=0.5, markersize=5, visible=False)
        lx, = ax.plot((self.ppm_f2[0], self.ppm_f2[-1]), (self.ppm_f1[0], self.ppm_f1[0]), '-', c='r',
                        lw=0.5, visible=False)
        ly, = ax.plot((self.ppm_f2[-1], self.ppm_f2[-1]), (self.ppm_f1[0], self.ppm_f1[-1]), '-', c='r',
                        lw=0.5, visible=False)

        # Distance measurement text placeholder
        text_measure = plt.text(0.75, 0.015, '', ha='left', va='bottom', transform=fig.transFigure, fontsize=12, color='r')
        _text = '\n'.join([
            r'$\Delta$'+f'F2: {0:12.3f} ppm | {0:12.3f} Hz',
            r'$\Delta$'+f'F1: {0:12.3f} ppm | {0:12.3f} Hz',
            ])
        text_measure.set_text(_text)

        # Make pretty scales
        misc.pretty_scale(ax, (max(self.ppm_f2), min(self.ppm_f2)), axis='x', n_major_ticks=n_xticks)
        misc.pretty_scale(ax, (max(self.ppm_f1), min(self.ppm_f1)), axis='y', n_major_ticks=n_yticks)
        ax.set_xlabel(X_label)
        ax.set_ylabel(Y_label)

        # Create buttons
        # define boxes for buttons
        iz_box = plt.axes([0.925, 0.80, 0.05, 0.05])
        dz_box = plt.axes([0.925, 0.75, 0.05, 0.05])
        iz_button = Button(iz_box, label=r'$\uparrow$')
        dz_button = Button(dz_box, label=r'$\downarrow$')
        misc.set_fontsizes(ax, 14)

        # Connect the widgets to slots
        scroll = fig.canvas.mpl_connect('scroll_event', on_scroll) 
        fig.canvas.mpl_connect('button_press_event', span.onclick)  
        fig.canvas.mpl_connect('button_release_event', span.onrelease)
            
        iz_button.on_clicked(increase_zoom)
        dz_button.on_clicked(decrease_zoom)

        # Crosshair for visualization
        cursor = Cursor(ax, useblit=True, c='tab:red', lw=0.8)

        plt.show()
        plt.close()


    def to_wav(self, filename=None, cutoff=None, rate=44100):
        """
        Converts the FID in an audio file by using misc.data2wav.
        ---------
        Parameters:
        - filename: str
            Path where to save the file. If None, self.filename is used
        - cutoff: float
            Clipping limits for the FID
        - rate: int
            Sampling rate in samples/sec
        """
        if filename is None:
            cwd = os.getcwd()
            filename = os.path.join(cwd, self.filename)
        # Make a shallow copy of the FID
        data = np.copy(self.fid)
        misc.data2wav(data, filename, cutoff, rate)


class pSpectrum_2D(Spectrum_2D):
    """
    Subclass of Spectrum_2D that allows to handle processed 2D NMR spectra.
    Reads the processed spectrum from Bruker.
    ---------
    Attributes:
    - datadir: str
        Path to the input file/dataset directory
    - filename: str
        Base of the name of the file, without extensions
    - acqus: dict
        Dictionary of acqusition parameters
    - ngdic: dict
        Generated by nmrglue.bruker.read, contains all the information on the spectrometer and on the spectrum.
    - procs: dict
        Dictionary of processing parameters
    - S: 2darray
        Complex (or hypercomplex, depending on FnMODE) spectrum
    - rr: 2darray
        Real part F2, real part F1
    - ii: 2darray
        Imaginary part F2, imaginary part F1
    - ir: 2darray
        Real part F2, imaginary part F1. Only exist if F1 is acquired in phase-sensitive mode
    - ri: 2darray
        Imaginary part F2, real part F1. Only exist if F1 is acquired in phase-sensitive mode
    - freq_f1: 1darray
        Frequency scale of the indirect dimension, in Hz
    - freq_f2: 1darray
        Frequency scale of the direct dimension, in Hz
    - ppm_f1: 1darray
        ppm scale of the indirect dimension
    - ppm_f2: 1darray
        ppm scale of the direct dimension
    - trf1: dict
        Projections of the indirect dimension, as 1darrays. Keys: 'ppm_f2' where they were taken
    - trf2: dict
        Projections of the direct dimension, as 1darrays. Keys: 'ppm_f1' where they were taken
    - Trf1: dict
        Projections of the indirect dimension, as pSpectrum_1D objects. Keys: 'ppm_f2' where they were taken
    - Trf2: dict
        Projections of the direct dimension, as pSpectrum_1D objects. Keys: 'ppm_f1' where they were taken
    - integrals: dict
        Dictionary where to save the regions and values of the integrals.
    """

    def __init__(self, in_file):
        """
        Initialize the class. 
        -------
        Parameters:
        - in_file: str
            Path to the spectrum. Here, the 'pdata/#' folder must be specified.
        """
        self.datadir = os.path.abspath(in_file) # Get the file position
        if not os.path.isdir(self.datadir):
            self.datadir = os.path.dirname(self.datadir)
        self.filename = os.path.basename(in_file).rsplit('.', 1)[0] # Get the filename
        # If filename is a directory, write things inside it
        if os.path.isdir('/'.join([self.datadir, self.filename])):
            self.datadir = os.path.join(self.datadir, self.filename) # i.e. add filename to datadir

        # Read the dictionary
        with warnings.catch_warnings():   # Suppress errors due to CONVDTA in TopSpin
            warnings.simplefilter("ignore")
            dic, _ = ng.bruker.read(in_file.split('pdata')[0], cplex=True)
            # Read the files
            _, self.rr = ng.bruker.read_pdata(in_file, bin_files=['2rr'])
            _, self.ii = ng.bruker.read_pdata(in_file, bin_files=['2ii'])
            # Check for existence of hypercomplex data parts
            listdir = os.listdir(in_file)
            if ('2ir' in listdir and '2ri' in listdir): # Read them
                _, self.ir = ng.bruker.read_pdata(in_file, bin_files=['2ir'])
                _, self.ri = ng.bruker.read_pdata(in_file, bin_files=['2ri'])
                self.S = processing.repack_2D(self.rr, self.ir, self.ri, self.ii)
            else:    # Copy rr in ir and ri in ii
                self.ir = np.array(np.copy(self.rr))
                self.ri = np.copy(self.ii)
                self.S = self.rr + 1j*self.ii

            # Make the acqus dir
            self.acqus = misc.makeacqus_2D(dic)
            self.acqus['BYTORDA'] = dic['acqus']['BYTORDA']
            self.acqus['DTYPA'] = dic['acqus']['DTYPA']
            self.ngdic = dic
            del dic

        # Look for group delay points
        try:
            self.acqus['GRPDLY'] = int(self.ngdic['acqus']['GRPDLY'])
        except:
            self.acqus['GRPDLY'] = 0

        # initialize the procs dictionary with default values
        if os.path.exists(os.path.join(self.datadir, f'{self.filename}.procs')):
            self.procs = self.read_procs()
        else:
            proc_init_2D = (
                    [dict(wf0), dict(wf0)],     # window function
                    [None, None],   # zero-fill
                    [0.5, 0.5],     # fcor
                    [0,0]           # tdeff
                    )

            self.procs = {}
            for k, key in enumerate(proc_keys_1D):
                self.procs[key] = proc_init_2D[k]         # Processing parameters
            self.procs['wf'][0]['sw'] = round(self.acqus['SW1'], 4)
            self.procs['wf'][1]['sw'] = round(self.acqus['SW2'], 4)

            # Then, phases
            self.procs['p0_1'] = 0
            self.procs['p1_1'] = 0
            self.procs['pv_1'] = round(self.acqus['o1p'], 2)
            self.procs['p0_2'] = 0
            self.procs['p1_2'] = 0
            self.procs['pv_2'] = round(self.acqus['o2p'], 2)
            # Calibration
            self.procs['cal_1'] = 0
            self.procs['cal_2'] = 0
            self.write_procs()

        # Calculates the frequency and ppm scales
        self.freq_f1 = processing.make_scale(self.rr.shape[0], dw=self.acqus['dw1'])
        if self.acqus['SFO1'] < 0:
            self.freq_f1 = self.freq_f1[::-1]
        self.ppm_f1 = misc.freq2ppm(self.freq_f1, B0=self.acqus['SFO1'], o1p=self.acqus['o1p'])

        self.freq_f2 = processing.make_scale(self.rr.shape[1], dw=self.acqus['dw2'])
        if self.acqus['SFO2'] < 0:
            self.freq_f2 = self.freq_f2[::-1]
        self.ppm_f2 = misc.freq2ppm(self.freq_f2, B0=self.acqus['SFO2'], o1p=self.acqus['o2p']) 

        # Create empty dictionary where to save the projections
        self.trf1 = {}
        self.trf2 = {}
        self.Trf1 = {}
        self.Trf2 = {}

class Pseudo_2D(Spectrum_2D):
    """ 
    Subclass of Spectrum_2D to simulate and handle pseudo-2D experiments.
    Basically, they share more or less the same attributes, but some methods were adapted in order to suit well with a not-Fourier-transformed indirect dimension.
    ---------
    Attributes:
    - datadir: str
        Path to the input file/dataset directory
    - filename: str
        Base of the name of the file, without extensions
    - fid: 2darray
        FID. For simulated data, this must be explicitely set!
    - acqus: dict
        Dictionary of acqusition parameters
    - ngdic: dict
        Created only if it is an experimental spectrum. Generated by nmrglue.bruker.read, contains all the information on the spectrometer and on the spectrum.
    - procs: dict
        Dictionary of processing parameters
    - S: 2darray
        Complex spectrum
    - rr: 2darray
        Real part F2, real part F1
    - ii: 2darray
        Imaginary part F2, imaginary part F1
    - freq_f1: 1darray
        Indeces of the experiments, works as placeholder
    - freq_f2: 1darray
        Frequency scale of the direct dimension, in Hz
    - ppm_f1: 1darray
        Indeces of the experiments, works as placeholder
    - ppm_f2: 1darray
        ppm scale of the direct dimension
    - trf1: dict
        Projections of the indirect dimension, as 1darrays. Keys: 'ppm_f2' where they were taken
    - trf2: dict
        Projections of the direct dimension, as 1darrays. Keys: 'ppm_f1' where they were taken
    - Trf1: dict
        Projections of the indirect dimension, as pSpectrum_1D objects. Keys: 'ppm_f2' where they were taken
    - Trf2: dict
        Projections of the direct dimension, as pSpectrum_1D objects. Keys: 'ppm_f1' where they were taken
    - integrals: dict
        Dictionary where to save the regions and values of the integrals.
    - F: fit.Voigt_Fit_P2D object
        Interface for lineshape deconvolution.
    """

    def __str__(self):
        doc = '-'*64
        doc += '\nPseudo_2D object.\n'
        if 'ngdic' in self.__dict__.keys():
            doc += f'Read from "{self.datadir}"\n'
        else:
            doc += f'Simulated from "{self.datadir}"\n'
        doc += f'It is a {self.acqus["nuc"]} spectrum recorded over a\nsweep width of {self.acqus["SWp"]} ppm, centered at {self.acqus["o1p"]} ppm.\n'
        if self.fid is None:
            doc += 'The FID is not present yet.'
        else:
            N = self.fid.shape
            doc += f'The FID consists of {N[0]} experiments, each one is {N[1]} points long.\n'
        doc += '-'*64
        return doc

    def __init__(self, in_file, pv=False, isexp=True):
        """
        Initialize the class. 
        -------
        Parameters:
        - in_file: str
            path to file to read, or to the folder of the spectrum
        - pv: bool
            True if you want to use pseudo-voigt lineshapes for simulation, False for Voigt
        - isexp: bool
            True if this is an experimental dataset, False if it is simulated
        """
        ## Set up the filenames
        if isinstance(in_file, dict):   # Simulated data with already given acqus dictionary
            self.datadir = os.getcwd()      # Current directory
            self.filename = 'dict'  # Filename is just "dict"
        else:                           # You need to actually read a file
            self.datadir = os.path.abspath(in_file) # Get the file position
            if not os.path.isdir(self.datadir):
                self.datadir = os.path.dirname(self.datadir)
            self.filename = os.path.basename(in_file).rsplit('.', 1)[0] # Get the filename
            # If filename is a directory, write things inside it
            if os.path.isdir(os.path.join(self.datadir, self.filename)) and isexp:
                self.datadir = os.path.join(self.datadir, self.filename) # i.e. add filename to datadir

        if isexp is False:      # It is simulated experiment
            if isinstance(in_file, dict):       # acqus dictionary provided
                self.acqus = dict(in_file)  # Just read it
            else:   # acqus is in the file: read it
                self.acqus = sim.load_sim_1D(in_file)
            self.fid = None     # FID must be loaded with mount
        else:       # Experimental data
            with warnings.catch_warnings():   # Suppress errors due to CONVDTA in TopSpin
                warnings.simplefilter("ignore")
                # Read the FID
                dic, data = ng.bruker.read(in_file, cplex=True)
                self.fid = data
                # Make the acqus dictionary as it was 1D-like
                self.acqus = misc.makeacqus_1D(dic)
                self.acqus['BYTORDA'] = dic['acqus']['BYTORDA']
                self.acqus['DTYPA'] = dic['acqus']['DTYPA']
                self.ngdic = dic
                del dic
                del data

        # Try to find the group delay
        try:
            self.acqus['GRPDLY'] = int(self.ngdic['acqus']['GRPDLY'])
        except:
            self.acqus['GRPDLY'] = 0

        if isinstance(self.fid, np.ndarray):
            try:
                self.acqus['TD1'] = self.ngdic['acqu2s']['TD']
                self.fid = np.reshape(self.fid, (self.acqus['TD1'], -1))
            except:
                self.acqus['TD1'] = 0
            self.acqus['TD1'] = self.fid.shape[0]
        else:
            self.acqus['TD1'] = 0

        ## Initalize the procs dictionary 
        # If there already is a procs dictionary saved as file, load it
        if os.path.exists(os.path.join(self.datadir, f'{self.filename}.procs')):
            self.procs = self.read_procs()
        # Otherwise, initialize it with default values
        else:
            self.procs = {}
            proc_init_1D = (dict(wf0), None, 0.5, 0)
            for k, key in enumerate(proc_keys_1D):
                self.procs[key] = proc_init_1D[k]         # Processing parameters
            self.procs['wf']['sw'] = round(self.acqus['SW'], 4)
            #   Then, phases
            self.procs['p0'] = 0
            self.procs['p1'] = 0
            self.procs['pv'] = round(self.acqus['o1p'], 2)
            #   Then, baseline
            self.procs['basl_c'] = None
            self.procs['cal'] = 0
            self.procs['roll_ppm'] = None

            self.write_procs()

    def convdta(self, scaling=1):
        """ Calls processing.convdta """
        self.fid = processing.convdta(self.fid, self.acqus['GRPDLY'], scaling)

    def add_noise(self, s_n=1):
        """
        Adds noise to the FID, using the function sim.noisegen.
        ------------------
        Parameters:
        - s_n: float
            Standard deviation of the noise
        """
        self.fid += sim.noisegen(self.fid.shape, self.acqus['o1'], self.acqus['t1'], s_n=s_n)

    def scan(self, ns=1, s_n=1):
        """
        Simulates the acquisition of ns scans, by adding a different realization of noise at each iteration.
        The function is supposed to start with the FID without noise at all. If not, the results will be biased.
        --------------------
        Parameters:
        - ns: int
            Number of scans to accumulate
        - s_n: float
            Standard deviation of the noise
        """
        clean_fid = np.copy(self.fid)   # Save a shallow copy of the FID without noise
        for k in range(ns):
            # Each scan is: clean FID + noise
            self.fid += clean_fid + sim.noisegen(self.fid.shape, self.acqus['o1'], self.acqus['t1'], s_n=s_n)
        
    def process(self):
        """
        Process only the direct dimension.
        Calls processing.fp on each transient.
        The parameters are read from the procs dictionary
        """
        # Make empty self.S whose dimensions are given by the zero-filling
        if self.procs['zf'] is None:
            self.S = np.zeros(self.fid.shape).astype(self.fid.dtype)
        else:
            self.S = np.zeros((self.fid.shape[0], self.procs['zf'])).astype(self.fid.dtype)

        # Do the processing
        for k in range(self.fid.shape[0]):
            self.S[k] = processing.fp(self.fid[k], wf=self.procs['wf'], zf=self.procs['zf'], fcor=self.procs['fcor'], tdeff=self.procs['tdeff'])

        # Make the frequency and ppm scales
        self.freq_f2 = processing.make_scale(self.S.shape[1], dw=self.acqus['dw'])
        if self.acqus['SFO1'] < 0:      # Correct for negative gamma nuclei
            self.freq_f2 = self.freq_f2[::-1]
        self.ppm_f2 = misc.freq2ppm(self.freq_f2, B0=self.acqus['SFO1'], o1p=self.acqus['o1p']) 

        if self.acqus['SFO1'] < 0:      # Correct for negative gamma nuclei
            self.S = self.S[:,::-1]

        # Unpack self.S
        self.rr = self.S.real
        self.ii = self.S.imag

        self.F = fit.Voigt_Fit_P2D(self.ppm_f2, self.S, self.acqus['t1'], self.acqus['SFO1'], self.acqus['o1p'], self.acqus['nuc'], self.filename)

        # Adjust the phase
        self.adjph(p0=self.procs['p0'], p1=self.procs['p1'], pv=self.procs['pv'], update=False)

        # Use number of the experiment as fake scale in F1
        self.freq_f1 = np.arange(self.S.shape[0])           # python numbering
        self.ppm_f1 = np.arange(self.S.shape[0]) + 1        # human numbering

        # Align the spectrum
        if self.procs['roll_ppm'] is not None:
            for k, experiment in enumerate(self.S):
                roll_pt = int(ppm_shift / self.procs['roll_ppm'][k])    # Compute the circular shift in points
                self.S[k] = np.roll(experiment, roll_pt)                # Apply it to each experiment
            # Unpack S
            self.rr = self.S.real
            self.ii = self.S.imag

        # Calibrate the f2 scale. 
        self.cal(self.procs['cal'], update=False)

        self.integrals = {}
        
        # Create empty dictionary where to save the projections
        self.trf1 = {}
        self.trf2 = {}
        self.Trf1 = {}
        self.Trf2 = {}


    def mount(self, fids=[], filename=None, newacqus=None):
        """
        Replaces the FID of the experiment with a custom one, made by stacking 1D experiments.
        If the default filename exists (i.e. '<self.filename>.npy'), the function loads it, otherwise calls processing.stack_fids to create it.
        The "fid" attribute is overwritten. The key TD1 of the acqus dictionary is updated to match the first dimension of the new FID.
        ----------
        Parameters:
        - fids: sequence of 1darray or Spectrum_1D objects
            FIDs to be stacked. It can be empty if the .npy file already exists.
        - filename: str or None
            Path to the filename, without the .npy extension. If it is None, the default filename is used.
        - newacqus: dict
            New acqus dictionary that replaces the actual one. If it is not a dictionary, no actions are performed.
        """
        # Adjust the filename
        if filename is None:    # Default
            filename = os.path.join(self.datadir, f'{self.filename}.npy')
        else:                   # Add the .npy extension
            filename = f'{filename}.npy'

        # Check if the .npy file already exists
        if not len(fids):
            if os.path.exists(filename):    # then load it
                self.fid = np.load(filename)
            else:
                raise ValueError('You passed no FIDs!')
        else:   # make it
            self.fid = processing.stack_fids(*fids, filename=filename)

        # Replace acqus only if newacqus is a dictionary
        if isinstance(newacqus, dict):
            self.acqus = dict(newacqus)

        # Update the TD1 key
        self.acqus['TD1'] = self.fid.shape[0]

    def cal(self, offset=None, isHz=False, update=True):
        """
        Calibration of the ppm and frequency scales according to a given value, or interactively. In this latter case, a reference peak must be chosen.
        Calls processing.calibration
        --------
        Parameters:
        - offset: float
            scale shift F2
        - isHz: tuple of bool
            True if offset is in frequency units, False if offset is in ppm
        - update: bool
            Choose if to update the procs dictionary or not
        """
        def _calibrate(ppm, trace, SFO1, o1p):
            """ Main function that calls the real calibration """
            offppm = processing.calibration(ppm, trace)
            offhz = misc.ppm2freq(offppm, SFO1, o1p)
            return offppm, offhz

        # Get the missing entries
        if offset is None: # Select the reference traces
            coord = misc.select_traces(self.ppm_f1, self.ppm_f2, self.rr, Neg=False, grid=False)
            ix, iy = coord[0][0], coord[0][1]   # Position of the first crosshair
            # F2 reference spectrum
            X = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, iy, column=False)
            # F1 reference spectrum
            Y = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, ix, column=True)

            ppm_f2 = np.copy(self.ppm_f2)
            offp2, offh2 = _calibrate(ppm_f2, X, self.acqus['SFO1'], self.acqus['o1p'])
        else:   # Calculate offh2 from offp2 or viceversa
            if isHz:    # offp2 is missing
                offh2 = offset
                offp2 = misc.freq2ppm(offh2, self.acqus['SFO1'], self.acqus['o1p']) 
            else:       # offh2 is missing
                offp2 = offset
                offh2 = misc.ppm2freq(offp2, self.acqus['SFO1'], self.acqus['o1p']) 
            
        # Apply the calibration
        self.freq_f2 += offh2
        self.ppm_f2 += offp2
        # Move the offsets to the center of the SW
        self.acqus['o1p'] += offp2
        self.acqus['o1'] += offh2
        # Update the procs dictionary
        if update:  
            self.procs['cal'] += offp2
        # Update the .procs file
        self.write_procs()

        # Update F
        self.F.ppm_scale = self.ppm_f2
        self.F.o1p = self.acqus['o1p']

    def adjph(self, expno=0, p0=None, p1=None, pv=None, update=True):
        """
        Adjusts the phases of the spectrum according to the given parameters, or interactively if they are left as default.
        -------
        Parameters:
        - expno: int
            Index of the experiment (python numbering) to use in the interactive panel
        - p0: float or None
            0-th order phase correction /°
        - p1: float or None
            1-st order phase correction /°
        - pv: float or None
            1-st order pivot /ppm
        - update: bool
            Choose if to upload the procs dictionary or not
        """
        # Get the reference spectrum
        S = self.S[expno]
        # Adjust the phases
        _, values = processing.ps(S, self.ppm_f2, p0=p0, p1=p1, pivot=pv)
        self.S, _ = processing.ps(self.S, self.ppm_f2, *values)
        
        # Unpack self.S
        self.rr = self.S.real
        self.ii = self.S.imag

        # Update the procs dictionary
        if update:
            self.procs['p0'] += round(values[0], 2)
            self.procs['p1'] += round(values[1], 2)
            if values[2] is not None:
                self.procs['pv'] = round(values[2], 5)
        # Update the .procs file
        self.write_procs()

        # Update the F attribute
        self.F.S = self.S
        self.F.ppm_scale = self.ppm_f2

    def pknl(self):
        """
        Reverses the effect of the digital filter by applying a first order phase correction.
        To be called after having processed the data by 'self.process()'
        """
        self.adjph(p1=-360 * self.acqus['GRPDLY'], update=False)

    def projf1(self, a, b=None):
        """
        Calculates the sum trace of the indirect dimension, from a to b in F2.
        Store the trace in the dictionary trf1 and as 1D spectrum in Trf1. The key is 'a' or 'a:b'
        Updates the Trf1[label].freq and Trf1[label].ppm with self.freq_f1 and self.ppm_f1 respectively.
        -------
        Parameters:
        - a: float
            ppm F2 value where to extract the trace.
        - b: float or None.
            If it is None, extract the trace in a. Else, sum from a to b in F2.
        """
        # make dictionary label
        if b is None:
            label = f'{a:.2f}'
        else:
            label = f'{a:.2f}:{b:.2f}'
        # Compute the trace
        f1 = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, a, b, column=True)
        # Store it
        self.trf1[label] = f1
        self.Trf1[label] = pSpectrum_1D(f1, acqus=self.acqus, procs=self.procs, istrace=True)
        # Overwrite f2 scale with f1 scale as this is f1 projection
        self.Trf1[label].freq = np.copy(self.freq_f1)
        self.Trf1[label].ppm = np.copy(self.ppm_f1)

    def projf2(self, a, b=None):
        """
        Calculates the sum trace of the direct dimension, from a to b in F1.
        Store the trace in the dictionary trf2 and as 1D spectrum in Trf2. The key is 'a' or 'a:b'
        -------
        Parameters:
        - a: float
            ppm F1 value where to extract the trace.
        - b: float or None.
            If it is None, extract the trace in a. Else, sum from a to b in F1.
        """
        # make dictionary label
        if b is None:
            label = f'{a:.2f}'
        else:
            label = f'{a:.2f}:{b:.2f}'
        # Compute the trace
        f2 = misc.get_trace(self.rr, self.ppm_f2, self.ppm_f1, a, b, column=False)
        # Store it
        self.trf2[label] = f2
        self.Trf2[label] = pSpectrum_1D(f2, acqus=self.acqus, procs=self.procs, istrace=True)

    def plot(self, Neg=True, lvl0=0.2, Y_label=''):
        """
        Plots the real part of the spectrum as a 2D contour plot.
        -------
        Parameters:
        - Neg: bool
            Plot (True) or not (False) the negative contours.
        - lvl0: float
            Starting contour value.
        - Y_label: str
            Custom label for vertical axis.
        """
        warnings.filterwarnings("ignore", message="No contour levels were found within the data range.")
        # Plots data, set Neg=True to see negative contours
        S = np.copy(self.rr)
        n_xticks, n_yticks = 10, 10

        X_label = r'$\delta\ $'+misc.nuc_format(self.acqus['nuc'])+' /ppm'

        cmaps = ['Blues_r', 'Reds_r']

        # flags for the activation of scroll zoom
        lvlstep = 0.02

        # define boxes for sliders
        iz_box = plt.axes([0.925, 0.80, 0.05, 0.05])
        dz_box = plt.axes([0.925, 0.75, 0.05, 0.05])

        # Functions connected to the sliders
        def increase_zoom(event):
            nonlocal lvlstep
            lvlstep *= 2

        def decrease_zoom(event):
            nonlocal lvlstep
            lvlstep /= 2

        def on_scroll(event):
            nonlocal lvl, cnt
            if Neg:
                nonlocal Ncnt
                
            act_xlim = ax.get_xlim()
            act_ylim = ax.get_ylim()

            if event.button == 'up':
                lvl += lvlstep 
            elif event.button == 'down':
                lvl += -lvlstep
            if lvl <= 0:
                lvl = 1e-6
            elif lvl > 1:
                lvl = 1

            if Neg:
                cnt, Ncnt = figures.redraw_contours(ax, self.ppm_f2, self.ppm_f1, S, lvl=lvl, cnt=cnt, Neg=Neg, Ncnt=Ncnt, lw=0.5, cmap=[cmaps[0], cmaps[1]])
            else:
                cnt, _ = figures.redraw_contours(ax, self.ppm_f2, self.ppm_f1, S, lvl=lvl, cnt=cnt, Neg=Neg, Ncnt=None, lw=0.5, cmap=[cmaps[0], cmaps[1]])

            misc.pretty_scale(ax, act_xlim, axis='x', n_major_ticks=n_xticks)
            misc.pretty_scale(ax, act_ylim, axis='y', n_major_ticks=n_yticks)
            ax.set_xlabel(X_label)
            ax.set_ylabel(Y_label)
            misc.set_fontsizes(ax, 14)
            lvl_text.set_text(f'{lvl:.5g}')
            fig.canvas.draw()

        # Make the figure
        fig = plt.figure(f'{self.filename}')
        fig.set_size_inches(15,8)
        plt.subplots_adjust(left = 0.10, bottom=0.10, right=0.90, top=0.95)
        ax = fig.add_subplot(1,1,1)

        contour_num = 16
        contour_factor = 1.40

        lvl = lvl0

        cnt = figures.ax2D(ax, self.ppm_f2, self.ppm_f1, S, lvl=lvl, cmap=cmaps[0])
        if Neg:
            Ncnt = figures.ax2D(ax, self.ppm_f2, self.ppm_f1, -S, lvl=lvl, cmap=cmaps[1])

        lvl_text = ax.text(0.925, 0.60, f'{lvl:.5g}', ha='left', va='center', transform=fig.transFigure, fontsize=12)

        # Make pretty x-scale
        misc.pretty_scale(ax, (max(self.ppm_f2), min(self.ppm_f2)), axis='x', n_major_ticks=n_xticks)
        misc.pretty_scale(ax, (max(self.ppm_f1), min(self.ppm_f1)), axis='y', n_major_ticks=n_yticks)
        ax.set_xlabel(X_label)
        ax.set_ylabel(Y_label)

        scale_factor = 1

        # Create buttons
        iz_button = Button(iz_box, label=r'$\uparrow$')
        dz_button = Button(dz_box, label=r'$\downarrow$')

        # Connect the widgets to functions
        scroll = fig.canvas.mpl_connect('scroll_event', on_scroll)
            
        iz_button.on_clicked(increase_zoom)
        dz_button.on_clicked(decrease_zoom)

        misc.set_fontsizes(ax, 14)

        cursor = Cursor(ax, useblit=True, c='tab:red', lw=0.8)

        plt.show()
        plt.close()
    
    def plot_md(self, which=None, lims=None):
        """ 
        Plot a number of experiments, superimposed.
        --------
        Parameters:
        - which: str or None
            List of experiment indexes, so that eval(which) is meaningful. None plots all of them
        - lims: tuple
            Region of the spectrum to show (ppm1, ppm2)
        """
        # Select which spectra to plot
        if which is None:   # All of them
            which_exp = np.arange(self.rr.shape[0])
        else:   # Just the ones in which
            which_exp = eval(which)
        # Make shallow copy of ppm scale
        ppm = np.copy(self.ppm_f2)
        # Organize the spectra to be plot
        S = [np.copy(self.rr[w]) for w in which_exp]

        # Cut the data according to lims
        if lims is not None:
            for k, s in enumerate(S): 
                _, S[k] = misc.trim_data(ppm, s, *lims)
            ppm, _ = misc.trim_data(ppm, s, *lims)

        # Make the figure
        figures.dotmd(ppm, S, labels=[f'{w}' for w in which_exp])

    def plot_stacked(self, which=None, lims=None):
        """ 
        Plot a number of experiments, stacked.
        --------
        Parameters:
        - which: str or None
            List of experiment indexes, so that eval(which) is meaningful. None plots all of them.
        - lims: tuple
            Region of the spectrum to show (ppm1, ppm2)
        """
        # Select which spectra to plot
        if which is None:   # All of them
            which_exp = np.arange(self.rr.shape[0])
        else:   # Just the ones in which
            which_exp = eval(which)
        # Make shallow copy of ppm scale
        ppm = np.copy(self.ppm_f2)
        # Organize the spectra to be plot
        S = [np.copy(self.rr[w]) for w in which_exp]

        # Cut the data according to lims
        if lims is not None:
            for k, s in enumerate(S): 
                _, S[k] = misc.trim_data(ppm, s, *lims)
            ppm, _ = misc.trim_data(ppm, s, *lims)

        X_label = r'$\delta\ $'+misc.nuc_format(self.acqus['nuc'])+' /ppm'

        # Make the figure
        figures.stacked_plot(
                ppm, S, 
                X_label=X_label, Y_label='Normalized intensity /a.u.',
                labels=[f'{w}' for w in which_exp])


    def integrate(self, which=0, lims=None):
        """
        Integrate the spectrum with a dedicated GUI.
        Calls processing.integral on each experiment, then saves the results in self.integrals.
        Therefore, the entries of self.integrals are sequences!
        If lims is not given, calls fit.integrate on the trace to select the regions to integrate.
        --------
        Parameters:
        - which: int
            Experiment index to show in interactive panel
        - lims: tuple
            Region of the spectrum to integrate (ppm1, ppm2)
        """
        # Select the integration region
        if lims is None:
            X_label = r'$\delta\,$'+misc.nuc_format(self.acqus['nuc'])+' /ppm'
            integrals = fit.integrate(self.ppm_f2, self.rr[which], X_label=X_label)
            for key, _ in integrals.items():
                if ':' in key:
                    lims = [eval(q) for q in key.split(':')] # trasforma stringa in float!!!
                    self.integrals[key] = [processing.integral(self.rr[k], self.ppm_f2, lims)[-1] for k in range(self.rr.shape[0])]
                else:
                    self.integrals[key] = np.array(integrals[key])

        else:
            self.integrals[f'{lims[0]:.2f}:{lims[1]:.2f}'] = np.array(processing.integral(self.rr, self.ppm_f2, lims)[...,-1])

    def write_integrals(self, filename='integrals.dat'):
        """
        Write the integrals in a file named filename.
        -------
        Parameters:
        - filename: str
            name of the file where to write the integrals.
        """
        @staticmethod
        def arr2string(array):
            if isinstance(array, (np.ndarray, list, tuple)):
                string = [f'{w:.4e}' for w in array]
            else: 
                string = [f'{array:.4e}']
            return string

        @staticmethod
        def write_arr(f, string):
            for w in string:
                f.write(f'{w}, ')

        f = open(filename, 'w')
        for key, value in self.integrals.items():
            f.write('{:12}\t'.format(key))
            if 'total' in key:
                write_arr(f, [value])
                f.write('\n')
            elif 'ref' in key:
                if 'pos' in key:
                    f.write('{}\n'.format(value))
                elif 'int' in key:
                    f.write('{:.4e}\n'.format(value))
                elif 'val' in key:
                    f.write('{:.3f}\n'.format(value))
            else:
                write_arr(f, arr2string(value))
                f.write('\n')
        f.close()

    def align(self, lims=None, u_off=0.5, ref_idx=0):
        """
        Aligns the spectrum to a reference signal in the reference spectrum (default: first one).
        ---------
        Parameters:
        - lims: tuple or None
            Reference signal region, in ppm. If None, you can select it interactively.
        - u_off: float
            Maximum displacement allowed, in ppm
        - ref_idx: int
            Index of the spectrum to be used as a reference (python numbering)
        """
        # Get the region of the reference peak
        if lims is None:
            lims = fit.get_region(self.ppm_f2, self.r, rev=True)
        # Align
        self.S, roll_pt, roll_ppm = processing.align(self.ppm_f2, self.S, lims, u_off, ref_idx)
        # Unpack S
        self.rr = self.S.real
        self.ii = self.S.imag
        # Update the procs dictionary
        self.procs['roll_ppm'] += roll_ppm
        # Update the .procs file
        self.write_procs()

        self.F.S = self.S

    def basl(self, from_procs=False, phase=True):
        """ 
        Apply baseline correction to the whole pseudo-2D by subtracting self.baseline from self.S. Then, self.S is unpacked in self.rr and self.ii.
        --------
        Parameters:
        - from_procs: bool
            If True, computes the baseline using the polynomion model reading self.procs['basl_c'] as coefficients
        - phase: bool
            Choose if to apply the same phase correction of the spectrum to the baseline. This should be done if the baseline was computed before the phase adjustment!
        """
        # Get the baseline, if there is not already
        if from_procs:
            x_basl = np.linspace(-1, 1, self.S.shape[-1])
            self.baseline = misc.polyn(x_basl, self.procs['basl_c'])
        # Apply the phase correction
        if phase:
            self.baseline, *_ = processing.ps(self.baseline, self.ppm_f2, p0=self.procs['p0'], p1=self.procs['p1'], pivot=self.procs['pv'])
        # Make an array of baseline
        full_baseline = np.array([self.baseline for w in range(self.S.shape[0])])
        # Apply the baseline correction
        self.S -= baseline
        self.rr = self.S.real
        self.ii = self.S.imag

        # Update F
        self.F.S = self.S


    def rpbc(self, ref_exp=0, **rpbc_kws):
        """
        Computes the phase angles and the baseline using processing.rpbc on a reference spectrum taken from self.S.
        Then applies the phase correction and subtracts the baseline, automatically, to all experiments of the pseudo-2D.
        The procs dictionary is then updated and saved.
        The polynomial baseline is computed according to the given coefficients and stored in self.baseline
        ---------------
        Parameters:
        - ref_exp: int
            Index of the reference experiment on which to apply the algorithm
        - rpbc_kws: keyworded arguments
            See processing.RPBC for details.
        """
        # Get the reference experiment
        experiment = np.copy(self.S[ref_exp])
        # Get phase angles and baseline coefficients with rpbc
        _, p0, p1, c = processing.rpbc(experiment, **rpbc_kws)
        # Apply phase to the whole self.S
        self.adjph(p0=p0, p1=p1)
        # Compute the baseline
        self.procs['basl_c'] = c
        self.baseline = misc.polyn(np.linspace(-1, 1, self.S.shape[-1]), c)
        # Apply it
        self.basl()

        # Update the .procs file
        self.write_procs()

    def to_wav(self, filename=None, cutoff=None, rate=44100):
        """
        Converts the FID in an audio file by using misc.data2wav.
        ---------
        Parameters:
        - filename: str
            Path where to save the file. If None, self.filename is used
        - cutoff: float
            Clipping limits for the FID
        - rate: int
            Sampling rate in samples/sec
        """
        if filename is None:
            cwd = os.getcwd()
            filename = os.path.join(cwd, self.filename)
        # Make a shallow copy of the FID
        data = np.copy(self.fid)
        misc.data2wav(data, filename, cutoff, rate)





