# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
r"""Version 2 of *HydPy-W-Land* is still under development and likely to change in the
future.

When applying |wland_v001| on the Kiel Catchment river basins, we realised some
strengths and some limitations of the `WALRUS`_ concept.  First, |wland_v001| does
not predict the high dynamics of the groundwater levels encountered at many
groundwater gauges.  We assume this shortcoming is connected with an observed
tendency to underestimate streamflow peaks.  Second, |wland_v001| tends to run dry
during summer conditions much too fast.

.. _`WALRUS`: https://www.wur.nl/en/Research-Results/Chair-groups/Environmental-Sciences/Hydrology-and-Quantitative-Water-Management-Group/Research/WALRUS-1.htm

So far, |wland_v002| offers a physically based approach that increases groundwater
dynamics.  See the documentation on method |Calc_GF_V1| for an in-depth
explanation.  The second problem seems to be related to a lack of "groundwater
recharge areas" where groundwater is less exposed to evapotranspiration.  We will
address this later.

The following integration tests are identical with the ones of |wland_v001|, except
that we set the additional parameter |ThetaR| to 0.01.  Compare the
:ref:`wland_v001_base_scenario` example of |wland_v001| and with the
:ref:`wland_v002_base_scenario` example of |wland_v002| to see how the differences
between both models affect groundwater dynamics.


Integration tests
=================

.. how_to_understand_integration_tests::

>>> from hydpy import IntegrationTest, Element, pub, round_
>>> pub.timegrids = "2017-02-10", "2017-04-10", "1d"
>>> from hydpy.models.wland_v002 import *
>>> parameterstep("1d")
>>> land = Element("land", outlets="outlet")
>>> land.model = model
>>> al(9.8)
>>> as_(0.2)
>>> nu(3)
>>> lt(FIELD, CONIFER, SEALED)
>>> aur(0.6, 0.3, 0.1)
>>> cp(0.8)
>>> cpet(0.9)
>>> cpetl.sealed = 0.7
>>> cpetl.conifer = 1.3
>>> cpetl.field = 0.73, 0.73, 0.77, 0.95, 1.19, 1.18, 1.19, 1.15, 0.97, 0.85, 0.78, 0.73
>>> cpes(jan=1.16, feb=1.22, mar=1.26, apr=1.28, may=1.28, jun=1.30,
...      jul=1.28, aug=1.28, sep=1.27, oct=1.23, nov=1.17, dec=1.14)
>>> lai.sealed = 10.0
>>> lai.conifer = 11.0
>>> lai.field = 0.4, 0.4, 0.3, 0.7, 3.0, 5.2, 4.6, 3.1, 1.3, 0.2, 0.0, 0.0
>>> ih(0.2)
>>> tt(0.0)
>>> ti(4.0)
>>> ddf(5.0)
>>> ddt(0.0)
>>> cw(400.0)
>>> cv(0.2)
>>> cg(200000.0)
>>> cgf(1.0)
>>> cq(0.5)
>>> cd(1500.0)
>>> cs(8.0)
>>> hsmin(0.0)
>>> xs(1.5)
>>> b(soil=SANDY_LOAM)
>>> psiae(soil=SANDY_LOAM)
>>> thetas(soil=SANDY_LOAM)
>>> thetar(0.01)
>>> zeta1(0.02)
>>> zeta2(400.0)
>>> sh(1.0)
>>> st(1.0)
>>> test = IntegrationTest(land)
>>> test.inits = ((states.ic, -3.0),
...               (states.sp, -3.0),
...               (states.dv, 140.0),
...               (states.dg, 1600.0),
...               (states.hq, 0.0),
...               (states.hs, -2.0))
>>> inputs.t.series = (
...     -2.8, -1.5, -0.9, -1.6, -1.3, 1.7, 4.4, 4.5, 3.4, 4.8, 6.7, 5.8, 6.5, 5.0, 3.0,
...     3.1, 7.1, 9.4, 4.6, 3.7, 4.7, 5.9, 7.7, 6.3, 3.7, 1.6, 4.0, 5.6, 5.8, 5.7, 4.6,
...     4.2, 7.4, 6.3, 8.7, 6.4, 5.2, 5.1, 8.7, 6.2, 5.9, 5.2, 5.2, 5.9, 6.7, 7.0, 8.3,
...     9.0, 12.4, 15.0, 11.8, 9.4, 8.1, 7.9, 7.5, 7.2, 8.1, 8.6, 10.5)
>>> inputs.p.series = (
...     0.0, 0.4, 0.0, 0.0, 0.0, 0.0, 0.2, 4.5, 0.0, 3.2, 4.6, 2.3, 18.0, 19.2, 0.4,
...     8.3, 5.3, 0.7, 2.7, 1.6, 2.5, 0.6, 0.2, 1.7, 0.3, 0.0, 1.8, 8.9, 0.0, 0.0,
...     0.0, 0.9, 0.1, 0.0, 0.0, 3.9, 8.7, 26.4, 11.5, 0.9, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 1.5, 0.3, 0.2, 4.5, 0.0, 0.0, 0.0, 0.4, 0.0, 0.0, 0.0, 0.0)
>>> inputs.pet.series = (
...     0.6, 0.8, 0.7, 0.4, 0.4, 0.4, 0.4, 0.3, 0.3, 0.4, 0.3, 0.6, 0.8, 0.5, 0.8,
...     0.5, 0.4, 1.3, 0.9, 0.7, 0.7, 1.1, 1.0, 0.8, 0.6, 0.7, 0.7, 0.5, 0.8, 1.0,
...     1.2, 0.9, 0.9, 1.2, 1.4, 1.1, 1.1, 0.5, 0.6, 1.5, 2.0, 1.6, 1.6, 1.2, 1.3,
...     1.6, 1.9, 0.8, 1.5, 2.7, 1.5, 1.6, 2.0, 2.1, 1.7, 1.7, 0.8, 1.3, 2.5)
>>> inputs.fxg.series = 0.0
>>> inputs.fxs.series = 0.0
>>> test.reset_inits()
>>> conditions = sequences.conditions

.. _wland_v002_base_scenario:

base scenario
_____________

.. integration-test::

    >>> test("wland_v002_base_scenario",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |    t |    p | pet | fxg | fxs |    pc |                 petl |    pes |                              tf |                           ei |                              rf |            sf |                              pm |                           am |    ps |       pv |       pq |      etv |       es |       et | fxs | fxg |         cdg |       fgs |       fqs |       rh |        r |                              ic |                              sp |         dv |          dg |       hq |         hs |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 0.6 | 0.0 | 0.0 |   0.0 | 0.3942  0.702  0.378 | 0.6588 |       0.0        0.0        0.0 |      0.0  0.000001       0.0 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.000117   0.000117   0.000117 |      0.0       0.0       0.0 |   0.0 |      0.0 |      0.0 |  0.49406 | 0.000073 | 0.435763 | 0.0 | 0.0 |  -37.315378 |  0.000762 |       0.0 |      0.0 |      0.0 |      -3.0  -3.000001       -3.0 |      -3.0       -3.0       -3.0 | 140.494822 | 1562.684622 |      0.0 |  -1.966476 |      0.0 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.000001        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.009388   0.009388   0.009388 |      0.0       0.0       0.0 |  0.32 |      0.0 |      0.0 | 0.658704 | 0.000253 | 0.580984 | 0.0 | 0.0 |  -15.511779 |  0.000468 |       0.0 |      0.0 |      0.0 | -2.680002  -2.680003  -2.680002 |      -3.0       -3.0       -3.0 | 141.153995 | 1547.172843 |      0.0 |  -1.626089 |      0.0 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 | 0.4599  0.819  0.441 | 0.7686 |       0.0        0.0        0.0 | 0.000002  0.000004  0.000002 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.069591   0.069591   0.069591 |      0.0       0.0       0.0 |   0.0 |      0.0 |      0.0 | 0.576325 | 0.000452 |  0.50833 | 0.0 | 0.0 |   -5.174815 |  0.000343 |       0.0 |      0.0 |      0.0 | -2.680004  -2.680007  -2.680004 |      -3.0       -3.0       -3.0 | 141.730663 | 1541.998028 |      0.0 |  -1.611403 |      0.0 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.006707   0.006707   0.006707 |      0.0       0.0       0.0 |   0.0 |      0.0 |      0.0 | 0.329312 | 0.000276 |  0.29046 | 0.0 | 0.0 |   -0.906637 |  0.000319 |       0.0 |      0.0 |      0.0 | -2.680005  -2.680009  -2.680005 |      -3.0       -3.0       -3.0 | 142.060293 | 1541.091391 |      0.0 |  -1.597615 |      0.0 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.018374   0.018374   0.018374 |      0.0       0.0       0.0 |   0.0 |      0.0 |      0.0 | 0.329299 | 0.000294 | 0.290449 | 0.0 | 0.0 |    0.730369 |  0.000316 |       0.0 |      0.0 |      0.0 | -2.680007  -2.680011  -2.680006 |      -3.0       -3.0       -3.0 | 142.389909 | 1541.821759 |      0.0 |  -1.583956 |      0.0 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |   8.50479    8.50479    8.50479 | 0.000009  0.000009  0.000009 |   0.0 | 0.000002 | 0.000007 | 0.329287 | 0.000313 | 0.290439 | 0.0 | 0.0 |     1.50547 |  0.000323 |  0.000003 |      0.0 |      0.0 | -2.680008  -2.680013  -2.680007 | -3.000008  -3.000009  -3.000009 | 142.719516 |  1543.32723 | 0.000003 |  -1.569858 |      0.0 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 0.4 | 0.0 | 0.0 |  0.16 | 0.2628  0.468  0.252 | 0.4392 |  0.000001        0.0        0.0 | 0.000002  0.000003  0.000002 |  0.000001        0.0        0.0 | 0.0  0.0  0.0 | 22.000001  22.000001  22.000001 | 0.000023  0.000023  0.000023 |  0.16 | 0.000007 | 0.000017 | 0.329273 | 0.000495 | 0.290431 | 0.0 | 0.0 |    1.871274 |  0.000317 |  0.000013 |      0.0 |      0.0 |  -2.52001  -2.520016  -2.520009 | -3.000031  -3.000032  -3.000032 |   143.0491 | 1545.198504 | 0.000007 |   -1.39572 |      0.0 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 0.3 | 0.0 | 0.0 |   3.6 | 0.1971  0.351  0.189 | 0.3294 |  0.716494   0.000865   0.002553 | 0.045813  0.103383  0.056188 |  0.716494   0.000865   0.002553 | 0.0  0.0  0.0 |      22.5       22.5       22.5 | 0.000023  0.000023  0.000023 |   3.6 | 0.135797 | 0.308217 | 0.182326 | 0.201009 | 0.227671 | 0.0 | 0.0 |   -10.37264 | -0.000181 |  0.080118 | 0.000267 | 0.000031 |  0.317682   0.975737    1.02125 | -3.000054  -3.000055  -3.000055 | 143.095448 | 1534.825864 | 0.228105 |    5.90774 | 0.000031 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 0.3 | 0.0 | 0.0 |   0.0 | 0.1971  0.351  0.189 | 0.3294 |       0.0        0.0        0.0 | 0.147906  0.341646  0.186301 |       0.0        0.0        0.0 | 0.0  0.0  0.0 | 17.000016  17.000016  17.000016 | 0.000018  0.000018  0.000018 |   0.0 | 0.000005 | 0.000013 | 0.035705 |   0.3294 | 0.243749 | 0.0 | 0.0 |    6.834913 | -0.003057 |  0.198423 | 0.005734 | 0.000664 |  0.169777    0.63409    0.83495 | -3.000072  -3.000072  -3.000072 |  143.12809 | 1541.660777 | 0.029696 |  14.879523 | 0.000664 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 0.4 | 0.0 | 0.0 |  2.56 | 0.2628  0.468  0.252 | 0.4392 |  2.023752   0.453078   0.899772 | 0.221728  0.465424  0.251476 |  2.023752   0.453078   0.899772 | 0.0  0.0  0.0 |      24.0       24.0       24.0 | 0.000025  0.000025  0.000025 |  2.56 | 0.425039 | 1.057641 | 0.028075 |   0.4392 | 0.325402 | 0.0 | 0.0 |  -28.776921 | -0.006828 |  0.568766 | 0.018477 | 0.002139 |  0.484297   2.275589   2.243702 | -3.000097  -3.000097  -3.000097 | 142.724298 | 1512.883856 | 0.518571 |  43.644875 | 0.002139 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 0.3 | 0.0 | 0.0 |  3.68 | 0.1971  0.351  0.189 | 0.3294 |  3.340021   2.972444   3.208119 | 0.184109  0.350996  0.188997 |  3.340021   2.972444   3.208119 | 0.0  0.0  0.0 |      33.5       33.5       33.5 | 0.000035  0.000035  0.000035 |  3.68 | 0.904019 | 2.402975 | 0.008612 |   0.3294 | 0.244155 | 0.0 | 0.0 |   -51.07395 |  -0.02901 |  1.792081 | 0.105583 |  0.01222 |  0.640166   2.632149   2.526586 | -3.000131  -3.000131  -3.000131 | 141.799881 | 1461.809906 | 1.129465 |  128.24896 |  0.01222 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 0.6 | 0.0 | 0.0 |  1.84 | 0.3942  0.702  0.378 | 0.6588 |  1.642137   1.409144    1.61889 | 0.363807  0.701991  0.377995 |  1.642137   1.409144    1.61889 | 0.0  0.0  0.0 |      29.0       29.0       29.0 |  0.00003   0.00003   0.00003 |  1.84 | 0.435606 | 1.177899 | 0.020151 |   0.6588 | 0.488296 | 0.0 | 0.0 |    1.249463 | -0.106365 |  1.656164 | 0.291665 | 0.033758 |  0.474222   2.361014   2.369701 | -3.000161  -3.000161  -3.000161 | 141.278061 | 1463.059369 |   0.6512 | 191.308269 | 0.033758 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 0.8 | 0.0 | 0.0 |  14.4 | 0.5256  0.936  0.504 | 0.8784 | 13.589441  13.048599  13.564535 | 0.504697  0.935996  0.503997 | 13.589441  13.048599  13.564535 | 0.0  0.0  0.0 |      32.5       32.5       32.5 | 0.000033  0.000033  0.000033 |  14.4 | 3.631402 | 10.15647 | 0.013861 |   0.8784 | 0.651129 | 0.0 | 0.0 | -194.346077 | -0.228002 |  6.266747 | 0.746768 | 0.086431 |  0.780084   2.776418   2.701169 | -3.000194  -3.000195  -3.000195 | 137.432518 | 1268.713292 | 4.540923 | 464.507177 | 0.086431 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 0.5 | 0.0 | 0.0 | 15.36 | 0.3285  0.585  0.315 |  0.549 | 14.927286  14.656218  14.931712 | 0.322035  0.584999  0.314999 | 14.927286  14.656218  14.931712 | 0.0  0.0  0.0 |      25.0       25.0       25.0 | 0.000026  0.000026  0.000026 | 15.36 | 3.794015 | 11.43182 | 0.004289 |    0.549 | 0.406979 | 0.0 | 0.0 |  -51.730494 | -1.302711 | 10.388708 | 2.284271 | 0.264383 |  0.890763   2.895201   2.814459 |  -3.00022  -3.000221  -3.000221 |  132.34008 | 1216.982799 | 5.584035 | 816.701805 | 0.264383 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.294506   0.228504   0.294804 | 0.495651  0.935979  0.503995 |  0.294506   0.228504   0.294804 | 0.0  0.0  0.0 |  15.00006   15.00006   15.00006 | 0.000015  0.000015  0.000015 |  0.32 | 0.066142 | 0.215223 | 0.019881 |   0.8784 | 0.651116 | 0.0 | 0.0 |   16.462031 | -2.377399 |  4.955454 | 3.331042 | 0.385537 |  0.420605   2.050718   2.335659 | -3.000236  -3.000236  -3.000236 | 129.916421 | 1233.444829 | 0.843804 | 787.565232 | 0.385537 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 0.5 | 0.0 | 0.0 |  6.64 | 0.3285  0.585  0.315 |  0.549 |   6.06746   5.421574   6.057767 | 0.308293  0.584994  0.314997 |   6.06746   5.421574   6.057767 | 0.0  0.0  0.0 | 15.500043  15.500043  15.500043 | 0.000016  0.000016  0.000016 |  6.64 | 1.364207 | 4.644955 | 0.013414 |    0.549 | 0.406946 | 0.0 | 0.0 |  -30.069173 |  -1.74167 |  3.313151 | 2.825528 | 0.327029 |  0.684852    2.68415   2.602895 | -3.000251  -3.000252  -3.000252 | 126.823959 | 1203.375656 | 2.175608 | 737.916583 | 0.327029 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 0.4 | 0.0 | 0.0 |  4.24 | 0.2628  0.468  0.252 | 0.4392 |  3.990587   3.797307   3.989263 | 0.251899  0.467998  0.251998 |  3.990587   3.797307   3.989263 | 0.0  0.0  0.0 |      35.5       35.5       35.5 | 0.000037  0.000037  0.000037 |  4.24 | 0.879995 | 3.140511 | 0.007238 |   0.4392 | 0.325572 | 0.0 | 0.0 |   -9.178509 | -1.569861 |  3.661626 | 2.715403 | 0.314283 |  0.682366   2.658845   2.601634 | -3.000288  -3.000288  -3.000288 | 124.381341 | 1194.197147 | 1.654493 | 716.136077 | 0.314283 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 1.3 | 0.0 | 0.0 |  0.56 | 0.8541  1.521  0.819 | 1.4274 |  0.424048   0.206921   0.404336 | 0.695516   1.52066  0.818961 |  0.424048   0.206921   0.404336 | 0.0  0.0  0.0 |      47.0       47.0       47.0 | 0.000048  0.000048  0.000048 |  0.56 | 0.076781 | 0.287884 | 0.105418 |   1.4274 | 1.057822 | 0.0 | 0.0 |   20.217281 | -1.271414 |  1.611646 | 2.406926 | 0.278579 |  0.122801   1.491264   1.938337 | -3.000336  -3.000337  -3.000337 | 123.138564 | 1214.414429 | 0.330732 | 617.823665 | 0.278579 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 0.9 | 0.0 | 0.0 |  2.16 | 0.5913  1.053  0.567 | 0.9882 |  1.504342   0.475474   1.340734 |  0.45371  1.052728  0.566963 |  1.504342   0.475474   1.340734 | 0.0  0.0  0.0 |      23.0       23.0       23.0 | 0.000024  0.000024  0.000024 |  2.16 |    0.249 | 0.955245 |  0.09146 |   0.9882 | 0.732278 | 0.0 | 0.0 |   21.087662 | -0.836028 |  0.799116 |  1.86769 | 0.216168 |  0.324749   2.123061    2.19064 |  -3.00036  -3.000361  -3.000361 | 122.144996 | 1235.502091 |  0.48686 | 527.898816 | 0.216168 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 0.7 | 0.0 | 0.0 |  1.28 | 0.4851  0.819  0.441 | 0.7938 |  0.950591   0.497004   0.875872 | 0.384057  0.818947  0.440979 |  0.950591   0.497004   0.875872 | 0.0  0.0  0.0 | 18.500006  18.500006  18.500006 | 0.000019  0.000019  0.000019 |  1.28 | 0.169301 | 0.654691 | 0.067123 |   0.7938 |  0.58489 | 0.0 | 0.0 |   29.527892 | -0.610875 |  0.795377 | 1.519409 | 0.175858 |  0.270102    2.08711   2.153789 | -3.000379   -3.00038   -3.00038 | 121.431943 | 1265.029982 | 0.346174 | 464.448441 | 0.175858 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 0.7 | 0.0 | 0.0 |   2.0 | 0.4851  0.819  0.441 | 0.7938 |  1.532173   1.008537   1.460236 | 0.393795  0.818966  0.440983 |  1.532173   1.008537   1.460236 | 0.0  0.0  0.0 |      23.5       23.5       23.5 | 0.000024  0.000024  0.000024 |   2.0 | 0.284453 | 1.111905 | 0.060652 |   0.7938 | 0.584915 | 0.0 | 0.0 |   13.288284 | -0.475427 |  0.920391 | 1.281983 | 0.148378 |  0.344134   2.259606   2.252569 | -3.000403  -3.000404  -3.000404 | 120.732714 | 1278.318266 | 0.537688 | 425.688335 | 0.148378 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 1.1 | 0.0 | 0.0 |  0.48 | 0.7623  1.287  0.693 | 1.2474 |  0.288966    0.08905   0.242285 | 0.505891  1.286378  0.692918 |  0.288966    0.08905   0.242285 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |  0.48 | 0.046248 |  0.18273 | 0.170509 |   1.2474 | 0.918902 | 0.0 | 0.0 |   27.773851 | -0.399929 |  0.578736 | 1.121226 | 0.129772 |  0.029276   1.364178   1.797366 | -3.000434  -3.000434  -3.000434 | 120.457046 | 1306.092118 | 0.141683 | 379.580818 | 0.129772 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 1.0 | 0.0 | 0.0 |  0.16 |  0.693   1.17   0.63 |  1.134 |  0.056032    0.00073   0.020468 |  0.28711  1.128812  0.629377 |  0.056032    0.00073   0.020468 | 0.0  0.0  0.0 |      38.5       38.5       38.5 |  0.00004   0.00004   0.00004 |  0.16 | 0.007808 | 0.028897 | 0.283266 |    1.134 | 0.834891 | 0.0 | 0.0 |   23.029559 | -0.301853 |  0.140113 | 0.912742 | 0.105641 | -0.153865   0.394636   1.307521 | -3.000473  -3.000474  -3.000474 | 120.430652 | 1329.121677 | 0.030467 | 326.523545 | 0.105641 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 0.8 | 0.0 | 0.0 |  1.36 | 0.5544  0.936  0.504 | 0.9072 |  0.699353   0.001276   0.261852 | 0.321216  0.881369   0.50364 |  0.699353   0.001276   0.261852 | 0.0  0.0  0.0 |      31.5       31.5       31.5 | 0.000032  0.000032  0.000032 |  1.36 | 0.096732 | 0.359154 | 0.173021 |   0.9072 | 0.668103 | 0.0 | 0.0 |   10.406657 | -0.217545 |  0.212161 | 0.739738 | 0.085618 |  0.185566   0.871991   1.902029 | -3.000506  -3.000506  -3.000506 | 120.289397 | 1339.528335 | 0.177459 | 290.791657 | 0.085618 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 0.6 | 0.0 | 0.0 |  0.24 | 0.4158  0.702  0.378 | 0.6804 |   0.13171   0.000231   0.068138 | 0.255779  0.664282  0.377897 |   0.13171   0.000231   0.068138 | 0.0  0.0  0.0 | 18.500006  18.500006  18.500006 | 0.000019  0.000019  0.000019 |  0.24 | 0.018186 |  0.06956 | 0.118812 |   0.6804 | 0.501131 | 0.0 | 0.0 |   13.902789 | -0.170184 |  0.194342 | 0.632213 | 0.073173 |  0.038077   0.447478   1.695993 | -3.000525  -3.000525  -3.000525 | 120.219838 | 1353.431124 | 0.052677 | 260.758231 | 0.073173 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 | 0.4851  0.819  0.441 | 0.7938 |       0.0        0.0        0.0 | 0.203146  0.529544  0.440411 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  8.006707   8.006707   8.006707 | 0.000008  0.000008  0.000008 |   0.0 | 0.000002 | 0.000007 | 0.283401 |   0.7938 | 0.584131 | 0.0 | 0.0 |   11.248881 | -0.128411 |   0.04549 |  0.52902 | 0.061229 | -0.165069  -0.082066   1.255582 | -3.000533  -3.000533  -3.000533 | 120.374826 | 1364.680004 | 0.007194 | 230.079543 | 0.061229 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 0.7 | 0.0 | 0.0 |  1.44 | 0.4851  0.819  0.441 | 0.7938 |  0.767777   0.000464   0.302021 | 0.289622  0.646586  0.440662 |  0.767777   0.000464   0.302021 | 0.0  0.0  0.0 | 20.000002  20.000002  20.000002 | 0.000021  0.000021  0.000021 |  1.44 | 0.106174 | 0.395471 | 0.187092 |   0.7938 | 0.584471 | 0.0 | 0.0 |    0.341947 | -0.094452 |   0.20907 | 0.450415 | 0.052131 |  0.217532   0.710883   1.952899 | -3.000553  -3.000554  -3.000554 | 120.361293 | 1365.021951 | 0.193595 | 214.284134 | 0.052131 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 0.5 | 0.0 | 0.0 |  7.12 | 0.3465  0.585  0.315 |  0.567 |  6.357929   4.564264   6.159258 | 0.317365  0.584263  0.314994 |  6.357929   4.564264   6.159258 | 0.0  0.0  0.0 |      28.0       28.0       28.0 | 0.000029  0.000029  0.000029 |  7.12 | 1.182345 | 4.735881 | 0.019596 |    0.567 | 0.417878 | 0.0 | 0.0 |  -82.382297 |  -0.09919 |  2.708243 | 0.555379 |  0.06428 |  0.662237   2.682356   2.598647 | -3.000582  -3.000583  -3.000583 | 119.099354 | 1282.639654 | 2.221233 | 321.397786 |  0.06428 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 | 0.5544  0.936  0.504 | 0.9072 |       0.0        0.0        0.0 | 0.473399   0.93593  0.503987 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.0       29.0       29.0 |  0.00003   0.00003   0.00003 |   0.0 | 0.000006 | 0.000024 | 0.053829 |   0.9072 | 0.668534 | 0.0 | 0.0 |   31.104587 |  -0.27678 |  1.920644 | 0.919603 | 0.106435 |  0.188839   1.746427    2.09466 | -3.000612  -3.000612  -3.000612 | 118.876398 | 1313.744242 | 0.300613 |  356.41602 | 0.106435 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 1.0 | 0.0 | 0.0 |   0.0 |  0.693   1.17   0.63 |  1.134 |       0.0        0.0        0.0 | 0.342819  1.155656  0.629755 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      28.5       28.5       28.5 | 0.000029  0.000029  0.000029 |   0.0 | 0.000006 | 0.000024 | 0.237378 |    1.134 | 0.835104 | 0.0 | 0.0 |   16.461632 | -0.266718 |  0.259927 | 0.849051 |  0.09827 |  -0.15398   0.590771   1.464905 | -3.000641  -3.000642  -3.000642 | 118.847052 | 1330.205874 |  0.04071 |  313.80365 |  0.09827 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.194384  0.813489  0.748184 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      23.0       23.0       23.0 | 0.000024  0.000024  0.000024 |   0.0 | 0.000005 | 0.000019 | 0.619401 |   1.3608 | 1.000313 | 0.0 | 0.0 |   14.359158 |  -0.18955 |  0.035263 | 0.687592 | 0.079582 | -0.348364  -0.222718   0.716721 | -3.000665  -3.000665  -3.000665 | 119.276898 | 1344.565033 | 0.005466 | 271.431958 | 0.079582 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 0.9 | 0.0 | 0.0 |  0.72 | 0.6237  1.053  0.567 | 1.0206 |  0.198314   0.000023   0.002957 | 0.206879  0.452527  0.552684 |  0.198314   0.000023   0.002957 | 0.0  0.0  0.0 | 21.000001  21.000001  21.000001 | 0.000022  0.000022  0.000022 |  0.72 | 0.027036 |  0.09498 | 0.476296 |   1.0206 | 0.749357 | 0.0 | 0.0 |   10.433321 | -0.133723 |  0.052482 | 0.559927 | 0.064806 | -0.033557   0.044731    0.88108 | -3.000687  -3.000687  -3.000687 | 119.592435 | 1354.998354 | 0.047964 | 239.809465 | 0.064806 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 0.9 | 0.0 | 0.0 |  0.08 | 0.6237  1.053  0.567 | 1.0206 |  0.023913   0.000002   0.000192 | 0.223902  0.377698  0.535186 |  0.023913   0.000002   0.000192 | 0.0  0.0  0.0 |      37.0       37.0       37.0 | 0.000038  0.000038  0.000038 |  0.08 | 0.003281 | 0.011453 | 0.489829 |   1.0206 | 0.747587 | 0.0 | 0.0 |   10.547625 | -0.098581 |  0.048575 | 0.469435 | 0.054333 | -0.201372  -0.252968   0.425702 | -3.000725  -3.000725  -3.000725 | 119.980402 | 1365.545979 | 0.010841 | 213.429845 | 0.054333 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.171771  0.216652  0.491877 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      31.5       31.5       31.5 | 0.000032  0.000032  0.000032 |   0.0 | 0.000007 | 0.000026 | 0.832568 |   1.3608 | 0.974442 | 0.0 | 0.0 |    9.399994 | -0.072149 |   0.00917 | 0.393345 | 0.045526 | -0.373143   -0.46962  -0.066175 | -3.000757  -3.000757  -3.000757 | 120.740815 | 1374.945973 | 0.001698 | 189.669362 | 0.045526 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 1.4 | 0.0 | 0.0 |   0.0 | 0.9702  1.638  0.882 | 1.5876 |       0.0        0.0        0.0 | 0.116297  0.128142  0.249606 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      43.5       43.5       43.5 | 0.000045  0.000045  0.000045 |   0.0 | 0.000009 | 0.000036 | 1.068503 |   1.5876 |  1.10469 | 0.0 | 0.0 |    9.198092 | -0.052742 |  0.001156 | 0.330509 | 0.038253 | -0.489441  -0.597762   -0.31578 | -3.000802  -3.000802  -3.000802 | 121.756567 | 1384.144065 | 0.000578 | 169.287036 | 0.038253 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 1.1 | 0.0 | 0.0 |  3.12 | 0.7623  1.287  0.693 | 1.2474 |  1.818463   0.016206   0.250957 | 0.484994  0.989435  0.602563 |  1.818463   0.016206   0.250957 | 0.0  0.0  0.0 |      32.0       32.0       32.0 | 0.000033  0.000033  0.000033 |  3.12 | 0.258106 | 0.888773 | 0.282973 |   1.2474 | 0.909652 | 0.0 | 0.0 |  -12.429032 | -0.040447 |  0.436309 | 0.301186 | 0.034859 |  0.327102   1.516597     1.9507 | -3.000834  -3.000835  -3.000835 | 121.740987 | 1371.715033 | 0.453041 | 175.695773 | 0.034859 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 1.1 | 0.0 | 0.0 |  6.96 | 0.7623  1.287  0.693 | 1.2474 |  6.071563   4.670998   5.753257 | 0.685968  1.286933  0.692984 |  6.071563   4.670998   5.753257 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |  6.96 | 1.175796 | 4.561373 | 0.050718 |   1.2474 | 0.919301 | 0.0 | 0.0 |  -76.879449 | -0.066745 |  2.896137 | 0.467407 | 0.054098 |   0.52957   2.518667   2.464459 | -3.000861  -3.000862  -3.000862 | 120.549163 | 1294.835584 | 2.118277 | 297.005263 | 0.054098 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 0.5 | 0.0 | 0.0 | 21.12 | 0.3465  0.585  0.315 |  0.567 | 20.402784  20.094611  20.410298 | 0.337475  0.584999  0.314999 | 20.402784  20.094611  20.410298 | 0.0  0.0  0.0 |      25.5       25.5       25.5 | 0.000026  0.000026  0.000026 | 21.12 |    4.078 | 16.64091 | 0.005996 |    0.567 | 0.417923 | 0.0 | 0.0 | -185.213968 |  -0.50063 | 11.201721 | 1.596319 | 0.184759 |  0.909311   2.959057   2.859162 | -3.000887  -3.000888  -3.000888 | 115.976529 | 1109.621616 | 7.557466 |  764.54885 | 0.184759 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 0.6 | 0.0 | 0.0 |   9.2 | 0.4158  0.702  0.378 | 0.6804 |  8.931935   8.697486   8.951417 | 0.406537  0.701998  0.377999 |  8.931935   8.697486   8.951417 | 0.0  0.0  0.0 |      43.5       43.5       43.5 | 0.000045  0.000045  0.000045 |   9.2 | 1.660716 | 7.368949 | 0.006156 |   0.6804 | 0.501513 | 0.0 | 0.0 |  -25.508851 | -2.410978 | 10.723004 |  3.84388 | 0.444894 |   0.77084   2.759573   2.729747 | -3.000932  -3.000932  -3.000932 | 111.910991 | 1084.112766 | 4.203411 | 999.977506 | 0.444894 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 1.5 | 0.0 | 0.0 |  0.72 | 1.0395  1.755  0.945 |  1.701 |  0.554126   0.276001   0.548122 | 0.842875  1.754558  0.944963 |  0.554126   0.276001   0.548122 | 0.0  0.0  0.0 |      31.0       31.0       31.0 | 0.000032  0.000032  0.000032 |  0.72 | 0.081905 | 0.396405 | 0.130834 |    1.701 | 1.253473 | 0.0 | 0.0 |    4.324072 | -2.598522 |  3.885166 | 4.067216 | 0.470743 |  0.093839   1.449014   1.956662 | -3.000964  -3.000964  -3.000964 | 109.361398 | 1088.436838 |  0.71465 | 871.414002 | 0.470743 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |  1.386   2.34   1.26 |  2.268 |       0.0        0.0        0.0 | 0.443211   1.68771  1.251505 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |   0.0 | 0.000005 | 0.000026 | 0.843462 |    2.268 | 1.668736 | 0.0 | 0.0 |   17.323598 | -1.448092 |  0.617802 | 2.979626 | 0.344864 | -0.349371  -0.238696   0.705157 | -3.000994  -3.000994  -3.000994 | 108.756763 | 1105.760436 | 0.096874 | 686.576121 | 0.344864 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.139085  0.270011  0.749585 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |   0.0 | 0.000005 | 0.000023 | 1.176986 |   1.8144 | 1.309014 | 0.0 | 0.0 |   49.332105 | -0.755004 |  0.083772 | 2.101813 | 0.243265 | -0.488456  -0.508708  -0.044428 | -3.001021  -3.001021  -3.001021 | 109.178741 | 1155.092541 | 0.013125 | 550.480234 | 0.243265 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.087831  0.124898  0.285202 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |   0.0 | 0.000005 | 0.000022 | 1.259234 |   1.8144 | 1.263247 | 0.0 | 0.0 |   49.278807 |  -0.44948 |  0.011353 | 1.536662 | 0.177854 | -0.576287  -0.633605  -0.329631 | -3.001047  -3.001048  -3.001048 |  109.98849 | 1204.371348 | 0.001795 | 452.566967 | 0.177854 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.049333  0.062911  0.109412 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |   0.0 | 0.000005 | 0.000025 | 0.965599 |   1.3608 |   0.9371 | 0.0 | 0.0 |   38.991441 | -0.290326 |  0.001575 | 1.164145 | 0.134739 | -0.625621  -0.696516  -0.439043 | -3.001078  -3.001078  -3.001078 | 110.663757 | 1243.362789 | 0.000245 | 380.272758 | 0.134739 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 | 0.9009  1.521  0.819 | 1.4742 |       0.0        0.0        0.0 | 0.043679  0.052843  0.081011 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      33.5       33.5       33.5 | 0.000034  0.000034  0.000034 |   0.0 | 0.000006 | 0.000029 | 1.057594 |   1.4742 |  1.01144 | 0.0 | 0.0 |    29.28041 | -0.194016 |  0.000223 | 0.907791 | 0.105068 | -0.669299  -0.749359  -0.520054 | -3.001112  -3.001112  -3.001112 | 111.527329 | 1272.643199 |  0.00005 | 324.863855 | 0.105068 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 |  0.04431  0.051555  0.072303 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      35.0       35.0       35.0 | 0.000036  0.000036  0.000036 |   0.0 | 0.000007 |  0.00003 |  1.31233 |   1.8144 |  1.24206 | 0.0 | 0.0 |   22.837178 | -0.131211 |   0.00006 | 0.723661 | 0.083757 | -0.713609  -0.800914  -0.592357 | -3.001148  -3.001148  -3.001148 | 112.708442 | 1295.480378 |  0.00002 | 281.082904 | 0.083757 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 1.9 | 0.0 | 0.0 |   0.0 | 1.3167  2.223  1.197 | 2.1546 |       0.0        0.0        0.0 |  0.04334  0.048893  0.064061 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      41.5       41.5       41.5 | 0.000043  0.000043  0.000043 |   0.0 | 0.000008 | 0.000035 | 1.568521 |   2.1546 | 1.472664 | 0.0 | 0.0 |   19.142277 | -0.089595 |  0.000038 | 0.586811 | 0.067918 | -0.756949  -0.849808  -0.656417 |  -3.00119  -3.001191  -3.001191 |  114.18736 | 1314.622654 | 0.000018 | 245.638485 | 0.067918 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 0.8 | 0.0 | 0.0 |   1.2 | 0.5544  0.936  0.504 | 0.9072 |  0.261855   0.000022   0.000144 | 0.145067  0.238297  0.204217 |  0.261855   0.000022   0.000144 | 0.0  0.0  0.0 |      45.0       45.0       45.0 | 0.000046  0.000046  0.000046 |   1.2 | 0.032993 | 0.127486 | 0.503791 |   0.9072 | 0.637859 | 0.0 | 0.0 |    11.98165 | -0.062939 |  0.052181 | 0.488188 | 0.056503 |  0.036129   0.111873   0.339221 | -3.001236  -3.001237  -3.001237 | 114.595218 | 1326.604304 | 0.075323 | 221.303174 | 0.056503 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 1.5 | 0.0 | 0.0 |  0.24 | 1.0395  1.755  0.945 |  1.701 |  0.076999   0.000006    0.00005 | 0.397315  0.624883  0.602032 |  0.076999   0.000006    0.00005 | 0.0  0.0  0.0 |      62.0       62.0       62.0 | 0.000064  0.000064  0.000064 |  0.24 | 0.009772 | 0.037475 | 0.802147 |    1.701 |  1.21785 | 0.0 | 0.0 |   11.000393 | -0.046015 |   0.08814 | 0.422891 | 0.048946 | -0.198185  -0.273016  -0.022861 |   -3.0013    -3.0013    -3.0013 | 115.341579 | 1337.604697 | 0.024658 | 200.987241 | 0.048946 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 2.7 | 0.0 | 0.0 |  0.16 | 1.8711  3.159  1.701 | 3.0618 |  0.023623   0.000001   0.000007 | 0.346813  0.402176  0.468844 |  0.023623   0.000001   0.000007 | 0.0  0.0  0.0 |      75.0       75.0       75.0 | 0.000077  0.000077  0.000077 |  0.16 |  0.00307 | 0.011489 | 1.928512 |   3.0618 | 2.130296 | 0.0 | 0.0 |   11.865254 | -0.032074 |  0.029402 | 0.361631 | 0.041855 |  -0.40862  -0.515194  -0.331711 | -3.001377  -3.001377  -3.001377 | 117.234947 | 1349.469952 | 0.006745 |  180.03015 | 0.041855 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 1.5 | 0.0 | 0.0 |   3.6 | 1.2825  1.755  0.945 |  1.728 |    1.9917   0.031322   0.415839 | 0.874859   1.42793  0.831929 |    1.9917   0.031322   0.415839 | 0.0  0.0  0.0 |      59.0       59.0       59.0 |  0.00006   0.00006   0.00006 |   3.6 | 0.264801 |  1.00774 | 0.379453 |    1.728 | 1.384995 | 0.0 | 0.0 |   -9.849407 | -0.023634 |  0.504803 | 0.332943 | 0.038535 |  0.324821   1.625554    2.02052 | -3.001437  -3.001438  -3.001438 | 117.325965 | 1339.620545 | 0.509683 | 188.948043 | 0.038535 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 |  1.368  1.872  1.008 | 1.8432 |       0.0        0.0        0.0 | 0.610003  1.639835  1.005964 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      47.0       47.0       47.0 | 0.000048  0.000048  0.000048 |   0.0 |  0.00001 |  0.00004 | 0.580673 |   1.8432 | 1.488396 | 0.0 | 0.0 |   14.747024 | -0.038379 |  0.440782 | 0.366954 | 0.042471 | -0.285183  -0.014281   1.014556 | -3.001485  -3.001486  -3.001486 | 117.868251 | 1354.367569 |  0.06894 | 188.662969 | 0.042471 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |   1.71   2.34   1.26 |  2.304 |       0.0        0.0        0.0 | 0.226994  0.478502  1.030203 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      40.5       40.5       40.5 | 0.000041  0.000041  0.000041 |   0.0 | 0.000008 | 0.000034 | 1.603398 |    2.304 | 1.835389 | 0.0 | 0.0 |   11.605931 | -0.036231 |  0.059676 | 0.332845 | 0.038524 | -0.512177  -0.492783  -0.015647 | -3.001527  -3.001527  -3.001527 | 119.435409 |   1365.9735 | 0.009298 | 171.043041 | 0.038524 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 2.1 | 0.0 | 0.0 |   0.0 | 1.7955  2.457  1.323 | 2.4192 |       0.0        0.0        0.0 | 0.119524  0.161068  0.355469 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      39.5       39.5       39.5 |  0.00004   0.00004   0.00004 |   0.0 | 0.000008 | 0.000033 | 1.875642 |   2.4192 |  1.85517 | 0.0 | 0.0 |   12.526556 | -0.027831 |  0.008696 | 0.284393 | 0.032916 | -0.631701  -0.653851  -0.371116 | -3.001567  -3.001568  -3.001568 | 121.283211 | 1378.500056 | 0.000635 | 153.602915 | 0.032916 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 1.7 | 0.0 | 0.0 |  0.32 | 1.4535  1.989  1.071 | 1.9584 |  0.014432   0.000001   0.000008 | 0.119617  0.145282  0.209544 |  0.014432   0.000001   0.000008 | 0.0  0.0  0.0 |      37.5       37.5       37.5 | 0.000038  0.000038  0.000038 |  0.32 | 0.002057 | 0.006848 |  1.49806 |   1.9584 |  1.49404 | 0.0 | 0.0 |   12.228509 | -0.022533 |  0.004273 | 0.243282 | 0.028158 | -0.445751  -0.479133  -0.260668 | -3.001606  -3.001606  -3.001606 | 122.756681 | 1390.728566 |  0.00321 | 139.016072 | 0.028158 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 1.7 | 0.0 | 0.0 |   0.0 | 1.4535  1.989  1.071 | 1.9584 |       0.0        0.0        0.0 | 0.126278  0.143979  0.175695 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      36.0       36.0       36.0 | 0.000037  0.000037  0.000037 |   0.0 | 0.000008 |  0.00003 | 1.493898 |   1.9584 | 1.490585 | 0.0 | 0.0 |   11.557872 | -0.019112 |  0.003012 | 0.209656 | 0.024266 | -0.572028  -0.623112  -0.436363 | -3.001642  -3.001643  -3.001643 | 124.231458 | 1402.286437 | 0.000228 | 125.879609 | 0.024266 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 |  0.684  0.936  0.504 | 0.9216 |       0.0        0.0        0.0 | 0.042012  0.045674  0.053504 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      40.5       40.5       40.5 | 0.000041  0.000041  0.000041 |   0.0 | 0.000009 | 0.000033 | 0.721842 |   0.9216 | 0.698471 | 0.0 | 0.0 |    9.523123 | -0.016684 |  0.000212 | 0.181968 | 0.021061 |  -0.61404  -0.668787  -0.489867 | -3.001684  -3.001684  -3.001684 | 124.936606 | 1411.809561 | 0.000049 | 115.134203 | 0.021061 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 | 1.1115  1.521  0.819 | 1.4976 |       0.0        0.0        0.0 | 0.055222  0.058913  0.067492 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      43.0       43.0       43.0 | 0.000044  0.000044  0.000044 |   0.0 |  0.00001 | 0.000035 | 1.186647 |   1.4976 |  1.13298 | 0.0 | 0.0 |    8.230108 |  -0.01426 |  0.000056 | 0.158913 | 0.018393 | -0.669262  -0.727699  -0.557359 | -3.001728  -3.001728  -3.001728 | 126.108984 | 1420.039669 | 0.000028 | 105.064852 | 0.018393 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 2.5 | 0.0 | 0.0 |   0.0 | 2.1375  2.925  1.575 |   2.88 |       0.0        0.0        0.0 | 0.078869  0.082516  0.092006 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      52.5       52.5       52.5 | 0.000054  0.000054  0.000054 |   0.0 | 0.000012 | 0.000043 | 2.310038 |     2.88 | 2.174705 | 0.0 | 0.0 |   10.517266 | -0.012375 |  0.000048 | 0.137392 | 0.015902 | -0.748131  -0.810216  -0.649366 | -3.001781  -3.001782  -3.001782 | 128.406634 | 1430.556935 | 0.000022 |  94.771882 | 0.015902 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_v002_seepage:

seepage
_______

>>> inputs.fxg.series = 10.0

.. integration-test::

    >>> test("wland_v002_seepage",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |    t |    p | pet |  fxg | fxs |    pc |                 petl |    pes |                              tf |                           ei |                              rf |            sf |                              pm |                           am |    ps |       pv |        pq |      etv |       es |       et | fxs |       fxg |         cdg |       fgs |       fqs |       rh |        r |                              ic |                              sp |          dv |          dg |       hq |          hs |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 0.6 | 10.0 | 0.0 |   0.0 | 0.3942  0.702  0.378 | 0.6588 |       0.0        0.0        0.0 |      0.0  0.000001       0.0 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.000117   0.000117   0.000117 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.494343 | 0.400191 | 0.444015 | 0.0 | 11.337868 | -441.309773 |  0.327496 |       0.0 | 0.001252 | 0.000145 |      -3.0  -3.000001       -3.0 |      -3.0       -3.0       -3.0 |   129.48397 | 1158.690227 |      0.0 |   11.979809 | 0.000145 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 0.8 | 10.0 | 0.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.000001        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.009388   0.009388   0.009388 |      0.0       0.0       0.0 |  0.32 |      0.0 |       0.0 | 0.659727 |   0.8784 | 0.599448 | 0.0 | 11.337868 |   -75.61766 |  0.677019 |       0.0 | 0.018407 |  0.00213 | -2.680002  -2.680003  -2.680002 |      -3.0       -3.0       -3.0 |  119.482847 | 1083.072567 |      0.0 |   40.357603 |  0.00213 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 0.7 | 10.0 | 0.0 |   0.0 | 0.4599  0.819  0.441 | 0.7686 |       0.0        0.0        0.0 | 0.000002  0.000004  0.000002 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.069591   0.069591   0.069591 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.577679 |   0.7686 | 0.524888 | 0.0 | 11.337868 |  -69.597799 |  0.891538 |       0.0 |  0.06084 | 0.007042 | -2.680004  -2.680007  -2.680004 |      -3.0       -3.0       -3.0 |  109.614196 | 1013.474769 |      0.0 |   75.863798 | 0.007042 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 0.4 | 10.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.006707   0.006707   0.006707 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.330299 |   0.4392 | 0.300109 | 0.0 | 11.337868 |  -70.558321 |  1.110172 |       0.0 | 0.131061 | 0.015169 | -2.680005  -2.680009  -2.680005 |      -3.0       -3.0       -3.0 |   99.716798 |  942.916447 |      0.0 |  117.830156 | 0.015169 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 0.4 | 10.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.018374   0.018374   0.018374 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.330458 |   0.4392 |  0.30025 | 0.0 | 11.337868 |  -71.337687 |  1.339503 |       0.0 | 0.231332 | 0.026775 | -2.680007  -2.680011  -2.680006 |      -3.0       -3.0       -3.0 |   90.048891 |  871.578761 |      0.0 |  164.896456 | 0.026775 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 0.4 | 10.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |   8.50479    8.50479    8.50479 | 0.000009  0.000009  0.000009 |   0.0 | 0.000001 |  0.000008 | 0.330587 |   0.4392 | 0.300363 | 0.0 | 11.337868 |  -71.733386 |  1.575409 |  0.000004 | 0.361738 | 0.041868 | -2.680008  -2.680013  -2.680007 | -3.000008  -3.000009  -3.000009 |   80.617017 |  799.845375 | 0.000003 |  215.846115 | 0.041868 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 0.4 | 10.0 | 0.0 |  0.16 | 0.2628  0.468  0.252 | 0.4392 |  0.000001        0.0        0.0 | 0.000002  0.000003  0.000002 |  0.000001        0.0        0.0 | 0.0  0.0  0.0 | 22.000001  22.000001  22.000001 | 0.000023  0.000023  0.000023 |  0.16 | 0.000002 |  0.000021 |  0.33069 |   0.4392 | 0.300455 | 0.0 | 11.337868 |  -72.274211 |  1.817813 |  0.000015 | 0.521079 |  0.06031 |  -2.52001  -2.520016  -2.520009 | -3.000031  -3.000032  -3.000032 |   71.427649 |  727.571164 |  0.00001 |  269.679233 |  0.06031 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 0.3 | 10.0 | 0.0 |   3.6 | 0.1971  0.351  0.189 | 0.3294 |  0.715966   0.000866   0.002554 | 0.045788  0.103395  0.056195 |  0.715966   0.000866   0.002554 | 0.0  0.0  0.0 |      22.5       22.5       22.5 | 0.000023  0.000023  0.000023 |   3.6 | 0.029375 |   0.40368 |  0.18317 |   0.3294 | 0.230973 | 0.0 | 11.337868 |  -74.370245 |  2.062218 |  0.105045 | 0.715074 | 0.082763 |  0.318236   0.975723   1.021242 | -3.000054  -3.000055  -3.000055 |   62.305793 |  653.200919 | 0.298644 |  333.287169 | 0.082763 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 0.3 | 10.0 | 0.0 |   0.0 | 0.1971  0.351  0.189 | 0.3294 |       0.0        0.0        0.0 |  0.14799  0.341643    0.1863 |       0.0        0.0        0.0 | 0.0  0.0  0.0 | 17.000016  17.000016  17.000016 | 0.000018  0.000018  0.000018 |   0.0 | 0.000001 |  0.000017 | 0.035821 |   0.3294 | 0.243901 | 0.0 | 11.337868 |  -76.641186 |  2.288689 |  0.258175 | 0.971891 | 0.112487 |  0.170246   0.634081   0.834941 | -3.000072  -3.000072  -3.000072 |   53.292434 |  576.559733 | 0.040486 |  397.944963 | 0.112487 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 0.4 | 10.0 | 0.0 |  2.56 | 0.2628  0.468  0.252 | 0.4392 |  2.024047   0.452944   0.899585 | 0.221752   0.46542  0.251473 |  2.024047   0.452944   0.899585 | 0.0  0.0  0.0 |      24.0       24.0       24.0 | 0.000025  0.000025  0.000025 |  2.56 | 0.053722 |  1.391945 | 0.028199 |   0.4392 | 0.325524 | 0.0 | 11.337868 |  -80.483142 |  2.531986 |  0.752747 | 1.262124 | 0.146079 |  0.484447   2.275716   2.243883 | -3.000097  -3.000097  -3.000097 |   44.461029 |  496.076591 | 0.679683 |  485.504799 | 0.146079 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 0.3 | 10.0 | 0.0 |  3.68 | 0.1971  0.351  0.189 | 0.3294 |  3.340109   2.972326    3.20822 | 0.184113  0.350996  0.188997 |  3.340109   2.972326    3.20822 | 0.0  0.0  0.0 |      33.5       33.5       33.5 | 0.000035  0.000035  0.000035 |  3.68 | 0.078869 |  3.145638 | 0.008653 |   0.3294 | 0.244193 | 0.0 | 11.337868 |  -87.800946 |  2.582104 |  2.345023 | 1.796926 | 0.207977 |  0.640224   2.632393   2.526666 | -3.000131  -3.000131  -3.000131 |   35.635049 |  408.275645 | 1.480298 |  627.786042 | 0.207977 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 0.6 | 10.0 | 0.0 |  1.84 | 0.3942  0.702  0.378 | 0.6588 |  1.642161   1.408849   1.618933 |  0.36381  0.701991  0.377995 |  1.642161   1.408849   1.618933 | 0.0  0.0  0.0 |      29.0       29.0       29.0 |  0.00003   0.00003   0.00003 |  1.84 | 0.023727 |  1.548519 |  0.02025 |   0.6588 | 0.488386 | 0.0 | 11.337868 |  -95.251406 |  2.593033 |  2.171734 | 2.461989 | 0.284952 |  0.474253   2.361553   2.369738 | -3.000161  -3.000161  -3.000161 |   26.886736 |  313.024239 | 0.857083 |  726.635535 | 0.284952 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 0.8 | 10.0 | 0.0 |  14.4 | 0.5256  0.936  0.504 | 0.8784 | 13.589478  13.049179  13.564578 | 0.504698  0.935996  0.503997 | 13.589478  13.049179  13.564578 | 0.0  0.0  0.0 |      32.5       32.5       32.5 | 0.000033  0.000033  0.000033 |  14.4 | 0.104482 | 13.330898 | 0.013929 |   0.8784 |  0.65119 | 0.0 | 11.337868 | -118.316903 |  2.297264 |  8.236517 | 3.557066 | 0.411697 |  0.780078   2.776379   2.701163 | -3.000194  -3.000195  -3.000195 |   17.755577 |  194.707336 | 5.951463 | 1067.202505 | 0.411697 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 0.5 | 10.0 | 0.0 | 15.36 | 0.3285  0.585  0.315 |  0.549 | 14.927269  14.656171  14.931694 | 0.322035  0.584999  0.314999 | 14.927269  14.656171  14.931694 | 0.0  0.0  0.0 |      25.0       25.0       25.0 | 0.000026  0.000026  0.000026 | 15.36 | 0.038178 | 14.812047 | 0.004309 |    0.549 | 0.406996 | 0.0 | 11.337868 | -171.354429 |  0.761944 | 13.535909 | 6.280161 | 0.726871 |  0.890774   2.895209    2.81447 |  -3.00022  -3.000221  -3.000221 |    7.145783 |   23.352907 | 7.227601 | 1464.866745 | 0.726871 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 0.8 | 10.0 | 0.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.294508   0.228504   0.294805 | 0.495654  0.935979  0.503995 |  0.294508   0.228504   0.294805 | 0.0  0.0  0.0 |  15.00006   15.00006   15.00006 | 0.000015  0.000015  0.000015 |  0.32 |  0.00005 |  0.274707 | 0.019964 |   0.8784 |  0.65119 | 0.0 | 11.337868 |  -25.762383 |  0.470716 |  6.412004 | 7.794024 | 0.902086 |  0.420612   2.050726   2.335669 | -3.000236  -3.000236  -3.000236 |   -3.701454 |   -2.409476 | 1.090304 | 1409.553961 | 0.902086 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 0.5 | 10.0 | 0.0 |  6.64 | 0.3285  0.585  0.315 |  0.549 |  6.067423   5.421468   6.057732 | 0.308292  0.584994  0.314997 |  6.067423   5.421468   6.057732 | 0.0  0.0  0.0 | 15.500043  15.500043  15.500043 | 0.000016  0.000016  0.000016 |  6.64 |      0.0 |  5.872684 |  0.01347 |    0.549 | 0.406994 | 0.0 | 11.337868 |   -6.780221 |  4.500179 |   4.21241 | 7.327716 | 0.848115 |  0.684897   2.684264    2.60294 | -3.000251  -3.000252  -3.000252 |  -10.525673 |   -9.189697 | 2.750577 | 1454.125153 | 0.848115 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 0.4 | 10.0 | 0.0 |  4.24 | 0.2628  0.468  0.252 | 0.4392 |  3.990618   3.797448   3.989293 | 0.251901  0.467998  0.251998 |  3.990618   3.797448   3.989293 | 0.0  0.0  0.0 |      35.5       35.5       35.5 | 0.000037  0.000037  0.000037 |  4.24 |      0.0 |  3.932571 | 0.007265 |   0.4392 | 0.325597 | 0.0 | 11.337868 |   -7.072739 |  4.217228 |  4.611493 | 7.772177 | 0.899558 |  0.682378   2.658817   2.601648 | -3.000288  -3.000288  -3.000288 |  -17.639048 |  -16.262436 | 2.071655 | 1481.260033 | 0.899558 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 1.3 | 10.0 | 0.0 |  0.56 | 0.8541  1.521  0.819 | 1.4274 |  0.424053   0.206901   0.404342 | 0.695521   1.52066  0.818961 |  0.424053   0.206901   0.404342 | 0.0  0.0  0.0 |      47.0       47.0       47.0 | 0.000048  0.000048  0.000048 |  0.56 |      0.0 |  0.356985 | 0.105809 |   1.4274 |  1.05817 | 0.0 | 11.337868 |   -5.373285 |  6.507355 |  2.016054 | 7.818684 |  0.90494 |  0.122804   1.491256   1.938345 | -3.000336  -3.000337  -3.000337 |  -22.363753 |  -21.635721 | 0.412586 | 1475.219428 |  0.90494 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 0.9 | 10.0 | 0.0 |  2.16 | 0.5913  1.053  0.567 | 0.9882 |  1.504266   0.475364   1.340625 |  0.45369  1.052728  0.566963 |  1.504266   0.475364   1.340625 | 0.0  0.0  0.0 |      23.0       23.0       23.0 | 0.000024  0.000024  0.000024 |  2.16 |      0.0 |  1.179255 | 0.091812 |   0.9882 | 0.732576 | 0.0 | 11.337868 |   -3.307635 |  7.995769 |  0.990315 | 7.840678 | 0.907486 |  0.324849   2.123164   2.190757 |  -3.00036  -3.000361  -3.000361 |   -25.61404 |  -24.943356 | 0.601526 | 1485.496171 | 0.907486 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 0.7 | 10.0 | 0.0 |  1.28 | 0.4851  0.819  0.441 | 0.7938 |  0.950677   0.497096    0.87597 | 0.384085  0.818947  0.440979 |  0.950677   0.497096    0.87597 | 0.0  0.0  0.0 | 18.500006  18.500006  18.500006 | 0.000019  0.000019  0.000019 |  1.28 |      0.0 |  0.807151 | 0.067348 |   0.7938 | 0.585105 | 0.0 | 11.337868 |   -3.305348 |   8.00434 |  0.982014 | 7.909205 | 0.915417 |  0.270086    2.08712   2.153807 | -3.000379   -3.00038   -3.00038 |  -28.880221 |  -28.248704 | 0.426663 | 1491.632207 | 0.915417 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 0.7 | 10.0 | 0.0 |   2.0 | 0.4851  0.819  0.441 | 0.7938 |  1.532258   1.008873    1.46039 | 0.393812  0.818966  0.440983 |  1.532258   1.008873    1.46039 | 0.0  0.0  0.0 |      23.5       23.5       23.5 | 0.000024  0.000024  0.000024 |   2.0 |      0.0 |   1.36808 | 0.060859 |   0.7938 | 0.585107 | 0.0 | 11.337868 |   -3.267029 |  7.944702 |  1.133532 | 7.969103 |  0.92235 |  0.344016   2.259281   2.252434 | -3.000403  -3.000404  -3.000404 |  -32.212529 |  -31.515733 | 0.661211 | 1500.287678 |  0.92235 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 1.1 | 10.0 | 0.0 |  0.48 | 0.7623  1.287  0.693 | 1.2474 |  0.289004   0.088949   0.242189 | 0.505952  1.286374  0.692918 |  0.289004   0.088949   0.242189 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |  0.48 |      0.0 |  0.224336 | 0.171079 |   1.2474 | 0.919439 | 0.0 | 11.337868 |   -3.014921 |   8.35274 |  0.711968 | 8.009542 |  0.92703 |  0.029061   1.363959   1.797327 | -3.000434  -3.000434  -3.000434 |  -35.026579 |  -34.530654 | 0.173579 | 1502.285426 |  0.92703 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 1.0 | 10.0 | 0.0 |  0.16 |  0.693   1.17   0.63 |  1.134 |  0.055998   0.000737   0.020493 | 0.286944  1.128621  0.629375 |  0.055998   0.000737   0.020493 | 0.0  0.0  0.0 |      38.5       38.5       38.5 |  0.00004   0.00004   0.00004 |  0.16 |      0.0 |  0.035909 |  0.28445 |    1.134 | 0.835782 | 0.0 | 11.337868 |   -2.163172 |  8.999878 |  0.171979 | 8.029476 | 0.929337 | -0.153882     0.3946   1.307459 | -3.000473  -3.000474  -3.000474 |   -37.08012 |  -36.693825 | 0.037509 | 1505.159235 | 0.929337 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 0.8 | 10.0 | 0.0 |  1.36 | 0.5544  0.936  0.504 | 0.9072 |  0.699272   0.001281   0.261612 | 0.321169  0.881292  0.503638 |  0.699272   0.001281   0.261612 | 0.0  0.0  0.0 |      31.5       31.5       31.5 | 0.000032  0.000032  0.000032 |  1.36 |      0.0 |  0.446141 | 0.173696 |   0.9072 | 0.668648 | 0.0 | 11.337868 |   -2.128036 |  8.949278 |  0.264427 | 8.061201 | 0.933009 |  0.185678   0.872028   1.902208 | -3.000506  -3.000506  -3.000506 |  -39.295014 |  -38.821861 | 0.219222 | 1510.172073 | 0.933009 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 0.6 | 10.0 | 0.0 |  0.24 | 0.4158  0.702  0.378 | 0.6804 |  0.131675   0.000231   0.068176 | 0.255712  0.664055  0.377897 |  0.131675   0.000231   0.068176 | 0.0  0.0  0.0 | 18.500006  18.500006  18.500006 | 0.000019  0.000019  0.000019 |  0.24 |      0.0 |  0.085911 | 0.119356 |   0.6804 | 0.501505 | 0.0 | 11.337868 |     -2.2859 |  8.983596 |  0.239648 | 8.092409 | 0.936621 |  0.038291   0.447742   1.696135 | -3.000525  -3.000525  -3.000525 |  -41.529931 |  -41.107761 | 0.065486 | 1513.030538 | 0.936621 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 0.7 | 10.0 | 0.0 |   0.0 | 0.4851  0.819  0.441 | 0.7938 |       0.0        0.0        0.0 | 0.203127  0.528999  0.440409 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  8.006707   8.006707   8.006707 | 0.000008  0.000008  0.000008 |   0.0 |      0.0 |  0.000008 | 0.284608 |   0.7938 | 0.585025 | 0.0 | 11.337868 |   -1.909938 |  9.215405 |  0.056508 |  8.11516 | 0.939255 | -0.164836  -0.081258   1.255726 | -3.000533  -3.000533  -3.000533 |  -43.367786 |  -43.017699 | 0.008986 | 1515.646959 | 0.939255 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 0.7 | 10.0 | 0.0 |  1.44 | 0.4851  0.819  0.441 | 0.7938 |  0.768015   0.000467   0.302361 | 0.289689  0.646975   0.44066 |  0.768015   0.000467   0.302361 | 0.0  0.0  0.0 | 20.000002  20.000002  20.000002 | 0.000021  0.000021  0.000021 |  1.44 |      0.0 |  0.491206 |  0.18759 |   0.7938 | 0.585062 | 0.0 | 11.337868 |   -2.002463 |  9.033775 |  0.260782 | 8.143847 | 0.942575 |  0.217459   0.711301   1.952705 | -3.000553  -3.000554  -3.000554 |  -45.484289 |  -45.020162 |  0.23941 | 1520.268637 | 0.942575 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 0.5 | 10.0 | 0.0 |  7.12 | 0.3465  0.585  0.315 |  0.567 |  6.357862   4.564533   6.159076 | 0.317363  0.584264  0.314994 |  6.357862   4.564533   6.159076 | 0.0  0.0  0.0 |      28.0       28.0       28.0 | 0.000029  0.000029  0.000029 |  7.12 |      0.0 |  5.800014 | 0.019668 |    0.567 | 0.417939 | 0.0 | 11.337868 |   -4.423548 |  5.966304 |  3.318541 | 8.245545 | 0.954346 |  0.662235   2.682504   2.598635 | -3.000582  -3.000583  -3.000583 |  -50.836186 |   -49.44371 | 2.720883 | 1540.266902 | 0.954346 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 0.8 | 10.0 | 0.0 |   0.0 | 0.5544  0.936  0.504 | 0.9072 |       0.0        0.0        0.0 | 0.473407   0.93593  0.503987 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.0       29.0       29.0 |  0.00003   0.00003   0.00003 |   0.0 |      0.0 |   0.00003 | 0.054013 |   0.9072 | 0.668701 | 0.0 | 11.337868 |   -5.347613 |  6.689537 |  2.353198 | 8.293707 |  0.95992 |  0.188828   1.746574   2.094648 | -3.000612  -3.000612  -3.000612 |  -55.430505 |  -54.791324 | 0.367714 | 1534.989641 |  0.95992 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 1.0 | 10.0 | 0.0 |   0.0 |  0.693   1.17   0.63 |  1.134 |       0.0        0.0        0.0 | 0.342769  1.155579  0.629754 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      28.5       28.5       28.5 | 0.000029  0.000029  0.000029 |   0.0 |      0.0 |  0.000029 | 0.238268 |    1.134 | 0.835837 | 0.0 | 11.337868 |   -2.288399 |  9.101612 |  0.317906 | 8.286048 | 0.959033 | -0.153941   0.590995   1.464894 | -3.000641  -3.000642  -3.000642 |  -57.428493 |  -57.079722 | 0.049838 | 1536.511712 | 0.959033 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 1.2 | 10.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.194386  0.813195  0.748164 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      23.0       23.0       23.0 | 0.000024  0.000024  0.000024 |   0.0 |      0.0 |  0.000024 | 0.621679 |   1.3608 | 1.002235 | 0.0 | 11.337868 |   -1.408605 |  9.427044 |  0.043089 | 8.299972 | 0.960645 | -0.348328  -0.222201    0.71673 | -3.000665  -3.000665  -3.000665 |  -58.717639 |  -58.488328 | 0.006772 | 1537.996327 | 0.960645 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 0.9 | 10.0 | 0.0 |  0.72 | 0.6237  1.053  0.567 | 1.0206 |  0.198895   0.000023    0.00296 | 0.207398  0.453561  0.552677 |  0.198895   0.000023    0.00296 | 0.0  0.0  0.0 | 21.000001  21.000001  21.000001 | 0.000022  0.000022  0.000022 |  0.72 |      0.0 |  0.119662 | 0.477299 |   1.0206 | 0.750849 | 0.0 | 11.337868 |   -1.371286 |  9.406449 |  0.066023 | 8.314634 | 0.962342 |  -0.03462   0.044215   0.881093 | -3.000687  -3.000687  -3.000687 |  -60.171759 |  -59.859614 | 0.060411 | 1540.023583 | 0.962342 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 0.9 | 10.0 | 0.0 |  0.08 | 0.6237  1.053  0.567 | 1.0206 |  0.023841   0.000002   0.000192 | 0.223276  0.376887  0.534972 |  0.023841   0.000002   0.000192 | 0.0  0.0  0.0 |      37.0       37.0       37.0 | 0.000038  0.000038  0.000038 |  0.08 |      0.0 |  0.014363 | 0.492272 |   1.0206 | 0.749114 | 0.0 | 11.337868 |   -1.451433 |  9.435494 |  0.060692 | 8.330074 | 0.964129 | -0.201738  -0.252674   0.425928 | -3.000725  -3.000725  -3.000725 |  -61.581862 |  -61.311047 | 0.014082 | 1541.658457 | 0.964129 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 1.2 | 10.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.171522   0.21667    0.4912 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      31.5       31.5       31.5 | 0.000032  0.000032  0.000032 |   0.0 |      0.0 |  0.000032 | 0.835747 |   1.3608 | 0.977039 | 0.0 | 11.337868 |   -1.070293 |  9.504299 |  0.012139 | 8.342553 | 0.965573 |  -0.37326  -0.469344  -0.065272 | -3.000757  -3.000757  -3.000757 |  -62.579684 |   -62.38134 | 0.001975 | 1542.904392 | 0.965573 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 1.4 | 10.0 | 0.0 |   0.0 | 0.9702  1.638  0.882 | 1.5876 |       0.0        0.0        0.0 | 0.116223  0.128208  0.249667 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      43.5       43.5       43.5 | 0.000045  0.000045  0.000045 |   0.0 |      0.0 |  0.000045 |  1.07248 |   1.5876 | 1.108179 | 0.0 | 11.337868 |   -0.788817 |  9.537175 |  0.001726 |   8.3518 | 0.966644 | -0.489483  -0.597552  -0.314939 | -3.000802  -3.000802  -3.000802 |  -63.307898 |  -63.170156 | 0.000294 | 1544.400748 | 0.966644 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 1.1 | 10.0 | 0.0 |  3.12 | 0.7623  1.287  0.693 | 1.2474 |   1.81855   0.016239   0.253053 | 0.484984  0.989407  0.602651 |   1.81855   0.016239   0.253053 | 0.0  0.0  0.0 |      32.0       32.0       32.0 | 0.000033  0.000033  0.000033 |  3.12 |      0.0 |   1.12134 | 0.284049 |   1.2474 | 0.910595 | 0.0 | 11.337868 |   -1.724515 |  8.935511 |  0.551642 |  8.37268 |  0.96906 |  0.326983   1.516802   1.949358 | -3.000834  -3.000835  -3.000835 |  -65.426207 |  -64.894671 | 0.569992 | 1548.725862 |  0.96906 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 1.1 | 10.0 | 0.0 |  6.96 | 0.7623  1.287  0.693 | 1.2474 |  6.071439   4.671048   5.751904 | 0.685957  1.286933  0.692984 |  6.071439   4.671048   5.751904 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |  6.96 |      0.0 |  5.619395 | 0.050913 |   1.2474 | 0.919467 | 0.0 | 11.337868 |   -4.678908 |  5.785492 |  3.579887 |  8.45148 | 0.978181 |  0.529587    2.51882    2.46447 | -3.000861  -3.000862  -3.000862 |   -70.92767 |  -69.573579 |   2.6095 | 1562.419129 | 0.978181 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 0.5 | 10.0 | 0.0 | 21.12 | 0.3465  0.585  0.315 |  0.567 | 20.402804  20.094766  20.410313 | 0.337475  0.584999  0.314999 | 20.402804  20.094766  20.410313 | 0.0  0.0  0.0 |      25.5       25.5       25.5 | 0.000026  0.000026  0.000026 | 21.12 |      0.0 |  20.31117 | 0.006017 |    0.567 | 0.417941 | 0.0 | 11.337868 |  -13.741511 | -5.099846 | 13.714547 | 8.666954 |  1.00312 |  0.909308   2.959056   2.859158 | -3.000887  -3.000888  -3.000888 |  -87.359368 |   -83.31509 | 9.206122 | 1596.734061 |  1.00312 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 0.6 | 10.0 | 0.0 |   9.2 | 0.4158  0.702  0.378 | 0.6804 |  8.931936   8.697492   8.951417 | 0.406537  0.701998  0.377999 |  8.931936   8.697492   8.951417 | 0.0  0.0  0.0 |      43.5       43.5       43.5 | 0.000045  0.000045  0.000045 |   9.2 |      0.0 |  8.863595 | 0.006176 |   0.6804 |  0.50153 | 0.0 | 11.337868 |  -17.221206 | -4.551639 | 13.006593 |  8.79988 | 1.018505 |  0.770835   2.759566   2.729743 | -3.000932  -3.000932  -3.000932 |   -103.2427 | -100.536296 | 5.063124 | 1601.855431 | 1.018505 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 1.5 | 10.0 | 0.0 |  0.72 | 1.0395  1.755  0.945 |  1.701 |  0.554136    0.27601   0.548116 | 0.842885  1.754558  0.944963 |  0.554136    0.27601   0.548116 | 0.0  0.0  0.0 |      31.0       31.0       31.0 | 0.000032  0.000032  0.000032 |  0.72 |      0.0 |  0.470128 | 0.131219 |    1.701 | 1.253818 | 0.0 | 11.337868 |   -8.343375 |  4.769682 |  4.675708 | 8.813693 | 1.020103 |  0.093815   1.448998   1.956663 | -3.000964  -3.000964  -3.000964 | -109.679667 | -108.879671 | 0.857544 |  1599.64249 | 1.020103 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 2.0 | 10.0 | 0.0 |   0.0 |  1.386   2.34   1.26 |  2.268 |       0.0        0.0        0.0 | 0.443224  1.687536  1.251484 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |   0.0 |      0.0 |   0.00003 | 0.845974 |    2.268 | 1.670906 | 0.0 | 11.337868 |   -1.958155 |  9.229906 |  0.741361 | 8.811762 |  1.01988 | -0.349409  -0.238537   0.705179 | -3.000994  -3.000994  -3.000994 | -110.941655 | -110.837826 | 0.116214 | 1600.151914 |  1.01988 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 1.6 | 10.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.139055  0.270034  0.749157 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |   0.0 |      0.0 |  0.000027 | 1.180443 |   1.8144 |  1.31201 | 0.0 | 11.337868 |   -0.312602 |  9.920932 |  0.100395 | 8.813482 | 1.020079 | -0.488464  -0.508571  -0.043978 | -3.001021  -3.001021  -3.001021 | -111.178149 | -111.150428 | 0.015846 | 1600.095851 | 1.020079 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 1.6 | 10.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.087824  0.124941  0.285357 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |   0.0 |      0.0 |  0.000027 | 1.262958 |   1.8144 | 1.266555 | 0.0 | 11.337868 |   -0.075795 | 10.021932 |  0.013699 | 8.814087 | 1.020149 | -0.576288  -0.633511  -0.329335 | -3.001047  -3.001048  -3.001048 | -111.231128 | -111.226223 | 0.002174 | 1600.215539 | 1.020149 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 1.2 | 10.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.049332  0.062932  0.109506 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |   0.0 |      0.0 |   0.00003 |   0.9685 |   1.3608 | 0.939674 | 0.0 | 11.337868 |   -0.280474 | 10.033511 |  0.001894 | 8.815368 | 1.020297 |  -0.62562  -0.696443  -0.438841 | -3.001078  -3.001078  -3.001078 | -111.566986 | -111.506697 |  0.00031 | 1600.656991 | 1.020297 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 1.3 | 10.0 | 0.0 |   0.0 | 0.9009  1.521  0.819 | 1.4742 |       0.0        0.0        0.0 | 0.043678  0.052857  0.081065 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      33.5       33.5       33.5 | 0.000034  0.000034  0.000034 |   0.0 |      0.0 |  0.000034 | 1.060824 |   1.4742 | 1.014298 | 0.0 | 11.337868 |   -0.257699 | 10.032794 |  0.000287 | 8.818047 | 1.020607 | -0.669298    -0.7493  -0.519906 | -3.001112  -3.001112  -3.001112 | -111.811236 | -111.764396 | 0.000057 | 1600.740704 | 1.020607 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 1.6 | 10.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.044309  0.051566  0.072339 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      35.0       35.0       35.0 | 0.000036  0.000036  0.000036 |   0.0 |      0.0 |  0.000036 | 1.316424 |   1.8144 | 1.245678 | 0.0 | 11.337868 |   -0.032242 | 10.045224 |   0.00007 | 8.819349 | 1.020758 | -0.713608  -0.800866  -0.592246 | -3.001148  -3.001148  -3.001148 | -111.787456 | -111.796638 | 0.000023 | 1600.956663 | 1.020758 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 1.9 | 10.0 | 0.0 |   0.0 | 1.3167  2.223  1.197 | 2.1546 |       0.0        0.0        0.0 | 0.043339  0.048902  0.064086 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      41.5       41.5       41.5 | 0.000043  0.000043  0.000043 |   0.0 |      0.0 |  0.000043 |  1.57355 |   2.1546 | 1.477104 | 0.0 | 11.337868 |    0.226424 |  10.03735 |  0.000044 | 8.818533 | 1.020664 | -0.756947  -0.849768  -0.656332 |  -3.00119  -3.001191  -3.001191 | -111.514425 | -111.570214 | 0.000022 | 1600.524687 | 1.020664 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 0.8 | 10.0 | 0.0 |   1.2 | 0.5544  0.936  0.504 | 0.9072 |  0.262344   0.000023   0.000145 | 0.145287  0.238686  0.204268 |  0.262344   0.000023   0.000145 | 0.0  0.0  0.0 |      45.0       45.0       45.0 | 0.000046  0.000046  0.000046 |   1.2 |      0.0 |  0.157474 | 0.505162 |   0.9072 | 0.639318 | 0.0 | 11.337868 |   -0.618551 |  9.938123 |  0.064378 | 8.820235 | 1.020861 |  0.035422   0.111523   0.339255 | -3.001236  -3.001237  -3.001237 | -112.409009 | -112.188765 | 0.093117 | 1601.231505 | 1.020861 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 1.5 | 10.0 | 0.0 |  0.24 | 1.0395  1.755  0.945 |  1.701 |  0.076871   0.000006    0.00005 | 0.396709  0.624107  0.601753 |  0.076871   0.000006    0.00005 | 0.0  0.0  0.0 |      62.0       62.0       62.0 | 0.000064  0.000064  0.000064 |  0.24 |      0.0 |  0.046193 | 0.805463 |    1.701 | 1.220163 | 0.0 | 11.337868 |    -0.72093 |  9.938832 |  0.108768 | 8.827675 | 1.021722 | -0.198158   -0.27259  -0.022548 |   -3.0013    -3.0013    -3.0013 | -113.002582 | -112.909695 | 0.030542 |  1602.01887 | 1.021722 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 2.7 | 10.0 | 0.0 |  0.16 | 1.8711  3.159  1.701 | 3.0618 |  0.023626   0.000001   0.000007 | 0.346844  0.403413  0.470504 |  0.023626   0.000001   0.000007 | 0.0  0.0  0.0 |      75.0       75.0       75.0 | 0.000077  0.000077  0.000077 |  0.16 |      0.0 |  0.014253 | 1.934631 |   3.0618 | 2.136238 | 0.0 | 11.337868 |    0.390604 | 10.022251 |  0.035203 | 8.828681 | 1.021838 | -0.408628  -0.516004  -0.333059 | -3.001377  -3.001377  -3.001377 | -112.383569 | -112.519091 | 0.009593 | 1601.389188 | 1.021838 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 1.5 | 10.0 | 0.0 |   3.6 | 1.2825  1.755  0.945 |  1.728 |    1.9917   0.031385    0.41475 | 0.874826  1.427544  0.831626 |    1.9917   0.031385    0.41475 | 0.0  0.0  0.0 |      59.0       59.0       59.0 |  0.00006   0.00006   0.00006 |   3.6 |      0.0 |  1.245971 | 0.380921 |    1.728 | 1.386128 | 0.0 | 11.337868 |   -1.037388 |  9.339768 |  0.628215 | 8.834756 | 1.022541 |  0.324846   1.625066   2.020565 | -3.001437  -3.001438  -3.001438 | -114.000748 | -113.556479 | 0.627348 | 1604.189705 | 1.022541 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 1.6 | 10.0 | 0.0 |   0.0 |  1.368  1.872  1.008 | 1.8432 |       0.0        0.0        0.0 | 0.609807  1.638192   1.00595 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      47.0       47.0       47.0 | 0.000048  0.000048  0.000048 |   0.0 |      0.0 |  0.000048 | 0.583379 |   1.8432 | 1.490182 | 0.0 | 11.337868 |   -1.576833 |  9.483413 |  0.542155 | 8.848908 | 1.024179 | -0.284961  -0.013126   1.014615 | -3.001485  -3.001486  -3.001486 | -115.271824 | -115.133312 | 0.085241 | 1604.685248 | 1.024179 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 2.0 | 10.0 | 0.0 |   0.0 |   1.71   2.34   1.26 |  2.304 |       0.0        0.0        0.0 | 0.227073  0.479042  1.028501 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      40.5       40.5       40.5 | 0.000041  0.000041  0.000041 |   0.0 |      0.0 |  0.000041 | 1.608883 |    2.304 | 1.840266 | 0.0 | 11.337868 |      0.0503 |  9.998478 |  0.073651 | 8.850708 | 1.024387 | -0.512034  -0.492168  -0.013886 | -3.001527  -3.001527  -3.001527 | -115.002331 | -115.083012 | 0.011632 | 1604.387632 | 1.024387 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 2.1 | 10.0 | 0.0 |   0.0 | 1.7955  2.457  1.323 | 2.4192 |       0.0        0.0        0.0 | 0.119567  0.161335  0.357054 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      39.5       39.5       39.5 |  0.00004   0.00004   0.00004 |   0.0 |      0.0 |   0.00004 | 1.882447 |   2.4192 | 1.861431 | 0.0 | 11.337868 |    0.558378 | 10.063876 |  0.010068 | 8.846739 | 1.023928 | -0.631601  -0.653502   -0.37094 | -3.001567  -3.001568  -3.001568 | -114.393877 | -114.524634 | 0.001604 | 1603.941739 | 1.023928 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 1.7 | 10.0 | 0.0 |  0.32 | 1.4535  1.989  1.071 | 1.9584 |  0.014452   0.000001   0.000008 | 0.119771  0.145598  0.209777 |  0.014452   0.000001   0.000008 | 0.0  0.0  0.0 |      37.5       37.5       37.5 | 0.000038  0.000038  0.000038 |  0.32 |      0.0 |  0.008711 | 1.503569 |   1.9584 | 1.499105 | 0.0 | 11.337868 |    0.303241 | 10.045085 |  0.005861 |  8.84257 | 1.023446 | -0.445823  -0.479102  -0.260724 | -3.001606  -3.001606  -3.001606 | -114.183091 | -114.221393 | 0.004454 | 1603.450291 | 1.023446 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 1.7 | 10.0 | 0.0 |   0.0 | 1.4535  1.989  1.071 | 1.9584 |       0.0        0.0        0.0 | 0.126231  0.143967   0.17563 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      36.0       36.0       36.0 | 0.000037  0.000037  0.000037 |   0.0 |      0.0 |  0.000037 | 1.499805 |   1.9584 | 1.495758 | 0.0 | 11.337868 |    0.208967 | 10.057296 |  0.003867 |  8.84045 |   1.0232 | -0.572055  -0.623069  -0.436354 | -3.001642  -3.001643  -3.001643 | -113.963859 | -114.012426 | 0.000623 | 1603.185642 |   1.0232 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 0.8 | 10.0 | 0.0 |   0.0 |  0.684  0.936  0.504 | 0.9216 |       0.0        0.0        0.0 | 0.042007  0.045682  0.053505 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      40.5       40.5       40.5 | 0.000041  0.000041  0.000041 |   0.0 |      0.0 |  0.000041 | 0.724743 |   0.9216 |  0.70103 | 0.0 | 11.337868 |   -0.402346 | 10.058632 |  0.000562 | 8.840831 | 1.023244 | -0.614062  -0.668751  -0.489859 | -3.001684  -3.001684  -3.001684 | -114.518352 | -114.414771 | 0.000103 |  1603.83572 | 1.023244 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 1.3 | 10.0 | 0.0 |   0.0 | 1.1115  1.521  0.819 | 1.4976 |       0.0        0.0        0.0 | 0.055216   0.05892  0.067492 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      43.0       43.0       43.0 | 0.000044  0.000044  0.000044 |   0.0 |      0.0 |  0.000044 | 1.191509 |   1.4976 | 1.137267 | 0.0 | 11.337868 |   -0.173302 | 10.061586 |  0.000114 | 8.844119 | 1.023625 | -0.669278   -0.72767  -0.557351 | -3.001728  -3.001728  -3.001728 | -114.603125 | -114.588074 | 0.000033 |  1603.85368 | 1.023625 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 2.5 | 10.0 | 0.0 |   0.0 | 2.1375  2.925  1.575 |   2.88 |       0.0        0.0        0.0 | 0.078861  0.082522  0.092005 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      52.5       52.5       52.5 | 0.000054  0.000054  0.000054 |   0.0 |      0.0 |  0.000054 | 2.319839 |     2.88 | 2.183346 | 0.0 | 11.337868 |    0.826182 |  10.06762 |  0.000059 | 8.841465 | 1.023318 |  -0.74814  -0.810193  -0.649357 | -3.001781  -3.001782  -3.001782 | -113.553535 | -113.761891 | 0.000028 | 1602.885344 | 1.023318 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_v002_surface_water_supply:

surface water supply
____________________

>>> inputs.fxg.series =  0.0
>>> inputs.fxs.series = 10.0

.. integration-test::

    >>> test("wland_v002_surface_water_supply",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |    t |    p | pet | fxg |  fxs |    pc |                 petl |    pes |                              tf |                           ei |                              rf |            sf |                              pm |                           am |    ps |       pv |        pq |      etv |       es |       et |   fxs | fxg |         cdg |        fgs |       fqs |       rh |        r |                              ic |                              sp |         dv |          dg |       hq |          hs |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 0.6 | 0.0 | 10.0 |   0.0 | 0.3942  0.702  0.378 | 0.6588 |       0.0        0.0        0.0 |      0.0  0.000001       0.0 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.000117   0.000117   0.000117 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.494067 | 0.653311 | 0.448834 | 500.0 | 0.0 |  -72.847619 |  -0.427176 |       0.0 | 0.555336 | 0.064275 |      -3.0  -3.000001       -3.0 |      -3.0       -3.0       -3.0 | 140.066891 | 1527.152381 |      0.0 |  450.741423 | 0.064275 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 0.8 | 0.0 | 10.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.000001        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.009388   0.009388   0.009388 |      0.0       0.0       0.0 |  0.32 |      0.0 |       0.0 | 0.658793 |   0.8784 | 0.598625 | 500.0 | 0.0 | -136.550308 |  -1.822213 |       0.0 | 2.136493 | 0.247279 | -2.680002  -2.680003  -2.680002 |      -3.0       -3.0       -3.0 | 138.903471 | 1390.602073 |      0.0 |  762.998807 | 0.247279 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 0.7 | 0.0 | 10.0 |   0.0 | 0.4599  0.819  0.441 | 0.7686 |       0.0        0.0        0.0 | 0.000002  0.000004  0.000002 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.069591   0.069591   0.069591 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.576555 |   0.7686 | 0.523896 | 500.0 | 0.0 | -121.984043 |  -2.976519 |       0.0 | 3.521857 | 0.407622 | -2.680004  -2.680007  -2.680004 |      -3.0       -3.0       -3.0 | 136.503506 |  1268.61803 |      0.0 |  954.872858 | 0.407622 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 0.4 | 0.0 | 10.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.006707   0.006707   0.006707 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 |  0.32956 |   0.4392 | 0.299457 | 500.0 | 0.0 |  -34.467874 |  -3.878822 |       0.0 | 4.445772 | 0.514557 | -2.680005  -2.680009  -2.680005 |      -3.0       -3.0       -3.0 | 132.954245 | 1234.150156 |      0.0 | 1061.089015 | 0.514557 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 0.4 | 0.0 | 10.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  0.018374   0.018374   0.018374 |      0.0       0.0       0.0 |   0.0 |      0.0 |       0.0 | 0.329681 |   0.4392 | 0.299564 | 500.0 | 0.0 |   -32.62078 |  -4.421922 |       0.0 | 4.969671 | 0.575193 | -2.680007  -2.680011  -2.680006 |      -3.0       -3.0       -3.0 | 128.862004 | 1201.529376 |      0.0 | 1117.159516 | 0.575193 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 0.4 | 0.0 | 10.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |   8.50479    8.50479    8.50479 | 0.000009  0.000009  0.000009 |   0.0 | 0.000002 |  0.000007 | 0.329803 |   0.4392 | 0.299672 | 500.0 | 0.0 |  -31.773217 |  -4.649602 |  0.000004 | 5.260599 | 0.608866 | -2.680008  -2.680013  -2.680007 | -3.000008  -3.000009  -3.000009 | 124.542203 | 1169.756159 | 0.000003 | 1148.643122 | 0.608866 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 0.4 | 0.0 | 10.0 |  0.16 | 0.2628  0.468  0.252 | 0.4392 |  0.000001        0.0        0.0 | 0.000002  0.000003  0.000002 |  0.000001        0.0        0.0 | 0.0  0.0  0.0 | 22.000001  22.000001  22.000001 | 0.000023  0.000023  0.000023 |  0.16 | 0.000005 |  0.000019 | 0.329919 |   0.4392 | 0.299774 | 500.0 | 0.0 |  -31.175524 |  -4.714985 |  0.000013 | 5.435694 | 0.629131 |  -2.52001  -2.520016  -2.520009 | -3.000031  -3.000032  -3.000032 | 120.157132 | 1138.580635 | 0.000008 | 1168.649061 | 0.629131 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 0.3 | 0.0 | 10.0 |   3.6 | 0.1971  0.351  0.189 | 0.3294 |  0.716522   0.000865   0.002553 | 0.045818  0.103401  0.056198 |  0.716522   0.000865   0.002553 | 0.0  0.0  0.0 |      22.5       22.5       22.5 | 0.000023  0.000023  0.000023 |   3.6 | 0.093016 |  0.346736 | 0.182733 |   0.3294 | 0.230606 | 500.0 | 0.0 |  -34.331941 |  -4.716092 |  0.090039 | 5.569049 | 0.644566 |   0.31765   0.975719    1.02124 | -3.000054  -3.000055  -3.000055 | 115.530757 | 1104.248695 | 0.256705 | 1189.899481 | 0.644566 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 0.3 | 0.0 | 10.0 |   0.0 | 0.1971  0.351  0.189 | 0.3294 |       0.0        0.0        0.0 | 0.147887  0.341617  0.186298 |       0.0        0.0        0.0 | 0.0  0.0  0.0 | 17.000016  17.000016  17.000016 | 0.000018  0.000018  0.000018 |   0.0 | 0.000003 |  0.000015 | 0.035821 |   0.3294 | 0.243833 | 500.0 | 0.0 |   -31.54252 |   -4.73086 |  0.221365 | 5.722816 | 0.662363 |  0.169764   0.634101   0.834942 | -3.000072  -3.000072  -3.000072 | 110.835715 | 1072.706174 | 0.035355 | 1205.645253 | 0.662363 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 0.4 | 0.0 | 10.0 |  2.56 | 0.2628  0.468  0.252 | 0.4392 |  2.023732   0.453003   0.899733 | 0.221727  0.465424  0.251476 |  2.023732   0.453003   0.899733 | 0.0  0.0  0.0 |      24.0       24.0       24.0 | 0.000025  0.000025  0.000025 |  2.56 | 0.254687 |   1.21092 | 0.028158 |   0.4392 | 0.325473 | 500.0 | 0.0 |  -38.682073 |  -4.710334 |  0.652981 | 5.862583 |  0.67854 |  0.484305   2.275674   2.243733 | -3.000097  -3.000097  -3.000097 | 105.898852 | 1034.024101 | 0.593294 | 1238.907249 |  0.67854 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 0.3 | 0.0 | 10.0 |  3.68 | 0.1971  0.351  0.189 | 0.3294 |  3.340013   2.972315   3.208118 | 0.184109  0.350996  0.188997 |  3.340013   2.972315   3.208118 | 0.0  0.0  0.0 |      33.5       33.5       33.5 | 0.000035  0.000035  0.000035 |  3.68 | 0.499651 |  2.766863 | 0.008639 |   0.3294 | 0.244178 | 500.0 | 0.0 |   -45.73599 |  -4.991424 |  2.057405 | 6.254213 | 0.723867 |  0.640182   2.632363   2.526618 | -3.000131  -3.000131  -3.000131 | 100.416416 |  988.288111 | 1.302752 | 1310.238237 | 0.723867 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 0.6 | 0.0 | 10.0 |  1.84 | 0.3942  0.702  0.378 | 0.6588 |  1.642124   1.408767    1.61889 | 0.363805  0.701991  0.377995 |  1.642124   1.408767    1.61889 | 0.0  0.0  0.0 |      29.0       29.0       29.0 |  0.00003   0.00003   0.00003 |  1.84 | 0.219347 |  1.372411 | 0.020219 |   0.6588 | 0.488355 | 500.0 | 0.0 |  -40.305953 |  -5.298444 |   1.91591 | 6.672962 | 0.772334 |  0.474254   2.361605   2.369733 | -3.000161  -3.000161  -3.000161 |  94.918844 |  947.982158 | 0.759252 | 1337.989595 | 0.772334 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 0.8 | 0.0 | 10.0 |  14.4 | 0.5256  0.936  0.504 | 0.8784 |  13.58948  13.049203  13.564576 | 0.504698  0.935996  0.503997 |  13.58948  13.049203  13.564576 | 0.0  0.0  0.0 |      32.5       32.5       32.5 | 0.000033  0.000033  0.000033 |  14.4 | 1.641743 | 11.947372 | 0.013907 |   0.8784 | 0.651171 | 500.0 | 0.0 | -117.971595 |  -7.702903 |  7.360121 | 7.422974 |  0.85914 |  0.780076   2.776406    2.70116 | -3.000194  -3.000195  -3.000195 |  85.588105 |  830.010563 | 5.346503 | 1501.310427 |  0.85914 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 0.5 | 0.0 | 10.0 | 15.36 | 0.3285  0.585  0.315 |  0.549 | 14.927268   14.65619  14.931693 | 0.322035  0.584999  0.314999 | 14.927268   14.65619  14.931693 | 0.0  0.0  0.0 |      25.0       25.0       25.0 | 0.000026  0.000026  0.000026 | 15.36 | 1.319604 |  13.65877 | 0.004304 |    0.549 | 0.406992 | 500.0 | 0.0 | -180.915873 |  -16.25481 | 12.327521 | 8.015642 | 0.927736 |  0.890773   2.895217   2.814468 |  -3.00022  -3.000221  -3.000221 |  68.017995 |   649.09469 | 6.677751 | 1502.550734 | 0.927736 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 0.8 | 0.0 | 10.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.294507   0.228506   0.294804 | 0.495652  0.935979  0.503995 |  0.294507   0.228506   0.294804 | 0.0  0.0  0.0 |  15.00006   15.00006   15.00006 | 0.000015  0.000015  0.000015 |  0.32 |  0.01628 |  0.260099 |  0.01995 |   0.8784 | 0.651176 | 500.0 | 0.0 |  -15.447994 |  -8.917671 |  5.927614 | 8.005939 | 0.926613 |  0.420615   2.050731   2.335669 | -3.000236  -3.000236  -3.000236 |  59.103993 |  633.646696 | 1.010236 | 1498.879183 | 0.926613 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 0.5 | 0.0 | 10.0 |  6.64 | 0.3285  0.585  0.315 |  0.549 |  6.067446   5.421533   6.057752 | 0.308293  0.584994  0.314997 |  6.067446   5.421533   6.057752 | 0.0  0.0  0.0 | 15.500043  15.500043  15.500043 | 0.000016  0.000016  0.000016 |  6.64 |  0.27656 |  5.623815 |  0.01346 |    0.549 | 0.406985 | 500.0 | 0.0 |      -57.21 |  -6.802647 |   4.00067 |  8.00239 | 0.926203 |  0.684876   2.684204    2.60292 | -3.000251  -3.000252  -3.000252 |  52.038246 |  576.436696 | 2.633381 | 1500.886779 | 0.926203 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 0.4 | 0.0 | 10.0 |  4.24 | 0.2628  0.468  0.252 | 0.4392 |  3.990605   3.797404    3.98928 |   0.2519  0.467998  0.251998 |  3.990605   3.797404    3.98928 | 0.0  0.0  0.0 |      35.5       35.5       35.5 | 0.000037  0.000037  0.000037 |  4.24 | 0.139369 |  3.807117 | 0.007261 |   0.4392 | 0.325593 | 500.0 | 0.0 |  -64.760529 |  -7.279615 |  4.436518 | 8.006113 | 0.926633 |  0.682371   2.658802   2.601641 | -3.000288  -3.000288  -3.000288 |  44.626523 |  511.676167 |  2.00398 | 1500.740299 | 0.926633 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 1.3 | 0.0 | 10.0 |  0.56 | 0.8541  1.521  0.819 | 1.4274 |  0.424023   0.206937   0.404317 | 0.695482   1.52066  0.818961 |  0.424023   0.206937   0.404317 | 0.0  0.0  0.0 |      47.0       47.0       47.0 | 0.000048  0.000048  0.000048 |  0.56 | 0.009705 |  0.348241 | 0.105777 |   1.4274 | 1.058119 | 500.0 | 0.0 |  -14.723343 |  -4.556864 |  1.951674 | 7.994099 | 0.925243 |  0.122867   1.491205   1.938363 | -3.000336  -3.000337  -3.000337 |  40.165732 |  496.952824 | 0.400546 | 1494.842335 | 0.925243 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 0.9 | 0.0 | 10.0 |  2.16 | 0.5913  1.053  0.567 | 0.9882 |  1.504371   0.475372   1.340728 | 0.453718  1.052728  0.566963 |  1.504371   0.475372   1.340728 | 0.0  0.0  0.0 |      23.0       23.0       23.0 | 0.000024  0.000024  0.000024 |  2.16 | 0.026054 |  1.155882 | 0.091745 |   0.9882 | 0.732534 | 500.0 | 0.0 |  -24.398354 |  -3.548412 |  0.967053 | 7.921821 | 0.916877 |  0.324777   2.123105   2.190672 |  -3.00036  -3.000361  -3.000361 |  36.683011 |  472.554469 | 0.589375 | 1490.823663 | 0.916877 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 0.7 | 0.0 | 10.0 |  1.28 | 0.4851  0.819  0.441 | 0.7938 |  0.950551   0.496994   0.875855 | 0.384044  0.818947  0.440979 |  0.950551   0.496994   0.875855 | 0.0  0.0  0.0 | 18.500006  18.500006  18.500006 | 0.000019  0.000019  0.000019 |  1.28 | 0.015043 |  0.793494 | 0.067343 |   0.7938 | 0.585077 | 500.0 | 0.0 |  -30.731545 |  -3.357869 |  0.963106 | 7.941661 | 0.919174 |  0.270182   2.087164   2.153838 | -3.000379   -3.00038   -3.00038 |  33.377441 |  441.822925 | 0.419764 |  1493.33698 | 0.919174 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 0.7 | 0.0 | 10.0 |   2.0 | 0.4851  0.819  0.441 | 0.7938 |  1.532261   1.008675   1.460322 | 0.393813  0.818966  0.440983 |  1.532261   1.008675   1.460322 | 0.0  0.0  0.0 |      23.5       23.5       23.5 | 0.000024  0.000024  0.000024 |   2.0 | 0.020967 |  1.349145 | 0.060831 |   0.7938 | 0.585083 | 500.0 | 0.0 |  -38.506098 |  -3.405203 |  1.116525 | 7.980625 | 0.923684 |  0.344108   2.259522   2.252532 | -3.000403  -3.000404  -3.000404 |  30.012102 |  403.316826 | 0.652384 | 1500.052177 | 0.923684 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 1.1 | 0.0 | 10.0 |  0.48 | 0.7623  1.287  0.693 | 1.2474 |  0.288959   0.089034   0.242267 |  0.50588  1.286378  0.692918 |  0.288959   0.089034   0.242267 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |  0.48 | 0.002808 |  0.221815 | 0.171053 |   1.2474 | 0.919375 | 500.0 | 0.0 |  -26.576927 |  -3.097933 |  0.702183 |  7.99349 | 0.925172 |  0.029269    1.36411   1.797347 | -3.000434  -3.000434  -3.000434 |  27.082413 |  376.739899 | 0.172016 | 1497.398366 | 0.925172 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 1.0 | 0.0 | 10.0 |  0.16 |  0.693   1.17   0.63 |  1.134 |  0.056014   0.000733   0.020497 | 0.287017  1.128526  0.629375 |  0.056014   0.000733   0.020497 | 0.0  0.0  0.0 |      38.5       38.5       38.5 |  0.00004   0.00004   0.00004 |  0.16 |  0.00039 |  0.035566 |  0.28432 |    1.134 | 0.835682 | 500.0 | 0.0 |  -23.262947 |  -2.675598 |  0.170167 | 7.946202 | 0.919699 | -0.153762   0.394851   1.307475 | -3.000473  -3.000474  -3.000474 |  24.690745 |  353.476952 | 0.037416 | 1489.458574 | 0.919699 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 0.8 | 0.0 | 10.0 |  1.36 | 0.5544  0.936  0.504 | 0.9072 |  0.699695   0.001278   0.261984 | 0.321352  0.881396  0.503639 |  0.699695   0.001278   0.261984 | 0.0  0.0  0.0 |      31.5       31.5       31.5 | 0.000032  0.000032  0.000032 |  1.36 | 0.003937 |  0.442888 | 0.173473 |   0.9072 | 0.668589 | 500.0 | 0.0 |    -24.4205 |  -2.482883 |  0.261788 | 7.931138 | 0.917956 |  0.185191   0.872178   1.901852 | -3.000506  -3.000506  -3.000506 |  22.377399 |  329.056452 | 0.218516 | 1496.686962 | 0.917956 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 0.6 | 0.0 | 10.0 |  0.24 | 0.4158  0.702  0.378 | 0.6804 |  0.131638   0.000231   0.068103 | 0.255659  0.664309  0.377897 |  0.131638   0.000231   0.068103 | 0.0  0.0  0.0 | 18.500006  18.500006  18.500006 | 0.000019  0.000019  0.000019 |  0.24 | 0.000611 |  0.085331 | 0.119263 |   0.6804 | 0.501467 | 500.0 | 0.0 |   -29.10995 |  -2.464377 |  0.239262 | 7.993808 | 0.925209 |  0.037894   0.447637   1.695851 | -3.000525  -3.000525  -3.000525 |  20.031674 |  299.946502 | 0.064585 | 1499.600993 | 0.925209 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 0.7 | 0.0 | 10.0 |   0.0 | 0.4851  0.819  0.441 | 0.7938 |       0.0        0.0        0.0 |  0.20305  0.529438  0.440409 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |  8.006707   8.006707   8.006707 | 0.000008  0.000008  0.000008 |   0.0 |      0.0 |  0.000008 | 0.284415 |   0.7938 | 0.584938 | 500.0 | 0.0 |  -26.041777 |  -2.311714 |  0.055803 | 7.997079 | 0.925588 | -0.165156    -0.0818   1.255442 | -3.000533  -3.000533  -3.000533 |  18.004375 |  273.904724 |  0.00879 | 1499.741012 | 0.925588 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 0.7 | 0.0 | 10.0 |  1.44 | 0.4851  0.819  0.441 | 0.7938 |  0.767704   0.000465   0.301922 | 0.289598  0.646694  0.440662 |  0.767704   0.000465   0.301922 | 0.0  0.0  0.0 | 20.000002  20.000002  20.000002 | 0.000021  0.000021  0.000021 |  1.44 | 0.002214 |  0.488982 | 0.187681 |   0.7938 | 0.585007 | 500.0 | 0.0 |  -37.249078 |  -2.550243 |  0.259203 | 8.001458 | 0.926095 |  0.217542   0.711041   1.952859 | -3.000553  -3.000554  -3.000554 |    15.6396 |  236.655646 |  0.23857 | 1500.549574 | 0.926095 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 0.5 | 0.0 | 10.0 |  7.12 | 0.3465  0.585  0.315 |  0.567 |  6.357943   4.564431    6.15922 | 0.317366  0.584263  0.314994 |  6.357943   4.564431    6.15922 | 0.0  0.0  0.0 |      28.0       28.0       28.0 | 0.000029  0.000029  0.000029 |  7.12 | 0.015029 |   5.78652 |  0.01966 |    0.567 | 0.417934 | 500.0 | 0.0 | -129.250916 |  -5.886208 |   3.30999 | 8.029164 | 0.929301 |  0.662233   2.682347   2.598645 | -3.000582  -3.000583  -3.000583 |   9.758023 |   107.40473 |   2.7151 | 1508.252101 | 0.929301 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 0.8 | 0.0 | 10.0 |   0.0 | 0.5544  0.936  0.504 | 0.9072 |       0.0        0.0        0.0 | 0.473397   0.93593  0.503987 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.0       29.0       29.0 |  0.00003   0.00003   0.00003 |   0.0 |      0.0 |   0.00003 | 0.054005 |   0.9072 | 0.668688 | 500.0 | 0.0 |  -88.397265 |  -4.705385 |  2.347669 | 8.073352 | 0.934416 |  0.188836   1.746417   2.094657 | -3.000612  -3.000612  -3.000612 |   5.106642 |   19.007465 | 0.367461 | 1511.205602 | 0.934416 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 1.0 | 0.0 | 10.0 |   0.0 |  0.693   1.17   0.63 |  1.134 |       0.0        0.0        0.0 | 0.342784  1.155482  0.629754 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      28.5       28.5       28.5 | 0.000029  0.000029  0.000029 |   0.0 |      0.0 |  0.000029 |  0.23823 |    1.134 | 0.835784 | 500.0 | 0.0 |  -19.312197 |  -2.329407 |  0.317599 | 8.122931 | 0.940154 | -0.153948   0.590935   1.464903 | -3.000641  -3.000642  -3.000642 |   3.015465 |   -0.304731 | 0.049891 | 1516.760535 | 0.940154 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 1.2 | 0.0 | 10.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.194399   0.81325  0.748162 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      23.0       23.0       23.0 | 0.000024  0.000024  0.000024 |   0.0 |      0.0 |  0.000024 |   0.6215 |   1.3608 | 1.002101 | 500.0 | 0.0 |   -0.325402 |  -2.140023 |  0.043137 |   8.1329 | 0.941308 | -0.348347  -0.222315   0.716742 | -3.000665  -3.000665  -3.000665 |   1.496942 |   -0.630134 | 0.006778 | 1516.493407 | 0.941308 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 0.9 | 0.0 | 10.0 |  0.72 | 0.6237  1.053  0.567 | 1.0206 |    0.1985   0.000023   0.002958 | 0.207048  0.453034  0.552683 |    0.1985   0.000023   0.002958 | 0.0  0.0  0.0 | 21.000001  21.000001  21.000001 | 0.000022  0.000022  0.000022 |  0.72 | 0.000001 |  0.119424 | 0.477594 |   1.0206 | 0.750749 | 500.0 | 0.0 |   -0.250094 |  -2.167611 |  0.065913 | 8.134747 | 0.941522 | -0.033896   0.044627   0.881101 | -3.000687  -3.000687  -3.000687 |  -0.193076 |   -0.880228 | 0.060289 | 1517.093562 | 0.941522 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 0.9 | 0.0 | 10.0 |  0.08 | 0.6237  1.053  0.567 | 1.0206 |  0.023892   0.000002   0.000192 | 0.223715  0.377517  0.535147 |  0.023892   0.000002   0.000192 | 0.0  0.0  0.0 |      37.0       37.0       37.0 | 0.000038  0.000038  0.000038 |  0.08 |      0.0 |  0.014393 | 0.491656 |   1.0206 | 0.749032 | 500.0 | 0.0 |   -0.758079 |  -2.160047 |  0.060766 | 8.136398 | 0.941713 | -0.201502  -0.252892   0.425762 | -3.000725  -3.000725  -3.000725 |  -1.861467 |   -1.638307 | 0.013915 | 1517.052572 | 0.941713 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 1.2 | 0.0 | 10.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.171645  0.216554  0.491468 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      31.5       31.5       31.5 | 0.000032  0.000032  0.000032 |   0.0 |      0.0 |  0.000032 | 0.835519 |   1.3608 | 0.976902 | 500.0 | 0.0 |   -1.232209 |  -2.078392 |  0.012022 | 8.138485 | 0.941954 | -0.373147  -0.469446  -0.065706 | -3.000757  -3.000757  -3.000757 |   -3.10434 |   -2.870517 | 0.001926 | 1517.699466 | 0.941954 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 1.4 | 0.0 | 10.0 |   0.0 | 0.9702  1.638  0.882 | 1.5876 |       0.0        0.0        0.0 | 0.116208   0.12809  0.249038 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      43.5       43.5       43.5 | 0.000045  0.000045  0.000045 |   0.0 |      0.0 |  0.000045 | 1.072297 |   1.5876 | 1.107912 | 500.0 | 0.0 |   -1.030082 |  -2.060898 |  0.001665 | 8.143034 | 0.942481 | -0.489355  -0.597536  -0.314744 | -3.000802  -3.000802  -3.000802 |  -4.092942 |   -3.900599 | 0.000305 | 1518.156133 | 0.942481 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 1.1 | 0.0 | 10.0 |  3.12 | 0.7623  1.287  0.693 | 1.2474 |  1.818576   0.016236   0.252127 | 0.484984  0.989168   0.60256 |  1.818576   0.016236   0.252127 | 0.0  0.0  0.0 |      32.0       32.0       32.0 | 0.000033  0.000033  0.000033 |  3.12 |      0.0 |  1.121262 | 0.284068 |   1.2474 | 0.910533 | 500.0 | 0.0 |   -1.948576 |  -2.617969 |  0.551846 | 8.165185 | 0.945045 |  0.327084    1.51706    1.95057 | -3.000834  -3.000835  -3.000835 |  -6.426843 |   -5.849175 | 0.569721 | 1523.357543 | 0.945045 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 1.1 | 0.0 | 10.0 |  6.96 | 0.7623  1.287  0.693 | 1.2474 |  6.071563   4.671479   5.753171 | 0.685968  1.286933  0.692984 |  6.071563   4.671479   5.753171 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |  6.96 |      0.0 |  5.619726 | 0.050896 |   1.2474 | 0.919458 | 500.0 | 0.0 |   -4.899013 |  -5.781779 |   3.58005 | 8.249952 | 0.954856 |  0.529553   2.518648   2.464415 | -3.000861  -3.000862  -3.000862 | -12.157726 |  -10.748188 | 2.609397 | 1537.018557 | 0.954856 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 0.5 | 0.0 | 10.0 | 21.12 | 0.3465  0.585  0.315 |  0.567 | 20.402771  20.094596  20.410261 | 0.337475  0.584999  0.314999 | 20.402771  20.094596  20.410261 | 0.0  0.0  0.0 |      25.5       25.5       25.5 | 0.000026  0.000026  0.000026 | 21.12 |      0.0 | 20.311094 | 0.006016 |    0.567 | 0.417941 | 500.0 | 0.0 |  -14.082185 | -16.797206 | 13.714397 | 8.441139 | 0.976984 |  0.909307   2.959053   2.859155 | -3.000887  -3.000888  -3.000888 | -28.948917 |  -24.830373 | 9.206093 | 1566.763247 | 0.976984 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 0.6 | 0.0 | 10.0 |   9.2 | 0.4158  0.702  0.378 | 0.6804 |  8.931946    8.69755   8.951424 | 0.406537  0.701998  0.377999 |  8.931946    8.69755   8.951424 | 0.0  0.0  0.0 |      43.5       43.5       43.5 | 0.000045  0.000045  0.000045 |   9.2 |      0.0 |   8.86362 | 0.006175 |   0.6804 | 0.501529 | 500.0 | 0.0 |  -17.635243 | -16.311367 | 13.007069 | 8.530553 | 0.987333 |  0.770823   2.759505   2.729733 | -3.000932  -3.000932  -3.000932 | -45.254109 |  -42.465616 | 5.062644 |  1566.77034 | 0.987333 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 1.5 | 0.0 | 10.0 |  0.72 | 1.0395  1.755  0.945 |  1.701 |  0.554102   0.275967   0.548112 | 0.842841  1.754558  0.944963 |  0.554102   0.275967   0.548112 | 0.0  0.0  0.0 |      31.0       31.0       31.0 | 0.000032  0.000032  0.000032 |  0.72 |      0.0 |  0.470094 | 0.131237 |    1.701 | 1.253808 | 500.0 | 0.0 |   -8.813486 |  -7.046507 |  4.674931 | 8.500201 |  0.98382 |   0.09388   1.448981   1.956658 | -3.000964  -3.000964  -3.000964 | -52.169378 |  -51.279102 | 0.857807 |  1559.09992 |  0.98382 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 2.0 | 0.0 | 10.0 |   0.0 |  1.386   2.34   1.26 |  2.268 |       0.0        0.0        0.0 | 0.443363  1.687857  1.251508 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |   0.0 |      0.0 |   0.00003 | 0.845708 |    2.268 |  1.67085 | 500.0 | 0.0 |   -2.389655 |  -2.535234 |  0.742034 | 8.470464 | 0.980378 | -0.349483  -0.238877    0.70515 | -3.000994  -3.000994  -3.000994 | -53.858904 |  -53.668757 | 0.115804 | 1557.864535 | 0.980378 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 1.6 | 0.0 | 10.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.139004  0.269691  0.748739 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |   0.0 |      0.0 |  0.000027 |   1.1805 |   1.8144 | 1.311889 | 500.0 | 0.0 |   -0.706265 |  -1.797319 |  0.099931 |  8.46822 | 0.980118 | -0.488488  -0.508568  -0.043589 | -3.001021  -3.001021  -3.001021 | -54.475724 |  -54.375022 |   0.0159 | 1558.273946 | 0.980118 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 1.6 | 0.0 | 10.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.087812  0.124929  0.285484 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      26.0       26.0       26.0 | 0.000027  0.000027  0.000027 |   0.0 |      0.0 |  0.000027 | 1.262874 |   1.8144 | 1.266483 | 500.0 | 0.0 |   -0.457858 |    -1.7031 |  0.013728 | 8.470891 | 0.980427 | -0.576299  -0.633497  -0.329073 | -3.001047  -3.001048  -3.001048 | -54.915949 |   -54.83288 | 0.002198 | 1558.480961 | 0.980427 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 1.2 | 0.0 | 10.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.049329  0.062934  0.109594 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      29.5       29.5       29.5 |  0.00003   0.00003   0.00003 |   0.0 |      0.0 |   0.00003 | 0.968428 |   1.3608 | 0.939618 | 500.0 | 0.0 |   -0.660248 |  -1.685812 |  0.001914 | 8.474692 | 0.980867 | -0.625629  -0.696431  -0.438667 | -3.001078  -3.001078  -3.001078 | -55.633333 |  -55.493129 | 0.000315 | 1559.135035 | 0.980867 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 1.3 | 0.0 | 10.0 |   0.0 | 0.9009  1.521  0.819 | 1.4742 |       0.0        0.0        0.0 | 0.043675  0.052857  0.081106 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      33.5       33.5       33.5 | 0.000034  0.000034  0.000034 |   0.0 |      0.0 |  0.000034 | 1.060748 |   1.4742 | 1.014233 | 500.0 | 0.0 |   -0.636106 |  -1.675527 |   0.00029 |  8.47968 | 0.981444 | -0.669304  -0.749288  -0.519773 | -3.001112  -3.001112  -3.001112 | -56.248113 |  -56.129235 | 0.000059 | 1559.800305 | 0.981444 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 1.6 | 0.0 | 10.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.044308  0.051568  0.072372 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      35.0       35.0       35.0 | 0.000036  0.000036  0.000036 |   0.0 |      0.0 |  0.000036 | 1.316329 |   1.8144 | 1.245597 | 500.0 | 0.0 |   -0.404731 |  -1.672468 |  0.000071 |  8.48371 | 0.981911 | -0.713612  -0.800856  -0.592145 | -3.001148  -3.001148  -3.001148 | -56.604251 |  -56.533965 | 0.000024 |  1560.04801 | 0.981911 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 1.9 | 0.0 | 10.0 |   0.0 | 1.3167  2.223  1.197 | 2.1546 |       0.0        0.0        0.0 | 0.043336  0.048901  0.064103 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      41.5       41.5       41.5 | 0.000043  0.000043  0.000043 |   0.0 |      0.0 |  0.000043 | 1.573439 |   2.1546 | 1.477006 | 500.0 | 0.0 |    -0.14367 |  -1.664506 |  0.000044 | 8.485696 | 0.982141 | -0.756948  -0.849756  -0.656248 |  -3.00119  -3.001191  -3.001191 | -56.695318 |  -56.677635 | 0.000022 | 1560.206078 | 0.982141 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 0.8 | 0.0 | 10.0 |   1.2 | 0.5544  0.936  0.504 | 0.9072 |  0.262511   0.000023   0.000146 | 0.145361  0.238849  0.204304 |  0.262511   0.000023   0.000146 | 0.0  0.0  0.0 |      45.0       45.0       45.0 | 0.000046  0.000046  0.000046 |   1.2 |      0.0 |  0.157574 | 0.505022 |   0.9072 | 0.639289 | 500.0 | 0.0 |   -0.984411 |  -1.760531 |  0.064376 | 8.490732 | 0.982724 |   0.03518   0.111372   0.339302 | -3.001236  -3.001237  -3.001237 | -57.950827 |  -57.662046 |  0.09322 | 1561.477279 | 0.982724 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 1.5 | 0.0 | 10.0 |  0.24 | 1.0395  1.755  0.945 |  1.701 |  0.076819   0.000006    0.00005 | 0.396454   0.62374  0.601591 |  0.076819   0.000006    0.00005 | 0.0  0.0  0.0 |      62.0       62.0       62.0 | 0.000064  0.000064  0.000064 |  0.24 |      0.0 |  0.046162 |   0.8057 |    1.701 | 1.220098 | 500.0 | 0.0 |   -1.088514 |  -1.771559 |  0.108784 | 8.500021 | 0.983799 | -0.198093  -0.272374  -0.022339 |   -3.0013    -3.0013    -3.0013 | -58.916686 |   -58.75056 | 0.030597 | 1562.219894 | 0.983799 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 2.7 | 0.0 | 10.0 |  0.16 | 1.8711  3.159  1.701 | 3.0618 |   0.02362   0.000001   0.000007 | 0.346755  0.403415  0.470442 |   0.02362   0.000001   0.000007 | 0.0  0.0  0.0 |      75.0       75.0       75.0 | 0.000077  0.000077  0.000077 |  0.16 |      0.0 |   0.01425 | 1.934558 |   3.0618 | 2.136116 | 500.0 | 0.0 |    0.026639 |  -1.670686 |  0.035209 | 8.502756 | 0.984115 | -0.408469   -0.51579  -0.332788 | -3.001377  -3.001377  -3.001377 | -58.652814 |  -58.723921 | 0.009638 | 1562.228245 | 0.984115 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 1.5 | 0.0 | 10.0 |   3.6 | 1.2825  1.755  0.945 |  1.728 |  1.991769   0.031289   0.414313 | 0.874718  1.427468  0.831559 |  1.991769   0.031289   0.414313 | 0.0  0.0  0.0 |      59.0       59.0       59.0 |  0.00006   0.00006   0.00006 |   3.6 |      0.0 |   1.24594 | 0.380993 |    1.728 | 1.386098 | 500.0 | 0.0 |   -1.376047 |   -2.34437 |  0.628185 | 8.515631 | 0.985605 |  0.325045   1.625453   2.021341 | -3.001437  -3.001438  -3.001438 | -60.616191 |  -60.099968 | 0.627393 | 1565.713052 | 0.985605 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 1.6 | 0.0 | 10.0 |   0.0 |  1.368  1.872  1.008 | 1.8432 |       0.0        0.0        0.0 | 0.610023  1.639364  1.005966 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      47.0       47.0       47.0 | 0.000048  0.000048  0.000048 |   0.0 |      0.0 |  0.000048 | 0.582807 |   1.8432 | 1.490151 | 500.0 | 0.0 |   -1.950523 |   -2.23164 |  0.542349 | 8.529724 | 0.987237 | -0.284978  -0.013911   1.015375 | -3.001485  -3.001486  -3.001486 | -62.265024 |   -62.05049 | 0.085092 | 1565.543451 | 0.987237 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 2.0 | 0.0 | 10.0 |   0.0 |   1.71   2.34   1.26 |  2.304 |       0.0        0.0        0.0 | 0.227067  0.478502  1.028598 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      40.5       40.5       40.5 | 0.000041  0.000041  0.000041 |   0.0 |      0.0 |  0.000041 | 1.608966 |    2.304 | 1.840186 | 500.0 | 0.0 |   -0.311128 |  -1.696484 |  0.073518 | 8.530078 | 0.987278 | -0.512045  -0.492413  -0.013223 | -3.001527  -3.001527  -3.001527 | -62.352542 |  -62.361618 | 0.011615 | 1565.522979 | 0.987278 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 2.1 | 0.0 | 10.0 |   0.0 | 1.7955  2.457  1.323 | 2.4192 |       0.0        0.0        0.0 | 0.119557  0.161202  0.357479 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      39.5       39.5       39.5 |  0.00004   0.00004   0.00004 |   0.0 |      0.0 |   0.00004 |  1.88238 |   2.4192 | 1.861369 | 500.0 | 0.0 |    0.207253 |  -1.632475 |  0.010045 | 8.528991 | 0.987152 | -0.631602  -0.653615  -0.370703 | -3.001567  -3.001568  -3.001568 | -62.102637 |  -62.154366 | 0.001611 | 1565.154291 | 0.987152 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 1.7 | 0.0 | 10.0 |  0.32 | 1.4535  1.989  1.071 | 1.9584 |  0.014458   0.000001   0.000008 |  0.11982  0.145604  0.209947 |  0.014458   0.000001   0.000008 | 0.0  0.0  0.0 |      37.5       37.5       37.5 | 0.000038  0.000038  0.000038 |  0.32 |      0.0 |  0.008714 |  1.50344 |   1.9584 | 1.499039 | 500.0 | 0.0 |   -0.046755 |  -1.633392 |  0.005869 | 8.528018 | 0.987039 | -0.445879   -0.47922  -0.260658 | -3.001606  -3.001606  -3.001606 | -62.232589 |  -62.201121 | 0.004456 | 1565.370004 | 0.987039 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 1.7 | 0.0 | 10.0 |   0.0 | 1.4535  1.989  1.071 | 1.9584 |       0.0        0.0        0.0 | 0.126183  0.143877  0.175605 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      36.0       36.0       36.0 | 0.000037  0.000037  0.000037 |   0.0 |      0.0 |  0.000037 | 1.499774 |   1.9584 | 1.495674 | 500.0 | 0.0 |   -0.138086 |  -1.621338 |  0.003857 | 8.528824 | 0.987132 | -0.572062  -0.623097  -0.436263 | -3.001642  -3.001643  -3.001643 | -62.354152 |  -62.339207 | 0.000636 |  1565.65837 | 0.987132 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 0.8 | 0.0 | 10.0 |   0.0 |  0.684  0.936  0.504 | 0.9216 |       0.0        0.0        0.0 | 0.042006  0.045676  0.053523 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      40.5       40.5       40.5 | 0.000041  0.000041  0.000041 |   0.0 |      0.0 |  0.000041 | 0.724701 |   0.9216 | 0.700992 | 500.0 | 0.0 |   -0.744902 |  -1.632209 |  0.000573 | 8.532009 | 0.987501 | -0.614068  -0.668774  -0.489786 | -3.001684  -3.001684  -3.001684 |  -63.26166 |  -63.084109 | 0.000105 |    1566.184 | 0.987501 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 1.3 | 0.0 | 10.0 |   0.0 | 1.1115  1.521  0.819 | 1.4976 |       0.0        0.0        0.0 | 0.055212  0.058911  0.067506 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      43.0       43.0       43.0 | 0.000044  0.000044  0.000044 |   0.0 |      0.0 |  0.000044 | 1.191442 |   1.4976 | 1.137205 | 500.0 | 0.0 |   -0.514108 |  -1.619655 |  0.000115 | 8.537344 | 0.988119 |  -0.66928  -0.727685  -0.557292 | -3.001728  -3.001728  -3.001728 | -63.689872 |  -63.598217 | 0.000033 | 1566.398068 | 0.988119 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 2.5 | 0.0 | 10.0 |   0.0 | 2.1375  2.925  1.575 |   2.88 |       0.0        0.0        0.0 |  0.07886  0.082517  0.092024 |       0.0        0.0        0.0 | 0.0  0.0  0.0 |      52.5       52.5       52.5 | 0.000054  0.000054  0.000054 |   0.0 |      0.0 |  0.000054 | 2.319702 |     2.88 | 2.183225 | 500.0 | 0.0 |    0.487713 |  -1.597736 |  0.000059 | 8.537439 |  0.98813 |  -0.74814  -0.810202  -0.649315 | -3.001781  -3.001782  -3.001782 | -62.967907 |  -63.110504 | 0.000028 | 1566.188859 |  0.98813 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_v002_snowfall:

snowfall
________

>>> inputs.fxs.series =  0.0
>>> inputs.t.series -= 8.0

.. integration-test::

    >>> test("wland_v002_snowfall",
    ...      axis1=(fluxes.pc, states.sp,),
    ...      axis2=(inputs.t,))
    |                date |     t |    p | pet | fxg | fxs |    pc |                 petl |    pes |                              tf |                           ei |                           rf |                              sf |                              pm |                              am |    ps |       pv |        pq |      etv |       es |       et | fxs | fxg |         cdg |       fgs |       fqs |       rh |        r |                              ic |                              sp |         dv |          dg |        hq |          hs |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.6 | 0.0 | 0.0 |   0.0 | 0.3942  0.702  0.378 | 0.6588 |       0.0        0.0        0.0 |      0.0  0.000001       0.0 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 |  0.49406 | 0.000073 | 0.435763 | 0.0 | 0.0 |  -37.315378 |  0.000762 |       0.0 |      0.0 |      0.0 |      -3.0  -3.000001       -3.0 |      -3.0       -3.0       -3.0 | 140.494822 | 1562.684622 |       0.0 |   -1.966476 |      0.0 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.000001        0.0        0.0 | 0.000001  0.000002  0.000001 |      0.0       0.0       0.0 |  0.000001        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |  0.32 |      0.0 |       0.0 | 0.658704 | 0.000253 | 0.580984 | 0.0 | 0.0 |  -15.511778 |  0.000468 |       0.0 |      0.0 |      0.0 | -2.680002  -2.680003  -2.680002 | -2.999999       -3.0       -3.0 | 141.153995 | 1547.172844 |       0.0 |    -1.62609 |      0.0 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 | 0.4599  0.819  0.441 | 0.7686 |       0.0        0.0        0.0 | 0.000002  0.000004  0.000002 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 | 0.576325 | 0.000452 |  0.50833 | 0.0 | 0.0 |   -5.174814 |  0.000343 |       0.0 |      0.0 |      0.0 | -2.680004  -2.680007  -2.680004 | -2.999999       -3.0       -3.0 | 141.730663 |  1541.99803 |       0.0 |   -1.611406 |      0.0 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 | 0.329312 | 0.000276 |  0.29046 | 0.0 | 0.0 |   -0.906638 |  0.000319 |       0.0 |      0.0 |      0.0 | -2.680005  -2.680009  -2.680005 | -2.999999       -3.0       -3.0 | 142.060293 | 1541.091392 |       0.0 |   -1.597619 |      0.0 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 | 0.329299 | 0.000294 | 0.290449 | 0.0 | 0.0 |    0.730368 |  0.000316 |       0.0 |      0.0 |      0.0 | -2.680007  -2.680011  -2.680006 | -2.999999       -3.0       -3.0 | 142.389909 | 1541.821761 |       0.0 |   -1.583961 |      0.0 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000002  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 | 0.329287 | 0.000313 | 0.290439 | 0.0 | 0.0 |     1.50565 |  0.000323 |       0.0 |      0.0 |      0.0 | -2.680008  -2.680013  -2.680007 | -2.999999       -3.0       -3.0 | 142.719519 | 1543.327411 |       0.0 |   -1.570033 |      0.0 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.4 | 0.0 | 0.0 |  0.16 | 0.2628  0.468  0.252 | 0.4392 |  0.000001        0.0        0.0 | 0.000002  0.000003  0.000002 |      0.0       0.0       0.0 |  0.000001        0.0        0.0 |  0.000008   0.000008   0.000008 |       0.0        0.0        0.0 |  0.16 |      0.0 |       0.0 | 0.329273 | 0.000494 | 0.290431 | 0.0 | 0.0 |    1.871668 |  0.000317 |       0.0 |      0.0 |      0.0 |  -2.52001  -2.520016  -2.520009 | -2.999999       -3.0       -3.0 | 143.049109 | 1545.199079 |       0.0 |   -1.396551 |      0.0 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.3 | 0.0 | 0.0 |   3.6 | 0.1971  0.351  0.189 | 0.3294 |  0.715908   0.000866   0.002554 | 0.045779  0.103421  0.056212 |      0.0       0.0       0.0 |  0.715908   0.000866   0.002554 |  0.000011   0.000011   0.000011 |       0.0        0.0        0.0 |   3.6 |      0.0 |       0.0 | 0.182336 | 0.202501 | 0.227703 | 0.0 | 0.0 |    1.796541 | -0.000086 |       0.0 | 0.000099 | 0.000011 |  0.318303   0.975698   1.021225 | -2.284091  -2.999134  -2.997446 | 143.231359 |  1546.99562 |       0.0 |    1.992203 | 0.000011 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.3 | 0.0 | 0.0 |   0.0 | 0.1971  0.351  0.189 | 0.3294 |       0.0        0.0        0.0 | 0.148002  0.341645    0.1863 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 | 0.035641 | 0.329309 | 0.243747 | 0.0 | 0.0 |    1.094193 | -0.000447 |       0.0 | 0.000336 | 0.000039 |  0.170301   0.634053   0.834925 | -2.284091  -2.999134  -2.997446 | 143.266553 | 1548.089813 |       0.0 |     1.62642 | 0.000039 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.4 | 0.0 | 0.0 |  2.56 | 0.2628  0.468  0.252 | 0.4392 |  2.024091   0.452925   0.899571 | 0.221756   0.46542  0.251473 |      0.0       0.0       0.0 |  2.024091   0.452925   0.899571 |   0.00003    0.00003    0.00003 |  0.000001        0.0        0.0 |  2.56 |      0.0 |       0.0 | 0.028057 | 0.439174 |   0.3254 | 0.0 | 0.0 |    0.621041 | -0.000682 |       0.0 | 0.000609 | 0.000071 |  0.484454   2.275708    2.24388 |     -0.26   -2.54621  -2.097875 | 143.293928 | 1548.710853 |       0.0 |      3.6867 | 0.000071 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.3 | 0.0 | 0.0 |  3.68 | 0.1971  0.351  0.189 | 0.3294 |  3.340123   2.972407   3.208228 | 0.184113  0.350996  0.188997 | 0.584521  0.520171   0.56144 |  2.755601   2.452236   2.646788 |  0.018374   0.018374   0.018374 |  0.016188   0.000752   0.003746 |  3.68 | 0.163238 |  0.426307 | 0.008608 |   0.3294 | 0.244154 | 0.0 | 0.0 |  -11.702771 | -0.002522 |  0.238148 | 0.004421 | 0.000512 |  0.640218   2.632305   2.526656 |  2.479413  -0.094726   0.545167 | 143.136777 | 1537.008082 |  0.188159 |   18.374314 | 0.000512 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.6 | 0.0 | 0.0 |  1.84 | 0.3942  0.702  0.378 | 0.6588 |  1.642164   1.409265   1.618931 | 0.363811  0.701991  0.377995 |      0.0       0.0       0.0 |  1.642164   1.409265   1.618931 |  0.000888   0.000888   0.000888 |  0.000888   0.000771   0.000879 |  1.84 | 0.000241 |  0.000635 | 0.020144 |   0.6588 | 0.488293 | 0.0 | 0.0 |    5.452886 | -0.007584 |  0.164022 | 0.015901 |  0.00184 |  0.474243   2.361049    2.36973 |  4.120689   1.313768   2.163219 | 143.149095 | 1542.460968 |  0.024772 |   26.463072 |  0.00184 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.8 | 0.0 | 0.0 |  14.4 | 0.5256  0.936  0.504 | 0.8784 | 13.589457  13.048462  13.564559 | 0.504697  0.935996  0.503997 | 1.698682  1.631058   1.69557 | 11.890775  11.417404  11.868989 |  0.009388   0.009388   0.009388 |  0.009388   0.009388   0.009388 |  14.4 |  0.47743 |  1.257785 | 0.013856 |   0.8784 | 0.651125 | 0.0 | 0.0 |  -32.799642 | -0.016485 |  0.729031 | 0.044196 | 0.005115 |  0.780089   2.776592   2.701174 | 16.002075  12.721784   14.02282 | 142.669036 | 1509.661326 |  0.553526 |   72.770427 | 0.005115 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.5 | 0.0 | 0.0 | 15.36 | 0.3285  0.585  0.315 |  0.549 | 14.927277  14.656358  14.931703 | 0.322035  0.584999  0.314999 |      0.0       0.0       0.0 | 14.927277  14.656358  14.931703 |   0.00006    0.00006    0.00006 |   0.00006    0.00006    0.00006 | 15.36 | 0.000017 |  0.000045 | 0.004285 |    0.549 | 0.406976 | 0.0 | 0.0 |   14.914236 | -0.051056 |  0.478611 | 0.121448 | 0.014056 |  0.890776   2.895235   2.814472 | 30.929293  27.378082  28.954463 | 142.622249 | 1524.575562 |   0.07496 |  102.709372 | 0.014056 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504 | 0.8784 |  0.294509   0.228516   0.294807 | 0.495655  0.935979  0.503995 |      0.0       0.0       0.0 |  0.294509   0.228516   0.294807 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |  0.32 |      0.0 |       0.0 | 0.019855 |   0.8784 | 0.651095 | 0.0 | 0.0 |    6.184724 | -0.063559 |  0.065203 | 0.136902 | 0.015845 |  0.420612   2.050739    2.33567 | 31.223802  27.606598  29.249269 | 142.578545 | 1530.760286 |  0.009757 |   95.697907 | 0.015845 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.5 | 0.0 | 0.0 |  6.64 | 0.3285  0.585  0.315 |  0.549 |  6.067393   5.421308   6.057702 | 0.308291  0.584994  0.314997 |      0.0       0.0       0.0 |  6.067393   5.421308   6.057702 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |  6.64 |      0.0 |       0.0 | 0.013397 |    0.549 | 0.406929 | 0.0 | 0.0 |    3.019403 | -0.059994 |  0.008441 | 0.126512 | 0.014643 |  0.684928   2.684437   2.602971 | 37.291195  33.027906  35.306971 | 142.531948 | 1533.779689 |  0.001316 |   93.231176 | 0.014643 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.4 | 0.0 | 0.0 |  4.24 | 0.2628  0.468  0.252 | 0.4392 |  3.990639   3.797605   3.989314 | 0.251901  0.467998  0.251998 | 1.097426  1.044342  1.097061 |  2.893213   2.753264   2.892253 |  0.069591   0.069591   0.069591 |  0.069591   0.069591   0.069591 |  4.24 | 0.323242 |  0.860137 | 0.007225 |   0.4392 | 0.325561 | 0.0 | 0.0 |  -22.238877 | -0.060446 |  0.489402 | 0.137313 | 0.015893 |  0.682388   2.658834   2.601659 | 40.114817  35.711579  38.129633 | 142.155485 | 1511.540812 |   0.37205 |  111.481389 | 0.015893 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 1.3 | 0.0 | 0.0 |  0.56 | 0.8541  1.521  0.819 | 1.4274 |  0.424043   0.206924    0.40434 | 0.695509   1.52066  0.818961 | 0.360436  0.175885  0.343689 |  0.063606   0.031039   0.060651 |  7.013137   7.013137   7.013137 |  7.013137   7.013137   7.013137 |  0.56 | 2.025397 |  5.493677 |  0.10525 |   1.4274 |  1.05767 | 0.0 | 0.0 | -138.102988 | -0.093547 |  3.451139 | 0.318931 | 0.036913 |  0.122836    1.49125   1.938358 | 33.165286  28.729481  31.177147 | 140.141792 | 1373.437824 |  2.414588 |   259.64785 | 0.036913 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.9 | 0.0 | 0.0 |  2.16 | 0.5913  1.053  0.567 | 0.9882 |  1.504354   0.475414   1.340729 | 0.453714  1.052728  0.566963 |      0.0       0.0       0.0 |  1.504354   0.475414   1.340729 |  0.000016   0.000016   0.000016 |  0.000016   0.000016   0.000016 |  2.16 | 0.000004 |  0.000012 |  0.09131 |   0.9882 | 0.732148 | 0.0 | 0.0 |   56.694584 | -0.321424 |  2.087825 | 0.728738 | 0.084345 |  0.324768   2.123108   2.190666 | 34.669625  29.204879   32.51786 | 139.911674 | 1430.132408 |  0.326774 |  312.511393 | 0.084345 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.7 | 0.0 | 0.0 |  1.28 | 0.4851  0.819  0.441 | 0.7938 |  0.950596   0.497035   0.875887 | 0.384059  0.818947  0.440979 |      0.0       0.0       0.0 |  0.950596   0.497035   0.875887 |  0.000001   0.000001   0.000001 |  0.000001   0.000001   0.000001 |  1.28 |      0.0 |  0.000001 | 0.067011 |   0.7938 | 0.584792 | 0.0 | 0.0 |   25.126112 | -0.354391 |  0.282534 | 0.700675 | 0.081097 |  0.270113   2.087125   2.153799 |  35.62022  29.701913  33.393747 | 139.624294 |  1455.25852 |  0.044241 |   276.17937 | 0.081097 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.7 | 0.0 | 0.0 |   2.0 | 0.4851  0.819  0.441 | 0.7938 |  1.532198   1.008874   1.460256 |   0.3938  0.818966  0.440983 |      0.0       0.0       0.0 |  1.532198   1.008874   1.460256 |  0.000022   0.000022   0.000022 |  0.000022   0.000022   0.000022 |   2.0 | 0.000006 |  0.000016 | 0.060547 |   0.7938 | 0.584825 | 0.0 | 0.0 |   16.232114 | -0.283702 |  0.038211 | 0.567088 | 0.065635 |  0.344114   2.259285    2.25256 | 37.152396  30.710765  34.853981 | 139.401133 | 1471.490634 |  0.006046 |  238.392309 | 0.065635 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 1.1 | 0.0 | 0.0 |  0.48 | 0.7623  1.287  0.693 | 1.2474 |  0.288984   0.088968    0.24228 |  0.50592  1.286377  0.692918 |      0.0       0.0       0.0 |  0.288984   0.088968    0.24228 |  0.001244   0.001244   0.001244 |  0.001244   0.001244   0.001244 |  0.48 | 0.000337 |  0.000941 | 0.170201 |   1.2474 | 0.918647 | 0.0 | 0.0 |   11.679971 | -0.220478 |  0.005791 | 0.453811 | 0.052524 |   0.02921   1.363939   1.797362 | 37.440136  30.798489  35.095016 | 139.350519 | 1483.170605 |  0.001196 |   205.49501 | 0.052524 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 1.0 | 0.0 | 0.0 |  0.16 |  0.693   1.17   0.63 |  1.134 |  0.056025   0.000729   0.020468 | 0.287077  1.128774  0.629377 | 0.023811   0.00031  0.008699 |  0.032214   0.000419   0.011769 |   0.45975    0.45975    0.45975 |   0.45975    0.45975    0.45975 |  0.16 | 0.128756 |  0.359119 | 0.282818 |    1.134 | 0.834465 | 0.0 | 0.0 |   -0.407847 | -0.173929 |  0.205383 | 0.375959 | 0.043514 | -0.153891   0.394436   1.307517 | 37.012601  30.339159  34.647035 | 139.330653 | 1482.762758 |  0.154932 |  188.116565 | 0.043514 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.8 | 0.0 | 0.0 |  1.36 | 0.5544  0.936  0.504 | 0.9072 |  0.699103   0.001275   0.261794 | 0.321123  0.881327   0.50364 | 0.052433  0.000096  0.019635 |  0.646671    0.00118   0.242159 |   0.00479    0.00479    0.00479 |   0.00479    0.00479    0.00479 |  1.36 | 0.010769 |   0.02855 | 0.172802 |   0.9072 | 0.667842 | 0.0 | 0.0 |    9.627456 | -0.152868 |  0.148761 | 0.335142 |  0.03879 |  0.185883   0.871833   1.902084 | 37.654481  30.335549  34.884404 | 139.339817 | 1492.390214 |  0.034721 |  172.360088 |  0.03879 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.6 | 0.0 | 0.0 |  0.24 | 0.4158  0.702  0.378 | 0.6804 |  0.131772    0.00023   0.068149 |  0.25588  0.664259  0.377898 |      0.0       0.0       0.0 |  0.131772    0.00023   0.068149 |  0.000001   0.000001   0.000001 |  0.000001   0.000001   0.000001 |  0.24 |      0.0 |  0.000001 | 0.118549 |   0.6804 | 0.500951 | 0.0 | 0.0 |    6.985571 | -0.129433 |  0.030202 | 0.286145 | 0.033119 |  0.038231   0.447344   1.696037 | 37.786253  30.335778  34.952553 | 139.328933 | 1499.375785 |   0.00452 |  153.384352 | 0.033119 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 | 0.4851  0.819  0.441 | 0.7938 |       0.0        0.0        0.0 |  0.20321  0.529519  0.440411 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 |      0.0 |       0.0 | 0.282879 |   0.7938 | 0.583701 | 0.0 | 0.0 |    5.255944 |  -0.10604 |  0.003816 | 0.239392 | 0.027707 | -0.164979  -0.082174   1.255626 | 37.786253  30.335778  34.952553 | 139.505772 | 1504.631729 |  0.000703 |  136.131576 | 0.027707 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.7 | 0.0 | 0.0 |  1.44 | 0.4851  0.819  0.441 | 0.7938 |  0.767711   0.000464    0.30128 | 0.289599  0.646696   0.44066 |      0.0       0.0       0.0 |  0.767711   0.000464    0.30128 |  0.000002   0.000002   0.000002 |  0.000002   0.000002   0.000002 |  1.44 | 0.000001 |  0.000002 | 0.186747 |   0.7938 | 0.584185 | 0.0 | 0.0 |    4.348785 | -0.088092 |  0.000609 | 0.202495 | 0.023437 |  0.217711   0.710666   1.953686 | 38.553961  30.336241  35.253831 | 139.604426 | 1508.980514 |  0.000095 |   122.79801 | 0.023437 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.5 | 0.0 | 0.0 |  7.12 | 0.3465  0.585  0.315 |  0.567 |  6.357962   4.562585   6.159877 | 0.317367  0.584262  0.314994 |      0.0       0.0       0.0 |  6.357962   4.562585   6.159877 |  0.000453   0.000453   0.000453 |  0.000453   0.000453   0.000453 |  7.12 | 0.000123 |  0.000342 | 0.019561 |    0.567 | 0.417847 | 0.0 | 0.0 |    2.705199 | -0.078007 |  0.000277 | 0.180515 | 0.020893 |  0.662382   2.683819   2.598816 | 44.911471  34.898373  41.413255 | 139.545857 | 1511.685713 |  0.000161 |  116.898665 | 0.020893 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 | 0.5544  0.936  0.504 | 0.9072 |       0.0        0.0        0.0 | 0.472491  0.935923  0.503987 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000888   0.000888   0.000888 |  0.000888   0.000888   0.000888 |   0.0 | 0.000241 |  0.000671 | 0.054335 |   0.9072 | 0.668444 | 0.0 | 0.0 |    1.826143 | -0.068386 |  0.000521 | 0.160597 | 0.018588 |  0.189891   1.747896   2.094829 | 44.910583  34.897485  41.412366 | 139.531565 | 1513.511856 |  0.000311 |   104.97133 | 0.018588 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 1.0 | 0.0 | 0.0 |   0.0 |  0.693   1.17   0.63 |  1.134 |       0.0        0.0        0.0 | 0.343246  1.155778  0.629754 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000634   0.000634   0.000634 |  0.000634   0.000634   0.000634 |   0.0 | 0.000172 |  0.000479 | 0.236615 |    1.134 | 0.834717 | 0.0 | 0.0 |    1.985034 | -0.056854 |  0.000541 | 0.136952 | 0.015851 | -0.153355   0.592118   1.465074 | 44.909949  34.896851  41.411732 | 139.711154 |  1515.49689 |  0.000249 |   94.509017 | 0.015851 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.194698  0.817474  0.748186 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000016   0.000016   0.000016 |  0.000016   0.000016   0.000016 |   0.0 | 0.000004 |  0.000012 | 0.616708 |   1.3608 | 0.999295 | 0.0 | 0.0 |     2.91962 | -0.047849 |  0.000217 | 0.117121 | 0.013556 | -0.348054  -0.225356   0.716888 | 44.909933  34.896836  41.411717 | 140.280009 |  1518.41651 |  0.000044 |   85.192647 | 0.013556 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.9 | 0.0 | 0.0 |  0.72 | 0.6237  1.053  0.567 | 1.0206 |  0.198569   0.000023   0.002961 | 0.207131  0.451043  0.552689 |      0.0       0.0       0.0 |  0.198569   0.000023   0.002961 |  0.000004   0.000004   0.000004 |  0.000004   0.000004   0.000004 |  0.72 | 0.000001 |  0.000003 | 0.475714 |   1.0206 | 0.748555 | 0.0 | 0.0 |    3.744556 | -0.041511 |  0.000039 | 0.101377 | 0.011733 | -0.033754   0.043578   0.881239 | 45.108498  34.896855  41.414673 | 140.714211 | 1522.161066 |  0.000007 |   77.994486 | 0.011733 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.9 | 0.0 | 0.0 |  0.08 | 0.6237  1.053  0.567 | 1.0206 |  0.023902   0.000002   0.000192 | 0.223811  0.376938  0.535207 | 0.008366  0.000001  0.000067 |  0.015537   0.000001   0.000125 |  0.184058   0.184058   0.184058 |  0.184058   0.184058   0.184058 |  0.08 | 0.052369 |  0.141952 | 0.489198 |   1.0206 | 0.746756 | 0.0 | 0.0 |    -0.26656 | -0.037103 |  0.080358 | 0.091091 | 0.010543 | -0.201467  -0.253362   0.425841 | 44.939977  34.712798   41.23074 | 141.113938 | 1521.894506 |  0.061601 |   74.800647 | 0.010543 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.171721  0.216415  0.491956 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |   0.00479    0.00479    0.00479 |   0.00479    0.00479    0.00479 |   0.0 | 0.001333 |   0.00359 | 0.831056 |   1.3608 | 0.973017 | 0.0 | 0.0 |    5.774035 | -0.035608 |  0.055331 | 0.085726 | 0.009922 | -0.373188  -0.469777  -0.066116 | 44.935187  34.708008   41.22595 | 141.908053 | 1527.668541 |  0.009861 |   70.294439 | 0.009922 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 1.4 | 0.0 | 0.0 |   0.0 | 0.9702  1.638  0.882 | 1.5876 |       0.0        0.0        0.0 | 0.116277  0.128063  0.249379 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  3.633659   3.633659   3.633659 |  3.633659   3.633659   3.633659 |   0.0 |  1.01639 |  2.718908 | 1.066482 |   1.5876 | 1.102849 | 0.0 | 0.0 |  -68.147557 | -0.040542 |  1.551842 | 0.130011 | 0.015048 | -0.489465   -0.59784  -0.315494 | 41.301528  31.074349  37.592291 | 141.917603 | 1459.520984 |  1.176926 |  136.458656 | 0.015048 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 1.1 | 0.0 | 0.0 |  3.12 | 0.7623  1.287  0.693 | 1.2474 |  1.818476   0.016201   0.251089 | 0.484996  0.989423  0.602617 | 0.181848   0.00162  0.025109 |  1.636628   0.014581    0.22598 |  0.006707   0.006707   0.006707 |  0.006707   0.006707   0.006707 |  3.12 | 0.036007 |  0.086406 | 0.282439 |   1.2474 | 0.909184 | 0.0 | 0.0 |   31.776788 | -0.114115 |   1.06035 |  0.28374 |  0.03284 |  0.327064   1.516536     1.9508 | 42.931449  31.082224  37.811565 |  142.04992 | 1491.297772 |  0.202982 |  171.068945 |  0.03284 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 1.1 | 0.0 | 0.0 |  6.96 | 0.7623  1.287  0.693 | 1.2474 |  6.071649   4.671871   5.753559 | 0.685976  1.286933  0.692984 |      0.0       0.0       0.0 |  6.071649   4.671871   5.753559 |  0.000117   0.000117   0.000117 |  0.000117   0.000117   0.000117 |  6.96 | 0.000033 |  0.000088 | 0.050614 |   1.2474 | 0.919214 | 0.0 | 0.0 |   15.954879 | -0.142678 |  0.175529 | 0.302005 | 0.034954 |  0.529438   2.517732   2.464257 | 49.002981  35.753978  43.565006 | 141.957824 | 1507.252651 |   0.02754 |  163.990141 | 0.034954 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.5 | 0.0 | 0.0 | 21.12 | 0.3465  0.585  0.315 |  0.567 | 20.402847  20.094414   20.41026 | 0.337476  0.584999  0.314999 |      0.0       0.0       0.0 | 20.402847  20.094414   20.41026 |  0.000084   0.000084   0.000084 |  0.000084   0.000084   0.000084 | 21.12 | 0.000023 |  0.000063 | 0.005982 |    0.567 | 0.417912 | 0.0 | 0.0 |    6.993098 |  -0.14477 |  0.023825 | 0.290815 | 0.033659 |  0.909116   2.958319   2.858998 | 69.405744  55.848308  63.975182 | 141.819012 | 1514.245749 |  0.003778 |  164.785417 | 0.033659 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.6 | 0.0 | 0.0 |   9.2 | 0.4158  0.702  0.378 | 0.6804 |  8.931806   8.696853   8.951311 | 0.406533  0.701998  0.377999 | 6.028969  5.870376  6.042135 |  2.902837   2.826477   2.909176 |  3.633659   3.633659   3.633659 |  3.633659   3.633659   3.633659 |   9.2 | 2.637844 |  7.242307 | 0.006145 |   0.6804 |   0.5015 | 0.0 | 0.0 | -191.125786 | -0.165239 |  4.116173 | 0.499492 | 0.057812 |  0.770777   2.759468   2.729688 | 68.674922  55.041126    63.2507 | 139.022073 | 1323.119962 |  3.129912 |  342.735875 | 0.057812 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 1.5 | 0.0 | 0.0 |  0.72 | 1.0395  1.755  0.945 |  1.701 |  0.554111   0.275961     0.5481 | 0.842857  1.754558  0.944963 | 0.027706  0.013798  0.027405 |  0.526406   0.262163   0.520695 |   0.00342    0.00342    0.00342 |   0.00342    0.00342    0.00342 |  0.72 | 0.007126 |   0.02051 |  0.13054 |    1.701 | 1.253203 | 0.0 | 0.0 |    64.34888 | -0.503561 |  2.719178 | 1.075018 | 0.124423 |  0.093809   1.448949   1.956626 | 69.197908  55.299869  63.767974 | 138.641926 | 1387.468843 |  0.431244 |  399.036649 | 0.124423 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |  1.386   2.34   1.26 |  2.268 |       0.0        0.0        0.0 | 0.443267  1.687835  1.251504 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.001244   0.001244   0.001244 |  0.001244   0.001244   0.001244 |   0.0 | 0.000334 |  0.000944 | 0.841351 |    2.268 | 1.666944 | 0.0 | 0.0 |   32.301781 | -0.515823 |  0.373347 |  0.99021 | 0.114608 | -0.349458  -0.238886   0.705122 | 69.196663  55.298624  63.766729 | 138.967121 | 1419.770623 |   0.05884 |   342.80437 | 0.114608 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.139047  0.269888  0.749587 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000117   0.000117   0.000117 |  0.000117   0.000117   0.000117 |   0.0 | 0.000032 |  0.000089 | 1.174146 |   1.8144 | 1.306451 | 0.0 | 0.0 |   26.852792 | -0.390592 |  0.050859 | 0.768609 | 0.088959 | -0.488506  -0.508774  -0.044465 | 69.196546  55.298507  63.766612 | 139.750642 | 1446.623416 |   0.00807 |  287.826438 | 0.088959 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.087816   0.12487  0.285174 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000117   0.000117   0.000117 |  0.000117   0.000117   0.000117 |   0.0 | 0.000032 |  0.000089 | 1.256072 |   1.8144 | 1.260438 | 0.0 | 0.0 |   22.666928 | -0.295917 |   0.00707 | 0.594096 | 0.068761 | -0.576322  -0.633644  -0.329639 | 69.196428  55.298389  63.766495 | 140.710766 | 1469.290343 |  0.001089 |  243.603727 | 0.068761 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.049327  0.062902   0.10941 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.001244   0.001244   0.001244 |  0.001244   0.001244   0.001244 |   0.0 | 0.000344 |  0.000934 |  0.96312 |   1.3608 | 0.934907 | 0.0 | 0.0 |   18.317657 | -0.230729 |   0.00146 | 0.466831 | 0.054031 | -0.625648  -0.696546  -0.439048 | 69.195184  55.297145   63.76525 | 141.442812 |    1487.608 |  0.000564 |  208.797731 | 0.054031 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 | 0.9009  1.521  0.819 | 1.4742 |       0.0        0.0        0.0 | 0.043674  0.052837  0.081009 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.018374   0.018374   0.018374 |  0.018374   0.018374   0.018374 |   0.0 | 0.005137 |  0.013751 | 1.054831 |   1.4742 | 1.008998 | 0.0 | 0.0 |   14.489647 | -0.184232 |  0.007862 | 0.373372 | 0.043214 | -0.669322  -0.749383  -0.520058 |  69.17681  55.278771  63.746876 | 142.308274 | 1502.097647 |  0.006453 |  180.915523 | 0.043214 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.044306   0.05155  0.072302 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |      0.05       0.05       0.05 |      0.05       0.05       0.05 |   0.0 | 0.014157 |  0.037259 | 1.308831 |   1.8144 |  1.23897 | 0.0 | 0.0 |   12.214636 | -0.150417 |  0.026991 | 0.303555 | 0.035134 | -0.713628  -0.800933   -0.59236 |  69.12681  55.228771  63.696876 | 143.452531 | 1514.312282 |  0.016721 |  158.612527 | 0.035134 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 1.9 | 0.0 | 0.0 |   0.0 | 1.3167  2.223  1.197 | 2.1546 |       0.0        0.0        0.0 | 0.043336   0.04889   0.06406 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |   1.95975    1.95975    1.95975 |   1.95975    1.95975    1.95975 |   0.0 | 0.561887 |  1.454051 | 1.564285 |   2.1546 | 1.468925 | 0.0 | 0.0 |  -28.268903 | -0.132292 |  0.840283 | 0.288218 | 0.033359 | -0.756964  -0.849823   -0.65642 |  67.16706  53.269021  61.737126 | 144.322637 |  1486.04338 |   0.63049 |  177.386794 | 0.033359 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.8 | 0.0 | 0.0 |   1.2 | 0.5544  0.936  0.504 | 0.9072 |  0.261946   0.000022   0.000144 | 0.145107   0.23828  0.204215 |  0.19646  0.000017  0.000108 |  0.065487   0.000006   0.000036 |      5.05       5.05       5.05 |      5.05       5.05       5.05 |   1.2 | 1.483567 |  3.832681 |  0.50244 |   0.9072 | 0.636687 | 0.0 | 0.0 |  -83.807551 | -0.181114 |  2.706216 | 0.457216 | 0.052919 |  0.035982   0.111875   0.339221 | 62.182546  48.219026  56.687162 | 143.160396 | 1402.235829 |  1.756955 |  279.436232 | 0.052919 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 1.5 | 0.0 | 0.0 |  0.24 | 1.0395  1.755  0.945 |  1.701 |  0.076968   0.000006    0.00005 | 0.397174  0.624732  0.601928 | 0.076968  0.000006   0.00005 |       0.0        0.0        0.0 | 22.000001  22.000001  22.000001 | 22.000001  22.000001  22.000001 |  0.24 | 6.040982 | 16.609304 | 0.800552 |    1.701 | 1.216305 | 0.0 | 0.0 | -164.872077 | -0.644895 | 10.921826 | 1.428194 |   0.1653 | -0.198159  -0.272863  -0.022757 | 40.182546  26.219026  34.687162 | 137.275071 | 1237.363752 |  7.444434 |  713.295105 |   0.1653 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 2.7 | 0.0 | 0.0 |  0.16 | 1.8711  3.159  1.701 | 3.0618 |  0.023632   0.000001   0.000007 | 0.346942  0.403332   0.47052 | 0.023632  0.000001  0.000007 |       0.0        0.0        0.0 |      35.0       35.0       35.0 |      35.0  27.003244  34.732152 |  0.16 | 8.019558 | 25.370766 |  1.92555 |   3.0618 | 2.128263 | 0.0 | 0.0 | -110.533139 | -3.867888 | 21.446337 | 4.674145 | 0.540989 | -0.408734  -0.516195  -0.333283 |  5.182546  -0.784218   -0.04499 | 127.313175 | 1126.830613 | 11.368862 | 1356.982728 | 0.540989 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 1.5 | 0.0 | 0.0 |   3.6 | 1.2825  1.755  0.945 |  1.728 |  1.991548   0.031215   0.413719 | 0.874807  1.427486  0.831576 | 1.991548  0.031215  0.413719 |       0.0        0.0        0.0 | 19.000004  19.000004  19.000004 |  6.071372   0.262718   0.920078 |   3.6 |  1.21334 |  3.967306 | 0.379423 |    1.728 | 1.384773 | 0.0 | 0.0 |  -21.217785 | -7.094506 | 12.663141 | 7.209863 | 0.834475 |  0.324911   1.625103   2.021421 | -0.888827  -1.046935  -0.965068 | 119.384752 | 1105.612828 |  2.673027 | 1305.987767 | 0.834475 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 |  1.368  1.872  1.008 | 1.8432 |       0.0        0.0        0.0 | 0.610094  1.639722  1.005972 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  7.013137   7.013137   7.013137 |  0.093304   0.050378   0.069838 |   0.0 | 0.015615 |  0.064026 | 0.580684 |   1.8432 | 1.488426 | 0.0 | 0.0 |    12.10703 | -4.257845 |  2.349429 | 5.267133 | 0.609622 | -0.285183  -0.014619   1.015449 | -0.982131  -1.097314  -1.034907 | 115.691976 | 1117.719858 |  0.387624 |  968.138953 | 0.609622 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |   1.71   2.34   1.26 |  2.304 |       0.0        0.0        0.0 | 0.226983  0.478112  1.029561 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  1.298601   1.298601   1.298601 |  0.013653   0.008179   0.010807 |   0.0 | 0.002265 |  0.009687 | 1.603904 |    2.304 | 1.835652 | 0.0 | 0.0 |   21.791594 | -1.982554 |  0.340629 | 3.354159 | 0.388213 | -0.512166  -0.492731  -0.014112 | -0.995783  -1.105493  -1.045714 | 115.311061 | 1139.511452 |  0.056683 |  727.387168 | 0.388213 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 2.1 | 0.0 | 0.0 |   0.0 | 1.7955  2.457  1.323 | 2.4192 |       0.0        0.0        0.0 | 0.119526  0.161082  0.357094 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.798601   0.798601   0.798601 |  0.007994   0.004882   0.006391 |   0.0 |  0.00134 |  0.005694 | 1.876255 |   2.4192 | 1.855875 | 0.0 | 0.0 |   42.113429 | -0.979669 |  0.052254 | 2.255785 | 0.261086 | -0.631692  -0.653813  -0.371206 | -1.003777  -1.110375  -1.052105 | 116.206307 | 1181.624881 |  0.010123 |  571.535766 | 0.261086 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 1.7 | 0.0 | 0.0 |  0.32 | 1.4535  1.989  1.071 | 1.9584 |  0.014438   0.000001   0.000008 | 0.119666  0.145372  0.209579 | 0.005414       0.0  0.000003 |  0.009024   0.000001   0.000005 |  0.251908   0.251908   0.251908 |  0.002507   0.001518   0.001978 |  0.32 | 0.001132 |  0.004387 | 1.498583 |   1.9584 |  1.49456 | 0.0 | 0.0 |   52.858212 | -0.577839 |  0.011107 | 1.611433 | 0.186508 | -0.445796  -0.479186  -0.260793 | -0.997259  -1.111892  -1.054078 | 117.125919 | 1234.483093 |  0.003402 |  464.387312 | 0.186508 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 1.7 | 0.0 | 0.0 |   0.0 | 1.4535  1.989  1.071 | 1.9584 |       0.0        0.0        0.0 | 0.126256  0.143948  0.175627 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.096611   0.096611   0.096611 |  0.000976   0.000579   0.000754 |   0.0 | 0.000168 |  0.000684 | 1.494572 |   1.9584 | 1.491151 | 0.0 | 0.0 |   44.104767 | -0.383111 |  0.003346 | 1.199918 | 0.138879 | -0.572052  -0.623134   -0.43642 | -0.998235  -1.112471  -1.054832 | 118.237212 |  1278.58786 |   0.00074 |  385.701757 | 0.138879 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 |  0.684  0.936  0.504 | 0.9216 |       0.0        0.0        0.0 | 0.042008   0.04567  0.053493 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  1.298601   1.298601   1.298601 |  0.012716   0.007641    0.00989 |   0.0 | 0.002218 |  0.008914 | 0.722182 |   0.9216 | 0.698767 | 0.0 | 0.0 |   32.640212 |  -0.26865 |  0.005812 | 0.922225 | 0.106739 |  -0.61406  -0.668804  -0.489913 | -1.010951  -1.120113  -1.064722 | 118.688526 | 1311.228072 |  0.003842 |  327.106236 | 0.106739 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 | 1.1115  1.521  0.819 | 1.4976 |       0.0        0.0        0.0 | 0.055218  0.058909  0.067481 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  3.184058   3.184058   3.184058 |  0.028378   0.017679   0.022514 |   0.0 | 0.005048 |  0.020038 | 1.187233 |   1.4976 | 1.133492 | 0.0 | 0.0 |    24.00745 | -0.193043 |  0.015007 | 0.727937 | 0.084252 | -0.669278  -0.727713  -0.557394 | -1.039329  -1.137792  -1.087236 | 119.677668 | 1335.235522 |  0.008873 |  281.433974 | 0.084252 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 2.5 | 0.0 | 0.0 |   0.0 | 2.1375  2.925  1.575 |   2.88 |       0.0        0.0        0.0 | 0.078864  0.082512  0.091995 |      0.0       0.0       0.0 |       0.0        0.0        0.0 | 12.500323  12.500323  12.500323 |   0.08546   0.058198   0.071096 |   0.0 | 0.015908 |  0.061528 | 2.311243 |     2.88 | 2.175763 | 0.0 | 0.0 |   20.526782 | -0.141945 |    0.0439 | 0.586074 | 0.067833 | -0.748142  -0.810225  -0.649389 | -1.124788  -1.195991  -1.158332 | 121.831059 | 1355.762304 |  0.026501 |  245.141598 | 0.067833 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_v002_no_vadose_zone:

No vadose zone
______________

>>> lt(SEALED)

.. integration-test::

    >>> test("wland_v002_no_vadose_zone",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |     t |    p | pet | fxg | fxs |    pc |                petl |    pes |                              tf |                           ei |                           rf |                              sf |                              pm |                              am |    ps |  pv |        pq | etv |       es |       et | fxs | fxg | cdg | fgs |       fqs |        rh |        r |                              ic |                              sp |  dv |  dg |        hq |          hs |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.6 | 0.0 | 0.0 |   0.0 | 0.378  0.378  0.378 | 0.6588 |       0.0        0.0        0.0 |      0.0       0.0       0.0 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 | 0.000067 | 0.000002 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 |      -3.0       -3.0       -3.0 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |   -2.000067 |      0.0 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 | 0.504  0.504  0.504 | 0.8784 |       0.0        0.0        0.0 | 0.000001  0.000001  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |  0.32 | 0.0 |       0.0 | 0.0 |  0.00024 | 0.000006 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680002  -2.680002  -2.680002 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |   -1.680307 |      0.0 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 | 0.441  0.441  0.441 | 0.7686 |       0.0        0.0        0.0 | 0.000002  0.000002  0.000002 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 |  0.00034 | 0.000009 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680004  -2.680004  -2.680004 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |   -1.680647 |      0.0 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.252  0.252  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000001  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 | 0.000194 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680005  -2.680005  -2.680005 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |   -1.680842 |      0.0 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.252  0.252  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000001  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 | 0.000194 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680006  -2.680006  -2.680006 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |   -1.681036 |      0.0 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 | 0.252  0.252  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000001  0.000001  0.000001 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 | 0.000194 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680007  -2.680007  -2.680007 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |    -1.68123 |      0.0 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.4 | 0.0 | 0.0 |  0.16 | 0.252  0.252  0.252 | 0.4392 |       0.0        0.0        0.0 | 0.000002  0.000002  0.000002 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000008   0.000008   0.000008 |       0.0        0.0        0.0 |  0.16 | 0.0 |       0.0 | 0.0 | 0.000299 | 0.000008 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.520009  -2.520009  -2.520009 |      -3.0       -3.0       -3.0 | nan | nan |       0.0 |   -1.521529 |      0.0 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.3 | 0.0 | 0.0 |   3.6 | 0.189  0.189  0.189 | 0.3294 |  0.002575   0.002575   0.002575 |  0.05696   0.05696   0.05696 |      0.0       0.0       0.0 |  0.002575   0.002575   0.002575 |  0.000011   0.000011   0.000011 |       0.0        0.0        0.0 |   3.6 | 0.0 |       0.0 | 0.0 | 0.190061 | 0.059622 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000086 |  0.00001 |  1.020457   1.020457   1.020457 | -2.997425  -2.997425  -2.997425 | nan | nan |       0.0 |    1.884092 |  0.00001 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.3 | 0.0 | 0.0 |   0.0 | 0.189  0.189  0.189 | 0.3294 |       0.0        0.0        0.0 | 0.186128  0.186128  0.186128 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 |  0.32923 |  0.18899 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |   0.00031 | 0.000036 |  0.834329   0.834329   0.834329 | -2.997425  -2.997425  -2.997425 | nan | nan |       0.0 |    1.539384 | 0.000036 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.4 | 0.0 | 0.0 |  2.56 | 0.252  0.252  0.252 | 0.4392 |  0.899276   0.899276   0.899276 | 0.251443  0.251443  0.251443 |      0.0       0.0       0.0 |  0.899276   0.899276   0.899276 |   0.00003    0.00003    0.00003 |       0.0        0.0        0.0 |  2.56 | 0.0 |       0.0 | 0.0 |  0.43916 | 0.255198 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000585 | 0.000068 |  2.243609   2.243609   2.243609 | -2.098149  -2.098149  -2.098149 | nan | nan |       0.0 |    3.630963 | 0.000068 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.3 | 0.0 | 0.0 |  3.68 | 0.189  0.189  0.189 | 0.3294 |  3.208267   3.208267   3.208267 | 0.188997  0.188997  0.188997 | 0.561447  0.561447  0.561447 |   2.64682    2.64682    2.64682 |  0.018374   0.018374   0.018374 |  0.003734   0.003734   0.003734 |  3.68 | 0.0 |  0.565181 | 0.0 |   0.3294 | 0.191805 | 0.0 | 0.0 | 0.0 | 0.0 |  0.312595 |  0.005503 | 0.000637 |  2.526345   2.526345   2.526345 |  0.544937   0.544937   0.544937 | nan | nan |  0.252586 |   22.023581 | 0.000637 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.6 | 0.0 | 0.0 |  1.84 | 0.378  0.378  0.378 | 0.6588 |  1.618792   1.618792   1.618792 | 0.377995  0.377995  0.377995 |      0.0       0.0       0.0 |  1.618792   1.618792   1.618792 |  0.000888   0.000888   0.000888 |  0.000879   0.000879   0.000879 |  1.84 | 0.0 |  0.000879 | 0.0 |   0.6588 | 0.383611 | 0.0 | 0.0 | 0.0 | 0.0 |  0.219014 |   0.02176 | 0.002519 |  2.369558   2.369558   2.369558 |   2.16285    2.16285    2.16285 | nan | nan |  0.034451 |    32.84847 | 0.002519 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.8 | 0.0 | 0.0 |  14.4 | 0.504  0.504  0.504 | 0.8784 | 13.564241  13.564241  13.564241 | 0.503997  0.503997  0.503997 |  1.69553   1.69553   1.69553 | 11.868711  11.868711  11.868711 |  0.009388   0.009388   0.009388 |  0.009388   0.009388   0.009388 |  14.4 | 0.0 |  1.704919 | 0.0 |   0.8784 | 0.511485 | 0.0 | 0.0 | 0.0 | 0.0 |  0.989471 |  0.061663 | 0.007137 |   2.70132    2.70132    2.70132 | 14.022173  14.022173  14.022173 | nan | nan |  0.749898 |   91.771019 | 0.007137 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.5 | 0.0 | 0.0 | 15.36 | 0.315  0.315  0.315 |  0.549 | 14.931774  14.931774  14.931774 | 0.314999  0.314999  0.314999 |      0.0       0.0       0.0 | 14.931774  14.931774  14.931774 |   0.00006    0.00006    0.00006 |   0.00006    0.00006    0.00006 | 15.36 | 0.0 |   0.00006 | 0.0 |    0.549 | 0.319679 | 0.0 | 0.0 | 0.0 | 0.0 |  0.648952 |  0.172668 | 0.019985 |  2.814547   2.814547   2.814547 | 28.953887  28.953887  28.953887 | nan | nan |  0.101005 |  129.747281 | 0.019985 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 | 0.504  0.504  0.504 | 0.8784 |  0.294814   0.294814   0.294814 | 0.503995  0.503995  0.503995 |      0.0       0.0       0.0 |  0.294814   0.294814   0.294814 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |  0.32 | 0.0 |       0.0 | 0.0 |   0.8784 | 0.511484 | 0.0 | 0.0 | 0.0 | 0.0 |  0.085293 |  0.197859 |   0.0229 |  2.335738   2.335738   2.335738 | 29.248701  29.248701  29.248701 | nan | nan |  0.015712 |  123.475311 |   0.0229 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.5 | 0.0 | 0.0 |  6.64 | 0.315  0.315  0.315 |  0.549 |  6.056957   6.056957   6.056957 | 0.314997  0.314997  0.314997 |      0.0       0.0       0.0 |  6.056957   6.056957   6.056957 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |  6.64 | 0.0 |       0.0 | 0.0 |    0.549 | 0.319677 | 0.0 | 0.0 | 0.0 | 0.0 |  0.013676 |  0.186196 |  0.02155 |  2.603784   2.603784   2.603784 | 35.305657  35.305657  35.305657 | nan | nan |  0.002036 |   120.92665 |  0.02155 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.4 | 0.0 | 0.0 |  4.24 | 0.252  0.252  0.252 | 0.4392 |  3.989852   3.989852   3.989852 | 0.251998  0.251998  0.251998 | 1.097209  1.097209  1.097209 |  2.892643   2.892643   2.892643 |  0.069591   0.069591   0.069591 |  0.069591   0.069591   0.069591 |  4.24 | 0.0 |    1.1668 | 0.0 |   0.4392 | 0.255742 | 0.0 | 0.0 | 0.0 | 0.0 |  0.663756 |  0.205341 | 0.023766 |  2.601934   2.601934   2.601934 | 38.128709  38.128709  38.128709 | nan | nan |   0.50508 |  146.984456 | 0.023766 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 1.3 | 0.0 | 0.0 |  0.56 | 0.819  0.819  0.819 | 1.4274 |  0.404441   0.404441   0.404441 | 0.818961  0.818961  0.818961 | 0.343774  0.343774  0.343774 |  0.060666   0.060666   0.060666 |  7.013137   7.013137   7.013137 |  7.013137   7.013137   7.013137 |  0.56 | 0.0 |  7.356912 | 0.0 |   1.4274 |  0.83113 | 0.0 | 0.0 | 0.0 | 0.0 |  4.631053 |   0.49186 | 0.056928 |  1.938532   1.938532   1.938532 | 31.176238  31.176238  31.176238 | nan | nan |  3.230939 |  348.445657 | 0.056928 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.9 | 0.0 | 0.0 |  2.16 | 0.567  0.567  0.567 | 0.9882 |   1.34089    1.34089    1.34089 | 0.566963  0.566963  0.566963 |      0.0       0.0       0.0 |   1.34089    1.34089    1.34089 |  0.000016   0.000016   0.000016 |  0.000016   0.000016   0.000016 |  2.16 | 0.0 |  0.000016 | 0.0 |   0.9882 | 0.575388 | 0.0 | 0.0 | 0.0 | 0.0 |  2.793519 |  1.150341 | 0.133141 |  2.190679   2.190679   2.190679 | 32.517112  32.517112  32.517112 | nan | nan |  0.437436 |  428.982847 | 0.133141 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.7 | 0.0 | 0.0 |  1.28 | 0.441  0.441  0.441 | 0.7938 |  0.875908   0.875908   0.875908 | 0.440979  0.440979  0.440979 |      0.0       0.0       0.0 |  0.875908   0.875908   0.875908 |  0.000001   0.000001   0.000001 |  0.000001   0.000001   0.000001 |  1.28 | 0.0 |  0.000001 | 0.0 |   0.7938 | 0.448036 | 0.0 | 0.0 | 0.0 | 0.0 |  0.378436 |   1.15153 | 0.133279 |  2.153792   2.153792   2.153792 | 33.393019  33.393019  33.393019 | nan | nan |     0.059 |  390.435891 | 0.133279 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.7 | 0.0 | 0.0 |   2.0 | 0.441  0.441  0.441 | 0.7938 |  1.460731   1.460731   1.460731 | 0.440983  0.440983  0.440983 |      0.0       0.0       0.0 |  1.460731   1.460731   1.460731 |  0.000022   0.000022   0.000022 |  0.000022   0.000022   0.000022 |   2.0 | 0.0 |  0.000022 | 0.0 |   0.7938 |  0.44804 | 0.0 | 0.0 | 0.0 | 0.0 |  0.049835 |  0.971353 | 0.112425 |  2.252077   2.252077   2.252077 | 34.853729  34.853729  34.853729 | nan | nan |  0.009187 |  345.516369 | 0.112425 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 1.1 | 0.0 | 0.0 |  0.48 | 0.693  0.693  0.693 | 1.2474 |  0.242299   0.242299   0.242299 | 0.692918  0.692918  0.692918 |      0.0       0.0       0.0 |  0.242299   0.242299   0.242299 |  0.001244   0.001244   0.001244 |  0.001244   0.001244   0.001244 |  0.48 | 0.0 |  0.001244 | 0.0 |   1.2474 | 0.704008 | 0.0 | 0.0 | 0.0 | 0.0 |  0.006954 |  0.805985 | 0.093285 |   1.79686    1.79686    1.79686 | 35.094783  35.094783  35.094783 | nan | nan |  0.003477 |   304.79046 | 0.093285 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 1.0 | 0.0 | 0.0 |  0.16 |  0.63   0.63   0.63 |  1.134 |  0.020512   0.020512   0.020512 | 0.629363  0.629363  0.629363 | 0.008717  0.008717  0.008717 |  0.011794   0.011794   0.011794 |   0.45975    0.45975    0.45975 |   0.45975    0.45975    0.45975 |  0.16 | 0.0 |  0.468467 | 0.0 |    1.134 | 0.639455 | 0.0 | 0.0 | 0.0 | 0.0 |  0.271901 |  0.686722 | 0.079482 |  1.306986   1.306986   1.306986 | 34.646828  34.646828  34.646828 | nan | nan |  0.200043 |  282.803486 | 0.079482 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.8 | 0.0 | 0.0 |  1.36 | 0.504  0.504  0.504 | 0.9072 |  0.260975   0.260975   0.260975 | 0.503638  0.503638  0.503638 | 0.019573  0.019573  0.019573 |  0.241402   0.241402   0.241402 |   0.00479    0.00479    0.00479 |   0.00479    0.00479    0.00479 |  1.36 | 0.0 |  0.024363 | 0.0 |   0.9072 | 0.511709 | 0.0 | 0.0 | 0.0 | 0.0 |  0.184323 |  0.621006 | 0.071876 |  1.902373   1.902373   1.902373 | 34.883439  34.883439  34.883439 | nan | nan |  0.040083 |  261.237836 | 0.071876 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.6 | 0.0 | 0.0 |  0.24 | 0.378  0.378  0.378 | 0.6804 |  0.068207   0.068207   0.068207 | 0.377898  0.377898  0.377898 |      0.0       0.0       0.0 |  0.068207   0.068207   0.068207 |  0.000001   0.000001   0.000001 |  0.000001   0.000001   0.000001 |  0.24 | 0.0 |  0.000001 | 0.0 |   0.6804 | 0.383948 | 0.0 | 0.0 | 0.0 | 0.0 |  0.033848 |  0.539065 | 0.062392 |  1.696269   1.696269   1.696269 | 34.951645  34.951645  34.951645 | nan | nan |  0.006235 |  235.502773 | 0.062392 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 | 0.441  0.441  0.441 | 0.7938 |       0.0        0.0        0.0 | 0.440409  0.440409  0.440409 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |       0.0        0.0        0.0 |   0.0 | 0.0 |       0.0 | 0.0 |   0.7938 | 0.447476 | 0.0 | 0.0 | 0.0 | 0.0 |  0.008314 |  0.460003 | 0.053241 |   1.25586    1.25586    1.25586 | 34.951645  34.951645  34.951645 | nan | nan | -0.002078 |  212.116224 | 0.053241 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.7 | 0.0 | 0.0 |  1.44 | 0.441  0.441  0.441 | 0.7938 |  0.301973   0.301973   0.301973 | 0.440654  0.440654  0.440654 |      0.0       0.0       0.0 |  0.301973   0.301973   0.301973 |  0.000002   0.000002   0.000002 |  0.000002   0.000002   0.000002 |  1.44 | 0.0 |  0.000002 | 0.0 |   0.7938 | 0.447717 | 0.0 | 0.0 | 0.0 | 0.0 | -0.001793 |   0.39616 | 0.045852 |  1.953233   1.953233   1.953233 | 35.253616  35.253616  35.253616 | nan | nan | -0.000284 |  192.866568 | 0.045852 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.5 | 0.0 | 0.0 |  7.12 | 0.315  0.315  0.315 |  0.567 |  6.159253   6.159253   6.159253 | 0.314994  0.314994  0.314994 |      0.0       0.0       0.0 |  6.159253   6.159253   6.159253 |  0.000453   0.000453   0.000453 |  0.000453   0.000453   0.000453 |  7.12 | 0.0 |  0.000453 | 0.0 |    0.567 | 0.320034 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000011 |   0.35272 | 0.040824 |  2.598986   2.598986   2.598986 | 41.412416  41.412416  41.412416 | nan | nan |  0.000157 |  181.784116 | 0.040824 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 | 0.504  0.504  0.504 | 0.9072 |       0.0        0.0        0.0 | 0.503987  0.503987  0.503987 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000888   0.000888   0.000888 |  0.000888   0.000888   0.000888 |   0.0 | 0.0 |  0.000888 | 0.0 |   0.9072 | 0.512051 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000506 |  0.314282 | 0.036375 |  2.094999   2.094999   2.094999 | 41.411528  41.411528  41.411528 | nan | nan |   0.00054 |  165.187617 | 0.036375 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 1.0 | 0.0 | 0.0 |   0.0 |  0.63   0.63   0.63 |  1.134 |       0.0        0.0        0.0 | 0.629751  0.629751  0.629751 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000634   0.000634   0.000634 |  0.000634   0.000634   0.000634 |   0.0 | 0.0 |  0.000634 | 0.0 |    1.134 | 0.639836 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000931 |  0.272705 | 0.031563 |  1.465249   1.465249   1.465249 | 41.410894  41.410894  41.410894 | nan | nan |  0.000243 |  150.463971 | 0.031563 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.756  0.756  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.748148  0.748148  0.748148 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000016   0.000016   0.000016 |  0.000016   0.000016   0.000016 |   0.0 | 0.0 |  0.000016 | 0.0 |   1.3608 | 0.760401 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000329 |  0.237333 | 0.027469 |  0.717101   0.717101   0.717101 | 41.410879  41.410879  41.410879 | nan | nan | -0.000071 |   137.25262 | 0.027469 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.9 | 0.0 | 0.0 |  0.72 | 0.567  0.567  0.567 | 1.0206 |  0.002965   0.002965   0.002965 | 0.552718  0.552718  0.552718 |      0.0       0.0       0.0 |  0.002965   0.002965   0.002965 |  0.000004   0.000004   0.000004 |  0.000004   0.000004   0.000004 |  0.72 | 0.0 |  0.000004 | 0.0 |   1.0206 | 0.562076 | 0.0 | 0.0 | 0.0 | 0.0 | -0.000093 |  0.208385 | 0.024119 |  0.881418   0.881418   0.881418 |  41.41384   41.41384   41.41384 | nan | nan |  0.000026 |  126.528211 | 0.024119 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.9 | 0.0 | 0.0 |  0.08 | 0.567  0.567  0.567 | 1.0206 |  0.000192   0.000192   0.000192 | 0.535229  0.535229  0.535229 | 0.000067  0.000067  0.000067 |  0.000125   0.000125   0.000125 |  0.184058   0.184058   0.184058 |  0.184058   0.184058   0.184058 |  0.08 | 0.0 |  0.184125 | 0.0 |   1.0206 | 0.544936 | 0.0 | 0.0 | 0.0 | 0.0 |  0.106415 |  0.188324 | 0.021797 |  0.425997   0.425997   0.425997 | 41.229907  41.229907  41.229907 | nan | nan |  0.077736 |  121.385715 | 0.021797 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.756  0.756  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.492029  0.492029  0.492029 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |   0.00479    0.00479    0.00479 |   0.00479    0.00479    0.00479 |   0.0 | 0.0 |   0.00479 | 0.0 |   1.3608 | 0.509404 | 0.0 | 0.0 | 0.0 | 0.0 |  0.068412 |  0.177555 |  0.02055 | -0.066032  -0.066032  -0.066032 | 41.225117  41.225117  41.225117 | nan | nan |  0.014115 |  114.499349 |  0.02055 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 1.4 | 0.0 | 0.0 |   0.0 | 0.882  0.882  0.882 | 1.5876 |       0.0        0.0        0.0 | 0.249425  0.249425  0.249425 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  3.633659   3.633659   3.633659 |  3.633659   3.633659   3.633659 |   0.0 | 0.0 |  3.633659 | 0.0 |   1.5876 | 0.276189 | 0.0 | 0.0 | 0.0 | 0.0 |  2.075009 |   0.24936 | 0.028861 | -0.315457  -0.315457  -0.315457 | 37.591458  37.591458  37.591458 | nan | nan |  1.572764 |   202.11922 | 0.028861 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 1.1 | 0.0 | 0.0 |  3.12 | 0.693  0.693  0.693 | 1.2474 |  0.251138   0.251138   0.251138 | 0.602565  0.602565  0.602565 | 0.025114  0.025114  0.025114 |  0.226024   0.226024   0.226024 |  0.006707   0.006707   0.006707 |  0.006707   0.006707   0.006707 |  3.12 | 0.0 |  0.031821 | 0.0 |   1.2474 | 0.615461 | 0.0 | 0.0 | 0.0 | 0.0 |  1.369598 |  0.496609 | 0.057478 |  1.950841   1.950841   1.950841 | 37.810775  37.810775  37.810775 | nan | nan |  0.234987 |  246.271666 | 0.057478 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 1.1 | 0.0 | 0.0 |  6.96 | 0.693  0.693  0.693 | 1.2474 |  5.752427   5.752427   5.752427 | 0.692984  0.692984  0.692984 |      0.0       0.0       0.0 |  5.752427   5.752427   5.752427 |  0.000117   0.000117   0.000117 |  0.000117   0.000117   0.000117 |  6.96 | 0.0 |  0.000117 | 0.0 |   1.2474 | 0.704072 | 0.0 | 0.0 | 0.0 | 0.0 |   0.20355 |  0.520082 | 0.060195 |   2.46543    2.46543    2.46543 | 43.563084  43.563084  43.563084 | nan | nan |  0.031555 |  235.954109 | 0.060195 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.5 | 0.0 | 0.0 | 21.12 | 0.315  0.315  0.315 |  0.567 | 20.411176  20.411176  20.411176 | 0.314999  0.314999  0.314999 |      0.0       0.0       0.0 | 20.411176  20.411176  20.411176 |  0.000084   0.000084   0.000084 |  0.000084   0.000084   0.000084 | 21.12 | 0.0 |  0.000084 | 0.0 |    0.567 | 0.320039 | 0.0 | 0.0 | 0.0 | 0.0 |  0.027353 |  0.495111 | 0.057305 |  2.859255   2.859255   2.859255 | 63.974177  63.974177  63.974177 | nan | nan |  0.004285 |  233.091874 | 0.057305 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.6 | 0.0 | 0.0 |   9.2 | 0.378  0.378  0.378 | 0.6804 |  8.951477   8.951477   8.951477 | 0.377999  0.377999  0.377999 | 6.042247  6.042247  6.042247 |   2.90923    2.90923    2.90923 |  3.633659   3.633659   3.633659 |  3.633659   3.633659   3.633659 |   9.2 | 0.0 |  9.675906 | 0.0 |   0.6804 | 0.384047 | 0.0 | 0.0 | 0.0 | 0.0 |  5.502898 |  0.817301 | 0.094595 |   2.72978    2.72978    2.72978 | 63.249748  63.249748  63.249748 | nan | nan |  4.177293 |  470.388437 | 0.094595 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 1.5 | 0.0 | 0.0 |  0.72 | 0.945  0.945  0.945 |  1.701 |  0.548136   0.548136   0.548136 | 0.944963  0.944963  0.944963 | 0.027407  0.027407  0.027407 |  0.520729   0.520729   0.520729 |   0.00342    0.00342    0.00342 |   0.00342    0.00342    0.00342 |  0.72 | 0.0 |  0.030827 | 0.0 |    1.701 | 0.960084 | 0.0 | 0.0 | 0.0 | 0.0 |  3.630626 |  1.749968 | 0.202543 |  1.956681   1.956681   1.956681 | 63.767056  63.767056  63.767056 | nan | nan |  0.577494 |  559.809702 | 0.202543 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |  1.26   1.26   1.26 |  2.268 |       0.0        0.0        0.0 | 1.251526  1.251526  1.251526 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.001244   0.001244   0.001244 |  0.001244   0.001244   0.001244 |   0.0 | 0.0 |  0.001244 | 0.0 |    2.268 | 1.271856 | 0.0 | 0.0 | 0.0 | 0.0 |  0.500436 |  1.687171 | 0.195274 |  0.705154   0.705154   0.705154 | 63.765812  63.765812  63.765812 | nan | nan |  0.078302 |  497.704544 | 0.195274 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008 | 1.8144 |       0.0        0.0        0.0 |  0.74934   0.74934   0.74934 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000117   0.000117   0.000117 |  0.000117   0.000117   0.000117 |   0.0 | 0.0 |  0.000117 | 0.0 |   1.8144 | 0.770641 | 0.0 | 0.0 | 0.0 | 0.0 |   0.06619 |  1.374635 | 0.159101 | -0.044186  -0.044186  -0.044186 | 63.765695  63.765695  63.765695 | nan | nan |   0.01223 |  430.401718 | 0.159101 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.286063  0.286063  0.286063 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.000117   0.000117   0.000117 |  0.000117   0.000117   0.000117 |   0.0 | 0.0 |  0.000117 | 0.0 |   1.8144 |  0.31663 | 0.0 | 0.0 | 0.0 | 0.0 |  0.008232 |  1.107468 | 0.128179 | -0.330248  -0.330248  -0.330248 | 63.765577  63.765577  63.765577 | nan | nan |  0.004116 |  373.617276 | 0.128179 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 | 0.756  0.756  0.756 | 1.3608 |       0.0        0.0        0.0 | 0.109167  0.109167  0.109167 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.001244   0.001244   0.001244 |  0.001244   0.001244   0.001244 |   0.0 | 0.0 |  0.001244 | 0.0 |   1.3608 | 0.134199 | 0.0 | 0.0 | 0.0 | 0.0 |  0.005903 |  0.901366 | 0.104325 | -0.439415  -0.439415  -0.439415 | 63.764333  63.764333  63.764333 | nan | nan | -0.000542 |  327.477421 | 0.104325 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 | 0.819  0.819  0.819 | 1.4742 |       0.0        0.0        0.0 | 0.080905  0.080905  0.080905 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.018374   0.018374   0.018374 |  0.018374   0.018374   0.018374 |   0.0 | 0.0 |  0.018374 | 0.0 |   1.4742 | 0.108771 | 0.0 | 0.0 | 0.0 | 0.0 |  0.011888 |  0.743612 | 0.086066 |  -0.52032   -0.52032   -0.52032 | 63.745958  63.745958  63.745958 | nan | nan |  0.005944 |  289.405125 | 0.086066 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008 | 1.8144 |       0.0        0.0        0.0 | 0.072233  0.072233  0.072233 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |      0.05       0.05       0.05 |      0.05       0.05       0.05 |   0.0 | 0.0 |      0.05 | 0.0 |   1.8144 | 0.107076 | 0.0 | 0.0 | 0.0 | 0.0 |  0.032214 |  0.622194 | 0.072013 | -0.592552  -0.592552  -0.592552 | 63.695958  63.695958  63.695958 | nan | nan |   0.02373 |  258.059532 | 0.072013 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 1.9 | 0.0 | 0.0 |   0.0 | 1.197  1.197  1.197 | 2.1546 |       0.0        0.0        0.0 | 0.064013  0.064013  0.064013 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |   1.95975    1.95975    1.95975 |   1.95975    1.95975    1.95975 |   0.0 | 0.0 |   1.95975 | 0.0 |   2.1546 | 0.105825 | 0.0 | 0.0 | 0.0 | 0.0 |  1.132357 |  0.590953 | 0.068397 | -0.656565  -0.656565  -0.656565 | 61.736209  61.736209  61.736209 | nan | nan |  0.851123 |  281.842756 | 0.068397 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.8 | 0.0 | 0.0 |   1.2 | 0.504  0.504  0.504 | 0.9072 |  0.000144   0.000144   0.000144 | 0.204166  0.204166  0.204166 | 0.000108  0.000108  0.000108 |  0.000036   0.000036   0.000036 |      5.05       5.05       5.05 |      5.05       5.05       5.05 |   1.2 | 0.0 |  5.050108 | 0.0 |   0.9072 | 0.218227 | 0.0 | 0.0 | 0.0 | 0.0 |  3.602778 |  0.866616 | 0.100303 |  0.339124   0.339124   0.339124 | 56.686245  56.686245  56.686245 | nan | nan |  2.298453 |  415.340864 | 0.100303 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 1.5 | 0.0 | 0.0 |  0.24 | 0.945  0.945  0.945 |  1.701 |   0.00005    0.00005    0.00005 | 0.601868  0.601868  0.601868 |  0.00005   0.00005   0.00005 |       0.0        0.0        0.0 | 22.000001  22.000001  22.000001 | 22.000001  22.000001  22.000001 |  0.24 | 0.0 | 22.000051 | 0.0 |    1.701 |  0.62385 | 0.0 | 0.0 | 0.0 | 0.0 | 14.476228 |  2.422961 | 0.280435 | -0.022794  -0.022794  -0.022794 | 34.686244  34.686244  34.686244 | nan | nan |  9.822275 | 1002.066976 | 0.280435 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 2.7 | 0.0 | 0.0 |  0.16 | 1.701  1.701  1.701 | 3.0618 |  0.000007   0.000007   0.000007 | 0.470491  0.470491  0.470491 | 0.000007  0.000007  0.000007 |       0.0        0.0        0.0 |      35.0       35.0       35.0 | 34.731609  34.731609  34.731609 |  0.16 | 0.0 | 34.731616 | 0.0 |   3.0618 | 0.522317 | 0.0 | 0.0 | 0.0 | 0.0 | 28.356183 |  7.950003 | 0.920139 | -0.333291  -0.333291  -0.333291 | -0.045365  -0.045365  -0.045365 | nan | nan | 16.197708 | 1991.118031 | 0.920139 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 1.5 | 0.0 | 0.0 |   3.6 | 0.945  0.945  0.945 |  1.728 |  0.413728   0.413728   0.413728 | 0.831573  0.831573  0.831573 | 0.413728  0.413728  0.413728 |       0.0        0.0        0.0 | 19.000004  19.000004  19.000004 |  0.919718   0.919718   0.919718 |   3.6 | 0.0 |  1.333446 | 0.0 |    1.728 | 0.849501 | 0.0 | 0.0 | 0.0 | 0.0 | 14.790422 | 13.481342 |  1.56034 |  2.021409   2.021409   2.021409 | -0.965084  -0.965084  -0.965084 | nan | nan |  2.740732 | 2043.653612 |  1.56034 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008 | 1.8432 |       0.0        0.0        0.0 | 1.005972  1.005972  1.005972 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  7.013137   7.013137   7.013137 |  0.069834   0.069834   0.069834 |   0.0 | 0.0 |  0.069834 | 0.0 |   1.8432 | 1.022717 | 0.0 | 0.0 | 0.0 | 0.0 |  2.410784 | 10.827169 | 1.253145 |  1.015436   1.015436   1.015436 | -1.034918  -1.034918  -1.034918 | nan | nan |  0.399782 | 1618.580384 | 1.253145 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |  1.26   1.26   1.26 |  2.304 |       0.0        0.0        0.0 | 1.029652  1.029652  1.029652 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  1.298601   1.298601   1.298601 |  0.010807   0.010807   0.010807 |   0.0 | 0.0 |  0.010807 | 0.0 |    2.304 | 1.055139 | 0.0 | 0.0 | 0.0 | 0.0 |  0.351386 |  7.464324 | 0.863926 | -0.014216  -0.014216  -0.014216 | -1.045725  -1.045725  -1.045725 | nan | nan |  0.059203 | 1260.278116 | 0.863926 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 2.1 | 0.0 | 0.0 |   0.0 | 1.323  1.323  1.323 | 2.4192 |       0.0        0.0        0.0 |  0.35708   0.35708   0.35708 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.798601   0.798601   0.798601 |  0.006391   0.006391   0.006391 |   0.0 | 0.0 |  0.006391 | 0.0 |   2.4192 | 0.398323 | 0.0 | 0.0 | 0.0 | 0.0 |  0.053701 |  5.194291 | 0.601191 | -0.371296  -0.371296  -0.371296 | -1.052115  -1.052115  -1.052115 | nan | nan |  0.011893 | 1000.775704 | 0.601191 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 1.7 | 0.0 | 0.0 |  0.32 | 1.071  1.071  1.071 | 1.9584 |  0.000008   0.000008   0.000008 | 0.210309  0.210309  0.210309 | 0.000003  0.000003  0.000003 |  0.000005   0.000005   0.000005 |  0.251908   0.251908   0.251908 |  0.001978   0.001978   0.001978 |  0.32 | 0.0 |  0.001981 | 0.0 |   1.9584 |  0.24527 | 0.0 | 0.0 | 0.0 | 0.0 |  0.009252 |  3.736082 | 0.432417 | -0.261613  -0.261613  -0.261613 | -1.054088  -1.054088  -1.054088 | nan | nan |  0.004622 |   812.78656 | 0.432417 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 1.7 | 0.0 | 0.0 |   0.0 | 1.071  1.071  1.071 | 1.9584 |       0.0        0.0        0.0 | 0.175292  0.175292  0.175292 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  0.096611   0.096611   0.096611 |  0.000754   0.000754   0.000754 |   0.0 | 0.0 |  0.000754 | 0.0 |   1.9584 | 0.210954 | 0.0 | 0.0 | 0.0 | 0.0 |  0.003584 |  2.771673 | 0.320795 | -0.436905  -0.436905  -0.436905 | -1.054842  -1.054842  -1.054842 | nan | nan |  0.001792 |  672.420148 | 0.320795 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 | 0.504  0.504  0.504 | 0.9216 |       0.0        0.0        0.0 | 0.053397  0.053397  0.053397 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  1.298601   1.298601   1.298601 |  0.009889   0.009889   0.009889 |   0.0 | 0.0 |  0.009889 | 0.0 |   0.9216 | 0.070761 | 0.0 | 0.0 | 0.0 | 0.0 |  0.007862 |  2.112451 | 0.244497 | -0.490302  -0.490302  -0.490302 | -1.064732  -1.064732  -1.064732 | nan | nan |  0.003819 |  566.261243 | 0.244497 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 | 0.819  0.819  0.819 | 1.4976 |       0.0        0.0        0.0 | 0.067384  0.067384  0.067384 |      0.0       0.0       0.0 |       0.0        0.0        0.0 |  3.184058   3.184058   3.184058 |  0.022513   0.022513   0.022513 |   0.0 | 0.0 |  0.022513 | 0.0 |   1.4976 | 0.095989 | 0.0 | 0.0 | 0.0 | 0.0 |  0.017946 |  1.648182 | 0.190762 | -0.557687  -0.557687  -0.557687 | -1.087244  -1.087244  -1.087244 | nan | nan |  0.008385 |  483.233915 | 0.190762 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 2.5 | 0.0 | 0.0 |   0.0 | 1.575  1.575  1.575 |   2.88 |       0.0        0.0        0.0 | 0.091899  0.091899  0.091899 |      0.0       0.0       0.0 |       0.0        0.0        0.0 | 12.500323  12.500323  12.500323 |  0.071094   0.071094   0.071094 |   0.0 | 0.0 |  0.071094 | 0.0 |     2.88 | 0.147661 | 0.0 | 0.0 | 0.0 | 0.0 |  0.049923 |  1.310392 | 0.151666 | -0.649586  -0.649586  -0.649586 | -1.158338  -1.158338  -1.158338 | nan | nan |  0.029557 |  417.280525 | 0.151666 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_v002_no_land_area:

No land area
____________

>>> nu(0)
>>> al(0.0)
>>> as_(10.0)
>>> test.reset_inits()
>>> conditions = sequences.conditions

.. integration-test::

    >>> test("wland_v002_no_land_area",
    ...      axis1=(fluxes.pc, fluxes.et, fluxes.rh),
    ...      axis2=(states.hs,))
    |                date |     t |    p | pet | fxg | fxs |    pc | petl |    pes | tf | ei | rf | sf | pm | am |    ps |  pv |  pq | etv |       es |       et | fxs | fxg | cdg | fgs | fqs |       rh |        r | ic | sp |  dv |  dg |  hq |        hs |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.6 | 0.0 | 0.0 |   0.0 |    - | 0.6588 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 | 0.000067 | 0.000067 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -2.000067 |      0.0 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 |    - | 0.8784 |  - |  - |  - |  - |  - |  - |  0.32 | 0.0 | 0.0 | 0.0 |  0.00024 |  0.00024 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -1.680307 |      0.0 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 |    - | 0.7686 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |  0.00034 |  0.00034 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -1.680647 |      0.0 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 |    - | 0.4392 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 | 0.000194 | 0.000194 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -1.680841 |      0.0 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 |    - | 0.4392 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 | 0.000194 | 0.000194 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -1.681035 |      0.0 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.4 | 0.0 | 0.0 |   0.0 |    - | 0.4392 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 | 0.000194 | 0.000194 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -1.681229 |      0.0 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.4 | 0.0 | 0.0 |  0.16 |    - | 0.4392 |  - |  - |  - |  - |  - |  - |  0.16 | 0.0 | 0.0 | 0.0 | 0.000299 | 0.000299 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |      0.0 |  - |  - | nan | nan | 0.0 | -1.521528 |      0.0 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.3 | 0.0 | 0.0 |   3.6 |    - | 0.3294 |  - |  - |  - |  - |  - |  - |   3.6 | 0.0 | 0.0 | 0.0 | 0.188582 | 0.188582 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.000088 |  0.00001 |  - |  - | nan | nan | 0.0 |  1.889802 |  0.00001 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.3 | 0.0 | 0.0 |   0.0 |    - | 0.3294 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 | 0.329245 | 0.329245 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.000313 | 0.000036 |  - |  - | nan | nan | 0.0 |  1.560243 | 0.000036 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.4 | 0.0 | 0.0 |  2.56 |    - | 0.4392 |  - |  - |  - |  - |  - |  - |  2.56 | 0.0 | 0.0 | 0.0 | 0.439031 | 0.439031 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.000621 | 0.000072 |  - |  - | nan | nan | 0.0 |  3.680592 | 0.000072 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.3 | 0.0 | 0.0 |  3.68 |    - | 0.3294 |  - |  - |  - |  - |  - |  - |  3.68 | 0.0 | 0.0 | 0.0 |   0.3294 |   0.3294 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.00177 | 0.000205 |  - |  - | nan | nan | 0.0 |  7.029422 | 0.000205 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.6 | 0.0 | 0.0 |  1.84 |    - | 0.6588 |  - |  - |  - |  - |  - |  - |  1.84 | 0.0 | 0.0 | 0.0 |   0.6588 |   0.6588 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.002902 | 0.000336 |  - |  - | nan | nan | 0.0 |   8.20772 | 0.000336 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.8 | 0.0 | 0.0 |  14.4 |    - | 0.8784 |  - |  - |  - |  - |  - |  - |  14.4 | 0.0 | 0.0 | 0.0 |   0.8784 |   0.8784 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.008592 | 0.000994 |  - |  - | nan | nan | 0.0 | 21.720728 | 0.000994 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.5 | 0.0 | 0.0 | 15.36 |    - |  0.549 |  - |  - |  - |  - |  - |  - | 15.36 | 0.0 | 0.0 | 0.0 |    0.549 |    0.549 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.022164 | 0.002565 |  - |  - | nan | nan | 0.0 | 36.509564 | 0.002565 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.8 | 0.0 | 0.0 |  0.32 |    - | 0.8784 |  - |  - |  - |  - |  - |  - |  0.32 | 0.0 | 0.0 | 0.0 |   0.8784 |   0.8784 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.030012 | 0.003474 |  - |  - | nan | nan | 0.0 | 35.921151 | 0.003474 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.5 | 0.0 | 0.0 |  6.64 |    - |  0.549 |  - |  - |  - |  - |  - |  - |  6.64 | 0.0 | 0.0 | 0.0 |    0.549 |    0.549 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.033553 | 0.003883 |  - |  - | nan | nan | 0.0 | 41.978598 | 0.003883 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.4 | 0.0 | 0.0 |  4.24 |    - | 0.4392 |  - |  - |  - |  - |  - |  - |  4.24 | 0.0 | 0.0 | 0.0 |   0.4392 |   0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.040028 | 0.004633 |  - |  - | nan | nan | 0.0 | 45.739371 | 0.004633 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 1.3 | 0.0 | 0.0 |  0.56 |    - | 1.4274 |  - |  - |  - |  - |  - |  - |  0.56 | 0.0 | 0.0 | 0.0 |   1.4274 |   1.4274 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.041965 | 0.004857 |  - |  - | nan | nan | 0.0 | 44.830005 | 0.004857 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.9 | 0.0 | 0.0 |  2.16 |    - | 0.9882 |  - |  - |  - |  - |  - |  - |  2.16 | 0.0 | 0.0 | 0.0 |   0.9882 |   0.9882 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.042121 | 0.004875 |  - |  - | nan | nan | 0.0 | 45.959685 | 0.004875 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.7 | 0.0 | 0.0 |  1.28 |    - | 0.7938 |  - |  - |  - |  - |  - |  - |  1.28 | 0.0 | 0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.043217 | 0.005002 |  - |  - | nan | nan | 0.0 | 46.402668 | 0.005002 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.7 | 0.0 | 0.0 |   2.0 |    - | 0.7938 |  - |  - |  - |  - |  - |  - |   2.0 | 0.0 | 0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.044351 | 0.005133 |  - |  - | nan | nan | 0.0 | 47.564517 | 0.005133 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 1.1 | 0.0 | 0.0 |  0.48 |    - | 1.2474 |  - |  - |  - |  - |  - |  - |  0.48 | 0.0 | 0.0 | 0.0 |   1.2474 |   1.2474 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.044597 | 0.005162 |  - |  - | nan | nan | 0.0 |  46.75252 | 0.005162 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 1.0 | 0.0 | 0.0 |  0.16 |    - |  1.134 |  - |  - |  - |  - |  - |  - |  0.16 | 0.0 | 0.0 | 0.0 |    1.134 |    1.134 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.043306 | 0.005012 |  - |  - | nan | nan | 0.0 | 45.735214 | 0.005012 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.8 | 0.0 | 0.0 |  1.36 |    - | 0.9072 |  - |  - |  - |  - |  - |  - |  1.36 | 0.0 | 0.0 | 0.0 |   0.9072 |   0.9072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.042879 | 0.004963 |  - |  - | nan | nan | 0.0 | 46.145135 | 0.004963 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.6 | 0.0 | 0.0 |  0.24 |    - | 0.6804 |  - |  - |  - |  - |  - |  - |  0.24 | 0.0 | 0.0 | 0.0 |   0.6804 |   0.6804 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.042828 | 0.004957 |  - |  - | nan | nan | 0.0 | 45.661907 | 0.004957 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.7 | 0.0 | 0.0 |   0.0 |    - | 0.7938 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.041909 | 0.004851 |  - |  - | nan | nan | 0.0 | 44.826198 | 0.004851 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.7 | 0.0 | 0.0 |  1.44 |    - | 0.7938 |  - |  - |  - |  - |  - |  - |  1.44 | 0.0 | 0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.041748 | 0.004832 |  - |  - | nan | nan | 0.0 |  45.43065 | 0.004832 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.5 | 0.0 | 0.0 |  7.12 |    - |  0.567 |  - |  - |  - |  - |  - |  - |  7.12 | 0.0 | 0.0 | 0.0 |    0.567 |    0.567 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.046858 | 0.005423 |  - |  - | nan | nan | 0.0 | 51.936792 | 0.005423 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 |    - | 0.9072 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   0.9072 |   0.9072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.050832 | 0.005883 |  - |  - | nan | nan | 0.0 | 50.978759 | 0.005883 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 1.0 | 0.0 | 0.0 |   0.0 |    - |  1.134 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |    1.134 |    1.134 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.049255 | 0.005701 |  - |  - | nan | nan | 0.0 | 49.795504 | 0.005701 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 |    - | 1.3608 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.3608 |   1.3608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.047368 | 0.005482 |  - |  - | nan | nan | 0.0 | 48.387336 | 0.005482 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.9 | 0.0 | 0.0 |  0.72 |    - | 1.0206 |  - |  - |  - |  - |  - |  - |  0.72 | 0.0 | 0.0 | 0.0 |   1.0206 |   1.0206 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.046101 | 0.005336 |  - |  - | nan | nan | 0.0 | 48.040635 | 0.005336 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.9 | 0.0 | 0.0 |  0.08 |    - | 1.0206 |  - |  - |  - |  - |  - |  - |  0.08 | 0.0 | 0.0 | 0.0 |   1.0206 |   1.0206 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.04515 | 0.005226 |  - |  - | nan | nan | 0.0 | 47.054884 | 0.005226 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 |    - | 1.3608 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.3608 |   1.3608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.043461 |  0.00503 |  - |  - | nan | nan | 0.0 | 45.650624 |  0.00503 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 1.4 | 0.0 | 0.0 |   0.0 |    - | 1.5876 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.5876 |   1.5876 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.041347 | 0.004786 |  - |  - | nan | nan | 0.0 | 44.021677 | 0.004786 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 1.1 | 0.0 | 0.0 |  3.12 |    - | 1.2474 |  - |  - |  - |  - |  - |  - |  3.12 | 0.0 | 0.0 | 0.0 |   1.2474 |   1.2474 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.04149 | 0.004802 |  - |  - | nan | nan | 0.0 | 45.852787 | 0.004802 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 1.1 | 0.0 | 0.0 |  6.96 |    - | 1.2474 |  - |  - |  - |  - |  - |  - |  6.96 | 0.0 | 0.0 | 0.0 |   1.2474 |   1.2474 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.046842 | 0.005422 |  - |  - | nan | nan | 0.0 | 51.518546 | 0.005422 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.5 | 0.0 | 0.0 | 21.12 |    - |  0.567 |  - |  - |  - |  - |  - |  - | 21.12 | 0.0 | 0.0 | 0.0 |    0.567 |    0.567 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.067073 | 0.007763 |  - |  - | nan | nan | 0.0 | 72.004472 | 0.007763 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.6 | 0.0 | 0.0 |   9.2 |    - | 0.6804 |  - |  - |  - |  - |  - |  - |   9.2 | 0.0 | 0.0 | 0.0 |   0.6804 |   0.6804 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.091743 | 0.010618 |  - |  - | nan | nan | 0.0 | 80.432329 | 0.010618 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 1.5 | 0.0 | 0.0 |  0.72 |    - |  1.701 |  - |  - |  - |  - |  - |  - |  0.72 | 0.0 | 0.0 | 0.0 |    1.701 |    1.701 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.098337 | 0.011382 |  - |  - | nan | nan | 0.0 | 79.352992 | 0.011382 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |    - |  2.268 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |    2.268 |    2.268 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.095182 | 0.011016 |  - |  - | nan | nan | 0.0 |  76.98981 | 0.011016 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 |    - | 1.8144 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.8144 |   1.8144 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.091308 | 0.010568 |  - |  - | nan | nan | 0.0 | 75.084102 | 0.010568 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 |    - | 1.8144 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.8144 |   1.8144 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0879 | 0.010174 |  - |  - | nan | nan | 0.0 | 73.181802 | 0.010174 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 1.2 | 0.0 | 0.0 |   0.0 |    - | 1.3608 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.3608 |   1.3608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.084938 | 0.009831 |  - |  - | nan | nan | 0.0 | 71.736064 | 0.009831 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 |    - | 1.4742 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.4742 |   1.4742 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.082313 | 0.009527 |  - |  - | nan | nan | 0.0 | 70.179552 | 0.009527 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 |    - | 1.8144 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.8144 |   1.8144 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.079331 | 0.009182 |  - |  - | nan | nan | 0.0 | 68.285821 | 0.009182 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 1.9 | 0.0 | 0.0 |   0.0 |    - | 2.1546 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   2.1546 |   2.1546 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.075815 | 0.008775 |  - |  - | nan | nan | 0.0 | 66.055405 | 0.008775 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.8 | 0.0 | 0.0 |   1.2 |    - | 0.9072 |  - |  - |  - |  - |  - |  - |   1.2 | 0.0 | 0.0 | 0.0 |   0.9072 |   0.9072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.074113 | 0.008578 |  - |  - | nan | nan | 0.0 | 66.274092 | 0.008578 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 1.5 | 0.0 | 0.0 |  0.24 |    - |  1.701 |  - |  - |  - |  - |  - |  - |  0.24 | 0.0 | 0.0 | 0.0 |    1.701 |    1.701 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.073013 | 0.008451 |  - |  - | nan | nan | 0.0 | 64.740079 | 0.008451 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 2.7 | 0.0 | 0.0 |  0.16 |    - | 3.0618 |  - |  - |  - |  - |  - |  - |  0.16 | 0.0 | 0.0 | 0.0 |   3.0618 |   3.0618 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.06929 |  0.00802 |  - |  - | nan | nan | 0.0 |  61.76899 |  0.00802 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 1.5 | 0.0 | 0.0 |   3.6 |    - |  1.728 |  - |  - |  - |  - |  - |  - |   3.6 | 0.0 | 0.0 | 0.0 |    1.728 |    1.728 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.068327 | 0.007908 |  - |  - | nan | nan | 0.0 | 63.572662 | 0.007908 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 1.6 | 0.0 | 0.0 |   0.0 |    - | 1.8432 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.8432 |   1.8432 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.068237 | 0.007898 |  - |  - | nan | nan | 0.0 | 61.661225 | 0.007898 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 2.0 | 0.0 | 0.0 |   0.0 |    - |  2.304 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |    2.304 |    2.304 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.064772 | 0.007497 |  - |  - | nan | nan | 0.0 | 59.292453 | 0.007497 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 2.1 | 0.0 | 0.0 |   0.0 |    - | 2.4192 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   2.4192 |   2.4192 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.060918 | 0.007051 |  - |  - | nan | nan | 0.0 | 56.812335 | 0.007051 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 1.7 | 0.0 | 0.0 |  0.32 |    - | 1.9584 |  - |  - |  - |  - |  - |  - |  0.32 | 0.0 | 0.0 | 0.0 |   1.9584 |   1.9584 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.057657 | 0.006673 |  - |  - | nan | nan | 0.0 | 55.116278 | 0.006673 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 1.7 | 0.0 | 0.0 |   0.0 |    - | 1.9584 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.9584 |   1.9584 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.054817 | 0.006345 |  - |  - | nan | nan | 0.0 | 53.103062 | 0.006345 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.8 | 0.0 | 0.0 |   0.0 |    - | 0.9216 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   0.9216 |   0.9216 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.052558 | 0.006083 |  - |  - | nan | nan | 0.0 | 52.128904 | 0.006083 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 1.3 | 0.0 | 0.0 |   0.0 |    - | 1.4976 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |   1.4976 |   1.4976 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.050682 | 0.005866 |  - |  - | nan | nan | 0.0 | 50.580622 | 0.005866 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 2.5 | 0.0 | 0.0 |   0.0 |    - |   2.88 |  - |  - |  - |  - |  - |  - |   0.0 | 0.0 | 0.0 | 0.0 |     2.88 |     2.88 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.047417 | 0.005488 |  - |  - | nan | nan | 0.0 | 47.653205 | 0.005488 |

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from standard library
from typing import *

# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.core.typingtools import *

# ...from lland
from hydpy.models.wland import wland_model
from hydpy.models.wland import wland_solver
from hydpy.models.wland.wland_constants import *


class Model(modeltools.ELSModel):
    """The *HydPy-W-Land* model."""

    SOLVERPARAMETERS = (
        wland_solver.AbsErrorMax,
        wland_solver.RelErrorMax,
        wland_solver.RelDTMin,
        wland_solver.RelDTMax,
    )
    SOLVERSEQUENCES = ()
    INLET_METHODS = (
        wland_model.Calc_FR_V1,
        wland_model.Calc_PM_V1,
    )
    RECEIVER_METHODS = ()
    ADD_METHODS = (
        wland_model.Return_DVH_V2,
        wland_model.Return_ErrorDV_V1,
    )
    PART_ODE_METHODS = (
        wland_model.Calc_FXS_V1,
        wland_model.Calc_FXG_V1,
        wland_model.Calc_PC_V1,
        wland_model.Calc_PETL_V1,
        wland_model.Calc_PES_V1,
        wland_model.Calc_TF_V1,
        wland_model.Calc_EI_V1,
        wland_model.Calc_RF_V1,
        wland_model.Calc_SF_V1,
        wland_model.Calc_AM_V1,
        wland_model.Calc_PS_V1,
        wland_model.Calc_W_V1,
        wland_model.Calc_PV_V1,
        wland_model.Calc_PQ_V1,
        wland_model.Calc_Beta_V1,
        wland_model.Calc_ETV_V1,
        wland_model.Calc_ES_V1,
        wland_model.Calc_FQS_V1,
        wland_model.Calc_FGS_V1,
        wland_model.Calc_RH_V1,
        wland_model.Calc_DVEq_V3,
        wland_model.Calc_DGEq_V1,
        wland_model.Calc_GF_V1,
        wland_model.Calc_CDG_V2,
    )
    FULL_ODE_METHODS = (
        wland_model.Update_IC_V1,
        wland_model.Update_SP_V1,
        wland_model.Update_DV_V1,
        wland_model.Update_DG_V1,
        wland_model.Update_HQ_V1,
        wland_model.Update_HS_V1,
    )
    OUTLET_METHODS = (
        wland_model.Calc_ET_V1,
        wland_model.Calc_R_V1,
        wland_model.Pass_R_V1,
    )
    SENDER_METHODS = ()
    SUBMODELS = (wland_model.PegasusDGEq,)

    def check_waterbalance(
        self,
        initial_conditions: Dict[str, Dict[str, ArrayFloat]],
    ) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

        :math:`\sum_{t=t0}^{t1} \big( PC_t + FXG_t + FXS_t - ET_t - RH_t \big)
        + ASR \cdot \big( HS_{t0} - HS_{t1}\big)
        + ALR \cdot \left( \big( HQ_{t0} - HQ_{t1} \big) -
        AGR \cdot \big( DV_{t0} - DV_{t1} \big) +
        \sum_{k=1}^{NU} AUR^k \cdot \big( IC_{t0}^k -
        IC_{t1}^k + SP_{t0}^k - SP_{t1}^k \big) \right)`

        The returned error should always be in scale with numerical precision so
        that it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run
        via property |Sequences.conditions|.  See the integration tests of the
        application model |wland_v001| for some examples.
        """
        control = self.parameters.control
        derived = self.parameters.derived
        inputs = self.sequences.inputs
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["states"]
        ddv = (last.dv - first["dv"]) * derived.alr * derived.agr
        if numpy.isnan(ddv):
            ddv = 0.0
        return (
            sum(fluxes.pc.series)
            + sum(inputs.fxg.series)
            + sum(inputs.fxs.series)
            - sum(fluxes.et.series)
            - sum(fluxes.rh.series)
            - sum((last.ic - first["ic"]) * control.aur) * derived.alr
            - sum((last.sp - first["sp"]) * control.aur) * derived.alr
            - (last.hq - first["hq"]) * derived.alr
            - (last.hs - first["hs"]) * derived.asr
            + ddv
        )


tester = Tester()
cythonizer = Cythonizer()
