""":py:class:`~pulse2percept.datasets.load_perezfornos2012`"""
from os.path import dirname, join
import numpy as np

try:
    import pandas as pd
    has_pandas = True
except ImportError:
    has_pandas = False


def load_perezfornos2012(shuffle=False, subjects=None, figures=None,
                         random_state=0):
    """Load data from [PerezFornos2012]_

    Load the brightness associated with joystick position data described in
    [PerezFornos2012]_.
    Datapoints were extracted from Figures 3-7 of the paper.


    ===================   =====================
    Retinal implants:                  Argus II
    Subjects:                                 9
    Number of samples:                       45
    Number of features:                     158
    ===================   =====================

    The dataset includes the following features:

    ====================  ================================================
    Subject               Subject ID
    Figure                Reference figure from [PerezFornos2012]_
    time_series           Numpy array of the time series data associated 
                          with the figure.
                          Note: This was generated by linear interpolation 
                          from [-2.0, 75.5] in steps of 0.5
    ====================  ================================================

    .. versionadded:: 0.8

    Parameters
    ----------
    shuffle : boolean, optional
        If True, the rows of the DataFrame are shuffled.
    random_state : int | numpy.random.RandomState | None, optional
        Determines random number generation for dataset shuffling. Pass an int
        for reproducible output across multiple function calls.
    subjects : str | list of strings | None, optional
        Select data from a single subject or a list of subjects.
        By default, all subjects are selected.
    figures : str | list of strings | None, optional
        Select data from a single figure or a list of figures.
        By default, all figures are selected

    Returns
    -------
    data: pd.DataFrame
        The whole dataset is returned in a 45x158 Pandas DataFrame

    """
    if not has_pandas:
        raise ImportError("You do not have pandas installed. "
                          "You can install it via $ pip install pandas.")

    # Load data from CSV:
    module_path = dirname(__file__)
    file_path = join(module_path, 'data', 'perez-fornos-2012.csv')
    df = pd.read_csv(file_path)

    if subjects is not None:
        if isinstance(subjects, str):
            subjects = [subjects]
        df = df[df['Subject'].isin(subjects)]

    if figures is not None:
        if isinstance(figures, str):
            figures = [figures]
        df = df[df['Figure'].isin(figures)]

    if shuffle:
        df = df.sample(n=len(df), random_state=random_state)

    time_series_df = pd.DataFrame()
    time_steps = [str(time_step)
                  for time_step in np.arange(start=-2.0, stop=75.5, step=0.5)]

    for index, row in df.iterrows():
        time_series = np.array([])

        for time_step in time_steps:
            time_series = np.append(time_series, row[time_step])

        raw_spec = pd.DataFrame([{
            'figure': row['Figure'],
            'subject': row['Subject'],
            'time_series': time_series
        }])
        time_series_df = pd.concat([time_series_df, raw_spec], 
                                   verify_integrity=False, ignore_index=True)

    return time_series_df.reset_index(drop=True)
