"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === 'token') {
            return first.token;
        }
        return undefined;
    }
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    get length() {
        return this.fragments.length;
    }
    addLiteral(lit) {
        this.fragments.push({ type: 'literal', lit });
    }
    addToken(token) {
        this.fragments.push({ type: 'token', token });
    }
    addIntrinsic(value) {
        this.fragments.push({ type: 'intrinsic', value });
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === 'token') {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case 'literal':
                    ret.addLiteral(f.lit);
                    break;
                case 'token':
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case 'intrinsic':
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case 'literal': return fragment.lit;
        case 'token': return fragment.token.toString();
        case 'intrinsic': return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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