"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const app_1 = require("../lib/app");
function withApp(props, block) {
    const app = new app_1.App(Object.assign({ runtimeInfo: false, stackTraces: false }, props));
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        r2.node.addWarning('warning1');
        r2.node.addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStack(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.name, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, { Resources: { s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } } } });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.name, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, { Resources: { s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' } } });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context passed through CDK_CONTEXT has precedence'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App({
            context: {
                key1: 'val3',
                key2: 'val4'
            }
        });
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: "DummyResource",
                    Properties: {
                        Prog2: "Prog2"
                    }
                },
                s1c2r1D1791C01: {
                    Type: "ResourceType1"
                },
                s1c2r25F685FFF: {
                    Type: "ResourceType2",
                    Properties: {
                        FromContext: "HELLO"
                    }
                }
            }
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App({
            context: {
                foo: 'bar'
            }
        });
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: 'fake',
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: 'fake2',
                    props: {
                        foo: 'bar',
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: "missing-context-key",
                provider: 'fake',
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: "missing-context-key-2",
                provider: 'fake2',
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                    foo: 'bar',
                },
            }
        ]);
        test.done();
    },
    'runtime library versions disabled'(test) {
        const context = {};
        context[cxapi.DISABLE_VERSION_REPORTING] = true;
        const assembly = withApp(context, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        test.deepEqual(libs['jsii-runtime'], `Java/1.2.3.4`);
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        const version = require('../package.json').version;
        test.deepEqual(libs, {
            '@aws-cdk/core': version,
            '@aws-cdk/cx-api': version,
            'jsii-runtime': `node.js/${process.version}`
        });
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.name, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5hcHAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmFwcC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEseUNBQTBDO0FBRTFDLGdDQUFtRTtBQUNuRSxvQ0FBMkM7QUFFM0MsU0FBUyxPQUFPLENBQUMsS0FBZSxFQUFFLEtBQXlCO0lBQ3pELE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxpQkFDakIsV0FBVyxFQUFFLEtBQUssRUFDbEIsV0FBVyxFQUFFLEtBQUssSUFDZixLQUFLLEVBQ1IsQ0FBQztJQUVILEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUVYLE9BQU8sR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0FBQ3JCLENBQUM7QUFFRCxTQUFTLEtBQUssQ0FBQyxPQUFnQztJQUM3QyxPQUFPLE9BQU8sQ0FBQyxFQUFFLE9BQU8sRUFBRSxFQUFFLEdBQUcsQ0FBQyxFQUFFO1FBQ2hDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDNUYsSUFBSSxpQkFBVyxDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDM0YsTUFBTSxFQUFFLEdBQUcsSUFBSSxpQkFBVyxDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFaEcsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzNGLE1BQU0sRUFBRSxHQUFHLElBQUksV0FBVyxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUzQyxvQkFBb0I7UUFDcEIsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3JDLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQy9CLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQy9CLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFVBQVUsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLDhCQUE4QjtJQUN2RSxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRCxTQUFTLFVBQVUsQ0FBQyxJQUFZLEVBQUUsa0JBQTJCLEtBQUssRUFBRSxPQUFhO0lBQy9FLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNoQyxNQUFNLEtBQUssR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBRXRDLElBQUksQ0FBQyxlQUFlLEVBQUU7UUFDcEIsT0FBUSxLQUFhLENBQUMsUUFBUSxDQUFDO0tBQ2hDO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBdVJELE1BQU0sV0FBWSxTQUFRLGVBQVM7SUFDakMsWUFBWSxLQUFnQixFQUFFLEVBQVU7UUFDdEMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLGlCQUFXLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZELElBQUksaUJBQVcsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDdkgsQ0FBQztDQUNGO0FBNVJELGlCQUFTO0lBQ1AscURBQXFELENBQUMsSUFBVTtRQUM5RCxNQUFNLFFBQVEsR0FBRyxLQUFLLEVBQUUsQ0FBQztRQUN6QixPQUFRLFFBQWdCLENBQUMsR0FBRyxDQUFDO1FBRTdCLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFMUMsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztRQUNqRSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQ3pDLEVBQUUsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUU7Z0JBQy9ELElBQUksRUFBRSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDbkUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRTtZQUN2QyxTQUFTLEVBQUUsQ0FBQyxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDO1lBQ3hDLGNBQWMsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsQ0FBQztZQUM3RCxjQUFjLEVBQ1osQ0FBQyxFQUFFLElBQUksRUFBRSxtQkFBbUIsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFO2dCQUM1QyxFQUFFLElBQUksRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFO2dCQUM3QyxFQUFFLElBQUksRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLENBQUM7U0FDakQsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hGLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFDekMsRUFBRSxJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtnQkFDL0QsY0FBYyxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDekMsY0FBYyxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUU7WUFDdkMsY0FBYyxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxDQUFDO1lBQzdELGNBQWMsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQztZQUMxRCxpQkFBaUIsRUFDZixDQUFDLEVBQUUsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxDQUFDO1lBQ3pELGlCQUFpQixFQUNmLENBQUMsRUFBRSxJQUFJLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUM7U0FDMUQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2hCLENBQUM7SUFFRCwyQ0FBMkMsQ0FBQyxJQUFVO1FBQ3BELE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDOUMsSUFBSSxFQUFFLE1BQU07WUFDWixJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdkIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtREFBbUQsQ0FBQyxJQUFVO1FBQzVELE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDOUMsSUFBSSxFQUFFLE1BQU07WUFDWixJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksU0FBRyxDQUFDO1lBQ25CLE9BQU8sRUFBRTtnQkFDUCxJQUFJLEVBQUUsTUFBTTtnQkFDWixJQUFJLEVBQUUsTUFBTTthQUNiO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1FBQzVFLE1BQU0sTUFBTSxHQUFHLFVBQVUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxFQUFFLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFFOUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFO1lBQzlCLFNBQVMsRUFBRTtnQkFDVCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLFVBQVUsRUFBRTt3QkFDVixLQUFLLEVBQUUsT0FBTztxQkFDZjtpQkFDRjtnQkFDRCxjQUFjLEVBQUU7b0JBQ2QsSUFBSSxFQUFFLGVBQWU7aUJBQ3RCO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxJQUFJLEVBQUUsZUFBZTtvQkFDckIsVUFBVSxFQUFFO3dCQUNWLFdBQVcsRUFBRSxPQUFPO3FCQUNyQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZEQUE2RCxDQUFDLElBQVU7UUFDdEUsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFHLENBQUM7WUFDbkIsT0FBTyxFQUFFO2dCQUNQLEdBQUcsRUFBRSxLQUFLO2FBQ1g7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrR0FBa0csQ0FBQyxJQUFVO1FBQzNHLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdkIsSUFBSSxXQUFLLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ3RCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNGQUFzRixDQUFDLElBQVU7UUFFL0YsTUFBTSxLQUFNLFNBQVEsZUFBUztZQUNqQixRQUFRO2dCQUNoQixPQUFPLENBQUMsY0FBYyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDeEMsQ0FBQztTQUNGO1FBRUQsTUFBTSxNQUFPLFNBQVEsV0FBSztTQUV6QjtRQUVELE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFFdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3pDLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztRQUN4QixJQUFJLEtBQUssQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFeEIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEVBQUUsNkNBQTZDLENBQUMsQ0FBQztRQUU5RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsaUZBQWlGLENBQUMsSUFBVTtRQUMxRixNQUFNLE9BQVEsU0FBUSxXQUFLO1lBQ3pCLFlBQVksS0FBVSxFQUFFLEVBQVUsRUFBRSxLQUFrQjtnQkFDcEQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7Z0JBRXhCLElBQUksQ0FBQyxvQkFBb0IsQ0FBQztvQkFDeEIsR0FBRyxFQUFFLHFCQUFxQjtvQkFDMUIsUUFBUSxFQUFFLE1BQU07b0JBQ2hCLEtBQUssRUFBRTt3QkFDTCxPQUFPLEVBQUUsYUFBYTt3QkFDdEIsTUFBTSxFQUFFLFlBQVk7cUJBQ3JCO2lCQUNGLENBQ0EsQ0FBQztnQkFFRixJQUFJLENBQUMsb0JBQW9CLENBQUM7b0JBQ3hCLEdBQUcsRUFBRSx1QkFBdUI7b0JBQzVCLFFBQVEsRUFBRSxPQUFPO29CQUNqQixLQUFLLEVBQUU7d0JBQ0wsR0FBRyxFQUFFLEtBQUs7d0JBQ1YsT0FBTyxFQUFFLGFBQWE7d0JBQ3RCLE1BQU0sRUFBRSxZQUFZO3FCQUNyQjtpQkFDRixDQUNBLENBQUM7WUFDSixDQUFDO1NBQ0Y7UUFFRCxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxFQUFFO1lBQ2pDLElBQUksT0FBTyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM5QixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7WUFDeEM7Z0JBQ0UsR0FBRyxFQUFFLHFCQUFxQjtnQkFDMUIsUUFBUSxFQUFFLE1BQU07Z0JBQ2hCLEtBQUssRUFBRTtvQkFDTCxPQUFPLEVBQUUsYUFBYTtvQkFDdEIsTUFBTSxFQUFFLFlBQVk7aUJBQ3JCO2FBQ0Y7WUFDRDtnQkFDRSxHQUFHLEVBQUUsdUJBQXVCO2dCQUM1QixRQUFRLEVBQUUsT0FBTztnQkFDakIsS0FBSyxFQUFFO29CQUNMLE9BQU8sRUFBRSxhQUFhO29CQUN0QixNQUFNLEVBQUUsWUFBWTtvQkFDcEIsR0FBRyxFQUFFLEtBQUs7aUJBQ1g7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLE1BQU0sT0FBTyxHQUFRLEVBQUUsQ0FBQztRQUN4QixPQUFPLENBQUMsS0FBSyxDQUFDLHlCQUF5QixDQUFDLEdBQUcsSUFBSSxDQUFDO1FBRWhELE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLEVBQUU7WUFDdEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3ZDLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztRQUNuRSxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwQkFBMEIsQ0FBQyxJQUFVO1FBQ25DLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLENBQUMsRUFBRTtZQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDdkMsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxDQUFDO1FBRXBFLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNuRCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLFdBQVcsT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDbkUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZDQUE2QyxDQUFDLElBQVU7UUFDdEQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxVQUFVLEdBQUcsY0FBYyxDQUFDO1FBRXhDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLENBQUMsRUFBRTtZQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDdkMsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ3BFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRXJELE9BQU8sT0FBTyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUM7UUFDOUIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNFQUFzRSxDQUFDLElBQVU7UUFDL0UsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLEVBQUUsV0FBVyxFQUFFLElBQUksRUFBRSxFQUFFLEdBQUcsQ0FBQyxFQUFFO1lBQ3BELE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUN2QyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7UUFDbkUsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLElBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLElBQUksUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFcEUsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ25ELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFO1lBQ25CLGVBQWUsRUFBRSxPQUFPO1lBQ3hCLGlCQUFpQixFQUFFLE9BQU87WUFDMUIsY0FBYyxFQUFFLFdBQVcsT0FBTyxDQUFDLE9BQU8sRUFBRTtTQUM3QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOENBQThDLENBQUMsSUFBVTtRQUN2RCxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFO1lBQ25DLE1BQU0sUUFBUSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUN6QyxNQUFNLFdBQVcsR0FBRyxJQUFJLGlCQUFXLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxFQUFFLElBQUksRUFBRSx5QkFBeUIsRUFBRSxDQUFDLENBQUM7WUFFMUYsTUFBTSxXQUFXLEdBQUcsSUFBSSxXQUFLLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3BELElBQUksaUJBQVcsQ0FBQyxXQUFXLEVBQUUsS0FBSyxFQUFFLEVBQUUsSUFBSSxFQUFFLDRCQUE0QixFQUFFLENBQUMsQ0FBQztRQUM5RSxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxFQUFFO1lBQ2pGO2dCQUNFLElBQUksRUFBRSxPQUFPO2dCQUNiLFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLElBQUksRUFBRSx5QkFBeUIsRUFBRSxFQUFFLEVBQUU7YUFDdEU7WUFDRDtnQkFDRSxJQUFJLEVBQUUsdUJBQXVCO2dCQUM3QixRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxJQUFJLEVBQUUsNEJBQTRCLEVBQUUsRUFBRSxFQUFFO2FBQ3pFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNBLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgY3hhcGkgPSByZXF1aXJlKCdAYXdzLWNkay9jeC1hcGknKTtcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgQ29uc3RydWN0LCBTdGFjaywgU3RhY2tQcm9wcyB9IGZyb20gJy4uL2xpYic7XG5pbXBvcnQgeyBBcHAsIEFwcFByb3BzIH0gZnJvbSAnLi4vbGliL2FwcCc7XG5cbmZ1bmN0aW9uIHdpdGhBcHAocHJvcHM6IEFwcFByb3BzLCBibG9jazogKGFwcDogQXBwKSA9PiB2b2lkKTogY3hhcGkuQ2xvdWRBc3NlbWJseSB7XG4gIGNvbnN0IGFwcCA9IG5ldyBBcHAoe1xuICAgIHJ1bnRpbWVJbmZvOiBmYWxzZSxcbiAgICBzdGFja1RyYWNlczogZmFsc2UsXG4gICAgLi4ucHJvcHMsXG4gIH0pO1xuXG4gIGJsb2NrKGFwcCk7XG5cbiAgcmV0dXJuIGFwcC5zeW50aCgpO1xufVxuXG5mdW5jdGlvbiBzeW50aChjb250ZXh0PzogeyBba2V5OiBzdHJpbmddOiBhbnkgfSk6IGN4YXBpLkNsb3VkQXNzZW1ibHkge1xuICByZXR1cm4gd2l0aEFwcCh7IGNvbnRleHQgfSwgYXBwID0+IHtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnc3RhY2sxJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1JywgcmVnaW9uOiAndXMtZWFzdC0xJyB9IH0pO1xuICAgIG5ldyBDZm5SZXNvdXJjZShzdGFjazEsICdzMWMxJywgeyB0eXBlOiAnRHVtbXlSZXNvdXJjZScsIHByb3BlcnRpZXM6IHsgUHJvcDE6ICdQcm9wMScgfSB9KTtcbiAgICBjb25zdCByMiA9IG5ldyBDZm5SZXNvdXJjZShzdGFjazEsICdzMWMyJywgeyB0eXBlOiAnRHVtbXlSZXNvdXJjZScsIHByb3BlcnRpZXM6IHsgRm9vOiAxMjMgfSB9KTtcblxuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdzdGFjazInKTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2syLCAnczJjMScsIHsgdHlwZTogJ0R1bW15UmVzb3VyY2UnLCBwcm9wZXJ0aWVzOiB7IFByb2cyOiAnUHJvZzInIH0gfSk7XG4gICAgY29uc3QgYzEgPSBuZXcgTXlDb25zdHJ1Y3Qoc3RhY2syLCAnczFjMicpO1xuXG4gICAgLy8gYWRkIHNvbWUgbWV0YWRhdGFcbiAgICBzdGFjazEubm9kZS5hZGRNZXRhZGF0YSgnbWV0YScsIDExMSk7XG4gICAgcjIubm9kZS5hZGRXYXJuaW5nKCd3YXJuaW5nMScpO1xuICAgIHIyLm5vZGUuYWRkV2FybmluZygnd2FybmluZzInKTtcbiAgICBjMS5ub2RlLmFkZE1ldGFkYXRhKCdtZXRhJywgeyBrZXk6ICd2YWx1ZScgfSk7XG4gICAgYXBwLm5vZGUuYWRkTWV0YWRhdGEoJ2FwcGxldmVsJywgMTIzKTsgLy8gYXBwcyBjYW4gYWxzbyBoYXZlIG1ldGFkYXRhXG4gIH0pO1xufVxuXG5mdW5jdGlvbiBzeW50aFN0YWNrKG5hbWU6IHN0cmluZywgaW5jbHVkZU1ldGFkYXRhOiBib29sZWFuID0gZmFsc2UsIGNvbnRleHQ/OiBhbnkpOiBjeGFwaS5DbG91ZEZvcm1hdGlvblN0YWNrQXJ0aWZhY3Qge1xuICBjb25zdCByZXNwb25zZSA9IHN5bnRoKGNvbnRleHQpO1xuICBjb25zdCBzdGFjayA9IHJlc3BvbnNlLmdldFN0YWNrKG5hbWUpO1xuXG4gIGlmICghaW5jbHVkZU1ldGFkYXRhKSB7XG4gICAgZGVsZXRlIChzdGFjayBhcyBhbnkpLm1ldGFkYXRhO1xuICB9XG5cbiAgcmV0dXJuIHN0YWNrO1xufVxuXG5leHBvcnQgPSB7XG4gICdzeW50aGVzaXplcyBhbGwgc3RhY2tzIGFuZCByZXR1cm5zIHN5bnRoZXNpcyByZXN1bHQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByZXNwb25zZSA9IHN5bnRoKCk7XG4gICAgZGVsZXRlIChyZXNwb25zZSBhcyBhbnkpLmRpcjtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHJlc3BvbnNlLnN0YWNrcy5sZW5ndGgsIDIpO1xuXG4gICAgY29uc3Qgc3RhY2sxID0gcmVzcG9uc2Uuc3RhY2tzWzBdO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS5uYW1lLCAnc3RhY2sxJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sxLmVudmlyb25tZW50LmFjY291bnQsIDEyMzQ1KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEuZW52aXJvbm1lbnQucmVnaW9uLCAndXMtZWFzdC0xJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sxLmVudmlyb25tZW50Lm5hbWUsICdhd3M6Ly8xMjM0NS91cy1lYXN0LTEnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEudGVtcGxhdGUsIHsgUmVzb3VyY2VzOlxuICAgICAgeyBzMWMxOiB7IFR5cGU6ICdEdW1teVJlc291cmNlJywgUHJvcGVydGllczogeyBQcm9wMTogJ1Byb3AxJyB9IH0sXG4gICAgICAgIHMxYzI6IHsgVHlwZTogJ0R1bW15UmVzb3VyY2UnLCBQcm9wZXJ0aWVzOiB7IEZvbzogMTIzIH0gfSB9IH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS5tYW5pZmVzdC5tZXRhZGF0YSwge1xuICAgICAgJy9zdGFjazEnOiBbeyB0eXBlOiAnbWV0YScsIGRhdGE6IDExMSB9XSxcbiAgICAgICcvc3RhY2sxL3MxYzEnOiBbeyB0eXBlOiAnYXdzOmNkazpsb2dpY2FsSWQnLCBkYXRhOiAnczFjMScgfV0sXG4gICAgICAnL3N0YWNrMS9zMWMyJzpcbiAgICAgICAgW3sgdHlwZTogJ2F3czpjZGs6bG9naWNhbElkJywgZGF0YTogJ3MxYzInIH0sXG4gICAgICAgIHsgdHlwZTogJ2F3czpjZGs6d2FybmluZycsIGRhdGE6ICd3YXJuaW5nMScgfSxcbiAgICAgICAgeyB0eXBlOiAnYXdzOmNkazp3YXJuaW5nJywgZGF0YTogJ3dhcm5pbmcyJyB9XSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHN0YWNrMiA9IHJlc3BvbnNlLnN0YWNrc1sxXTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazIubmFtZSwgJ3N0YWNrMicpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMi5lbnZpcm9ubWVudC5uYW1lLCAnYXdzOi8vdW5rbm93bi1hY2NvdW50L3Vua25vd24tcmVnaW9uJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLnRlbXBsYXRlLCB7IFJlc291cmNlczpcbiAgICAgIHsgczJjMTogeyBUeXBlOiAnRHVtbXlSZXNvdXJjZScsIFByb3BlcnRpZXM6IHsgUHJvZzI6ICdQcm9nMicgfSB9LFxuICAgICAgICBzMWMycjFEMTc5MUMwMTogeyBUeXBlOiAnUmVzb3VyY2VUeXBlMScgfSxcbiAgICAgICAgczFjMnIyNUY2ODVGRkY6IHsgVHlwZTogJ1Jlc291cmNlVHlwZTInIH0gfSB9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazIubWFuaWZlc3QubWV0YWRhdGEsIHtcbiAgICAgICcvc3RhY2syL3MyYzEnOiBbeyB0eXBlOiAnYXdzOmNkazpsb2dpY2FsSWQnLCBkYXRhOiAnczJjMScgfV0sXG4gICAgICAnL3N0YWNrMi9zMWMyJzogW3sgdHlwZTogJ21ldGEnLCBkYXRhOiB7IGtleTogJ3ZhbHVlJyB9IH1dLFxuICAgICAgJy9zdGFjazIvczFjMi9yMSc6XG4gICAgICAgIFt7IHR5cGU6ICdhd3M6Y2RrOmxvZ2ljYWxJZCcsIGRhdGE6ICdzMWMycjFEMTc5MUMwMScgfV0sXG4gICAgICAnL3N0YWNrMi9zMWMyL3IyJzpcbiAgICAgICAgW3sgdHlwZTogJ2F3czpjZGs6bG9naWNhbElkJywgZGF0YTogJ3MxYzJyMjVGNjg1RkZGJyB9XSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xufSxcblxuJ2NvbnRleHQgY2FuIGJlIHBhc3NlZCB0aHJvdWdoIENES19DT05URVhUJyh0ZXN0OiBUZXN0KSB7XG4gIHByb2Nlc3MuZW52W2N4YXBpLkNPTlRFWFRfRU5WXSA9IEpTT04uc3RyaW5naWZ5KHtcbiAgICBrZXkxOiAndmFsMScsXG4gICAga2V5MjogJ3ZhbDInXG4gIH0pO1xuICBjb25zdCBwcm9nID0gbmV3IEFwcCgpO1xuICB0ZXN0LmRlZXBFcXVhbChwcm9nLm5vZGUudHJ5R2V0Q29udGV4dCgna2V5MScpLCAndmFsMScpO1xuICB0ZXN0LmRlZXBFcXVhbChwcm9nLm5vZGUudHJ5R2V0Q29udGV4dCgna2V5MicpLCAndmFsMicpO1xuICB0ZXN0LmRvbmUoKTtcbn0sXG5cbidjb250ZXh0IHBhc3NlZCB0aHJvdWdoIENES19DT05URVhUIGhhcyBwcmVjZWRlbmNlJyh0ZXN0OiBUZXN0KSB7XG4gIHByb2Nlc3MuZW52W2N4YXBpLkNPTlRFWFRfRU5WXSA9IEpTT04uc3RyaW5naWZ5KHtcbiAgICBrZXkxOiAndmFsMScsXG4gICAga2V5MjogJ3ZhbDInXG4gIH0pO1xuICBjb25zdCBwcm9nID0gbmV3IEFwcCh7XG4gICAgY29udGV4dDoge1xuICAgICAga2V5MTogJ3ZhbDMnLFxuICAgICAga2V5MjogJ3ZhbDQnXG4gICAgfVxuICB9KTtcbiAgdGVzdC5kZWVwRXF1YWwocHJvZy5ub2RlLnRyeUdldENvbnRleHQoJ2tleTEnKSwgJ3ZhbDEnKTtcbiAgdGVzdC5kZWVwRXF1YWwocHJvZy5ub2RlLnRyeUdldENvbnRleHQoJ2tleTInKSwgJ3ZhbDInKTtcbiAgdGVzdC5kb25lKCk7XG59LFxuXG4nY29udGV4dCBmcm9tIHRoZSBjb21tYW5kIGxpbmUgY2FuIGJlIHVzZWQgd2hlbiBjcmVhdGluZyB0aGUgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgY29uc3Qgb3V0cHV0ID0gc3ludGhTdGFjaygnc3RhY2syJywgZmFsc2UsIHsgY3R4MTogJ0hFTExPJyB9KTtcblxuICB0ZXN0LmRlZXBFcXVhbChvdXRwdXQudGVtcGxhdGUsIHtcbiAgICBSZXNvdXJjZXM6IHtcbiAgICAgIHMyYzE6IHtcbiAgICAgICAgVHlwZTogXCJEdW1teVJlc291cmNlXCIsXG4gICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICBQcm9nMjogXCJQcm9nMlwiXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBzMWMycjFEMTc5MUMwMToge1xuICAgICAgICBUeXBlOiBcIlJlc291cmNlVHlwZTFcIlxuICAgICAgfSxcbiAgICAgIHMxYzJyMjVGNjg1RkZGOiB7XG4gICAgICAgIFR5cGU6IFwiUmVzb3VyY2VUeXBlMlwiLFxuICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgRnJvbUNvbnRleHQ6IFwiSEVMTE9cIlxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9KTtcbiAgdGVzdC5kb25lKCk7XG59LFxuXG4nc2V0Q29udGV4dChrLHYpIGNhbiBiZSB1c2VkIHRvIHNldCBjb250ZXh0IHByb2dyYW1tYXRpY2FsbHknKHRlc3Q6IFRlc3QpIHtcbiAgY29uc3QgcHJvZyA9IG5ldyBBcHAoe1xuICAgIGNvbnRleHQ6IHtcbiAgICAgIGZvbzogJ2JhcidcbiAgICB9XG4gIH0pO1xuICB0ZXN0LmRlZXBFcXVhbChwcm9nLm5vZGUudHJ5R2V0Q29udGV4dCgnZm9vJyksICdiYXInKTtcbiAgdGVzdC5kb25lKCk7XG59LFxuXG4nc2V0Q29udGV4dChrLHYpIGNhbm5vdCBiZSBjYWxsZWQgYWZ0ZXIgc3RhY2tzIGhhdmUgYmVlbiBhZGRlZCBiZWNhdXNlIHN0YWNrcyBtYXkgdXNlIHRoZSBjb250ZXh0Jyh0ZXN0OiBUZXN0KSB7XG4gIGNvbnN0IHByb2cgPSBuZXcgQXBwKCk7XG4gIG5ldyBTdGFjayhwcm9nLCAnczEnKTtcbiAgdGVzdC50aHJvd3MoKCkgPT4gcHJvZy5ub2RlLnNldENvbnRleHQoJ2ZvbycsICdiYXInKSk7XG4gIHRlc3QuZG9uZSgpO1xufSxcblxuJ2FwcC5zeW50aCgpIHBlcmZvcm1zIHZhbGlkYXRpb24gZmlyc3QgYW5kIGlmIHRoZXJlIGFyZSBlcnJvcnMsIGl0IHJldHVybnMgdGhlIGVycm9ycycodGVzdDogVGVzdCkge1xuXG4gIGNsYXNzIENoaWxkIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICBwcm90ZWN0ZWQgdmFsaWRhdGUoKSB7XG4gICAgICByZXR1cm4gW2BFcnJvciBmcm9tICR7dGhpcy5ub2RlLmlkfWBdO1xuICAgIH1cbiAgfVxuXG4gIGNsYXNzIFBhcmVudCBleHRlbmRzIFN0YWNrIHtcblxuICB9XG5cbiAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuXG4gIGNvbnN0IHBhcmVudCA9IG5ldyBQYXJlbnQoYXBwLCAnUGFyZW50Jyk7XG4gIG5ldyBDaGlsZChwYXJlbnQsICdDMScpO1xuICBuZXcgQ2hpbGQocGFyZW50LCAnQzInKTtcblxuICB0ZXN0LnRocm93cygoKSA9PiBhcHAuc3ludGgoKSwgL1ZhbGlkYXRpb24gZmFpbGVkIHdpdGggdGhlIGZvbGxvd2luZyBlcnJvcnMvKTtcblxuICB0ZXN0LmRvbmUoKTtcbn0sXG5cbidhcHAuc3ludGhlc2l6ZVN0YWNrKHN0YWNrKSB3aWxsIHJldHVybiBhIGxpc3Qgb2YgbWlzc2luZyBjb250ZXh0dWFsIGluZm9ybWF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gIGNsYXNzIE15U3RhY2sgZXh0ZW5kcyBTdGFjayB7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IEFwcCwgaWQ6IHN0cmluZywgcHJvcHM/OiBTdGFja1Byb3BzKSB7XG4gICAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgICAgdGhpcy5yZXBvcnRNaXNzaW5nQ29udGV4dCh7XG4gICAgICAgIGtleTogJ21pc3NpbmctY29udGV4dC1rZXknLFxuICAgICAgICBwcm92aWRlcjogJ2Zha2UnLFxuICAgICAgICBwcm9wczoge1xuICAgICAgICAgIGFjY291bnQ6ICcxMjM0NTY4OTAxMicsXG4gICAgICAgICAgcmVnaW9uOiAnYWItbm9ydGgtMScsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgKTtcblxuICAgICAgdGhpcy5yZXBvcnRNaXNzaW5nQ29udGV4dCh7XG4gICAgICAgIGtleTogJ21pc3NpbmctY29udGV4dC1rZXktMicsXG4gICAgICAgIHByb3ZpZGVyOiAnZmFrZTInLFxuICAgICAgICBwcm9wczoge1xuICAgICAgICAgIGZvbzogJ2JhcicsXG4gICAgICAgICAgYWNjb3VudDogJzEyMzQ1Njg5MDEyJyxcbiAgICAgICAgICByZWdpb246ICdhYi1zb3V0aC0xJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIGNvbnN0IGFzc2VtYmx5ID0gd2l0aEFwcCh7fSwgYXBwID0+IHtcbiAgICBuZXcgTXlTdGFjayhhcHAsICdNeVN0YWNrJyk7XG4gIH0pO1xuXG4gIHRlc3QuZGVlcEVxdWFsKGFzc2VtYmx5Lm1hbmlmZXN0Lm1pc3NpbmcsIFtcbiAgICB7XG4gICAgICBrZXk6IFwibWlzc2luZy1jb250ZXh0LWtleVwiLFxuICAgICAgcHJvdmlkZXI6ICdmYWtlJyxcbiAgICAgIHByb3BzOiB7XG4gICAgICAgIGFjY291bnQ6ICcxMjM0NTY4OTAxMicsXG4gICAgICAgIHJlZ2lvbjogJ2FiLW5vcnRoLTEnLFxuICAgICAgfSxcbiAgICB9LFxuICAgIHtcbiAgICAgIGtleTogXCJtaXNzaW5nLWNvbnRleHQta2V5LTJcIixcbiAgICAgIHByb3ZpZGVyOiAnZmFrZTInLFxuICAgICAgcHJvcHM6IHtcbiAgICAgICAgYWNjb3VudDogJzEyMzQ1Njg5MDEyJyxcbiAgICAgICAgcmVnaW9uOiAnYWItc291dGgtMScsXG4gICAgICAgIGZvbzogJ2JhcicsXG4gICAgICB9LFxuICAgIH1cbiAgXSk7XG5cbiAgdGVzdC5kb25lKCk7XG59LFxuXG4ncnVudGltZSBsaWJyYXJ5IHZlcnNpb25zIGRpc2FibGVkJyh0ZXN0OiBUZXN0KSB7XG4gIGNvbnN0IGNvbnRleHQ6IGFueSA9IHt9O1xuICBjb250ZXh0W2N4YXBpLkRJU0FCTEVfVkVSU0lPTl9SRVBPUlRJTkddID0gdHJ1ZTtcblxuICBjb25zdCBhc3NlbWJseSA9IHdpdGhBcHAoY29udGV4dCwgYXBwID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdzdGFjazEnKTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdNeVJlc291cmNlJywgeyB0eXBlOiAnUmVzb3VyY2U6OlR5cGUnIH0pO1xuICB9KTtcblxuICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5ydW50aW1lLCB7IGxpYnJhcmllczoge30gfSk7XG4gIHRlc3QuZG9uZSgpO1xufSxcblxuJ3J1bnRpbWUgbGlicmFyeSB2ZXJzaW9ucycodGVzdDogVGVzdCkge1xuICBjb25zdCByZXNwb25zZSA9IHdpdGhBcHAoeyBydW50aW1lSW5mbzogdHJ1ZSB9LCBhcHAgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ3N0YWNrMScpO1xuICAgIG5ldyBDZm5SZXNvdXJjZShzdGFjaywgJ015UmVzb3VyY2UnLCB7IHR5cGU6ICdSZXNvdXJjZTo6VHlwZScgfSk7XG4gIH0pO1xuXG4gIGNvbnN0IGxpYnMgPSAocmVzcG9uc2UucnVudGltZSAmJiByZXNwb25zZS5ydW50aW1lLmxpYnJhcmllcykgfHwge307XG5cbiAgY29uc3QgdmVyc2lvbiA9IHJlcXVpcmUoJy4uL3BhY2thZ2UuanNvbicpLnZlcnNpb247XG4gIHRlc3QuZGVlcEVxdWFsKGxpYnNbJ0Bhd3MtY2RrL2NvcmUnXSwgdmVyc2lvbik7XG4gIHRlc3QuZGVlcEVxdWFsKGxpYnNbJ0Bhd3MtY2RrL2N4LWFwaSddLCB2ZXJzaW9uKTtcbiAgdGVzdC5kZWVwRXF1YWwobGlic1snanNpaS1ydW50aW1lJ10sIGBub2RlLmpzLyR7cHJvY2Vzcy52ZXJzaW9ufWApO1xuICB0ZXN0LmRvbmUoKTtcbn0sXG5cbidqc2lpLXJ1bnRpbWUgdmVyc2lvbiBsb2FkZWQgZnJvbSBKU0lJX0FHRU5UJyh0ZXN0OiBUZXN0KSB7XG4gIHByb2Nlc3MuZW52LkpTSUlfQUdFTlQgPSAnSmF2YS8xLjIuMy40JztcblxuICBjb25zdCByZXNwb25zZSA9IHdpdGhBcHAoeyBydW50aW1lSW5mbzogdHJ1ZSB9LCBhcHAgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ3N0YWNrMScpO1xuICAgIG5ldyBDZm5SZXNvdXJjZShzdGFjaywgJ015UmVzb3VyY2UnLCB7IHR5cGU6ICdSZXNvdXJjZTo6VHlwZScgfSk7XG4gIH0pO1xuXG4gIGNvbnN0IGxpYnMgPSAocmVzcG9uc2UucnVudGltZSAmJiByZXNwb25zZS5ydW50aW1lLmxpYnJhcmllcykgfHwge307XG4gIHRlc3QuZGVlcEVxdWFsKGxpYnNbJ2pzaWktcnVudGltZSddLCBgSmF2YS8xLjIuMy40YCk7XG5cbiAgZGVsZXRlIHByb2Nlc3MuZW52LkpTSUlfQUdFTlQ7XG4gIHRlc3QuZG9uZSgpO1xufSxcblxuJ3ZlcnNpb24gcmVwb3J0aW5nIGluY2x1ZGVzIG9ubHkgQGF3cy1jZGssIGF3cy1jZGsgYW5kIGpzaWkgbGlicmFyaWVzJyh0ZXN0OiBUZXN0KSB7XG4gIGNvbnN0IHJlc3BvbnNlID0gd2l0aEFwcCh7IHJ1bnRpbWVJbmZvOiB0cnVlIH0sIGFwcCA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnc3RhY2sxJyk7XG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrLCAnTXlSZXNvdXJjZScsIHsgdHlwZTogJ1Jlc291cmNlOjpUeXBlJyB9KTtcbiAgfSk7XG5cbiAgY29uc3QgbGlicyA9IChyZXNwb25zZS5ydW50aW1lICYmIHJlc3BvbnNlLnJ1bnRpbWUubGlicmFyaWVzKSB8fCB7fTtcblxuICBjb25zdCB2ZXJzaW9uID0gcmVxdWlyZSgnLi4vcGFja2FnZS5qc29uJykudmVyc2lvbjtcbiAgdGVzdC5kZWVwRXF1YWwobGlicywge1xuICAgICdAYXdzLWNkay9jb3JlJzogdmVyc2lvbixcbiAgICAnQGF3cy1jZGsvY3gtYXBpJzogdmVyc2lvbixcbiAgICAnanNpaS1ydW50aW1lJzogYG5vZGUuanMvJHtwcm9jZXNzLnZlcnNpb259YFxuICB9KTtcblxuICB0ZXN0LmRvbmUoKTtcbn0sXG5cbidkZWVwIHN0YWNrIGlzIHNob3duIGFuZCBzeW50aGVzaXplZCBwcm9wZXJseScodGVzdDogVGVzdCkge1xuICAvLyBXSEVOXG4gIGNvbnN0IHJlc3BvbnNlID0gd2l0aEFwcCh7fSwgKGFwcCkgPT4ge1xuICAgIGNvbnN0IHRvcFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG4gICAgY29uc3QgdG9wUmVzb3VyY2UgPSBuZXcgQ2ZuUmVzb3VyY2UodG9wU3RhY2ssICdSZXMnLCB7IHR5cGU6ICdDREs6OlRvcFN0YWNrOjpSZXNvdXJjZScgfSk7XG5cbiAgICBjb25zdCBib3R0b21TdGFjayA9IG5ldyBTdGFjayh0b3BSZXNvdXJjZSwgJ1N0YWNrJyk7XG4gICAgbmV3IENmblJlc291cmNlKGJvdHRvbVN0YWNrLCAnUmVzJywgeyB0eXBlOiAnQ0RLOjpCb3R0b21TdGFjazo6UmVzb3VyY2UnIH0pO1xuICB9KTtcblxuICAvLyBUSEVOXG4gIHRlc3QuZGVlcEVxdWFsKHJlc3BvbnNlLnN0YWNrcy5tYXAocyA9PiAoeyBuYW1lOiBzLm5hbWUsIHRlbXBsYXRlOiBzLnRlbXBsYXRlIH0pKSwgW1xuICAgIHtcbiAgICAgIG5hbWU6ICdTdGFjaycsXG4gICAgICB0ZW1wbGF0ZTogeyBSZXNvdXJjZXM6IHsgUmVzOiB7IFR5cGU6ICdDREs6OlRvcFN0YWNrOjpSZXNvdXJjZScgfSB9IH0sXG4gICAgfSxcbiAgICB7XG4gICAgICBuYW1lOiAnU3RhY2tSZXNTdGFjazdFNEFGQTg2JyxcbiAgICAgIHRlbXBsYXRlOiB7IFJlc291cmNlczogeyBSZXM6IHsgVHlwZTogJ0NESzo6Qm90dG9tU3RhY2s6OlJlc291cmNlJyB9IH0gfSxcbiAgICB9LFxuICBdKTtcblxuICB0ZXN0LmRvbmUoKTtcbn0sXG59O1xuXG5jbGFzcyBNeUNvbnN0cnVjdCBleHRlbmRzIENvbnN0cnVjdCB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbmV3IENmblJlc291cmNlKHRoaXMsICdyMScsIHsgdHlwZTogJ1Jlc291cmNlVHlwZTEnIH0pO1xuICAgIG5ldyBDZm5SZXNvdXJjZSh0aGlzLCAncjInLCB7IHR5cGU6ICdSZXNvdXJjZVR5cGUyJywgcHJvcGVydGllczogeyBGcm9tQ29udGV4dDogdGhpcy5ub2RlLnRyeUdldENvbnRleHQoJ2N0eDEnKSB9IH0pO1xuICB9XG59XG4iXX0=