"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
const testPort = 12321;
const testFunctionName = "something-unique";
const testClusterName = "something-else";
test("When provided a VPC, does not create a second VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
});
test("When provided an existingCache, does not create a second cache", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc, testPort);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingCache,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ElastiCache::CacheCluster", 1);
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        Port: testPort,
    });
});
test("When provided an existingFunction, does not create a second function", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: ".handler",
        vpc: existingVpc,
        functionName: testFunctionName,
    });
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingLambdaObj: existingFunction,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test custom environment variable name", () => {
    const stack = new cdk.Stack();
    const testEnvironmentVariableName = "CUSTOM_CLUSTER_NAME";
    new lib_1.LambdaToElasticachememcached(stack, "test-construct", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        cacheEndpointEnvironmentVariableName: testEnvironmentVariableName,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CUSTOM_CLUSTER_NAME: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
    });
});
test("Test setting custom function properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
            functionName: testFunctionName,
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test setting custom cache properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        cacheProps: {
            clusterName: testClusterName,
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        ClusterName: testClusterName,
    });
});
test("Test setting custom VPC properties", () => {
    const stack = new cdk.Stack();
    const testCidrBlock = "192.168.0.0/16";
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        vpcProps: {
            ipAddresses: ec2.IpAddresses.cidr(testCidrBlock),
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidrBlock,
    });
});
test("Test all default values", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.resourceCountIs("AWS::ElastiCache::CacheCluster", 1);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CACHE_ENDPOINT: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
        Handler: ".handler",
        Runtime: "nodejs16.x",
    });
    // All values taken from elasticache-defaults.ts
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        CacheNodeType: "cache.t3.medium",
        Engine: "memcached",
        NumCacheNodes: 2,
        Port: 11222,
        AZMode: "cross-az",
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Test for the proper self referencing security group', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: ".handler",
        },
        cacheProps: {
            port: 22223
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: 22223,
        ToPort: 22223,
        GroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
    });
});
test("Test error from existingCache and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existing function and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: ".handler",
        vpc: existingVpc,
    });
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingLambdaObj: existingFunction,
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existingCache and cacheProps", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            existingVpc,
            cacheProps: {
                numCacheNodes: 4,
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("Cannot specify existingCache and cacheProps");
});
test("Test error from trying to launch Redis", () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            cacheProps: {
                numCacheNodes: 4,
                engine: "redis",
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("This construct can only launch memcached clusters");
});
test("Test error from existingCache and no VPC", () => {
    const stack = new cdk.Stack();
    const vpc = defaults.getTestVpc(stack);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: ".handler",
            },
            vpcProps: {},
            existingVpc: vpc
        });
    };
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const testVpc = defaults.getTestVpc(stack);
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        vpc: testVpc,
    });
    const props = {
        existingVpc: testVpc,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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