"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const destinations = require("../lib");
/*
 * Stack verification steps:
 * * aws lambda invoke --function-name <deployed fn name> --invocation-type Event --payload '"OK"' response.json
 * * aws lambda invoke --function-name <deployed fn name> --invocation-type Event --payload '"NOT OK"' response.json
 */
class TestStack extends core_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const topic = new sns.Topic(this, 'Topic');
        const queue = new sqs.Queue(this, 'Queue');
        const fn = new lambda.Function(this, 'SnsSqs', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        if (event === 'OK') return 'success';
        throw new Error('failure');
      };`),
            onFailure: new destinations.SnsDestination(topic),
            onSuccess: new destinations.SqsDestination(queue),
            maxEventAge: core_1.Duration.hours(3),
            retryAttempts: 1
        });
        const onSuccessLambda = new lambda.Function(this, 'OnSucces', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        console.log(event);
      };`),
        });
        new lambda.Function(this, 'EventBusLambda', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        if (event === 'OK') return 'success';
        throw new Error('failure');
      };`),
            onFailure: new destinations.EventBridgeDestination(),
            onSuccess: new destinations.LambdaDestination(onSuccessLambda),
        });
        const version = fn.addVersion('MySpecialVersion');
        new lambda.Alias(this, 'MySpecialAlias', {
            aliasName: 'MySpecialAlias',
            version,
            onSuccess: new destinations.SqsDestination(queue),
            onFailure: new destinations.SnsDestination(topic),
            maxEventAge: core_1.Duration.hours(2),
            retryAttempts: 0
        });
    }
}
const app = new core_1.App();
new TestStack(app, 'aws-cdk-lambda-destinations');
app.synth();
//# sourceMappingURL=data:application/json;base64,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