# This file is part of daf_butler.
#
# Developed for the LSST Data Management System.
# This product includes software developed by the LSST Project
# (http://www.lsst.org).
# See the COPYRIGHT file at the top-level directory of this distribution
# for details of code ownership.
#
# This software is dual licensed under the GNU General Public License and also
# under a 3-clause BSD license. Recipients may choose which of these licenses
# to use; please see the files gpl-3.0.txt and/or bsd_license.txt,
# respectively.  If you choose the GPL option then the following text applies
# (but note that there is still no warranty even if you opt for BSD instead):
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import annotations

__all__ = (
    "DatabaseDimension",
    "DatabaseDimensionCombination",
    "DatabaseDimensionElement",
    "DatabaseTopologicalFamily",
)

from collections.abc import Iterable, Mapping, Set
from types import MappingProxyType
from typing import TYPE_CHECKING

from lsst.utils.classes import cached_getter

from .._named import NamedValueAbstractSet, NamedValueSet
from .._topology import TopologicalFamily, TopologicalSpace
from ._elements import Dimension, DimensionCombination, DimensionElement, KeyColumnSpec, MetadataColumnSpec
from .construction import DimensionConstructionBuilder, DimensionConstructionVisitor

if TYPE_CHECKING:
    from ._governor import GovernorDimension
    from ._universe import DimensionUniverse


class DatabaseTopologicalFamily(TopologicalFamily):
    """Database topological family implementation.

    A `TopologicalFamily` implementation for the `DatabaseDimension` and
    `DatabaseDimensionCombination` objects that have direct database
    representations.

    Parameters
    ----------
    name : `str`
        Name of the family.
    space : `TopologicalSpace`
        Space in which this family's regions live.
    members : `NamedValueAbstractSet` [ `DimensionElement` ]
        The members of this family, ordered according to the priority used
        in `choose` (first-choice member first).
    """

    def __init__(
        self,
        name: str,
        space: TopologicalSpace,
        *,
        members: NamedValueAbstractSet[DimensionElement],
    ):
        super().__init__(name, space)
        self.members = members

    def choose(self, endpoints: Set[str], universe: DimensionUniverse) -> DimensionElement:
        # Docstring inherited from TopologicalFamily.
        for member in self.members:
            if member.name in endpoints:
                return member
        raise RuntimeError(f"No recognized endpoints for {self.name} in {endpoints}.")

    @property
    @cached_getter
    def governor(self) -> GovernorDimension:
        """Return `GovernorDimension` common to all members of this family.

        (`GovernorDimension`).
        """
        governors = {m.governor for m in self.members}
        if None in governors:
            raise RuntimeError(
                f"Bad {self.space.name} family definition {self.name}: at least one member "
                f"in {self.members} has no GovernorDimension dependency."
            )
        try:
            (result,) = governors
        except ValueError:
            raise RuntimeError(
                f"Bad {self.space.name} family definition {self.name}: multiple governors {governors} "
                f"in {self.members}."
            ) from None
        return result  # type: ignore

    members: NamedValueAbstractSet[DimensionElement]
    """The members of this family, ordered according to the priority used in
    `choose` (first-choice member first).
    """


class DatabaseTopologicalFamilyConstructionVisitor(DimensionConstructionVisitor):
    """A construction visitor for `DatabaseTopologicalFamily`.

    This visitor depends on (and is thus visited after) its members.

    Parameters
    ----------
    name : `str`
        Name of the family.
    space : `TopologicalSpace`
        Space in which this family's regions live.
    members : `~collections.abc.Iterable` [ `str` ]
        The names of the members of this family, ordered according to the
        priority used in `choose` (first-choice member first).
    """

    def __init__(self, name: str, space: TopologicalSpace, members: Iterable[str]):
        super().__init__(name)
        self._space = space
        self._members = tuple(members)

    def hasDependenciesIn(self, others: Set[str]) -> bool:
        # Docstring inherited from DimensionConstructionVisitor.
        return not others.isdisjoint(self._members)

    def visit(self, builder: DimensionConstructionBuilder) -> None:
        # Docstring inherited from DimensionConstructionVisitor.
        members = NamedValueSet(builder.elements[name] for name in self._members)
        family = DatabaseTopologicalFamily(self.name, self._space, members=members.freeze())
        builder.topology[self._space].add(family)
        for member in members:
            assert isinstance(member, DatabaseDimension | DatabaseDimensionCombination)
            other = member._topology.setdefault(self._space, family)
            if other is not family:
                raise RuntimeError(
                    f"{member.name} is declared to be a member of (at least) two "
                    f"{self._space.name} families: {other.name} and {family.name}."
                )


class DatabaseDimensionElement(DimensionElement):
    """An intermediate base class for `DimensionElement` database classes.

    Theese classes are ones whose instances map directly to a database
    table or query.

    Parameters
    ----------
    name : `str`
        Name of the dimension.
    storage : `dict`
        Fully qualified name of the `DatabaseDimensionRecordStorage` subclass
        that will back this element in the registry (in a "cls" key) along
        with any other construction keyword arguments (in other keys).
    implied : `NamedValueAbstractSet` [ `Dimension` ]
        Other dimensions whose keys are included in this dimension's (logical)
        table as foreign keys.
    metadata_columns : `NamedValueAbstractSet` [ `MetadataColumnSpec` ]
        Field specifications for all non-key fields in this dimension's table.
    doc : `str`
        Extended description of this element.
    """

    def __init__(
        self,
        name: str,
        storage: dict,
        *,
        implied: NamedValueAbstractSet[Dimension],
        metadata_columns: NamedValueAbstractSet[MetadataColumnSpec],
        doc: str,
    ):
        self._name = name
        self._storage = storage
        self._implied = implied
        self._metadata_columns = metadata_columns
        self._topology: dict[TopologicalSpace, DatabaseTopologicalFamily] = {}
        self._doc = doc

    @property
    def name(self) -> str:
        # Docstring inherited from TopologicalRelationshipEndpoint.
        return self._name

    @property
    def implied(self) -> NamedValueAbstractSet[Dimension]:
        # Docstring inherited from DimensionElement.
        return self._implied

    @property
    def metadata_columns(self) -> NamedValueAbstractSet[MetadataColumnSpec]:
        # Docstring inherited from DimensionElement.
        return self._metadata_columns

    @property
    def implied_union_target(self) -> DimensionElement | None:
        # Docstring inherited from DimensionElement.
        # This is a bit encapsulation-breaking, but it'll all be cleaned up
        # soon when we get rid of the storage objects entirely.
        storage = self._storage.get("nested")
        if storage is None:
            storage = self._storage
        name = storage.get("view_of")
        return self.universe[name] if name is not None else None

    @property
    def is_cached(self) -> bool:
        # Docstring inherited.
        # This is a bit encapsulation-breaking, but it'll all be cleaned up
        # soon when we get rid of the storage objects entirely.
        return "caching" in self._storage["cls"]

    @property
    def documentation(self) -> str:
        # Docstring inherited from DimensionElement.
        return self._doc

    @property
    def topology(self) -> Mapping[TopologicalSpace, DatabaseTopologicalFamily]:
        # Docstring inherited from TopologicalRelationshipEndpoint
        return MappingProxyType(self._topology)

    @property
    def spatial(self) -> DatabaseTopologicalFamily | None:
        # Docstring inherited from TopologicalRelationshipEndpoint
        return self.topology.get(TopologicalSpace.SPATIAL)

    @property
    def temporal(self) -> DatabaseTopologicalFamily | None:
        # Docstring inherited from TopologicalRelationshipEndpoint
        return self.topology.get(TopologicalSpace.TEMPORAL)


class DatabaseDimension(Dimension, DatabaseDimensionElement):
    """A `Dimension` class that maps directly to a database table or query.

    Parameters
    ----------
    name : `str`
        Name of the dimension.
    storage : `dict`
        Fully qualified name of the `DatabaseDimensionRecordStorage` subclass
        that will back this element in the registry (in a "cls" key) along
        with any other construction keyword arguments (in other keys).
    required : `NamedValueSet` [ `Dimension` ]
        Other dimensions whose keys are part of the compound primary key for
        this dimension's (logical) table, as well as references to their own
        tables.  The ``required`` parameter does not include ``self`` (it
        can't, of course), but the corresponding attribute does - it is added
        by the constructor.
    implied : `NamedValueAbstractSet` [ `Dimension` ]
        Other dimensions whose keys are included in this dimension's (logical)
        table as foreign keys.
    metadata_columns : `NamedValueAbstractSet` [ `MetadataColumnSpec` ]
        Field specifications for all non-key fields in this dimension's table.
    unique_keys : `NamedValueAbstractSet` [ `KeyColumnSpec` ]
        Fields that can each be used to uniquely identify this dimension (given
        values for all required dimensions).  The first of these is used as
        (part of) this dimension's table's primary key, while others are used
        to define unique constraints.
    doc : `str`
        Extended description of this element.

    Notes
    -----
    `DatabaseDimension` objects may belong to a `TopologicalFamily`, but it is
    the responsibility of `DatabaseTopologicalFamilyConstructionVisitor` to
    update the `~TopologicalRelationshipEndpoint.topology` attribute of their
    members.
    """

    def __init__(
        self,
        name: str,
        storage: dict,
        *,
        required: NamedValueSet[Dimension],
        implied: NamedValueAbstractSet[Dimension],
        metadata_columns: NamedValueAbstractSet[MetadataColumnSpec],
        unique_keys: NamedValueAbstractSet[KeyColumnSpec],
        doc: str,
    ):
        super().__init__(name, storage=storage, implied=implied, metadata_columns=metadata_columns, doc=doc)
        required.add(self)
        self._required = required.freeze()
        self._unique_keys = unique_keys

    @property
    def required(self) -> NamedValueAbstractSet[Dimension]:
        # Docstring inherited from DimensionElement.
        return self._required

    @property
    def unique_keys(self) -> NamedValueAbstractSet[KeyColumnSpec]:
        # Docstring inherited from Dimension.
        return self._unique_keys


class DatabaseDimensionCombination(DimensionCombination, DatabaseDimensionElement):
    """A combination class that maps directly to a database table or query.

    Parameters
    ----------
    name : `str`
        Name of the dimension.
    storage : `dict`
        Fully qualified name of the `DatabaseDimensionRecordStorage` subclass
        that will back this element in the registry (in a "cls" key) along
        with any other construction keyword arguments (in other keys).
    required : `NamedValueAbstractSet` [ `Dimension` ]
        Dimensions whose keys define the compound primary key for this
        combinations's (logical) table, as well as references to their own
        tables.
    implied : `NamedValueAbstractSet` [ `Dimension` ]
        Dimensions whose keys are included in this combinations's (logical)
        table as foreign keys.
    metadata_columns : `NamedValueAbstractSet` [ `MetadataColumnSpec` ]
        Field specifications for all non-key fields in this combination's
        table.
    alwaysJoin : `bool`, optional
        If `True`, always include this element in any query or data ID in
        which its ``required`` dimensions appear, because it defines a
        relationship between those dimensions that must always be satisfied.
    populated_by : `Dimension` or `None`
        The dimension that this element's records are always inserted,
        exported, and imported alongside.
    doc : `str`
        Extended description of this element.

    Notes
    -----
    `DatabaseDimensionCombination` objects may belong to a `TopologicalFamily`,
    but it is the responsibility of
    `DatabaseTopologicalFamilyConstructionVisitor` to update the
    `~TopologicalRelationshipEndpoint.topology` attribute of their members.

    This class has a lot in common with `DatabaseDimension`, but they are
    expected to diverge in future changes, and the only way to make them share
    method implementations would be via multiple inheritance.  Given the
    trivial nature of all of those implementations, this does not seem worth
    the drawbacks (particularly the constraints it imposes on constructor
    signatures).
    """

    def __init__(
        self,
        name: str,
        storage: dict,
        *,
        required: NamedValueAbstractSet[Dimension],
        implied: NamedValueAbstractSet[Dimension],
        metadata_columns: NamedValueAbstractSet[MetadataColumnSpec],
        alwaysJoin: bool,
        populated_by: Dimension | None,
        doc: str,
    ):
        super().__init__(name, storage=storage, implied=implied, metadata_columns=metadata_columns, doc=doc)
        self._required = required
        self._alwaysJoin = alwaysJoin
        self._populated_by = populated_by

    @property
    def required(self) -> NamedValueAbstractSet[Dimension]:
        # Docstring inherited from DimensionElement.
        return self._required

    @property
    def alwaysJoin(self) -> bool:
        # Docstring inherited from DimensionElement.
        return self._alwaysJoin

    @property
    def defines_relationships(self) -> bool:
        # Docstring inherited from DimensionElement.
        return self._alwaysJoin or bool(self.implied)

    @property
    def populated_by(self) -> Dimension | None:
        # Docstring inherited.
        return self._populated_by


class DatabaseDimensionElementConstructionVisitor(DimensionConstructionVisitor):
    """Construction visitor for database dimension and dimension combination.

    Specifically, a construction visitor for `DatabaseDimension` and
    `DatabaseDimensionCombination`.

    Parameters
    ----------
    name : `str`
        Name of the dimension.
    storage : `dict`
        Fully qualified name of the `DatabaseDimensionRecordStorage` subclass
        that will back this element in the registry (in a "cls" key) along
        with any other construction keyword arguments (in other keys).
    required : `~collections.abc.Set` [ `Dimension` ]
        Names of dimensions whose keys define the compound primary key for this
        element's (logical) table, as well as references to their own
        tables.
    implied : `~collections.abc.Set` [ `Dimension` ]
        Names of dimension whose keys are included in this elements's
        (logical) table as foreign keys.
    doc : `str`
        Extended description of this element.
    metadata_columns : `~collections.abc.Iterable` [ `MetadataColumnSpec` ]
        Field specifications for all non-key fields in this element's table.
    unique_keys : `~collections.abc.Iterable` [ `KeyColumnSpec` ]
        Fields that can each be used to uniquely identify this dimension (given
        values for all required dimensions).  The first of these is used as
        (part of) this dimension's table's primary key, while others are used
        to define unique constraints.  Should be empty for
        `DatabaseDimensionCombination` definitions.
    alwaysJoin : `bool`, optional
        If `True`, always include this element in any query or data ID in
        which its ``required`` dimensions appear, because it defines a
        relationship between those dimensions that must always be satisfied.
        Should only be provided when a `DimensionCombination` is being
        constructed.
    populated_by : `Dimension`, optional
        The dimension that this element's records are always inserted,
        exported, and imported alongside.
    """

    def __init__(
        self,
        name: str,
        storage: dict,
        required: set[str],
        implied: set[str],
        doc: str,
        metadata_columns: Iterable[MetadataColumnSpec] = (),
        unique_keys: Iterable[KeyColumnSpec] = (),
        alwaysJoin: bool = False,
        populated_by: str | None = None,
    ):
        super().__init__(name)
        self._storage = storage
        self._required = required
        self._implied = implied
        self._metadata_columns = NamedValueSet(metadata_columns).freeze()
        self._unique_keys = NamedValueSet(unique_keys).freeze()
        self._alwaysJoin = alwaysJoin
        self._populated_by = populated_by
        self._doc = doc

    def hasDependenciesIn(self, others: Set[str]) -> bool:
        # Docstring inherited from DimensionConstructionVisitor.
        return not (self._required.isdisjoint(others) and self._implied.isdisjoint(others))

    def visit(self, builder: DimensionConstructionBuilder) -> None:
        # Docstring inherited from DimensionConstructionVisitor.
        # Expand required dependencies.
        for name in tuple(self._required):  # iterate over copy
            self._required.update(builder.dimensions[name].required.names)
        # Transform required and implied Dimension names into instances,
        # and reorder to match builder's order.
        required: NamedValueSet[Dimension] = NamedValueSet()
        implied: NamedValueSet[Dimension] = NamedValueSet()
        for dimension in builder.dimensions:
            if dimension.name in self._required:
                required.add(dimension)
            if dimension.name in self._implied:
                implied.add(dimension)

        if self._unique_keys:
            if self._alwaysJoin:
                raise RuntimeError(f"'alwaysJoin' is not a valid option for Dimension object {self.name}.")
            # Special handling for creating Dimension instances.
            dimension = DatabaseDimension(
                self.name,
                storage=self._storage,
                required=required,
                implied=implied.freeze(),
                metadata_columns=self._metadata_columns,
                unique_keys=self._unique_keys,
                doc=self._doc,
            )
            builder.dimensions.add(dimension)
            builder.elements.add(dimension)
        else:
            # Special handling for creating DimensionCombination instances.
            combination = DatabaseDimensionCombination(
                self.name,
                storage=self._storage,
                required=required,
                implied=implied.freeze(),
                doc=self._doc,
                metadata_columns=self._metadata_columns,
                alwaysJoin=self._alwaysJoin,
                populated_by=(
                    builder.dimensions[self._populated_by] if self._populated_by is not None else None
                ),
            )
            builder.elements.add(combination)
