"""Support for Nexia / Trane XL Thermostats."""

import logging

import aiohttp
from nexia.const import BRAND_NEXIA
from nexia.home import NexiaHome
from nexia.thermostat import NexiaThermostat

from homeassistant.const import CONF_PASSWORD, CONF_USERNAME
from homeassistant.core import HomeAssistant
from homeassistant.exceptions import ConfigEntryNotReady
from homeassistant.helpers import device_registry as dr
from homeassistant.helpers.aiohttp_client import async_get_clientsession

from .const import CONF_BRAND, DOMAIN, PLATFORMS
from .coordinator import NexiaDataUpdateCoordinator
from .types import NexiaConfigEntry
from .util import is_invalid_auth_code

_LOGGER = logging.getLogger(__name__)


async def async_setup_entry(hass: HomeAssistant, entry: NexiaConfigEntry) -> bool:
    """Configure the base Nexia device for Home Assistant."""

    conf = entry.data
    username = conf[CONF_USERNAME]
    password = conf[CONF_PASSWORD]
    brand = conf.get(CONF_BRAND, BRAND_NEXIA)

    state_file = hass.config.path(f"nexia_config_{username}.conf")
    session = async_get_clientsession(hass)
    nexia_home = NexiaHome(
        session,
        username=username,
        password=password,
        device_name=hass.config.location_name,
        state_file=state_file,
        brand=brand,
    )

    try:
        await nexia_home.login()
    except TimeoutError as ex:
        raise ConfigEntryNotReady(
            f"Timed out trying to connect to Nexia service: {ex}"
        ) from ex
    except aiohttp.ClientResponseError as http_ex:
        if is_invalid_auth_code(http_ex.status):
            _LOGGER.error(
                "Access error from Nexia service, please check credentials: %s", http_ex
            )
            return False
        raise ConfigEntryNotReady(f"Error from Nexia service: {http_ex}") from http_ex
    except aiohttp.ClientOSError as os_error:
        raise ConfigEntryNotReady(
            f"Error connecting to Nexia service: {os_error}"
        ) from os_error

    coordinator = NexiaDataUpdateCoordinator(hass, entry, nexia_home)
    await coordinator.async_config_entry_first_refresh()
    entry.runtime_data = coordinator
    await hass.config_entries.async_forward_entry_setups(entry, PLATFORMS)

    return True


async def async_unload_entry(hass: HomeAssistant, entry: NexiaConfigEntry) -> bool:
    """Unload a config entry."""
    return await hass.config_entries.async_unload_platforms(entry, PLATFORMS)


async def async_remove_config_entry_device(
    hass: HomeAssistant, entry: NexiaConfigEntry, device_entry: dr.DeviceEntry
) -> bool:
    """Remove a nexia config entry from a device."""
    coordinator = entry.runtime_data
    nexia_home = coordinator.nexia_home
    dev_ids = {dev_id[1] for dev_id in device_entry.identifiers if dev_id[0] == DOMAIN}
    for thermostat_id in nexia_home.get_thermostat_ids():
        if thermostat_id in dev_ids:
            return False
        thermostat: NexiaThermostat = nexia_home.get_thermostat_by_id(thermostat_id)
        for zone_id in thermostat.get_zone_ids():
            if zone_id in dev_ids:
                return False
    return True


async def async_migrate_entry(hass: HomeAssistant, entry: NexiaConfigEntry) -> bool:
    """Migrate entry."""

    _LOGGER.debug("Migrating from version %s", entry.version)

    if entry.version == 1:
        # 1 -> 2: Unique ID from integer to string
        if entry.minor_version == 1:
            minor_version = 2
            hass.config_entries.async_update_entry(
                entry, unique_id=str(entry.unique_id), minor_version=minor_version
            )

    _LOGGER.debug("Migration successful")

    return True
