"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const constructs_1 = require("constructs");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 *
 * @stability stable
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     *
     * @param scope The scope.
     * @stability stable
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     * @stability stable
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     * @stability stable
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     *
     * The toolkit will fail synthesis when errors are reported.
     *
     * @param message The error message.
     * @stability stable
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property` (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about alternatives.
     * @stability stable
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${constructs_1.Node.of(this.scope).path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        constructs_1.Node.of(this.scope).addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
_a = JSII_RTTI_SYMBOL_1;
Annotations[_a] = { fqn: "@aws-cdk/core.Annotations", version: "1.95.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW5ub3RhdGlvbnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbm5vdGF0aW9ucy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDJEQUEyRDtBQUMzRCwyQ0FBOEM7QUFFOUMsTUFBTSxtQkFBbUIsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7Ozs7OztBQUtyRSxNQUFhLFdBQVc7SUFTdEIsWUFBcUMsS0FBaUI7UUFBakIsVUFBSyxHQUFMLEtBQUssQ0FBWTtJQUV0RCxDQUFDOzs7Ozs7O0lBTk0sTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUFpQjtRQUNoQyxPQUFPLElBQUksV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2hDLENBQUM7Ozs7Ozs7Ozs7SUFjTSxVQUFVLENBQUMsT0FBZTtRQUMvQixJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDcEUsQ0FBQzs7Ozs7Ozs7O0lBU00sT0FBTyxDQUFDLE9BQWU7UUFDNUIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3BFLENBQUM7Ozs7Ozs7OztJQU9NLFFBQVEsQ0FBQyxPQUFlO1FBQzdCLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNyRSxDQUFDOzs7Ozs7Ozs7Ozs7OztJQWdCTSxjQUFjLENBQUMsR0FBVyxFQUFFLE9BQWU7UUFDaEQsTUFBTSxJQUFJLEdBQUcsV0FBVyxHQUFHLG1CQUFtQixPQUFPLHNEQUFzRCxDQUFDO1FBRTVHLHlDQUF5QztRQUN6QyxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsc0JBQXNCLEVBQUU7WUFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLGlCQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQztTQUN6RDtRQUVELDBCQUEwQjtRQUMxQixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUM7UUFDdEMsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ2hCLE9BQU87U0FDUjtRQUVELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdEIsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNmLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssVUFBVSxDQUFDLEtBQWEsRUFBRSxPQUFlO1FBQy9DLGlCQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7T0FFRztJQUNILElBQVksb0JBQW9CO1FBQzlCLElBQUksR0FBRyxHQUFJLElBQUksQ0FBQyxLQUFhLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ1IsR0FBRyxHQUFHLElBQUksR0FBRyxFQUFFLENBQUM7WUFDaEIsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDeEU7UUFFRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7O0FBakdILGtDQWtHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGN4c2NoZW1hIGZyb20gJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSc7XG5pbXBvcnQgeyBJQ29uc3RydWN0LCBOb2RlIH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbmNvbnN0IERFUFJFQ0FUSU9OU19TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9jb3JlLmRlcHJlY2F0aW9ucycpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFubm90YXRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgb2Yoc2NvcGU6IElDb25zdHJ1Y3QpIHtcbiAgICByZXR1cm4gbmV3IEFubm90YXRpb25zKHNjb3BlKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBzY29wZTogSUNvbnN0cnVjdCkge1xuXG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkV2FybmluZyhtZXNzYWdlOiBzdHJpbmcpIHtcbiAgICB0aGlzLmFkZE1lc3NhZ2UoY3hzY2hlbWEuQXJ0aWZhY3RNZXRhZGF0YUVudHJ5VHlwZS5XQVJOLCBtZXNzYWdlKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRJbmZvKG1lc3NhZ2U6IHN0cmluZyk6IHZvaWQge1xuICAgIHRoaXMuYWRkTWVzc2FnZShjeHNjaGVtYS5BcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlLklORk8sIG1lc3NhZ2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkRXJyb3IobWVzc2FnZTogc3RyaW5nKSB7XG4gICAgdGhpcy5hZGRNZXNzYWdlKGN4c2NoZW1hLkFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuRVJST1IsIG1lc3NhZ2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZERlcHJlY2F0aW9uKGFwaTogc3RyaW5nLCBtZXNzYWdlOiBzdHJpbmcpIHtcbiAgICBjb25zdCB0ZXh0ID0gYFRoZSBBUEkgJHthcGl9IGlzIGRlcHJlY2F0ZWQ6ICR7bWVzc2FnZX0uIFRoaXMgQVBJIHdpbGwgYmUgcmVtb3ZlZCBpbiB0aGUgbmV4dCBtYWpvciByZWxlYXNlYDtcblxuICAgIC8vIHRocm93IGlmIENES19CTE9DS19ERVBSRUNBVElPTlMgaXMgc2V0XG4gICAgaWYgKHByb2Nlc3MuZW52LkNES19CTE9DS19ERVBSRUNBVElPTlMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtOb2RlLm9mKHRoaXMuc2NvcGUpLnBhdGh9OiAke3RleHR9YCk7XG4gICAgfVxuXG4gICAgLy8gZGUtZHVwIGJhc2VkIG9uIGFwaSBrZXlcbiAgICBjb25zdCBzZXQgPSB0aGlzLmRlcHJlY2F0aW9uc1JlcG9ydGVkO1xuICAgIGlmIChzZXQuaGFzKGFwaSkpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLmFkZFdhcm5pbmcodGV4dCk7XG4gICAgc2V0LmFkZChhcGkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBtZXNzYWdlIG1ldGFkYXRhIGVudHJ5IHRvIHRoZSBjb25zdHJ1Y3Qgbm9kZSwgdG8gYmUgZGlzcGxheWVkIGJ5IHRoZSBDREsgQ0xJLlxuICAgKiBAcGFyYW0gbGV2ZWwgVGhlIG1lc3NhZ2UgbGV2ZWxcbiAgICogQHBhcmFtIG1lc3NhZ2UgVGhlIG1lc3NhZ2UgaXRzZWxmXG4gICAqL1xuICBwcml2YXRlIGFkZE1lc3NhZ2UobGV2ZWw6IHN0cmluZywgbWVzc2FnZTogc3RyaW5nKSB7XG4gICAgTm9kZS5vZih0aGlzLnNjb3BlKS5hZGRNZXRhZGF0YShsZXZlbCwgbWVzc2FnZSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgc2V0IG9mIGRlcHJlY2F0aW9ucyByZXBvcnRlZCBvbiB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHByaXZhdGUgZ2V0IGRlcHJlY2F0aW9uc1JlcG9ydGVkKCkge1xuICAgIGxldCBzZXQgPSAodGhpcy5zY29wZSBhcyBhbnkpW0RFUFJFQ0FUSU9OU19TWU1CT0xdO1xuICAgIGlmICghc2V0KSB7XG4gICAgICBzZXQgPSBuZXcgU2V0KCk7XG4gICAgICBPYmplY3QuZGVmaW5lUHJvcGVydHkodGhpcy5zY29wZSwgREVQUkVDQVRJT05TX1NZTUJPTCwgeyB2YWx1ZTogc2V0IH0pO1xuICAgIH1cblxuICAgIHJldHVybiBzZXQ7XG4gIH1cbn1cbiJdfQ==