"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const api = require("aws-cdk-lib/aws-apigateway");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    return new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
}
test('override cognito properties', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolProps = {
        userPoolName: 'test',
        userVerification: {}
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolProps
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPool', {
        AdminCreateUserConfig: {
            AllowAdminCreateUserOnly: true
        },
        EmailVerificationMessage: "The verification code to your new account is {####}",
        EmailVerificationSubject: "Verify your new account",
        SmsVerificationMessage: "The verification code to your new account is {####}",
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test",
        VerificationMessageTemplate: {
            DefaultEmailOption: "CONFIRM_WITH_CODE",
            EmailMessage: "The verification code to your new account is {####}",
            EmailSubject: "Verify your new account",
            SmsMessage: "The verification code to your new account is {####}"
        }
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.userPool !== null);
    expect(construct.userPoolClient !== null);
    expect(construct.apiGateway !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
    expect(construct.apiGatewayAuthorizer !== null);
});
test('override cognito cognitoUserPoolClientProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolClientProps = {
        authFlows: {
            userSrp: true
        }
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolClientProps
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPoolClient', {
        ExplicitAuthFlows: [
            "ALLOW_USER_SRP_AUTH",
            "ALLOW_REFRESH_TOKEN_AUTH"
        ],
    });
});
test('Check for default Cognito Auth Type', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('override Auth Type to COGNITO', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.COGNITO
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Try to override Auth Type to NONE', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                operationName: 'foo-bar'
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps to support CORS', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultCorsPreflightOptions: {
                allowOrigins: api.Cors.ALL_ORIGINS,
                allowMethods: api.Cors.ALL_METHODS
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "NONE"
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "NONE"
    });
});
test('Override apiGatewayProps with proxy = false and add POST method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('POST');
    // Super imporant to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "POST",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    template.hasResourceProperties('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
test('Override apiGatewayProps with proxy = false and add OPTIONS method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('OPTIONS');
    // Mandatory to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "NONE",
    });
    template.hasResourceProperties('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jb2duaXRvLWFwaWdhdGV3YXktbGFtYmRhLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsZ0NBQXVGO0FBQ3ZGLG1DQUFtQztBQUVuQyxpREFBaUQ7QUFDakQsa0RBQWtEO0FBQ2xELHVEQUFrRDtBQUVsRCxTQUFTLGFBQWEsQ0FBQyxLQUFnQjtJQUNyQyxNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixPQUFTLElBQUksaUNBQTJCLENBQUMsS0FBSyxFQUFFLGdDQUFnQyxFQUFFO1FBQ2hGLG1CQUFtQjtLQUNwQixDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtJQUN2QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLG9CQUFvQixHQUEwQjtRQUNsRCxZQUFZLEVBQUUsTUFBTTtRQUNwQixnQkFBZ0IsRUFBRSxFQUFFO0tBQ3JCLENBQUM7SUFFRixJQUFJLGlDQUEyQixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUN2RSxtQkFBbUI7UUFDbkIsb0JBQW9CO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx3QkFBd0IsRUFDckQ7UUFDRSxxQkFBcUIsRUFBRTtZQUNyQix3QkFBd0IsRUFBRSxJQUFJO1NBQy9CO1FBQ0Qsd0JBQXdCLEVBQUUscURBQXFEO1FBQy9FLHdCQUF3QixFQUFFLHlCQUF5QjtRQUNuRCxzQkFBc0IsRUFBRSxxREFBcUQ7UUFDN0UsY0FBYyxFQUFFO1lBQ2Qsb0JBQW9CLEVBQUUsVUFBVTtTQUNqQztRQUNELFlBQVksRUFBRSxNQUFNO1FBQ3BCLDJCQUEyQixFQUFFO1lBQzNCLGtCQUFrQixFQUFFLG1CQUFtQjtZQUN2QyxZQUFZLEVBQUUscURBQXFEO1lBQ25FLFlBQVksRUFBRSx5QkFBeUI7WUFDdkMsVUFBVSxFQUFFLHFEQUFxRDtTQUNsRTtLQUNGLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9EQUFvRCxFQUFFLEdBQUcsRUFBRTtJQUM5RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLEtBQUssR0FBcUMsRUFDL0MsQ0FBQztJQUVGLElBQUk7UUFDRixJQUFJLGlDQUEyQixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUNqRjtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUNqQztBQUNILENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtJQUM1QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLFNBQVMsR0FBZ0MsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRXBFLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3BDLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQzFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQzFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsd0JBQXdCLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDcEQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUM5QyxNQUFNLENBQUMsU0FBUyxDQUFDLG9CQUFvQixLQUFLLElBQUksQ0FBQyxDQUFDO0FBQ2xELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEdBQUcsRUFBRTtJQUN2RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLDBCQUEwQixHQUFHO1FBQ2pDLFNBQVMsRUFBRTtZQUNULE9BQU8sRUFBRSxJQUFJO1NBQ2Q7S0FDRixDQUFDO0lBRUYsSUFBSSxpQ0FBMkIsQ0FBQyxLQUFLLEVBQUUsZ0NBQWdDLEVBQUU7UUFDdkUsbUJBQW1CO1FBQ25CLDBCQUEwQjtLQUMzQixDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsOEJBQThCLEVBQUU7UUFDN0QsaUJBQWlCLEVBQUU7WUFDakIscUJBQXFCO1lBQ3JCLDBCQUEwQjtTQUMzQjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixJQUFJLGlDQUEyQixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUN2RSxtQkFBbUI7S0FDcEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELFVBQVUsRUFBRTtZQUNWLFlBQVksRUFBRTtnQkFDWixrREFBa0Q7Z0JBQ2xELGdCQUFnQjthQUNqQjtTQUNGO1FBQ0QsaUJBQWlCLEVBQUUsb0JBQW9CO1FBQ3ZDLFlBQVksRUFBRTtZQUNaLEdBQUcsRUFBRSxzREFBc0Q7U0FDNUQ7S0FDRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsVUFBVSxFQUFFO1lBQ1YsR0FBRyxFQUFFLHVEQUF1RDtTQUM3RDtRQUNELGlCQUFpQixFQUFFLG9CQUFvQjtRQUN2QyxZQUFZLEVBQUU7WUFDWixHQUFHLEVBQUUsc0RBQXNEO1NBQzVEO0tBQ0YsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0JBQStCLEVBQUUsR0FBRyxFQUFFO0lBQ3pDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sbUJBQW1CLEdBQXlCO1FBQ2hELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLElBQUksaUNBQTJCLENBQUMsS0FBSyxFQUFFLGdDQUFnQyxFQUFFO1FBQ3ZFLG1CQUFtQjtRQUNuQixlQUFlLEVBQUU7WUFDZixvQkFBb0IsRUFBRTtnQkFDcEIsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLGlCQUFpQixDQUFDLE9BQU87YUFDakQ7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx5QkFBeUIsRUFBRTtRQUN4RCxVQUFVLEVBQUU7WUFDVixZQUFZLEVBQUU7Z0JBQ1osa0RBQWtEO2dCQUNsRCxnQkFBZ0I7YUFDakI7U0FDRjtRQUNELGlCQUFpQixFQUFFLG9CQUFvQjtRQUN2QyxZQUFZLEVBQUU7WUFDWixHQUFHLEVBQUUsc0RBQXNEO1NBQzVEO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELFVBQVUsRUFBRTtZQUNWLEdBQUcsRUFBRSx1REFBdUQ7U0FDN0Q7UUFDRCxpQkFBaUIsRUFBRSxvQkFBb0I7UUFDdkMsWUFBWSxFQUFFO1lBQ1osR0FBRyxFQUFFLHNEQUFzRDtTQUM1RDtLQUNGLENBQUMsQ0FBQztBQUVMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixJQUFJLGlDQUEyQixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUN2RSxtQkFBbUI7UUFDbkIsZUFBZSxFQUFFO1lBQ2Ysb0JBQW9CLEVBQUU7Z0JBQ3BCLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJO2FBQzlDO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsVUFBVSxFQUFFO1lBQ1YsWUFBWSxFQUFFO2dCQUNaLGtEQUFrRDtnQkFDbEQsZ0JBQWdCO2FBQ2pCO1NBQ0Y7UUFDRCxpQkFBaUIsRUFBRSxvQkFBb0I7UUFDdkMsWUFBWSxFQUFFO1lBQ1osR0FBRyxFQUFFLHNEQUFzRDtTQUM1RDtLQUNGLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx5QkFBeUIsRUFBRTtRQUN4RCxVQUFVLEVBQUU7WUFDVixHQUFHLEVBQUUsdURBQXVEO1NBQzdEO1FBQ0QsaUJBQWlCLEVBQUUsb0JBQW9CO1FBQ3ZDLFlBQVksRUFBRTtZQUNaLEdBQUcsRUFBRSxzREFBc0Q7U0FDNUQ7S0FDRixDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwQkFBMEIsRUFBRSxHQUFHLEVBQUU7SUFDcEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxtQkFBbUIsR0FBeUI7UUFDaEQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QixDQUFDO0lBRUYsSUFBSSxpQ0FBMkIsQ0FBQyxLQUFLLEVBQUUsZ0NBQWdDLEVBQUU7UUFDdkUsbUJBQW1CO1FBQ25CLGVBQWUsRUFBRTtZQUNmLG9CQUFvQixFQUFFO2dCQUNwQixhQUFhLEVBQUUsU0FBUzthQUN6QjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELGFBQWEsRUFBRSxTQUFTO1FBQ3hCLFVBQVUsRUFBRTtZQUNWLFlBQVksRUFBRTtnQkFDWixrREFBa0Q7Z0JBQ2xELGdCQUFnQjthQUNqQjtTQUNGO1FBQ0QsaUJBQWlCLEVBQUUsb0JBQW9CO1FBQ3ZDLFlBQVksRUFBRTtZQUNaLEdBQUcsRUFBRSxzREFBc0Q7U0FDNUQ7S0FDRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsYUFBYSxFQUFFLFNBQVM7UUFDeEIsVUFBVSxFQUFFO1lBQ1YsR0FBRyxFQUFFLHVEQUF1RDtTQUM3RDtRQUNELGlCQUFpQixFQUFFLG9CQUFvQjtRQUN2QyxZQUFZLEVBQUU7WUFDWixHQUFHLEVBQUUsc0RBQXNEO1NBQzVEO0tBQ0YsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMENBQTBDLEVBQUUsR0FBRyxFQUFFO0lBQ3BELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sbUJBQW1CLEdBQXlCO1FBQ2hELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLElBQUksaUNBQTJCLENBQUMsS0FBSyxFQUFFLGdDQUFnQyxFQUFFO1FBQ3ZFLG1CQUFtQjtRQUNuQixlQUFlLEVBQUU7WUFDZiwyQkFBMkIsRUFBRTtnQkFDM0IsWUFBWSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVztnQkFDbEMsWUFBWSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVzthQUNuQztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELFVBQVUsRUFBRSxLQUFLO1FBQ2pCLFVBQVUsRUFBRTtZQUNWLFlBQVksRUFBRTtnQkFDWixrREFBa0Q7Z0JBQ2xELGdCQUFnQjthQUNqQjtTQUNGO1FBQ0QsaUJBQWlCLEVBQUUsb0JBQW9CO1FBQ3ZDLFlBQVksRUFBRTtZQUNaLEdBQUcsRUFBRSxzREFBc0Q7U0FDNUQ7S0FDRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsVUFBVSxFQUFFLEtBQUs7UUFDakIsVUFBVSxFQUFFO1lBQ1YsR0FBRyxFQUFFLHVEQUF1RDtTQUM3RDtRQUNELGlCQUFpQixFQUFFLG9CQUFvQjtRQUN2QyxZQUFZLEVBQUU7WUFDWixHQUFHLEVBQUUsc0RBQXNEO1NBQzVEO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELFVBQVUsRUFBRSxTQUFTO1FBQ3JCLFVBQVUsRUFBRTtZQUNWLFlBQVksRUFBRTtnQkFDWixrREFBa0Q7Z0JBQ2xELGdCQUFnQjthQUNqQjtTQUNGO1FBQ0QsaUJBQWlCLEVBQUUsTUFBTTtLQUMxQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsVUFBVSxFQUFFLFNBQVM7UUFDckIsVUFBVSxFQUFFO1lBQ1YsR0FBRyxFQUFFLHVEQUF1RDtTQUM3RDtRQUNELGlCQUFpQixFQUFFLE1BQU07S0FDMUIsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUVBQWlFLEVBQUUsR0FBRyxFQUFFO0lBQzNFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sbUJBQW1CLEdBQXlCO1FBQ2hELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLE1BQU0sQ0FBQyxHQUFHLElBQUksaUNBQTJCLENBQUMsS0FBSyxFQUFFLGdDQUFnQyxFQUFFO1FBQ2pGLG1CQUFtQjtRQUNuQixlQUFlLEVBQUU7WUFDZixLQUFLLEVBQUUsS0FBSztTQUNiO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQy9DLENBQUMsQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7SUFFcEIsc0VBQXNFO0lBQ3RFLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUVuQixNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsVUFBVSxFQUFFLE1BQU07UUFDbEIsVUFBVSxFQUFFO1lBQ1YsR0FBRyxFQUFFLHFEQUFxRDtTQUMzRDtRQUNELGlCQUFpQixFQUFFLG9CQUFvQjtRQUN2QyxZQUFZLEVBQUU7WUFDWixHQUFHLEVBQUUsc0RBQXNEO1NBQzVEO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDJCQUEyQixFQUFFO1FBQzFELFFBQVEsRUFBRSxLQUFLO0tBQ2hCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9FQUFvRSxFQUFFLEdBQUcsRUFBRTtJQUM5RSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLENBQUMsR0FBRyxJQUFJLGlDQUEyQixDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUNqRixtQkFBbUI7UUFDbkIsZUFBZSxFQUFFO1lBQ2YsS0FBSyxFQUFFLEtBQUs7U0FDYjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMvQyxDQUFDLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBRXZCLGlFQUFpRTtJQUNqRSxDQUFDLENBQUMsY0FBYyxFQUFFLENBQUM7SUFFbkIsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELFVBQVUsRUFBRSxTQUFTO1FBQ3JCLFVBQVUsRUFBRTtZQUNWLEdBQUcsRUFBRSxxREFBcUQ7U0FDM0Q7UUFDRCxpQkFBaUIsRUFBRSxNQUFNO0tBQzFCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQkFBMkIsRUFBRTtRQUMxRCxRQUFRLEVBQUUsS0FBSztLQUNoQixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IENvZ25pdG9Ub0FwaUdhdGV3YXlUb0xhbWJkYSwgQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQgKiBhcyBjb2duaXRvIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jb2duaXRvJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGFwaSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheSc7XG5pbXBvcnQgeyBUZW1wbGF0ZSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hc3NlcnRpb25zXCI7XG5cbmZ1bmN0aW9uIGRlcGxveU5ld0Z1bmMoc3RhY2s6IGNkay5TdGFjaykge1xuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICByZXR1cm4gICBuZXcgQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhKHN0YWNrLCAndGVzdC1jb2duaXRvLWFwaWdhdGV3YXktbGFtYmRhJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHNcbiAgfSk7XG59XG5cbnRlc3QoJ292ZXJyaWRlIGNvZ25pdG8gcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgY29nbml0b1VzZXJQb29sUHJvcHM6IGNvZ25pdG8uVXNlclBvb2xQcm9wcyA9IHtcbiAgICB1c2VyUG9vbE5hbWU6ICd0ZXN0JyxcbiAgICB1c2VyVmVyaWZpY2F0aW9uOiB7fVxuICB9O1xuXG4gIG5ldyBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wcyxcbiAgICBjb2duaXRvVXNlclBvb2xQcm9wc1xuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpDb2duaXRvOjpVc2VyUG9vbCcsXG4gICAge1xuICAgICAgQWRtaW5DcmVhdGVVc2VyQ29uZmlnOiB7XG4gICAgICAgIEFsbG93QWRtaW5DcmVhdGVVc2VyT25seTogdHJ1ZVxuICAgICAgfSxcbiAgICAgIEVtYWlsVmVyaWZpY2F0aW9uTWVzc2FnZTogXCJUaGUgdmVyaWZpY2F0aW9uIGNvZGUgdG8geW91ciBuZXcgYWNjb3VudCBpcyB7IyMjI31cIixcbiAgICAgIEVtYWlsVmVyaWZpY2F0aW9uU3ViamVjdDogXCJWZXJpZnkgeW91ciBuZXcgYWNjb3VudFwiLFxuICAgICAgU21zVmVyaWZpY2F0aW9uTWVzc2FnZTogXCJUaGUgdmVyaWZpY2F0aW9uIGNvZGUgdG8geW91ciBuZXcgYWNjb3VudCBpcyB7IyMjI31cIixcbiAgICAgIFVzZXJQb29sQWRkT25zOiB7XG4gICAgICAgIEFkdmFuY2VkU2VjdXJpdHlNb2RlOiBcIkVORk9SQ0VEXCJcbiAgICAgIH0sXG4gICAgICBVc2VyUG9vbE5hbWU6IFwidGVzdFwiLFxuICAgICAgVmVyaWZpY2F0aW9uTWVzc2FnZVRlbXBsYXRlOiB7XG4gICAgICAgIERlZmF1bHRFbWFpbE9wdGlvbjogXCJDT05GSVJNX1dJVEhfQ09ERVwiLFxuICAgICAgICBFbWFpbE1lc3NhZ2U6IFwiVGhlIHZlcmlmaWNhdGlvbiBjb2RlIHRvIHlvdXIgbmV3IGFjY291bnQgaXMgeyMjIyN9XCIsXG4gICAgICAgIEVtYWlsU3ViamVjdDogXCJWZXJpZnkgeW91ciBuZXcgYWNjb3VudFwiLFxuICAgICAgICBTbXNNZXNzYWdlOiBcIlRoZSB2ZXJpZmljYXRpb24gY29kZSB0byB5b3VyIG5ldyBhY2NvdW50IGlzIHsjIyMjfVwiXG4gICAgICB9XG4gICAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgZXhjZXB0aW9uIGZvciBNaXNzaW5nIGV4aXN0aW5nT2JqIGZyb20gcHJvcHMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IHByb3BzOiBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGFQcm9wcyA9IHtcbiAgfTtcblxuICB0cnkge1xuICAgIG5ldyBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEnLCBwcm9wcyk7XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICBleHBlY3QoZSkudG9CZUluc3RhbmNlT2YoRXJyb3IpO1xuICB9XG59KTtcblxudGVzdCgnY2hlY2sgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgY29uc3RydWN0OiBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEgPSBkZXBsb3lOZXdGdW5jKHN0YWNrKTtcblxuICBleHBlY3QoY29uc3RydWN0LnVzZXJQb29sICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC51c2VyUG9vbENsaWVudCAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuYXBpR2F0ZXdheSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QubGFtYmRhRnVuY3Rpb24gIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LmFwaUdhdGV3YXlDbG91ZFdhdGNoUm9sZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuYXBpR2F0ZXdheUxvZ0dyb3VwICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5hcGlHYXRld2F5QXV0aG9yaXplciAhPT0gbnVsbCk7XG59KTtcblxudGVzdCgnb3ZlcnJpZGUgY29nbml0byBjb2duaXRvVXNlclBvb2xDbGllbnRQcm9wcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgY29nbml0b1VzZXJQb29sQ2xpZW50UHJvcHMgPSB7XG4gICAgYXV0aEZsb3dzOiB7XG4gICAgICB1c2VyU3JwOiB0cnVlXG4gICAgfVxuICB9O1xuXG4gIG5ldyBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wcyxcbiAgICBjb2duaXRvVXNlclBvb2xDbGllbnRQcm9wc1xuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpDb2duaXRvOjpVc2VyUG9vbENsaWVudCcsIHtcbiAgICBFeHBsaWNpdEF1dGhGbG93czogW1xuICAgICAgXCJBTExPV19VU0VSX1NSUF9BVVRIXCIsXG4gICAgICBcIkFMTE9XX1JFRlJFU0hfVE9LRU5fQVVUSFwiXG4gICAgXSxcbiAgfSk7XG59KTtcblxudGVzdCgnQ2hlY2sgZm9yIGRlZmF1bHQgQ29nbml0byBBdXRoIFR5cGUnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xNl9YLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9O1xuXG4gIG5ldyBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wc1xuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgUmVzb3VyY2VJZDoge1xuICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFMYW1iZGFSZXN0QXBpMkUyNzI0MzFcIixcbiAgICAgICAgXCJSb290UmVzb3VyY2VJZFwiXG4gICAgICBdXG4gICAgfSxcbiAgICBBdXRob3JpemF0aW9uVHlwZTogXCJDT0dOSVRPX1VTRVJfUE9PTFNcIixcbiAgICBBdXRob3JpemVySWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFDb2duaXRvQXV0aG9yaXplcjE3MENBQ0M5XCJcbiAgICB9LFxuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6QXBpR2F0ZXdheTo6TWV0aG9kJywge1xuICAgIFJlc291cmNlSWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFMYW1iZGFSZXN0QXBpcHJveHkyM0UxREEyMFwiXG4gICAgfSxcbiAgICBBdXRob3JpemF0aW9uVHlwZTogXCJDT0dOSVRPX1VTRVJfUE9PTFNcIixcbiAgICBBdXRob3JpemVySWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFDb2duaXRvQXV0aG9yaXplcjE3MENBQ0M5XCJcbiAgICB9LFxuICB9KTtcblxufSk7XG5cbnRlc3QoJ292ZXJyaWRlIEF1dGggVHlwZSB0byBDT0dOSVRPJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBuZXcgQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhKHN0YWNrLCAndGVzdC1jb2duaXRvLWFwaWdhdGV3YXktbGFtYmRhJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHMsXG4gICAgYXBpR2F0ZXdheVByb3BzOiB7XG4gICAgICBkZWZhdWx0TWV0aG9kT3B0aW9uczoge1xuICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBpLkF1dGhvcml6YXRpb25UeXBlLkNPR05JVE9cbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICBSZXNvdXJjZUlkOiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGkyRTI3MjQzMVwiLFxuICAgICAgICBcIlJvb3RSZXNvdXJjZUlkXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIkNPR05JVE9fVVNFUl9QT09MU1wiLFxuICAgIEF1dGhvcml6ZXJJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUNvZ25pdG9BdXRob3JpemVyMTcwQ0FDQzlcIlxuICAgIH0sXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgUmVzb3VyY2VJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGlwcm94eTIzRTFEQTIwXCJcbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIkNPR05JVE9fVVNFUl9QT09MU1wiLFxuICAgIEF1dGhvcml6ZXJJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUNvZ25pdG9BdXRob3JpemVyMTcwQ0FDQzlcIlxuICAgIH0sXG4gIH0pO1xuXG59KTtcblxudGVzdCgnVHJ5IHRvIG92ZXJyaWRlIEF1dGggVHlwZSB0byBOT05FJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBuZXcgQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhKHN0YWNrLCAndGVzdC1jb2duaXRvLWFwaWdhdGV3YXktbGFtYmRhJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHMsXG4gICAgYXBpR2F0ZXdheVByb3BzOiB7XG4gICAgICBkZWZhdWx0TWV0aG9kT3B0aW9uczoge1xuICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBpLkF1dGhvcml6YXRpb25UeXBlLk5PTkVcbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICBSZXNvdXJjZUlkOiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGkyRTI3MjQzMVwiLFxuICAgICAgICBcIlJvb3RSZXNvdXJjZUlkXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIkNPR05JVE9fVVNFUl9QT09MU1wiLFxuICAgIEF1dGhvcml6ZXJJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUNvZ25pdG9BdXRob3JpemVyMTcwQ0FDQzlcIlxuICAgIH0sXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgUmVzb3VyY2VJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGlwcm94eTIzRTFEQTIwXCJcbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIkNPR05JVE9fVVNFUl9QT09MU1wiLFxuICAgIEF1dGhvcml6ZXJJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUNvZ25pdG9BdXRob3JpemVyMTcwQ0FDQzlcIlxuICAgIH0sXG4gIH0pO1xuXG59KTtcblxudGVzdCgnT3ZlcnJpZGUgYXBpR2F0ZXdheVByb3BzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBuZXcgQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhKHN0YWNrLCAndGVzdC1jb2duaXRvLWFwaWdhdGV3YXktbGFtYmRhJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHMsXG4gICAgYXBpR2F0ZXdheVByb3BzOiB7XG4gICAgICBkZWZhdWx0TWV0aG9kT3B0aW9uczoge1xuICAgICAgICBvcGVyYXRpb25OYW1lOiAnZm9vLWJhcidcbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICBPcGVyYXRpb25OYW1lOiBcImZvby1iYXJcIixcbiAgICBSZXNvdXJjZUlkOiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGkyRTI3MjQzMVwiLFxuICAgICAgICBcIlJvb3RSZXNvdXJjZUlkXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIkNPR05JVE9fVVNFUl9QT09MU1wiLFxuICAgIEF1dGhvcml6ZXJJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUNvZ25pdG9BdXRob3JpemVyMTcwQ0FDQzlcIlxuICAgIH0sXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgT3BlcmF0aW9uTmFtZTogXCJmb28tYmFyXCIsXG4gICAgUmVzb3VyY2VJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGlwcm94eTIzRTFEQTIwXCJcbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIkNPR05JVE9fVVNFUl9QT09MU1wiLFxuICAgIEF1dGhvcml6ZXJJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUNvZ25pdG9BdXRob3JpemVyMTcwQ0FDQzlcIlxuICAgIH0sXG4gIH0pO1xuXG59KTtcblxudGVzdCgnT3ZlcnJpZGUgYXBpR2F0ZXdheVByb3BzIHRvIHN1cHBvcnQgQ09SUycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgbmV3IENvZ25pdG9Ub0FwaUdhdGV3YXlUb0xhbWJkYShzdGFjaywgJ3Rlc3QtY29nbml0by1hcGlnYXRld2F5LWxhbWJkYScsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzLFxuICAgIGFwaUdhdGV3YXlQcm9wczoge1xuICAgICAgZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zOiB7XG4gICAgICAgIGFsbG93T3JpZ2luczogYXBpLkNvcnMuQUxMX09SSUdJTlMsXG4gICAgICAgIGFsbG93TWV0aG9kczogYXBpLkNvcnMuQUxMX01FVEhPRFNcbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICBIdHRwTWV0aG9kOiBcIkFOWVwiLFxuICAgIFJlc291cmNlSWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwidGVzdGNvZ25pdG9hcGlnYXRld2F5bGFtYmRhTGFtYmRhUmVzdEFwaTJFMjcyNDMxXCIsXG4gICAgICAgIFwiUm9vdFJlc291cmNlSWRcIlxuICAgICAgXVxuICAgIH0sXG4gICAgQXV0aG9yaXphdGlvblR5cGU6IFwiQ09HTklUT19VU0VSX1BPT0xTXCIsXG4gICAgQXV0aG9yaXplcklkOiB7XG4gICAgICBSZWY6IFwidGVzdGNvZ25pdG9hcGlnYXRld2F5bGFtYmRhQ29nbml0b0F1dGhvcml6ZXIxNzBDQUNDOVwiXG4gICAgfSxcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCcsIHtcbiAgICBIdHRwTWV0aG9kOiBcIkFOWVwiLFxuICAgIFJlc291cmNlSWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFMYW1iZGFSZXN0QXBpcHJveHkyM0UxREEyMFwiXG4gICAgfSxcbiAgICBBdXRob3JpemF0aW9uVHlwZTogXCJDT0dOSVRPX1VTRVJfUE9PTFNcIixcbiAgICBBdXRob3JpemVySWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFDb2duaXRvQXV0aG9yaXplcjE3MENBQ0M5XCJcbiAgICB9LFxuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6QXBpR2F0ZXdheTo6TWV0aG9kJywge1xuICAgIEh0dHBNZXRob2Q6IFwiT1BUSU9OU1wiLFxuICAgIFJlc291cmNlSWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwidGVzdGNvZ25pdG9hcGlnYXRld2F5bGFtYmRhTGFtYmRhUmVzdEFwaTJFMjcyNDMxXCIsXG4gICAgICAgIFwiUm9vdFJlc291cmNlSWRcIlxuICAgICAgXVxuICAgIH0sXG4gICAgQXV0aG9yaXphdGlvblR5cGU6IFwiTk9ORVwiXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgSHR0cE1ldGhvZDogXCJPUFRJT05TXCIsXG4gICAgUmVzb3VyY2VJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGlwcm94eTIzRTFEQTIwXCJcbiAgICB9LFxuICAgIEF1dGhvcml6YXRpb25UeXBlOiBcIk5PTkVcIlxuICB9KTtcblxufSk7XG5cbnRlc3QoJ092ZXJyaWRlIGFwaUdhdGV3YXlQcm9wcyB3aXRoIHByb3h5ID0gZmFsc2UgYW5kIGFkZCBQT1NUIG1ldGhvZCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgYyA9IG5ldyBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wcyxcbiAgICBhcGlHYXRld2F5UHJvcHM6IHtcbiAgICAgIHByb3h5OiBmYWxzZVxuICAgIH1cbiAgfSk7XG5cbiAgY29uc3QgciA9IGMuYXBpR2F0ZXdheS5yb290LmFkZFJlc291cmNlKCdmb28nKTtcbiAgci5hZGRNZXRob2QoJ1BPU1QnKTtcblxuICAvLyBTdXBlciBpbXBvcmFudCB0byBjYWxsIHRoaXMgbWV0aG9kIHRvIEFwcGx5IHRoZSBDb2duaXRvIEF1dGhvcml6ZXJzXG4gIGMuYWRkQXV0aG9yaXplcnMoKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBcGlHYXRld2F5OjpNZXRob2QnLCB7XG4gICAgSHR0cE1ldGhvZDogXCJQT1NUXCIsXG4gICAgUmVzb3VyY2VJZDoge1xuICAgICAgUmVmOiBcInRlc3Rjb2duaXRvYXBpZ2F0ZXdheWxhbWJkYUxhbWJkYVJlc3RBcGlmb284OUFDQTQzN1wiXG4gICAgfSxcbiAgICBBdXRob3JpemF0aW9uVHlwZTogXCJDT0dOSVRPX1VTRVJfUE9PTFNcIixcbiAgICBBdXRob3JpemVySWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFDb2duaXRvQXV0aG9yaXplcjE3MENBQ0M5XCJcbiAgICB9LFxuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6QXBpR2F0ZXdheTo6UmVzb3VyY2UnLCB7XG4gICAgUGF0aFBhcnQ6IFwiZm9vXCIsXG4gIH0pO1xufSk7XG5cbnRlc3QoJ092ZXJyaWRlIGFwaUdhdGV3YXlQcm9wcyB3aXRoIHByb3h5ID0gZmFsc2UgYW5kIGFkZCBPUFRJT05TIG1ldGhvZCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgYyA9IG5ldyBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNvZ25pdG8tYXBpZ2F0ZXdheS1sYW1iZGEnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wcyxcbiAgICBhcGlHYXRld2F5UHJvcHM6IHtcbiAgICAgIHByb3h5OiBmYWxzZVxuICAgIH1cbiAgfSk7XG5cbiAgY29uc3QgciA9IGMuYXBpR2F0ZXdheS5yb290LmFkZFJlc291cmNlKCdmb28nKTtcbiAgci5hZGRNZXRob2QoJ09QVElPTlMnKTtcblxuICAvLyBNYW5kYXRvcnkgdG8gY2FsbCB0aGlzIG1ldGhvZCB0byBBcHBseSB0aGUgQ29nbml0byBBdXRob3JpemVyc1xuICBjLmFkZEF1dGhvcml6ZXJzKCk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6QXBpR2F0ZXdheTo6TWV0aG9kJywge1xuICAgIEh0dHBNZXRob2Q6IFwiT1BUSU9OU1wiLFxuICAgIFJlc291cmNlSWQ6IHtcbiAgICAgIFJlZjogXCJ0ZXN0Y29nbml0b2FwaWdhdGV3YXlsYW1iZGFMYW1iZGFSZXN0QXBpZm9vODlBQ0E0MzdcIlxuICAgIH0sXG4gICAgQXV0aG9yaXphdGlvblR5cGU6IFwiTk9ORVwiLFxuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6QXBpR2F0ZXdheTo6UmVzb3VyY2UnLCB7XG4gICAgUGF0aFBhcnQ6IFwiZm9vXCIsXG4gIH0pO1xufSk7Il19