"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeBuildSpecs = exports.BuildSpec = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * BuildSpec for CodeBuild projects.
 */
class BuildSpec {
    /**
     *
     */
    constructor() {
    }
    /**
     *
     */
    static fromObject(value) {
        return new ObjectBuildSpec(value);
    }
    /**
     * Use a file from the source as buildspec.
     *
     * Use this if you want to use a file different from 'buildspec.yml'`
     */
    static fromSourceFilename(filename) {
        return new FilenameBuildSpec(filename);
    }
}
exports.BuildSpec = BuildSpec;
/**
 * BuildSpec that just returns the input unchanged
 */
class FilenameBuildSpec extends BuildSpec {
    constructor(filename) {
        super();
        this.filename = filename;
        this.isImmediate = false;
    }
    toBuildSpec() {
        return this.filename;
    }
    toString() {
        return `<buildspec file: ${this.filename}>`;
    }
}
/**
 * BuildSpec that understands about structure
 */
class ObjectBuildSpec extends BuildSpec {
    constructor(spec) {
        super();
        this.spec = spec;
        this.isImmediate = true;
    }
    toBuildSpec() {
        // We have to pretty-print the buildspec, otherwise
        // CodeBuild will not recognize it as an inline buildspec.
        return core_1.Lazy.stringValue({
            produce: (ctx) => core_1.Stack.of(ctx.scope).toJsonString(this.spec, 2),
        });
    }
}
/**
 * Merge two buildspecs into a new BuildSpec
 *
 * NOTE: will currently only merge commands, not artifact
 * declarations, environment variables, secrets, or any
 * other configuration elements.
 *
 * Internal for now because it's not complete/good enough
 * to expose on the objects directly, but we need to it to
 * keep feature-parity for Project.
 *
 * @internal
 */
function mergeBuildSpecs(lhs, rhs) {
    if (!(lhs instanceof ObjectBuildSpec) || !(rhs instanceof ObjectBuildSpec)) {
        throw new Error('Can only merge buildspecs created using BuildSpec.fromObject()');
    }
    return new ObjectBuildSpec(copyCommands(lhs.spec, rhs.spec));
}
exports.mergeBuildSpecs = mergeBuildSpecs;
/**
 * Extend buildSpec phases with the contents of another one
 */
function copyCommands(buildSpec, extend) {
    if (buildSpec.version === '0.1') {
        throw new Error('Cannot extend buildspec at version "0.1". Set the version to "0.2" or higher instead.');
    }
    const ret = Object.assign({}, buildSpec); // Return a copy
    ret.phases = Object.assign({}, ret.phases);
    for (const phaseName of Object.keys(extend.phases)) {
        const phase = ret.phases[phaseName] = Object.assign({}, ret.phases[phaseName]);
        phase.commands = [...phase.commands || [], ...extend.phases[phaseName].commands];
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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