"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     *
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    /**
     * Add a permission only if there's a policy attached.
     *
     * @param statement The permissions statement to add.
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state changes.
     *
     * You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *     const rule = project.onStateChange('OnBuildStarted', { target });
     *     rule.addEventPattern({
     *       detail: {
     *         'build-status': [
     *           "IN_PROGRESS",
     *           "SUCCEEDED",
     *           "FAILED",
     *           "STOPPED"
     *         ]
     *       }
     *     });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @param metricName The name of the metric.
     * @param props Customization properties.
     * @returns a CloudWatch metric associated with this build project.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.metric('Builds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.metric('Duration', {
            statistic: 'avg',
            ...props,
        });
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.metric('SucceededBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the number of builds that failed because of client error or because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.metric('FailedBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props.environment, environmentVariables),
            fileSystemLocations: core_1.Lazy.anyValue({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.stringValue({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            secondarySources: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.anyValue({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                    'codebuild:BatchPutCodeCoverages',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        const bindFunction = this.buildImage.bind;
        if (bindFunction) {
            bindFunction.call(this.buildImage, this, this, {});
        }
    }
    /**
     *
     */
    static fromProjectArn(scope, id, projectArn) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.projectArn = projectArn;
                this.projectName = core_1.Stack.of(scope).parseArn(projectArn).resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK, or in a different CDK Stack (and exported using the {@link export} method).
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param projectName the name of the project to import.
     * @returns a reference to the existing Project
     * @note if you're importing a CodeBuild Project for use
     * in a CodePipeline, make sure the existing Project
     * has permissions to access the S3 Bucket of that Pipeline -
     * otherwise, builds in that Pipeline will always fail.
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable}, which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty}, which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables.
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     */
    static serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            name,
            type: environmentVariables[name].type || BuildEnvironmentVariableType.PLAINTEXT,
            value: environmentVariables[name].value,
        }));
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add.
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in.
     * @param options additional options for the binding.
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @override true
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(env = {}, projectVars = {}) {
        var _a, _b;
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? ImagePullPrincipalType.CODEBUILD
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_a = this.buildImage.secretsManagerCredentials) === null || _a === void 0 ? void 0 : _a.grantRead(this);
        }
        const secret = this.buildImage.secretsManagerCredentials;
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: secret
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    // Secrets must be referenced by either the full ARN (with SecretsManager suffix), or by name.
                    // "Partial" ARNs (without the suffix) will fail a validation regex at deploy-time.
                    credential: (_b = secret.secretFullArn) !== null && _b !== void 0 ? _b : secret.secretName,
                }
                : undefined,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars ? Project.serializeEnvVariables(vars) : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(props) {
        /**
         * The type of build environment.
         */
        this.type = 'LINUX_CONTAINER';
        /**
         * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         */
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Linux build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @param id The image identifier.
     * @returns A Docker image provided by CodeBuild.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @example
     *
     * 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     */
    validate(_) {
        return [];
    }
    /**
     * Make a buildspec to run the indicated script.
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
/**
 *
 */
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
/**
 *
 */
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
/**
 *
 */
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/**
 * The `aws/codebuild/standard:4.0` build image.
 */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
/**
 *
 */
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
/**
 *
 */
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/**
 * The Amazon Linux 2 x86_64 standard image, version `3.0`.
 */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
/**
 *
 */
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * Environment type for Windows Docker images.
 */
var WindowsImageType;
(function (WindowsImageType) {
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(props) {
        var _a;
        /**
         * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         */
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_a = props.imageType) !== null && _a !== void 0 ? _a : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    /**
     * Make a buildspec to run the indicated script.
     */
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
/**
 * (deprecated) Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is based off Windows Server Core 2016.
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is based off Windows Server Core 2019.
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
/**
 *
 */
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
//# sourceMappingURL=data:application/json;base64,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