"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["UV"] = "uv.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile || (exports.DependenciesFile = DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv requirements > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                ...props?.poetryWithoutUrls ? ['--without-urls'] : [],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * Packaging with `uv`.
     */
    static withUv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.UV,
            exportCommand: `uv export --frozen --no-emit-workspace --no-dev --no-editable -o ${DependenciesFile.PIP}`,
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes, poetryWithoutUrls) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes, poetryWithoutUrls });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.UV))) {
            return this.withUv();
        }
        else {
            return this.withNoPackaging();
        }
    }
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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