"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionsBoundary = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const iam_generated_1 = require("./iam.generated");
/**
 * Modify the Permissions Boundaries of Users and Roles in a construct tree.
 *
 * ```ts
 * const policy = iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess');
 * iam.PermissionsBoundary.of(this).apply(policy);
 * ```
 *
 * @stability stable
 */
class PermissionsBoundary {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Access the Permissions Boundaries of a construct tree.
     *
     * @stability stable
     */
    static of(scope) {
        return new PermissionsBoundary(scope);
    }
    /**
     * Apply the given policy as Permissions Boundary to all Roles and Users in the scope.
     *
     * Will override any Permissions Boundaries configured previously; in case
     * a Permission Boundary is applied in multiple scopes, the Boundary applied
     * closest to the Role wins.
     *
     * @stability stable
     */
    apply(boundaryPolicy) {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyOverride('PermissionsBoundary', boundaryPolicy.managedPolicyArn);
                }
            },
        });
    }
    /**
     * Remove previously applied Permissions Boundaries.
     *
     * @stability stable
     */
    clear() {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyDeletionOverride('PermissionsBoundary');
                }
            },
        });
    }
}
exports.PermissionsBoundary = PermissionsBoundary;
_a = JSII_RTTI_SYMBOL_1;
PermissionsBoundary[_a] = { fqn: "@aws-cdk/aws-iam.PermissionsBoundary", version: "1.130.0" };
//# sourceMappingURL=data:application/json;base64,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