"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as 
published by the Free Software Foundation, either version 3 of 
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This module is intended to distributed as part of automatically generated code by the
peakrdl-python tool. It provides a set of base classes used by the autogenerated code
"""
from dataclasses import dataclass
from typing import TYPE_CHECKING
from typing import Optional, Union
from enum import Enum, auto

from ..lib.utility_functions import swap_msb_lsb_ordering

from ._callbacks import FieldReadCallback, FieldWriteCallback

from .base import Base

if TYPE_CHECKING:
    from .register import BaseRegister

# pylint: disable=too-many-instance-attributes,too-many-arguments

#pylint:disable-next=too-few-public-methods
class _FieldBase(Base):
    """
    class for all fields

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__ = ['__high', '__low', '__msb', '__lsb',
                 '__bitmask', '__inverse_bitmask',
                 '__msb0', '__lsb0',
                 '__parent_register','__parent_width',
                 '_read_callback', '_write_callback']

    def __init__(self, *, low: int, high: int, msb: int, lsb: int,
                 parent_register: 'BaseRegister', parent_width: int, inst_name: str):

        super().__init__(full_inst_name=parent_register.full_inst_name + '.' + inst_name)

        self.__low = low
        self.__high = high

        # there are a couple of properties that have been included because they may be needed in
        # the future but are currently unused
        # pylint: disable=unused-private-member
        self.__lsb = lsb
        self.__msb = msb

        if (msb == high) and (lsb == low):
            self.__lsb0 = True
            self.__msb0 = False
        elif (msb == low) and (lsb == high):
            self.__lsb0 = False
            self.__msb0 = True
        else:
            raise ValueError('msb/lsb are inconsistent with low/high')
        # pylint: enable=unused-private-member

        self.__parent_register = parent_register
        self.__parent_width = parent_width

        self.__bitmask = 0
        for bit_position in range(low, high+1):
            self.__bitmask |= (1 << bit_position)

        parent_max_value = (2 ** parent_width) - 1
        self.__inverse_bitmask = parent_max_value ^ self.__bitmask

        self._write_callback: Optional[FieldWriteCallback] = None
        self._read_callback: Optional[FieldReadCallback] = None

    @property
    def __width(self) -> int:
        return self.__high - self.__low + 1

    @property
    def value(self) -> int:
        """
        Access the register value without triggering the callbacks
        """

        reg_value = self.__parent_register.value

        if self.__msb0:
            return swap_msb_lsb_ordering(value=(reg_value & self.__bitmask) >> self.__low,
                                         width=self.__width)

        return (reg_value & self.__bitmask) >> self.__low

    @value.setter
    def value(self, value: int) -> None:

        if self.__msb0:
            value = swap_msb_lsb_ordering(value=value, width=self.__width)

        if (self.__high == (self.__parent_width - 1)) and (self.__low == 0):
            # special case where the field occupies the whole register,
            # there a straight write can be performed
            self.__parent_register.value = value
        else:
            # do a read, modify write
            reg_value = self.__parent_register.value
            self.__parent_register.value = (reg_value & self.__inverse_bitmask) | \
                                           (value << self.__low)

class ReadOnlyField(_FieldBase):
    """
    Simulation of a read-only field
    """
    __slots__: list[str] = []

    @property
    def read_callback(self) -> Optional[FieldReadCallback]:
        """
        Callback made during each read operation
        """
        return self._read_callback

    @read_callback.setter
    def read_callback(self, callback: Optional[FieldReadCallback]) -> None:
        self._read_callback = callback


class WriteOnlyField(_FieldBase):
    """
    Simulation of a write-only field
    """
    __slots__:list[str] = []




    @property
    def write_callback(self) -> Optional[FieldWriteCallback]:
        """
        Callback made during each write operation
        """
        return self._write_callback

    @write_callback.setter
    def write_callback(self, callback: Optional[FieldWriteCallback]) -> None:
        self._write_callback = callback

class ReadWriteField(ReadOnlyField, WriteOnlyField):
    """
    Simulation of a read/write field
    """
    __slots__:list[str] = []


Field = Union[ReadOnlyField, WriteOnlyField, ReadWriteField]


class FieldType(Enum):
    """
    types of register field to be simulated based on the software access
    """
    READONLY = auto()
    WRITEONLY = auto()
    READWRITE = auto()


_class_type_map: dict[FieldType, Union[type[ReadOnlyField],
                                       type[WriteOnlyField],
                                       type[ReadWriteField]]] = {
    FieldType.READONLY: ReadOnlyField,
    FieldType.WRITEONLY: WriteOnlyField,
    FieldType.READWRITE: ReadWriteField}

@dataclass
class FieldDefinition:
    """
    Class for an entry in the list of memories in the simulator
    """
    high: int
    low: int
    msb: int
    lsb: int
    inst_name: str
    field_type: FieldType

    @property
    def class_type(self) -> Union[type[ReadOnlyField], type[WriteOnlyField], type[ReadWriteField]]:
        """
        Classtype to be used base on the fieldtype
        """
        return _class_type_map[self.field_type]
