"""
Structured Output

Tool-based structured output using Pydantic models.
Forces the model to call a 'respond' tool with validated schema.
"""

from __future__ import annotations

from typing import Any

from pydantic import BaseModel, ValidationError

from .types import Tool


class ResponseTool(Tool):
    """
    Special tool for structured output.

    The model is forced to call this tool (via tool_choice).
    The tool's input is validated against the Pydantic schema.
    """

    _output_type: type[BaseModel]

    def __init__(self, output_type: type[BaseModel]):
        # Get schema from Pydantic model
        schema = output_type.model_json_schema()

        # Inline $defs if present (simpler schema)
        if "$defs" in schema:
            schema = _inline_defs(schema)

        super().__init__(
            name="respond",
            description=f"Submit your final response as {output_type.__name__}",
            input_schema=schema,
        )

        object.__setattr__(self, "_output_type", output_type)

    async def execute(self, input: dict[str, Any]) -> tuple[str, bool]:
        """
        Validate and return the structured output.

        Returns:
            ("Success", False) if valid
            (error_message, True) if validation fails
        """
        try:
            self._output_type.model_validate(input)
            return "Success", False
        except ValidationError as e:
            error_msg = f"Validation error: {e}. Fix the errors and call respond again."
            return error_msg, True


def _inline_defs(schema: dict[str, Any]) -> dict[str, Any]:
    """Inline $defs references for simpler schema.

    This is a simplified version - just removes $defs for now.
    Full implementation would recursively resolve $ref pointers.
    """
    result = dict(schema)
    result.pop("$defs", None)
    return result


__all__ = [
    "ResponseTool",
]
