import io
import typing
from dataclasses import dataclass, field

from . import builder, util

INDENT = "  "

MODIFY_NOTICE = "// DO NOT MODIFY -- This file is generated by type_spec\n"


base_name_map = {
    builder.BaseTypeName.s_boolean: "boolean",
    builder.BaseTypeName.s_date: "string",  # IMPROVE: Aliased DateStr
    builder.BaseTypeName.s_date_time: "string",  # IMPROVE: Aliased DateTimeStr
    # Decimal's are marked as to_string_values thus are strings in the front-end
    builder.BaseTypeName.s_decimal: "string",
    builder.BaseTypeName.s_dict: "PartialRecord",
    builder.BaseTypeName.s_integer: "number",
    builder.BaseTypeName.s_lossy_decimal: "number",
    builder.BaseTypeName.s_opaque_key: "string",
    builder.BaseTypeName.s_none: "null",
    builder.BaseTypeName.s_string: "string",
    # UNC: global types
    builder.BaseTypeName.s_json_value: "JsonValue",
}


@dataclass(kw_only=True)
class EmitTypescriptContext:
    out: io.StringIO
    namespace: builder.SpecNamespace
    namespaces: set[builder.SpecNamespace] = field(default_factory=set)


def ts_type_name(name: str) -> str:
    return "".join([x.title() for x in name.split("_")])


def resolve_namespace_ref(namespace: builder.SpecNamespace) -> str:
    return f"{ts_type_name(namespace.name)}T"


def ts_name(name: str, name_case: builder.NameCase) -> str:
    if name_case == builder.NameCase.preserve:
        return name
    bits = util.split_any_name(name)
    return "".join([bits[0], *[x.title() for x in bits[1:]]])


def emit_value_ts(
    ctx: EmitTypescriptContext, stype: builder.SpecType, value: typing.Any
) -> str:
    """Mimics emit_python even if not all types are used in TypeScript yet"""
    literal = builder.unwrap_literal_type(stype)
    if literal is not None:
        return emit_value_ts(ctx, literal.value_type, literal.value)

    if stype.is_base_type(builder.BaseTypeName.s_string):
        assert isinstance(value, str)
        return util.encode_common_string(value)
    elif stype.is_base_type(builder.BaseTypeName.s_integer):
        assert isinstance(value, int)
        return str(value)
    elif stype.is_base_type(builder.BaseTypeName.s_boolean):
        assert isinstance(value, bool)
        return "true" if value else "false"
    elif stype.is_base_type(builder.BaseTypeName.s_lossy_decimal):
        return str(value)
    elif stype.is_base_type(builder.BaseTypeName.s_decimal):
        return f"'{value}'"
    elif isinstance(stype, builder.SpecTypeInstance):
        if stype.defn_type.is_base_type(builder.BaseTypeName.s_list):
            sub_type = stype.parameters[0]
            return (
                "[" + ", ".join([emit_value_ts(ctx, sub_type, x) for x in value]) + "]"
            )

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_dict):
            key_type = stype.parameters[0]
            value_type = stype.parameters[1]
            return (
                "{\n\t"
                + ",\n\t".join(
                    "["
                    + emit_value_ts(ctx, key_type, dkey)
                    + "]: "
                    + emit_value_ts(ctx, value_type, dvalue)
                    for dkey, dvalue in value.items()
                )
                + "\n}"
            )

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_optional):
            sub_type = stype.parameters[0]
            if value is None:
                return "null"
            return emit_value_ts(ctx, sub_type, value)

    elif isinstance(stype, builder.SpecTypeDefnStringEnum):
        return f"{refer_to(ctx, stype)}.{ts_enum_name(value, stype.name_case)}"

    raise Exception("invalid constant type", value, stype)


def emit_type_ts(ctx: EmitTypescriptContext, stype: builder.SpecType) -> None:
    if not isinstance(stype, builder.SpecTypeDefn):
        return

    if stype.is_base or stype.is_predefined:
        return

    ctx.out.write("\n")
    ctx.out.write(MODIFY_NOTICE)

    if isinstance(stype, builder.SpecTypeDefnExternal):
        assert not stype.is_exported, "expecting private names"
        ctx.out.write(stype.external_map["ts"])
        ctx.out.write("\n")
        return

    assert stype.is_exported, "expecting exported names"
    if isinstance(stype, builder.SpecTypeDefnAlias):
        ctx.out.write(f"export type {stype.name} = {refer_to(ctx, stype.alias)}\n")
        return

    if isinstance(stype, builder.SpecTypeDefnUnion):
        ctx.out.write(
            f"export type {stype.name} = {refer_to(ctx, stype.get_backing_type())}\n"
        )
        return

    if isinstance(stype, builder.SpecTypeDefnStringEnum):
        ctx.out.write(f"export enum {stype.name} {{\n")
        assert stype.values
        for name, entry in stype.values.items():
            ctx.out.write(
                f'{INDENT}{ts_enum_name(name, stype.name_case)} = "{entry.value}",\n'
            )
        ctx.out.write("}\n")
        return

    assert isinstance(stype, builder.SpecTypeDefnObject)
    assert stype.base is not None

    base_type = ""
    if not stype.base.is_base:
        base_type = f"{refer_to(ctx, stype.base)} & "

    if stype.properties is None and base_type == "":
        ctx.out.write(f"export type {stype.name} = TEmpty\n")
    elif stype.properties is None:
        ctx.out.write(f"export type {stype.name} = {base_type}{{}}\n")
    else:
        if isinstance(stype, builder.SpecTypeDefnObject) and len(stype.parameters) > 0:
            full_type_name = f'{stype.name}<{", ".join(stype.parameters)}>'
        else:
            full_type_name = stype.name
        ctx.out.write(f"export type {full_type_name} = {base_type}{{")
        ctx.out.write("\n")
        for prop in stype.properties.values():
            ref_type = refer_to(ctx, prop.spec_type)
            prop_name = ts_name(prop.name, prop.name_case)
            if prop.has_default and not prop.parse_require:
                # For now, we'll assume the generated types with defaults are meant as
                # arguments, thus treat like extant==missing
                # IMPROVE: if we can decide they are meant as output instead, then
                # they should be marked as required
                ctx.out.write(f"{INDENT}{prop_name}?: {ref_type}")
            elif prop.extant == builder.PropertyExtant.missing:
                # Unlike optional below, missing does not imply null is possible. They
                # treated distinctly.
                ctx.out.write(f"{INDENT}{prop_name}?: {ref_type}")
            elif prop.extant == builder.PropertyExtant.optional:
                # Need to add in |null since Python side can produce null's right now
                # IMPROVE: It would be better if the serializer could instead omit the None's
                # Dropping the null should be forward compatible
                ctx.out.write(f"{INDENT}{prop_name}?: {ref_type} | null")
            else:
                ctx.out.write(f"{INDENT}{prop_name}: {ref_type}")
            ctx.out.write("\n")
        ctx.out.write("}\n")


def refer_to(ctx: EmitTypescriptContext, stype: builder.SpecType) -> str:
    return refer_to_impl(ctx, stype)[0]


def refer_to_impl(
    ctx: EmitTypescriptContext, stype: builder.SpecType
) -> tuple[str, bool]:
    """
    @return (string-specific, multiple-types)
    """
    if isinstance(stype, builder.SpecTypeInstance):
        if stype.defn_type.name == builder.BaseTypeName.s_list:
            spec, multi = refer_to_impl(ctx, stype.parameters[0])
            return f"({spec})[]" if multi else f"{spec}[]", False
        if stype.defn_type.name == builder.BaseTypeName.s_readonly_array:
            spec, multi = refer_to_impl(ctx, stype.parameters[0])
            return f"readonly ({spec})[]" if multi else f"readonly {spec}[]", False
        if stype.defn_type.name == builder.BaseTypeName.s_union:
            return (
                f'({" | ".join([refer_to(ctx, p) for p in stype.parameters])})',
                False,
            )
        if stype.defn_type.name == builder.BaseTypeName.s_literal:
            parts = []
            for parameter in stype.parameters:
                assert isinstance(parameter, builder.SpecTypeLiteralWrapper)
                parts.append(refer_to(ctx, parameter))
            return f'({" | ".join(parts)})', False
        if stype.defn_type.name == builder.BaseTypeName.s_optional:
            return f"{refer_to(ctx, stype.parameters[0])} | null", True
        if stype.defn_type.name == builder.BaseTypeName.s_tuple:
            return f"[{", ".join([refer_to(ctx, p) for p in stype.parameters])}]", False
        params = ", ".join([refer_to(ctx, p) for p in stype.parameters])
        return f"{refer_to(ctx, stype.defn_type)}<{params}>", False

    if isinstance(stype, builder.SpecTypeLiteralWrapper):
        return emit_value_ts(ctx, stype.value_type, stype.value), False

    if isinstance(stype, builder.SpecTypeGenericParameter):
        return stype.name, False

    assert isinstance(stype, builder.SpecTypeDefn)
    if stype.is_base:  # assume correct namespace
        if stype.name == builder.BaseTypeName.s_list:
            return "any[]", False  # TODO: generic type
        return base_name_map[builder.BaseTypeName(stype.name)], False

    if stype.namespace == ctx.namespace:
        return stype.name, False

    ctx.namespaces.add(stype.namespace)
    return f"{resolve_namespace_ref(stype.namespace)}.{stype.name}", False


def ts_enum_name(name: str, name_case: builder.NameCase) -> str:
    if name_case == builder.NameCase.js_upper:
        return name.upper()
    return ts_name(name, name_case)


def resolve_namespace_name(namespace: builder.SpecNamespace) -> str:
    return namespace.name


def emit_namespace_imports_ts(
    namespaces: set[builder.SpecNamespace],
    out: io.StringIO,
    current_namespace: builder.SpecNamespace,
) -> None:
    for ns in sorted(
        namespaces,
        key=lambda name: resolve_namespace_name(name),
    ):
        import_as = resolve_namespace_ref(ns)
        import_path = (
            "./"
            if len(current_namespace.path) == 1
            else "../" * (len(current_namespace.path) - 1)
        )
        import_from = f"{import_path}{resolve_namespace_name(ns)}"
        out.write(f'import * as {import_as} from "{import_from}"\n')  # noqa: E501
